/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_lcdproc.c,v $ -- LCDproc display interface
 * $Id: mod_lcdproc.c,v 1.26 2004/10/10 09:46:37 boucman Exp $
 *
 * Copyright (C) by Andreas Neuhaus
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <netdb.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <locale.h>

#include "config.h"
#include "irmp3config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_lcdproc.h"

/*************************************************************************
 * GLOBALS
 */
unsigned int   mod_lcdproc_wid = 0;		// display width
int   mod_lcdproc_hgt = 0;		// display height
int   mod_lcdproc_cellwid = 0;		// cell width
int   mod_lcdproc_cellhgt = 0;		// cell height

int   mod_lcdproc_mode = 0;		// display mode
char *mod_lcdproc_playmode = "";	// playing mode
int   mod_lcdproc_time1 = 0;		// playing time
int   mod_lcdproc_time2 = 0;		// "       "
char  mod_lcdproc_artist[128] = "";	// artist
char  mod_lcdproc_title[128]  = "";	// title
int   mod_lcdproc_repeat      = 0;	// repeat mode

char  mod_lcdproc_browserpwd[128] = "";        // browser parent working directory
char  mod_lcdproc_browsercursor[128] = "";     // browser cursor selection
int   mod_lcdproc_browsercursorpos = 0;        // current cursorpos in %
int   mod_lcdproc_browsercursorposenabled = 0; // scrollbar emulation enabled?

int   mod_lcdproc_shuffle     = 0;	// shuffle mode
int   mod_lcdproc_mute        = 0;	// mute mode
int   mod_lcdproc_sleep       = 0;	// sleep mode
time_t mod_lcdproc_tmptimeout = 0;	// timout for tmp status

time_t mod_lcdproc_browsertimeout = 0;         // timer for file browser
int    mod_lcdproc_browsertimeoutmax = 10;     // timeout for file browser

int   lcdproc_fd = 0;
fd_set mod_lcdproc_fdset;
int   mod_lcdproc_clockmode  = 0;		// flag for clock/song display
int   mod_lcdproc_clocktimer = 0;		// how many seconds of idle before clock starts
int   mod_lcdproc_songtime_hours = 0;		// display hours separately
int   mod_lcdproc_status_alignment = 0;		// align status right
int   mod_lcdproc_preserve_upper_right = 0;     // preserve upper right corner of display for heartbeat

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_lcdproc = {
	"mod_lcdproc",
	mod_lcdproc_deinit,	// deinit
	mod_lcdproc_reload,	// reload
	&mod_lcdproc_fdset,	// fd's to watch
	NULL,			// poll
	mod_lcdproc_update,	// update
	mod_lcdproc_message,	// message
	NULL,			// SIGCHLD handler
	mod_lcdproc_init,
	NULL,			// avoid warning
};


/*************************************************************************
 * DISCONNECT FROM LCDPROC SERVER
 */
void mod_lcdproc_disconnect (void)
{
	if (lcdproc_fd)
	{
		shutdown(lcdproc_fd, 2);
		close(lcdproc_fd);
	}
	log_printf(LOG_NOISYDEBUG,"mod_lcdproc_disconnect(): disconnecting from lcdproc.\n");
	lcdproc_fd = 0;
	FD_ZERO(&mod_lcdproc_fdset);
	mod_lcdproc.poll = NULL;
}

/*************************************************************************
 * CONNECT TO LCDPROC SERVER
 */
char *mod_lcdproc_connect (void)
{
	struct sockaddr_in server;
	struct hostent *hostinfo;
	int rc;
	static int connect_tries=0;
	fd_set fds_write;
	struct timeval timeout = { 2, 0 }; /* connect timeout seconds */
	int socket_err;
	socklen_t serrlen = sizeof socket_err;

	if((connect_tries%10)>0)
	{
		connect_tries++;
		return "LCDproc: not trying to reconnect yet";
	}
	connect_tries++;

	// close if already open
	mod_lcdproc_disconnect();

	// resolve hostname
	server.sin_family = AF_INET;
	server.sin_port = htons(config_getnum("lcdproc_port", 13666));
	hostinfo = gethostbyname(config_getstr("lcdproc_host", "localhost"));
	if (!hostinfo)
		return "LCDproc: Unknown host";
	server.sin_addr = *(struct in_addr *) hostinfo->h_addr;

	// create socket
	lcdproc_fd = socket(PF_INET, SOCK_STREAM, 0);
	if (lcdproc_fd < 0) {
		lcdproc_fd=0;
		mod_lcdproc_disconnect();
		return "LCDproc: error creating socket";
	}
	FD_SET(lcdproc_fd,&mod_lcdproc_fdset);
	log_printf(LOG_DEBUG, "mod_lcdproc_connect(): created socket on fd %d\n", lcdproc_fd);

	// make connect not blocking to allow faster connect timeouts
	fcntl(lcdproc_fd, F_SETFL, O_NONBLOCK);

	// connect
	rc = connect(lcdproc_fd, (struct sockaddr *) &server, sizeof(server));
	if (rc < 0 && errno!=EINPROGRESS) {
		mod_lcdproc_disconnect();
		return "LCDproc: unable to connect";
	}

	// Wait for the connect() to finish, or timeout
	FD_ZERO(&fds_write);
	FD_SET(lcdproc_fd, &fds_write);
	if (select(lcdproc_fd + 1, NULL, &fds_write, NULL, &timeout) < 0) {
                mod_lcdproc_disconnect();
                return "LCDproc: unable to connect";
	}

	// Check if the connect() has finished, and if so, whether it did so successfully
	if(FD_ISSET(lcdproc_fd, &fds_write)) {
		if(getsockopt(lcdproc_fd, SOL_SOCKET, SO_ERROR, &socket_err, &serrlen) == -1) {
                	mod_lcdproc_disconnect();
	                return "LCDproc: unable to connect";
        	}

		if (socket_err != 0) {
                	mod_lcdproc_disconnect();
	                return "LCDproc: unable to connect";
        	}
	} else {
               	mod_lcdproc_disconnect();
                return "LCDproc: unable to connect";
	}

	mod_lcdproc.poll = mod_lcdproc_poll;
	connect_tries=0;

	return NULL;
}


/*************************************************************************
 * SETUP LCDPROC SCREENS
 */
void mod_lcdproc_setup (void)
{
	char *keylist, *tmpstr;
	char *key;

	if (!lcdproc_fd)
		return;
	log_printf(LOG_DEBUG, "mod_lcdproc_setup(): setting up screens\n");
	sendtext(lcdproc_fd, "hello\n");
	sendtext(lcdproc_fd, "screen_add irmp3\n");
	sendtext(lcdproc_fd, "widget_add irmp3 time string\n");
	sendtext(lcdproc_fd, "widget_add irmp3 status string\n");
	sendtext(lcdproc_fd, "widget_add irmp3 artist scroller\n");
	sendtext(lcdproc_fd, "widget_add irmp3 title scroller\n");

        sendtext(lcdproc_fd, "widget_add irmp3 browserpwd scroller\n");
        sendtext(lcdproc_fd, "widget_add irmp3 browsercursor scroller\n");
        sendtext(lcdproc_fd, "widget_add irmp3 browsercursorpos string\n");

	sendtext(lcdproc_fd, "widget_add irmp3 tmptext string\n");
	sendtext(lcdproc_fd, "widget_add irmp3 tmpbar hbar\n");
	sendtext(lcdproc_fd, "widget_set irmp3 tmpbar 0 0 0\n");
	sendtext(lcdproc_fd, "widget_del irmp3 heartbeat\n");

        /* lcdproc now requires the client to request which keypresses it wishes
           to receive. 
           This code takes an option from the config file which lists the keys
           you wish to receive from lcdproc and sends a command to lcdproc to receive them.
           2002-04-01 Jason Lewis jlewis@NOlordsSPAM.com       */

        /* get any keys to be received from LCDd */
	keylist = (char *) malloc(strlen(config_getstr("lcdproc_keylist", ""))+1);        
	if (keylist != NULL)
		strcpy(keylist, config_getstr("lcdproc_keylist", ""));
	/* get each key in the list, should be separated by "," no spaces */
	key = strtok(keylist,",");
	while (key != NULL) {
		tmpstr = (char *) malloc(strlen("client_add_key ") + strlen(key) + 1); //
		if (tmpstr != NULL)
			sprintf(tmpstr,"client_add_key %s\n",key);

		log_printf(LOG_NOISYDEBUG, "mod_lcdproc_setup(): adding client key %s\n", key);
		sendtext(lcdproc_fd, tmpstr); // request key from lcdproc

		free(tmpstr); // don't forget to free the malloc

		key = strtok(NULL,",");  // get the next key
	}
	       
	free(keylist);

}


/*************************************************************************
 * SET A DISPLAY WIDGET
 */
void mod_lcdproc_setstring_x (char *name, int x, int y, int w, char *text, int align)
{
	char buf[128];
	int wid=w;

	if (!lcdproc_fd || !name)
		return;

	if(x <= 0)
		x = 1;

	if(wid <= 0)
		wid = mod_lcdproc_wid;

	if (mod_lcdproc_preserve_upper_right) 	{
		if (y == 1 && align == 1)
			wid--;
	}

	if (!text || !*text)
		sendtext(lcdproc_fd, "widget_set irmp3 %s 0 0 \" \"\n", name);
	else {
		memset(buf, ' ', wid);
		if (align == 1)		// right align
			strcpy(buf+wid-strlen(text), text);
		else if (align == 2)	// center align
			strcpy(buf+((wid-strlen(text))/2), text);
		else			// left align
			strcpy(buf, text);
		sendtext(lcdproc_fd, "widget_set irmp3 %s %d %d \"%s\"\n", name, x, y, buf);
	}
}

void mod_lcdproc_setstring (char *name, int y, char *text, int align)
{
	mod_lcdproc_setstring_x(name, 0, y, 0, text, align);
}

void mod_lcdproc_setscroller (char *name, int y, char *text)
{
	if (!lcdproc_fd || !name)
		return;

	if (!text || !*text)
		sendtext(lcdproc_fd, "widget_set irmp3 %s 1 1 1 1 m 2 \" \"\n", name);
	else
		sendtext(lcdproc_fd, "widget_set irmp3 %s 1 %d %d %d m 2 \"%s   \"\n", name, y, mod_lcdproc_wid, y, text);
}



/*************************************************************************
 * SHOW TEMPORARY STATUS TEXT/BAR
 */
void mod_lcdproc_settmp (char *text, int percent, int centerbar)
{
	int x, wid;
	char buf[128];

	if (!lcdproc_fd)
		return;

	if (!text && !percent && !centerbar) {
		mod_lcdproc_setstring("tmptext", 0, NULL, 0);
		sendtext(lcdproc_fd, "widget_set irmp3 tmpbar 0 0 0\n");
		return;
	}

	mod_lcdproc_tmptimeout = time(NULL) + 2;
	if (text) {
		strcpy(buf, text);
		while (strlen(buf) < mod_lcdproc_wid)
			strcat(buf, " ");
		mod_lcdproc_setstring("tmptext", mod_lcdproc_hgt, buf, 0);
	} else
		mod_lcdproc_setstring("tmptext", 0, NULL, 0);
	if (percent == 0) {
		sendtext(lcdproc_fd, "widget_set irmp3 tmpbar 0 0 0\n");
		return;
	}
	x = text ? strlen(text)+2 : 1;
	if (centerbar) {
		x += (mod_lcdproc_wid-x) / 2;
		percent *= 2;
	}
	wid = (mod_lcdproc_wid - x + 1) * mod_lcdproc_cellwid;
	sendtext(lcdproc_fd, "widget_set irmp3 tmpbar %d %d %d\n", x, mod_lcdproc_hgt, wid*percent/100);
}


/*************************************************************************
 * REFRESH THE DISPLAY
 */
void mod_lcdproc_refreshtime (void)
{
	char buf[128];

	if (mod_lcdproc_songtime_hours == 0) {
		if (mod_lcdproc_mode == 1) {     // display remaining time
			if(mod_lcdproc_time2 == -1) {
				sprintf(buf,"..-..:..");
			} else {
				snprintf(buf, sizeof(buf)-1, "%-5s-%02d:%02d", mod_lcdproc_playmode, mod_lcdproc_time2%3600/60, mod_lcdproc_time2%3600%60);
			}
		}else {                            // display current time
			if(mod_lcdproc_time1 == -1) {
				sprintf(buf,".. ..:..");
			} else {
				snprintf(buf, sizeof(buf)-1, "%-5s %02d:%02d", mod_lcdproc_playmode, mod_lcdproc_time1%3600/60, mod_lcdproc_time1%3600%60);
			}
		}
	} else {
		if (mod_lcdproc_mode == 1) {	   // display remaining time
			if(mod_lcdproc_time2 == -1) {
				sprintf(buf,"..-..:..");
			} else {
				snprintf(buf, sizeof(buf)-1, "%-5s -%d:%02d:%02d", mod_lcdproc_playmode, mod_lcdproc_time2/3600, mod_lcdproc_time2%3600/60, mod_lcdproc_time2%3600%60);
			}
		} else {                          // display current time
			if(mod_lcdproc_time2 == -1) {
				sprintf(buf,".. ..:..");
			} else {
				snprintf(buf, sizeof(buf)-1, "%-5s  %d:%02d:%02d", mod_lcdproc_playmode, mod_lcdproc_time1/3600, mod_lcdproc_time1%3600/60, mod_lcdproc_time1%3600%60);
			}
		}
	}
	if(mod_lcdproc_wid > 20)
		mod_lcdproc_setstring_x("time", 1, 1, mod_lcdproc_wid/2, buf, 2);
	else
		mod_lcdproc_setstring("time", 1, buf, 1);
}

void mod_lcdproc_refresh (void)
{
	char buf[128];
	int status_alignment;

	log_printf(LOG_DEBUG, "mod_lcdproc_refresh(): refreshing display\n");

	// refresh artist/title
	if (mod_lcdproc_hgt < 4) {
		// don't  display spacing " - " if at least on of artist/title is empty
		// nice on startup and for startup welcome message
		if (mod_lcdproc_artist[0] && mod_lcdproc_title[0])
		    snprintf(buf, sizeof(buf)-1, "%s - %s", mod_lcdproc_artist, mod_lcdproc_title);
		else
		    snprintf(buf, sizeof(buf)-1, "%s%s", mod_lcdproc_artist, mod_lcdproc_title);
		buf[sizeof(buf)-1] = 0;
		mod_lcdproc_setscroller("artist", 0, NULL);
		mod_lcdproc_setscroller("title", 2, buf);
	} else {
		mod_lcdproc_setscroller("artist", 2, mod_lcdproc_artist);
		mod_lcdproc_setscroller("title", 3, mod_lcdproc_title);
	}

	// automatically align status left for smaller displays
	if (mod_lcdproc_wid <= 20) {
	    status_alignment = 0;
	} else {
	    status_alignment = mod_lcdproc_status_alignment;
	}

	// refresh status
	if (mod_lcdproc_hgt > 2 || mod_lcdproc_wid > 20) {
		snprintf(buf, sizeof(buf)-1, "%s %s %s %s",
			mod_lcdproc_repeat==0 ? "    " : (mod_lcdproc_repeat==1 ? "Rpt1" : "Rpt+"),
			mod_lcdproc_mute ? "Mute" : "    ",
			mod_lcdproc_shuffle ? "Shfl" : "    ",
			mod_lcdproc_sleep ? "Slep" : "    ");
		buf[sizeof(buf)-1] = 0;
		if(mod_lcdproc_wid > 20)
			mod_lcdproc_setstring_x("status", 21, 1, mod_lcdproc_wid/2, buf, 2); 
		else
			mod_lcdproc_setstring("status", mod_lcdproc_hgt, buf, status_alignment); 
	} else {
		snprintf(buf, sizeof(buf)-1, "%c%c%c%c",
			mod_lcdproc_repeat==0 ? ' ' : (mod_lcdproc_repeat==1 ? 'r' : 'R'),
			mod_lcdproc_mute ?      'm' : ' ',
			mod_lcdproc_shuffle ?   '?' : ' ',
			mod_lcdproc_sleep ?     's' : ' ');
		buf[sizeof(buf)-1] = 0;
		mod_lcdproc_setstring("status", 1, buf, status_alignment);
	}
}

/*****************************************************************************
* SHOW TEMPORARY FILE BROWSER INFORMATION
*/
void mod_lcdproc_browser(int display)
{
       int y1, remain;
       char buf[1024];

       if (!lcdproc_fd)
           return;

       y1     = mod_lcdproc_hgt < 4 ? 1 : 2;
       remain = mod_lcdproc_browsercursorposenabled ? 3 : 0;

       if (display) {          // show browser output
           mod_lcdproc_browsertimeout = time(NULL) + mod_lcdproc_browsertimeoutmax;

           if (mod_lcdproc_browserpwd) {
                   strcpy(buf, mod_lcdproc_browserpwd);
                   while (strlen(buf) < mod_lcdproc_wid-remain)
                       strcat(buf, " ");
		   if (strlen(buf) > mod_lcdproc_wid-remain) {
			   sendtext(lcdproc_fd, "widget_set irmp3 browserpwd 1 %d %d %d m 2 \"%s   \"\n", y1, mod_lcdproc_wid-remain, y1, buf);
		   } else {
			   sendtext(lcdproc_fd, "widget_set irmp3 browserpwd 1 %d %d %d m 2 \"%s\"\n", y1, mod_lcdproc_wid-remain, y1, buf);
		   }

           } else
                   mod_lcdproc_setstring("browserpwd", 0, NULL, 0);


           if (mod_lcdproc_browsercursorposenabled)
                   sendtext(lcdproc_fd, "widget_set irmp3 browsercursorpos %d %d \" %02d\"\n", mod_lcdproc_wid-2, y1, mod_lcdproc_browsercursorpos);

           if (mod_lcdproc_browsercursor) {
                   strcpy(buf, mod_lcdproc_browsercursor);
                   while (strlen(buf) < mod_lcdproc_wid)
                           strcat(buf, " ");
		   if (strlen(buf) > mod_lcdproc_wid-remain) { 
			   sendtext(lcdproc_fd, "widget_set irmp3 browsercursor 1 %d %d %d m 2 \"%s   \"\n", y1+1, mod_lcdproc_wid, y1+1, buf);
		   } else {
			   sendtext(lcdproc_fd, "widget_set irmp3 browsercursor 1 %d %d %d m 2 \"%s\"\n", y1+1, mod_lcdproc_wid, y1+1, buf);
		   }

           } else
                   mod_lcdproc_setstring("browsercursor", 0, NULL, 0);

       } else {        // clear display
               sendtext(lcdproc_fd, "widget_set irmp3 browserpwd 0 0 0 0 h 0 \" \"\n");
               sendtext(lcdproc_fd, "widget_set irmp3 browsercursor 0 0 0 0 h 0 \" \"\n");
               sendtext(lcdproc_fd, "widget_set irmp3 browsercursorpos 0 0 \" \"\n");
               mod_lcdproc_refreshtime();
               mod_lcdproc_refresh();
       }
}


/*************************************************************************
 * POLL INPUT DATA
 */
void mod_lcdproc_poll (int fd)
{
	int rc;
	char s[512], buf[128];
	char *c0, *c1, *c2, *c3;

	// read LCDproc response
	rc = readline(fd, s, sizeof(s));
	if (rc < 0) {
		// close LCDproc connection
		mod_lcdproc_disconnect();
		log_printf(LOG_ERROR, "Lost connection to LCDproc server: %s\n", strerror(errno));
		return;
	}
	if (!*s)
		return;

	// process LCDproc response
	log_printf(LOG_NOISYDEBUG, "mod_lcdproc_poll(): LCDproc response: '%s'\n", s);
	c0 = s ? strdup(s) : NULL;
	c1 = strtok(c0, " \t");
	if(!c1) return;
	if ( !strcasecmp(c1, "connect")) {
		c2 = strtok(NULL, " \t");
		do {
			c3 =  strtok(NULL, " \t");
			if (c3 && !strcasecmp(c3, "wid")) {
				c3 = strtok(NULL, " \t");
				mod_lcdproc_wid = c3 ? atoi(c3) : 0;
			} else if (c3 && !strcasecmp(c3, "hgt")) {
				c3 = strtok(NULL, " \t");
				mod_lcdproc_hgt = c3 ? atoi(c3) : 0;
			} else if (c3 && !strcasecmp(c3, "cellwid")) {
				c3 = strtok(NULL, " \t");
				mod_lcdproc_cellwid = c3 ? atoi(c3) : 0;
			} else if (c3 && !strcasecmp(c3, "cellhgt")) {
				c3 = strtok(NULL, " \t");
				mod_lcdproc_cellhgt = c3 ? atoi(c3) : 0;
			}
		} while (c3);
		log_printf(LOG_DEBUG, "mod_lcdproc_poll(): display size %dx%d, cell size %dx%d\n", mod_lcdproc_wid, mod_lcdproc_hgt, mod_lcdproc_cellwid, mod_lcdproc_cellhgt);

	} else if ( !strcasecmp(c1, "listen")) {
		// ignore 'listen' response

	} else if ( !strcasecmp(c1, "ignore")) {
		// ignore 'ignore' response

	} else if ( !strcasecmp(c1, "key") ) {
		c2 = strtok(NULL, " \t");
		if(!c2) return;
		log_printf(LOG_DEBUG, "mod_lcdproc_poll(): got key '%s'\n", c2);
		snprintf(buf, sizeof(buf)-1, "lcdproc_key_%s", c2);
		c3 = config_getstr(buf, NULL);
		if (c3)
			mod_sendmsg(MSGTYPE_INPUT, c3);

	} else if ( !strcasecmp(c1, "bye")) {
		log_printf(LOG_ERROR, "LCDproc has terminated connection\n");
		mod_lcdproc_disconnect();

        } else if (!strcasecmp(c1, "success")) {
	        log_printf(LOG_NOISYDEBUG, "mod_lcdproc_poll(): LCDd returned success message\n");

	} else {
		log_printf(LOG_DEBUG, "mod_lcdproc_poll(): unknown response: '%s'\n", s);
	}
	free(c0);
}


/*************************************************************************
 * UPDATE
 */
void mod_lcdproc_update (void)
{
	time_t now;

	now = time(NULL);
	if (mod_lcdproc_tmptimeout && now >= mod_lcdproc_tmptimeout) {
		mod_lcdproc_tmptimeout = 0;
		mod_lcdproc_settmp(NULL, 0, 0);
	}

        if (mod_lcdproc_browsertimeout && now >= mod_lcdproc_browsertimeout) {
                mod_lcdproc_browsertimeout = 0;
                mod_lcdproc_browser(0);         // clear lcd from browser output
        }

}


/*************************************************************************
 * GO INTO CLOCK MODE
 */
void mod_lcdproc_clockstart()
{

    sendtext(lcdproc_fd, "screen_del irmp3\n");
    sendtext(lcdproc_fd, "screen_add lcdtime\n");
    sendtext(lcdproc_fd, "screen_set lcdtime -heartbeat off\n");
    sendtext(lcdproc_fd, "widget_add lcdtime time string\n");
    sendtext(lcdproc_fd, "widget_add lcdtime date string\n");
	mod_lcdproc_clockmode=1;
	mod_lcdproc_timedisplay();
}

/*************************************************************************
 * DISPLAY THE TIME
 */
void mod_lcdproc_timedisplay(void)
{
	time_t			t;
	struct tm *l_time;
	char			timestr[128];
	char			datestr[128];
	char			x=0,y=0,mx=0,my=0,dx=0;

	bzero(&timestr,sizeof(timestr));
	t=time(NULL);
								// this code makes the time march
								// around the display (to avoid burn-in)
	mx=mod_lcdproc_wid-10;		
	my=mod_lcdproc_hgt-1;
	if(mx<1) mx=1;				// maximum X starting point
	if(my<1) my=1;				// maximum Y starting point
	x=((t/60)%mx)+1;			// actual X starting point (based on time)
	y=((t/60/mx)%my)+1;			// actual Y starting point 	"	"	"
        
	l_time = localtime(&t);			
	setlocale(LC_TIME,"");		// Get the current locale from environment

	strftime(timestr,sizeof(timestr),config_getstr("lcdproc_timestr","%R"),l_time); 
	strftime(datestr,sizeof(datestr),config_getstr("lcdproc_datestr","%x"),l_time);

	dx = (strlen(datestr) - strlen(timestr)) / 2;

	sendtext(lcdproc_fd, "widget_set lcdtime date %d %d \"%s\"\n",x,y,datestr);
	sendtext(lcdproc_fd, "widget_set lcdtime time %d %d \"%s\"\n",x+dx,y+1,timestr);
}


	
	
/*************************************************************************
 * GET OUT OF CLOCK MODE
 */
void mod_lcdproc_clockstop()
{
    sendtext(lcdproc_fd, "screen_del lcdtime\n");
	mod_lcdproc_setup();
	mod_lcdproc_clockmode=0;
	mod_lcdproc_refresh();
	mod_lcdproc_refreshtime();
}
	


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_lcdproc_message (int msgtype, char *msg,const char __attribute__((unused))*sender)
{
	char *c1 = NULL, *c2 = NULL, *c3 = NULL;
	char buf[128];
	int i, j;
	mod_message_t *answer = NULL;
	mod_message_t *tmp_msg = NULL;

	if(!lcdproc_fd)
		mod_lcdproc_reload();		// try to reconnect


	// pre-process input mesage for handling IDLE messages.
	if (msgtype == MSGTYPE_EVENT && !strncasecmp(msg,"idle",4)) {
		if(mod_lcdproc_clockmode) { 	// clock is already started
			mod_lcdproc_timedisplay();
		} else if (mod_lcdproc_clocktimer && mod_lcdproc_clocktimer <= strtol(msg + 5,NULL,0) ) {
			mod_lcdproc_clockstart();
		}
	} else if (mod_lcdproc_clockmode)  {
		mod_lcdproc_clockstop();
	}

	// process input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = strtok(msg, " \t") ;
		if(!c1) return ;

		// displaymode
		if (!strcasecmp(c1, "displaymode")) {
			mod_lcdproc_mode++;
			if (mod_lcdproc_mode > 1)
				mod_lcdproc_mode = 0;
			mod_lcdproc_refreshtime();
		}

	// process player messages
	} else if (msgtype == MSGTYPE_PLAYER) {
		c1 = strtok(msg, " \t");
		if(!c1)return;
		c2 = strtok(NULL, " \t");
		c3 = strtok(NULL,"");
			// time info
		if (!strcasecmp(c1, "time")) {
			mod_lcdproc_time1 =  c2?atoi(c2) : 0;
			mod_lcdproc_time2 =  c3?atoi(c3) : 0;
			mod_lcdproc_refreshtime();

		}

		// process mixer messages
	} else if (msgtype == MSGTYPE_EVENT ) {
		c1 = strtok(msg, " \t");
		if(!c1) return;
	
		if (!strcasecmp(c1, "repeat")) {
			c2 = strtok(NULL, "");
			mod_lcdproc_repeat = c2 ? atoi(c2) : 0;
			mod_lcdproc_refresh();

		// shuffle mode
		} else if ( !strcasecmp(c1, "shuffle")) {
			c2 = strtok(NULL, "") ;
			mod_lcdproc_shuffle = c2 ? atoi(c2) : 0;
			mod_lcdproc_refresh();

		// sleep timer
		} else if ( !strcasecmp(c1, "sleep")) {
			c2 = strtok(NULL, " \t") ;
			c3 = strtok(NULL, "") ;
			i = c2 ? atoi(c2) : 0;
			j = c3 ? atoi(c3) : 0;
			if(!strcasecmp(c2,"trigered")) {
				mod_lcdproc_settmp("Sleep timer expired", 0, 0);
				mod_lcdproc_sleep = 0;
			} else if(!strcasecmp(c2,"decreased")) {
				mod_lcdproc_settmp("Fading out", 0, 0);
			}else if (!i) {
				mod_lcdproc_settmp("Sleep timer off", 0, 0);
				mod_lcdproc_sleep = 0;
			} else {
				if (j)
					sprintf(buf, "Sleep %d/%d min", i, j);
				else
					sprintf(buf, "Sleep %d min", i);
				mod_lcdproc_settmp(buf, 0, 0);
				mod_lcdproc_sleep = 1;
			}
			mod_lcdproc_refresh();
		} else if ( !strcasecmp(c1,"alarm")) {
			c2 = strtok(NULL, "");
			i = c2 ? atoi(c2) : 0;
			if (i) 
				mod_lcdproc_settmp("Alarms enabled",0,0);
			else
				mod_lcdproc_settmp("Alarms disabled",0,0);
		} else if ( !strcasecmp(c1,"loadplaylist")) {

			// display a playlist name here...
			c2 = strtok(NULL, "");
			strcpy (buf,c2?c2:"");
			mod_lcdproc_settmp(buf,0,0);
		} else if(!strcasecmp(c1,"mixer")) { 
			c2 = strtok(NULL, " \t");
			if(!c2) return;
			if ( !strcasecmp(c2, "volume")) {
				c3 = strtok(NULL,"");
				mod_lcdproc_settmp("Vol",  c3 ? atoi(c3) : 0, 0);
			} else if (!strcasecmp(c2, "balance")) {
				c3 = strtok(NULL,"");
				mod_lcdproc_settmp("Balance", c3 ?atoi(c3) : 0, 1);
			} else if (!strcasecmp(c2, "bass")) {
				c3 = strtok(NULL,"");
				mod_lcdproc_settmp("Bass", c3 ? atoi(c3) : 0, 0);
			} else if (!strcasecmp(c2, "treble")) {
				c3 = strtok(NULL,"");
				mod_lcdproc_settmp("Treble", c3 ? atoi(c3) : 0, 0);
			} else if (!strcasecmp(c2, "mute")) {
				c3 = strtok(NULL,"");
				mod_lcdproc_mute = c3 ? atoi(c3) : 0;
				mod_lcdproc_refresh();
			}
		} else if (!strcasecmp(c1, "play")) {
			mod_lcdproc_playmode = "PLAY";
			answer = mod_query(MSGTYPE_QUERY,"title");
			while( answer && (answer->msgtype != MSGTYPE_INFO || strncasecmp(answer->msg,"title",5))) {
				tmp_msg = answer;
				answer =  answer->next;
				free(tmp_msg->msg);
				free(tmp_msg);
			}
			if(!answer) {
				answer = mod_query(MSGTYPE_QUERY,"titleguess");
			}
			while( answer && (answer->msgtype != MSGTYPE_INFO || strncasecmp(answer->msg,"title",5))) {
				tmp_msg = answer;
				answer =  answer->next;
				free(tmp_msg->msg);
				free(tmp_msg);
			}
			strcpy(mod_lcdproc_title,  answer ? answer->msg +5: "");
			free_message(answer);
			answer = mod_query(MSGTYPE_QUERY,"artist");
			while( answer && (answer->msgtype != MSGTYPE_INFO || strncasecmp(answer->msg,"artist",6))) {
				tmp_msg = answer;
				answer =  answer->next;
				free(tmp_msg->msg);
				free(tmp_msg);
			}
			if(!answer) {
				answer = mod_query(MSGTYPE_QUERY,"artistguess");
			}
			while( answer && (answer->msgtype != MSGTYPE_INFO || strncasecmp(answer->msg,"artist",6))) {
				tmp_msg = answer;
				answer =  answer->next;
				free(tmp_msg->msg);
				free(tmp_msg);
			}
			strcpy(mod_lcdproc_artist,  answer ? answer->msg +7: "");
			free_message(answer);
			mod_lcdproc_refreshtime();
			mod_lcdproc_refresh();
		} else if (!strcasecmp(c1, "update")) {
			c2 = strtok(NULL," \t");
			if(!c2) return ;
			if (!strcasecmp(c2, "title")) {
				c3 = strtok(NULL,"");
				if(!c3) return ;
				strcpy(mod_lcdproc_title,c3);
				mod_lcdproc_refresh();
			} else if (!strcasecmp(c2, "artist")) {
				c3 = strtok(NULL,"");
				if(!c3) return ;
				strcpy(mod_lcdproc_artist,c3);
				mod_lcdproc_refresh();
			}	
		} else if (!strcasecmp(c1, "stop")) {
			mod_lcdproc_playmode = "STOP";
			mod_lcdproc_time1 = 0;
			mod_lcdproc_time2 = 0;
			mod_lcdproc_refreshtime();
			mod_lcdproc_refresh();

			// pause
		} else if ( !strcasecmp(c1, "unpause")) {
				mod_lcdproc_playmode = "PLAY";
				mod_lcdproc_refreshtime();
		} else if ( !strcasecmp(c1, "pause")) {
				mod_lcdproc_playmode = "PAUSE";
				mod_lcdproc_refreshtime();
		} else if (!strcasecmp(c1, "halt")) {
			mod_lcdproc_playmode = "HALT";
			mod_lcdproc_time1 = 0;
			mod_lcdproc_time2 = 0;
			mod_lcdproc_refreshtime();
			strcpy(mod_lcdproc_artist, "");
			strcpy(mod_lcdproc_title, "");
			mod_lcdproc_refresh();
		}

	} else if (msgtype == MSGTYPE_INFO) {
		c1 = strtok(msg, " \t");
		if(!c1) return;
		// song info
		if (!strcasecmp(c1, "title")) {
			c2 =  strtok(NULL, "");
			if(!c2) return;
			strcpy(mod_lcdproc_title, c2 ? c2 : "");
			mod_lcdproc_refresh();
		} else if (!strcasecmp(c1, "artist")) {
			c2 =  strtok(NULL, "");
			if(!c2) return;
			strcpy(mod_lcdproc_artist, c2 ? c2 : "");
			mod_lcdproc_refresh();
                // file browser info
		} else if ( !strcasecmp(c1, "browser")) {
			c2 =  strtok(NULL, " \t");
			if(!c2) return;
                        if (!strcasecmp(c2, "pwd")) {
				c3 =  strtok(NULL, "");
				if(!c3) return;
                                strcpy(mod_lcdproc_browserpwd, c3 ? c3 : "");
                                mod_lcdproc_browser(1);
                        } else if (!strcasecmp(c2, "cursor")) {
				c3 =  strtok(NULL, "");
				if(!c3) return;
                                strcpy(mod_lcdproc_browsercursor, c3 ? c3 : "");
                                mod_lcdproc_browser(1);
                        } else if (!strcasecmp(c2, "cursorpos")) {
				c3 =  strtok(NULL, "");
				if(!c3) return;
                                mod_lcdproc_browsercursorpos = c3 ? atoi(c3) : 0;
                                mod_lcdproc_browser(1);
                        } else if (!strcasecmp(c2, "clrscr")) {
                                mod_lcdproc_browser(0);
                        } else if ( !strcasecmp(c2, "info")) {     // short browser info
				c3 =  strtok(NULL, "");
                                strcpy(buf, c3 ? c3 : "");
                                mod_lcdproc_settmp(buf, 0, 0);
                        }
		}

	// process generic messages
	} else if (msgtype == MSGTYPE_EVENT) {
		c1 = strtok(msg, " \t");
		if(!c1) return;

		// repeat mode
	}
}



/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_lcdproc_init (void)
{
	char *s;

	FD_ZERO(&mod_lcdproc_fdset);


	// read some configuration values
	mod_lcdproc_songtime_hours       = (strcasecmp("yes",   config_getstr("lcdproc_songtime_hours", "no")) == 0);
	mod_lcdproc_status_alignment     = (strcasecmp("right", config_getstr("lcdproc_status_alignment", "right")) == 0);
	mod_lcdproc_preserve_upper_right = (strcasecmp("yes",   config_getstr("lcdproc_preserve_upper_right", "no")) == 0);

	log_printf(LOG_DEBUG, "mod_lcdproc_init(): socket connected\n");

        // show browser scrollbar emulation?
        mod_lcdproc_browsercursorposenabled = !strcasecmp(config_getstr("lcdproc_browser_scrollbar", "yes"), "yes") ? 1 : 0;
  
        // browser display timeout
        mod_lcdproc_browsertimeoutmax = config_getnum("lcdproc_browser_timeout", 10);
        if (mod_lcdproc_browsertimeoutmax < 2)          // minimum check
                mod_lcdproc_browsertimeoutmax = 2;

	// get clock timer
	mod_lcdproc_clocktimer=60*config_getnum("lcdproc_clock_timer",1);

	// connect to LCDproc server
	s = mod_lcdproc_connect();
	if (s)
		return s;
	// setup LCDproc
	mod_lcdproc_setup();


	//clear display
	mod_sendmsg(MSGTYPE_PLAYER, "stop");

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_lcdproc_deinit (void)
{
	// close LCDproc connection
	mod_lcdproc_disconnect();

	log_printf(LOG_DEBUG, "mod_lcdproc_deinit(): socket closed\n");
}


/*************************************************************************
 * MODULE RELOAD FUNCTION
 */
char *mod_lcdproc_reload (void)
{
	char *s;

	log_printf(LOG_DEBUG, "mod_lcdproc_reload(): reconnecting\n");

	// close and reconnect to LCDproc server
	s = mod_lcdproc_connect();
	if (s)
		return s;

	// setup LCDproc
	mod_lcdproc_setup();

	// refresh display
	mod_lcdproc_refreshtime();
	mod_lcdproc_refresh();

	return NULL;
}


/*************************************************************************
 * EOF
 */
