# arch-tag: b1ae35f1-d91b-49f0-a1ff-e163ddd8cefd
# Copyright (C) 2004 David Allouche <david@allouche.net>
#               2005 Canonical Limited.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""
Internal module providing name escaping functionality.

This module implements some of public interface for the
pybaz_ package. But for convenience reasons the author prefers
to store this code in a file separate from ``__init__.py``.

.. _pybaz: pybaz-module.html

This module is strictly internal and should never be used.
"""

__all__ = ['name_escape', 'name_unescape']

import errors

### Escaping Tables ###

__space_escape_table = { '\t': 'tab', '\n': 'nl', '\v': 'U+B',
                    '\f': 'np', '\r': 'cr', ' ': 'sp' }

def __make_name_escape_table():
    table = {}
    for N in range(256):
        C = chr(N)
        if C in __space_escape_table: val = '\\(%s)' % __space_escape_table[C]
        elif N < 32 or N > 127: val = '\\(U+%X)' % N
        elif C in '\\': val = '\\'+C
        else: val = C
        table[C] = val
    return table

__name_escape_table = __make_name_escape_table()

__space_unescape_table = {}
for k, v in __space_escape_table.items(): __space_unescape_table[v] = k


### Escaping Functions ###

def name_escape(name):
    """Escape a file name using the Arch syntax.

    :arg name: unescaped file name.
    :type name: str
    :return: escaped file name.
    :rtype: str
    """
    return ''.join([__name_escape_table[C] for C in name])


def name_unescape(name):
    """Unescape a file name using the Arch syntax.

    :arg name: escaped file name.
    :type name: str
    :return: unescaped file name.
    :rtype: str
    :raise errors.IllegalEscapeSequence: the syntax of ``name`` is incorrect.
    """
    result = []
    after_backslash = 'after_backslash'
    in_brackets = 'in_brackets'
    state = None
    escape = None # buffer for escape sequence
    for C in name:
        if state is None:
            if C != '\\':
                result.append(C)
                continue
            else:
                state = after_backslash
                continue
        elif state is after_backslash:
            if C == '(':
                escape = []
                state = in_brackets
                continue
            elif C in '\\"':
                state = None
                result.append(C)
                continue
            else:
                raise errors.IllegalEscapeSequence(name)
        elif state is in_brackets:
            if C != ')':
                escape.append(C)
                continue
            else:
                state = None
                if len(escape) < 2:
                    raise errors.IllegalEscapeSequence(name)
                escape_str = ''.join(escape)
                escape = None
                if escape_str[0:2] in ('U+', 'u+'):
                    try:
                        code = int(escape_str[2:], 16)
                    except ValueError:
                        raise errors.IllegalEscapeSequence(name)
                    if code > 255:
                        raise errors.IllegalEscapeSequence(name)
                    result.append(chr(code))
                    continue
                else:
                    escape_str = escape_str.lower()
                    try:
                        result.append(__space_unescape_table[escape_str])
                    except KeyError:
                        raise errors.IllegalEscapeSequence(name)
                    continue
        else:
            raise AssertionError, "The programmer is on crack!"
    if state is not None:
        raise errors.IllegalEscapeSequence(name)
    else:
        return ''.join(result)
