# arch-tag: 14b6cb0a-947a-4065-9b57-9f55a2ab8905
# Copyright (C) 2004 David Allouche <david@allouche.net>
#               2005 Canonical Limited.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for non-archived working trees.
"""

import os

import pybaz as arch
from pybaz import errors
from pybaz.pathname import DirName, FileName

import framework
from framework import TestCase
import fixtures


def sorted(seq):
    L = list(seq)
    L.sort()
    return L


class InitTree(TestCase):

    tests = []

    def init_tree(self):
        """init_tree works (w/o version) for unnested and nested trees."""
        os.mkdir(self.params.working_dir)
        arch.init_tree(self.params.working_dir)
        os.mkdir(self.params.nested_dir)
        arch.init_tree(self.params.nested_dir, nested=True)
    tests.append('init_tree')

    def init_tree_not_nested(self):
        """init_tree (w/o version) fails for nested if "nested" is not set."""
        os.mkdir(self.params.working_dir)
        arch.init_tree(self.params.working_dir)
        os.mkdir(self.params.nested_dir)
        self.failUnlessRaises(errors.ExecProblem,
                              lambda: arch.init_tree(self.params.nested_dir))
    tests.append('init_tree_not_nested')

    def init_tree_with_version(self):
        """init_tree with version set tree-version and create log-version."""
        os.mkdir(self.params.working_dir)
        wt = arch.init_tree(self.params.working_dir, self.params.version)
        self.failUnlessEqual(self.params.version, wt.tree_version)
        self.failUnlessEqual(1, len(list(wt.iter_log_versions())))
        self.failUnlessEqual(self.params.version, wt.iter_log_versions().next())
    tests.append('init_tree_with_version')

    def tree_root(self):
        """tree_root works"""
        os.mkdir(self.params.working_dir)
        tree = arch.init_tree(self.params.working_dir)
        subdir = tree/'subdir'
        os.mkdir(subdir)
        subdir_tree = arch.tree_root(subdir)
        self.assertEqual(tree, subdir_tree)
    tests.append('tree_root')


class WorkingTreeConstructor(framework.NewTestCase):
    """Test cases for the creation of a WorkingTree object."""

    fixture = fixtures.InventoryTreeFixture()

    def test_WorkingTree(self):
        """WorkingTree works on an existing working tree."""
        tree_path = str(self.fixture.tree)
        tree = arch.WorkingTree(tree_path)
        self.assertEqual(type(tree), arch.WorkingTree)
        self.assertEqual(str(tree), tree_path)

    def test_WorkingTreeError(self):
        """WorkingTree raises SourceTreeError when argument is not a root."""
        subdir_path = str(self.fixture.tree / 'subdir')
        #import pdb ; pdb.set_trace()
        os.mkdir(subdir_path)
        self.assertRaises(errors.SourceTreeError,
                          arch.WorkingTree, subdir_path)
        sibling_path = str(self.fixture.tree) + '-sibling'
        os.mkdir(sibling_path)
        self.assertRaises(errors.SourceTreeError,
                          arch.WorkingTree, sibling_path)
        not_here_path = str(self.fixture.tree) + '-not-here'
        self.assertRaises(errors.SourceTreeError,
                          arch.WorkingTree, not_here_path)


class InventoryTree(framework.NewTestCase):
    """Test cases that work on a tree but need no associated archive"""

    fixture = fixtures.InventoryTreeFixture()

    def __failInventory(self, msg):
        self.fail("%s\n" % (msg,)
          + ("inv1: %s\n" % (self.__inv1,))
          + ("inv2: %s\n" % (self.__inv2,)))

    def assertInventoryEqual(self, inv1, inv2):
        inv1, inv2 = list(inv1), list(inv2)
        self.__inv1, self.__inv2 = inv1, inv2
        if len(inv1) != len(inv2):
            self.__failInventory("Different Inventory lengths: "
                                 + ("%d != %d" % (len(inv1), len(inv2))))
        def istuple(x):
            return type(x) is tuple
        for i, (x1, x2) in enumerate(zip(inv1, inv2)):
            if istuple(x1) and istuple(x2):
                id1, f1 = x1
                id2, f2 = x2
                if id1 != id2:
                    self.__failInventory("Different inventory ids, "
                                         "index %d: %r, %r" % (i, id1, id2))
            elif not istuple(x1) and not istuple(x2):
                f1, f2 = x1, x2
            else:
                self.__failInventory(
                    "Different inventory types, index %d: %r, %r" % (i,x1,x2))
            t1, t2 = type(f1), type(f2)
            if t1 is not t2:
                self.__failInventory(
                    "Different types, index %d: %r != %r" % (i, t1, t2))
            s1, s2 = str(f1), str(f2)
            if s1 != s2:
                self.__failInventory(
                    "Different values, index %d: %r != %r" % (i, s1, s2))

    def test_inventory(self):
        """Basic inventory."""
        wt = self.fixture.tree
        open(wt/'ahoy', 'w').close()
        wt.add_tag(wt/'ahoy')
        os.mkdir(wt/'bar')
        wt.add_tag(wt/'bar')
        open(wt/'bar'/'baz', 'w').close()
        wt.add_tag(wt/'bar'/'baz')

        inv = wt.iter_inventory(source=True, both=True)
        expected = [FileName('ahoy'), DirName('bar'), FileName('bar/baz')]
        self.assertInventoryEqual(inv, expected)
        inv = wt.iter_inventory(source=True, files=True)
        expected = [FileName('ahoy'), FileName('bar/baz')]
        self.assertInventoryEqual(inv, expected)
        inv = wt.iter_inventory(source=True, directories=True)
        expected = [DirName('bar')]
        self.assertInventoryEqual(inv, expected)

    def test_inventory_symlink(self):
        """Basic inventory."""
        wt = self.fixture.tree
        # names make setup easier its not what matters for the test
        wt.tagging_method = 'names'
        open(wt/'file', 'w').close()
        os.mkdir(wt/'dir')
        os.symlink('file', wt/'file-link')
        os.symlink('dir', wt/'dir-link')
        os.symlink('broken', wt/'bad-link')

        inv = wt.iter_inventory(source=True, files=True)
        expected_files = sorted(
            [FileName(name) for name in
             ['file', 'file-link', 'dir-link', 'bad-link']])
        self.assertInventoryEqual(inv, expected_files)
        inv = wt.iter_inventory(source=True, directories=True)
        expected_dirs = [DirName('dir')]
        self.assertInventoryEqual(inv, expected_dirs)
        inv = wt.iter_inventory(source=True, both=True)
        expected_both = sorted(expected_files + expected_dirs)
        self.assertInventoryEqual(inv, expected_both)

    def test_inventory_ids(self):
        """Inventory with ids"""
        wt = self.fixture.tree
        wt.tagging_method = 'names'
        open(wt/'ahoy', 'w').close()
        os.mkdir(wt/'bar')
        open(wt/'bar'/'baz', 'w').close()

        inv = wt.iter_inventory_ids(source=True, both=True)
        expected = [('?./ahoy', FileName('ahoy')),
                    ('?./bar', DirName('bar')),
                    ('?./bar/baz', FileName('bar/baz'))]
        self.assertInventoryEqual(inv, expected)
        inv = wt.iter_inventory_ids(source=True, files=True)
        expected = [('?./ahoy', FileName('ahoy')),
                    ('?./bar/baz', FileName('bar/baz'))]
        self.assertInventoryEqual(inv, expected)
        inv = wt.iter_inventory_ids(source=True, directories=True)
        expected = [('?./bar', DirName('bar'))]
        self.assertInventoryEqual(inv, expected)

    def test_inventory_names(self):
        """iter_inventory names keyword overrides untagged-source setting."""
        wt = self.fixture.tree
        print >> open(wt/'{arch}'/'=tagging-method', 'w'),\
              '\n'.join(('explicit', 'untagged-source precious',''))
        foo = FileName('foo')
        print >> open(wt/foo, 'w')

        inv = wt.iter_inventory(precious=True, files=True)
        self.assertInventoryEqual(inv, [foo]) # foo is untagged precious
        inv = wt.iter_inventory(source=True, files=True)
        self.assertInventoryEqual(inv, []) # thus foo is not source
        inv = wt.iter_inventory(source=True, files=True, names=True)
        self.assertInventoryEqual(inv, [foo]) # however it has a source name

    def test_get_tag(self):
        """ArchSourceTree.get_tag works."""
        wt = self.fixture.tree
        wt.tagging_method = 'tagline'
        foo = FileName('foo')
        print >> open(wt/foo, 'w')
        self.assertEqual(None, wt.get_tag(foo))
        print >> open(wt/foo, 'w'), 'arch-tag: yadda'
        self.assertEqual('i_yadda', wt.get_tag(foo))
        wt.add_tag(foo)
        print >> open(wt/'.arch-ids'/foo+'.id', 'w'), 'yadda'
        self.assertEqual('x_yadda', wt.get_tag(foo))

    def test_inventory_file_space(self):
        """iter_inventory handles spaces in file and dir names correctly."""
        wt = self.fixture.tree
        filename = FileName('file name')
        open(wt/filename, 'w').write('')
        wt.add_tag(filename)
        dirname = DirName('dir name')
        os.mkdir(wt/dirname)
        wt.add_tag(dirname)
        nestedname = 'nested tree'
        os.mkdir(wt/nestedname)
        nested = arch.init_tree(wt/nestedname, nested=True)

        inv = wt.iter_inventory(source=True, files=True)
        self.assertInventoryEqual(inv, [filename])
        inv = wt.iter_inventory(source=True, directories=True)
        self.assertInventoryEqual(inv, [dirname])
        inv = wt.iter_inventory(source=True, both=True)
        self.assertInventoryEqual(inv, sorted([filename, dirname]))
        inv = wt.iter_inventory(trees=True)
        self.assertInventoryEqual(inv, [nested])

    def test_inventory_limit(self):
        """Inventory with a limit"""
        wt = self.fixture.tree
        open(wt/'ahoy', 'w').close()
        wt.add_tag(wt/'ahoy')
        os.mkdir(wt/'bar')
        wt.add_tag(wt/'bar')
        open(wt/'bar'/'baz', 'w').close()
        wt.add_tag(wt/'bar'/'baz')

        inv = wt.iter_inventory(limit='bar', source=True, both=True)
        self.assertInventoryEqual(inv, [FileName('bar/baz')])
        self.assertRaises(TypeError, wt.iter_inventory,
                          limit=('a', 'b'), source=True, both=True)
        self.assertRaises(ValueError, wt.iter_inventory,
                          limit='/tmp', source=True, both=True)

    def test_inventory_ids_limit(self):
        """Inventory with ids"""
        wt = self.fixture.tree
        wt.tagging_method = 'names'
        open(wt/'ahoy', 'w').close()
        os.mkdir(wt/'bar')
        open(wt/'bar'/'baz', 'w').close()

        inv = wt.iter_inventory_ids(limit='bar', source=True, both=True)
        name_ids_fixed = arch.compat.BazaarCommandVersion(
            'baz Bazaar version 1.3~200504021840')
        if arch.backend.version < name_ids_fixed:
            expected = [('?bar/baz', FileName('bar/baz'))]
        else:
            expected = [('?./bar/baz', FileName('bar/baz'))]
        self.assertInventoryEqual(inv, expected)
        self.assertRaises(TypeError, wt.iter_inventory_ids,
                          limit=('a', 'b'), source=True, both=True)
        self.assertRaises(ValueError, wt.iter_inventory_ids,
                          limit='/tmp', source=True, both=True)


class WorkingTree(TestCase):

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()
        self.params.create_working_tree()

    tests = []

    def log_versions(self):
        """Log version addition, iteration and deletion."""
        wt = self.params.working_tree
        self.failUnlessEqual(0, len(list(wt.iter_log_versions())))
        wt.add_log_version(self.params.version)
        self.failUnlessEqual(1, len(list(wt.iter_log_versions())))
        self.failUnless(isinstance(wt.iter_log_versions().next(),
                                   arch.Version))
        self.failUnlessEqual(self.params.version, wt.iter_log_versions().next())
        wt.remove_log_version(self.params.version)
        self.failUnlessEqual(0, len(list(wt.iter_log_versions())))
    tests.append('log_versions')

    def log_versions_limit(self):
        """Log version listing with a limit."""
        wt = self.params.working_tree
        vsn = self.params.version
        other_arch = arch.Version('alice@example/%s' % vsn.nonarch)
        other_cat = vsn.archive['dog']['brn']['1.0']
        other_brn = vsn.category['trk']['1.0']
        other_vsn = vsn.branch['2']
        all_versions = [vsn, other_arch, other_cat, other_brn, other_vsn]
        for V in all_versions: wt.add_log_version(V)
        all_versions.sort()
        expected = map(str, all_versions)
        expected.sort()
        result = map(str, wt.iter_log_versions())
        result.sort()
        self.assertEqual(expected, result)
        result = list(wt.iter_log_versions(other_arch.archive))
        self.assertEqual([other_arch], result)
        result = map(str, wt.iter_log_versions(vsn.archive))
        result.sort()
        expected = map(str, (vsn, other_cat, other_brn, other_vsn))
        expected.sort()
        self.assertEqual(expected, result)
        result = list(wt.iter_log_versions(other_cat.category))
        self.assertEqual([other_cat], result)
        result = map(str, wt.iter_log_versions(vsn.category))
        result.sort()
        expected = map(str, (vsn, other_brn, other_vsn))
        expected.sort()
        self.assertEqual(expected, result)
        result = list(wt.iter_log_versions(other_brn.branch))
        self.assertEqual([other_brn], result)
        result = map(str, wt.iter_log_versions(vsn.branch))
        result.sort()
        expected = map(str, (vsn, other_vsn))
        expected.sort()
        self.assertEqual(expected, result)
        result = list(wt.iter_log_versions(vsn))
        self.assertEqual([vsn], result)
    tests.append('log_versions_limit')

    def tree_version(self):
        """Setting and getting WorkingTree.version."""
        wt = self.params.working_tree
        self.failUnlessRaises(arch.errors.TreeVersionError,
                              lambda: wt.tree_version)
        try:
            unused = wt.tree_version
        except arch.errors.TreeVersionError, E:
            self.assertEqual(E.bad_version, None)
        open(wt/'{arch}'/'++default-version', 'w').write('fooo!!!\n')
        try:
            unused = wt.tree_version
        except arch.errors.TreeVersionError, E:
            self.assertEqual(E.bad_version, 'fooo!!!')
        self.failUnlessRaises(arch.errors.TreeVersionError,
                              lambda: wt.tree_version)
        wt.tree_version = self.params.version
        self.failUnless(isinstance(wt.tree_version, arch.Version))
        self.failUnlessEqual(self.params.version, wt.tree_version)
    tests.append('tree_version')


framework.register(__name__)
