%  Copyright (C) 2005 Florian Weimer
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs query manifest}
\label{query-manifest}
\begin{code}
module QueryManifest ( query_manifest ) where
import Monad ( liftM )
import DarcsArguments ( DarcsFlag(..), working_repo_dir,
                        files, directories, pending, nullFlag )
import DarcsCommands ( DarcsCommand(..), nodefaults )
import Repository ( amInRepository, slurp_pending, slurp_recorded,
                    withRepoLock )
import SlurpDirectory ( list_slurpy, list_slurpy_files, list_slurpy_dirs )
import Printer ( putDocLnWith )
import ColourPrinter ( fancyPrinters )
import FileName ( fp2fn )
import PatchCore ( fn2d )
\end{code}

\options{query manifest}

\haskell{query_manifest_help}

By default (and if the \verb!--pending! option is specified),
the effect of pending patches on the repository is taken into account.
In other words, if you add a file using {\tt darcs add}, it
immediately appears in the output of {\tt query manifest}, even if it
is not yet recorded.  If you specify the \verb!--no-pending! option,
{\tt query manifest} will only list recorded files (and directories).

The \verb!--files! and \verb!--directories! options control whether
files and directories are included in the output.  The
\verb!--no-files!  and \verb!--no-directories! options have the
reverse effect.  The default is to include files, but not directories.

If you specify the \verb!--null! option, the file names are written to
standard output in unescaped form, and separated by ASCII NUL bytes.
This format is suitable for further automatic processing (for example,
using \verb!xargs -0!).

\begin{code}
query_manifest_description :: String
query_manifest_description = "List version-controlled files in the working copy."
\end{code}

\begin{code}
query_manifest_help :: String
query_manifest_help =
 "The manifest command lists the version-controlled files in the\n"++
 "working copy.\n"
\end{code}

\begin{code}
query_manifest :: DarcsCommand
query_manifest = DarcsCommand {
  command_name = "manifest",
  command_help = query_manifest_help,
  command_description = query_manifest_description,
  command_extra_args = 0,
  command_extra_arg_help = [],
  command_command = manifest_cmd,
  command_prereq = amInRepository,
  command_get_arg_possibilities = return [],
  command_argdefaults = nodefaults,
  command_darcsoptions = [files, directories, pending, nullFlag,
                          working_repo_dir] }

manifest_cmd :: [DarcsFlag] -> [String] -> IO ()
manifest_cmd opts _ = withRepoLock $ \repository -> do
    list <- to_list `liftM` slurp repository
    mapM_ output list
    where slurp = if NoPending `notElem` opts
                  then slurp_pending else slurp_recorded
          files_dirs False False = \_ -> []
          files_dirs False True  = list_slurpy_dirs
          files_dirs True  False = list_slurpy_files
          files_dirs True  True  = list_slurpy
          to_list =
              files_dirs  (NoFiles `notElem` opts) (Directories `elem` opts)
          output_null name = do { putStr name ; putChar '\0' }
          output_nice = (putDocLnWith fancyPrinters) . fn2d . fp2fn
          output = if NullFlag `elem` opts then output_null else output_nice

\end{code}
