/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/irmp3mod.c,v $ -- Module handling functions
 * $Id: irmp3mod.c,v 1.20 2004/08/10 21:08:31 boucman Exp $
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>


#include "config.h"
#include "irmp3log.h"
#include "modules.inc"
#include "irmp3mod.h"
#ifdef MOD_NETCTL
#include "mod_netctl.h"
#endif

/*************************************************************************
 * GLOBALS
 */

fd_set	blank_fd;
mod_message_t* answer_head = NULL;
static int return_status = -1;
const char * current_module = NULL;


/*************************************************************************
 * LIST OF ALL MODULES
 */
mod_t *mod_list = NULL;
mod_message_t *mod_msgs = NULL;




/*************************************************************************
 * SEND MESSAGE TO ALL MODULES
 */
void mod_sendmsg (int msgtype, char *msg)
{
	mod_message_t *newmsg, *m;
	log_printf(LOG_NOISYDEBUG,"CORE: message %s type %d sent\n",msg,msgtype);
	if(answer_head) { // we are in a query
		// the action has to be taken immediatly, so transform this to a query call
		// however, we don't care about the answer
		free_message(mod_query(msgtype,msg));
		return;
	} else {

		// create new message entry
		newmsg = malloc(sizeof(mod_message_t));
		if (!newmsg)
			return;
		newmsg->msg = malloc(strlen(msg)+1);
		if (!newmsg->msg) {
			free(newmsg);
			return;
		}
		newmsg->msgtype = msgtype;
		newmsg->sender = current_module;
		strcpy(newmsg->msg, msg);
		newmsg->next = NULL;

		// append message to queue
		for (m=mod_msgs; m && m->next; m=m->next);
		if (m) {
			m->next = newmsg;
		}else {
			mod_msgs = newmsg;
		}
	}
	
}

void mod_sendmsgf (int msgtype, char *msg, ...)
{
	va_list ap;
	char buf[1024];
	va_start(ap, msg);
	vsnprintf(buf, sizeof(buf)-1, msg, ap);
	buf[sizeof(buf)-1] = 0;
	va_end(ap);
	mod_sendmsg(msgtype, buf);
}


/*************************************************************************
 * INIT ALL MODULES
 */
void mod_init (void)
{
	char *s;
	const char *last_context;
	int i;
	mod_t *last_registered=NULL;

	last_context = current_module;
	for(i=0;mod_list_init[i];i++) {
		log_printf(LOG_NOISYDEBUG,"CORE: initialising module %s\n",mod_list_init[i]->mod_name);
		if(!mod_list) {
			last_registered = mod_list_init[i];
			mod_list = last_registered;
		} else {
			last_registered->next = mod_list_init[i];
			last_registered = mod_list_init[i];
		}
		last_registered->next = NULL;
		if(last_registered->init){
			current_module = last_registered->mod_name;
			s = last_registered->init();
			if (s)
				log_printf(LOG_ERROR, "Error initializing module: %s\n", s);
		}
	}
	current_module = last_context;
}


/*************************************************************************
 * DEINIT ALL MODULES
 */
void mod_deinit (void)
{
	mod_t *m;
	const char * last_context;

	// call deinit function of all modules
	// and clean up module list
	last_context = current_module;
	for (m=mod_list; m; m=m->next) {
		if (m->deinit) {
			log_printf(LOG_NOISYDEBUG,"CORE: deinitialising module %s\n",m->mod_name);
			current_module = m->mod_name;
			m->deinit();
		}
	}
	current_module = last_context;
	mod_list = NULL;
}


/*************************************************************************
 * RELOAD ALL MODULES
 */
void mod_reload (void)
{
	mod_t *m;
	char *s;
	const char * last_context;

	// call reload function of all modules
	last_context = current_module;
	for (m=mod_list; m; m=m->next) {
		log_printf(LOG_NOISYDEBUG,"CORE: reinitialising module %s\n",m->mod_name);
		if (m->reload) {
			current_module = m->mod_name;
			s = m->reload();
			if (s)
				log_printf(LOG_ERROR, "Error reloading module: %s\n", s);
		}
	}
	current_module = last_context;
}


/*************************************************************************
 * DISPERSE MESSAGES AND UPDATE ALL MODULES
 */
void mod_update (void)
{
	mod_t *m;
	mod_message_t *msg;
	mod_message_t msgdup;
	const char * last_context;

	// process message queue
	last_context = current_module;
	while (mod_msgs) {
		msg = mod_msgs;

		// global messages we handle
		if (msg->msgtype == MSGTYPE_INPUT && !strcasecmp(msg->msg, "reload")) {
			raise(SIGHUP);

		// disperse message to modules, make a copy of the message
		// so that modules may alter it without affecting
		// other modules (think of strtok)
		} else {
			for (m=mod_list; m; m=m->next) {
				if (m->message) {
					log_printf(LOG_NOISYDEBUG,"CORE: sending message %s type %d to module %s\n",msg->msg,msg->msgtype,m->mod_name);
					current_module = m->mod_name;
					msgdup.msgtype = msg->msgtype;
					msgdup.msg = malloc(strlen(msg->msg)+1);
					msgdup.sender = msg->sender;
					strcpy(msgdup.msg, msg->msg);
					m->message(msgdup.msgtype, msgdup.msg,msgdup.sender);
					free(msgdup.msg);
				}
			}
		}

		// remove old message from queue
		mod_msgs = mod_msgs->next;
		free(msg->msg);
		free(msg);
	}

	// call update function of all modules
	for (m=mod_list; m; m=m->next) {
		if (m->update){
			current_module = m->mod_name;
			log_printf(LOG_NOISYDEBUG,"CORE: updating module %s\n",m->mod_name);

			m->update();
		}
	}
	current_module = last_context;
}

/*************************************************************************
 * HANDLE SIGCHLD CALLS FOR ALL MODULES
 */

void mod_sigchld()
{
	pid_t pid;
	mod_t *m;
	int status;
	const char * last_context;


	// this process was called because a child of IRMP3 just died and
	// we received a SIGCHLD signal.  wait() won't block.
	pid = waitpid(-1,&status,WNOHANG);
	if(!pid) return;
	
	log_printf(LOG_NOISYDEBUG, "mod_sigchld(): notified that child %d died.\n", pid);

	// call child function of all modules
	last_context = current_module;
	for (m=mod_list;m;m=m->next) {
		if (m->chld){
			current_module = m->mod_name;
			log_printf(LOG_NOISYDEBUG,"CORE: sgnaling child death to module %s\n",m->mod_name);
			m->chld(pid,status);
		}
	}
	current_module = last_context;
}


/*************************************************************************
 * POST A QUERY
 */

mod_message_t* mod_query(int msgtype, char* msg)
{
	mod_t *m;
	mod_message_t msgdup;
	mod_message_t* newmsg;
	mod_message_t* return_queue; // create a new queue
	mod_message_t* tmp_answer; // store previous head
	const char * last_context;
	last_context = current_module;
	if(answer_head) { // we are in a query, add this to the list of answers
		// create new message entry
		newmsg = malloc(sizeof(mod_message_t));
		if (!newmsg) {
			return NULL;
		}
		newmsg->msg = malloc(strlen(msg)+1);
		if (!newmsg->msg) {
			free(newmsg);
			return NULL;
		}
		newmsg->msgtype = msgtype;
		newmsg->sender = last_context;
		strcpy(newmsg->msg, msg);
		newmsg->next = NULL;
		answer_head->next = newmsg;
		answer_head = newmsg;
	}
	return_queue = malloc(sizeof(mod_message_t)); // create a new queue
	if(!return_queue) {
		return NULL;
	}
	return_queue->next= NULL;
	tmp_answer = answer_head;
	answer_head = return_queue; // the new queue is the active queue

	// process the message
	for (m=mod_list; m; m=m->next) {
		if (m->message) {
			log_printf(LOG_NOISYDEBUG,"CORE: sending query %s type %d to module %s\n",msg,msgtype,m->mod_name);
			current_module = m->mod_name;
			msgdup.msgtype = msgtype;
			msgdup.msg = malloc(strlen(msg)+1);
			msgdup.sender = last_context;
			strcpy(msgdup.msg, msg);
			m->message(msgdup.msgtype, msgdup.msg,msgdup.sender);
			free(msgdup.msg);
		}
	}
	current_module = last_context;
	answer_head = tmp_answer; // restore previous queue
	tmp_answer = return_queue; // remove the first (fake) elt of the queue
	return_queue = return_queue->next;
	free(tmp_answer);
	return return_queue;
}

mod_message_t* mod_queryf(int msgtype, char* msg,...)
{
	va_list ap;
	char buf[1024];
	va_start(ap, msg);
	vsnprintf(buf, sizeof(buf)-1, msg, ap);
	buf[sizeof(buf)-1] = 0;
	va_end(ap);
	return mod_query(msgtype, buf);
}
	
	

	

/*************************************************************************
 * FREE AN ANSWER
 */

void free_message(mod_message_t* to_free) 
{
	mod_message_t *freeing, *next;
	freeing = to_free;
	while(freeing) {
		next = freeing->next;
		free(freeing->msg);
		free(freeing);
		freeing = next;
	}
}
		



pid_t system_noblock(int* std_in, int* std_out, int* std_error,char *cmd,...)
{
	int fd_send[2], fd_recv[2], fd_error[2];
	char *args[50];
	va_list ap;
	pid_t child_pid;

	// make socketpair to communicate with player
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd_send) < 0) {
		log_printf(LOG_ERROR, "system_nonblock(): unable to setup send pipe (%s).\n", strerror(errno));
		return -1;
	}
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd_recv) < 0) {
		log_printf(LOG_ERROR, "system_nonblock(): unable to setup recv pipe (%s).\n", strerror(errno));
		return -1;
	}
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd_error) < 0) {
		log_printf(LOG_ERROR, "system_nonblock(): unable to setup error pipe (%s).\n", strerror(errno));
		return -1;
	}

	// handle signals
	signal(SIGPIPE, SIG_IGN);

        args[0] = "sh";
	args[1] = "-c";
	va_start(ap,cmd);
	vasprintf(&args[2],cmd,ap);
	va_end(ap);
	args[3] = NULL;
		
		
	// fork child process
	child_pid = fork();
	if (child_pid == 0) {
		// pipe in/output through socket to parent
		dup2(fd_send[0], STDIN_FILENO);
		close(fd_send[0]);
		close(fd_send[1]);
		dup2(fd_recv[0], STDOUT_FILENO);
		close(fd_recv[0]);
		close(fd_recv[1]);
		dup2(fd_error[0], STDERR_FILENO);
		close(fd_error[0]);
		close(fd_error[1]);
#ifdef MOD_NETCTL
		mod_netctl_close_all_clients();
#endif
		// spawn player
		execvp(args[0], args);
		// never reached if exec was ok
		exit(-1);
	} else if (child_pid== -1) {
		free(args[2]);
		close(fd_send[0]);
		close(fd_send[1]);
		close(fd_recv[0]);
		close(fd_recv[1]);
		close(fd_error[0]);
		close(fd_error[1]);
		return -1;
	}


	// parent continues here

	close(fd_send[0]);
	log_printf(LOG_DEBUG, "system_nonbloc(): child spawned pid %d, piped on fd %d, %d and %d\n", child_pid, fd_send[1], fd_recv[1], fd_error[1]);
	log_printf(LOG_DEBUG, "CMD: %s\n",args[2]);
	free(args[2]);
	if(std_in) {
		*std_in = fd_send[1];
	} else {
		close(fd_send[1]);
	}
	close(fd_recv[0]);
	if(std_out) {
		*std_out = fd_recv[1];
	} else {
		close(fd_recv[1]);
	}
		
	close(fd_error[0]);
	if(std_error) {
		*std_error = fd_error[1];
	} else {
		close(fd_error[1]);
	}

	return child_pid;
}


int system_block(int std_in, int* std_out, int* std_error,char*cmd,...)
{
	int fd_send[2], fd_recv[2], fd_error[2];
	char *args[50];
	va_list ap;
	pid_t wait_pid = -1;

	// make socketpair to communicate with child
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd_send) < 0) {
		log_printf(LOG_ERROR, "system_block(): unable to setup send pipe (%s).\n", strerror(errno));
		return -1;
	}
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd_recv) < 0) {
		log_printf(LOG_ERROR, "system_block(): unable to setup recv pipe (%s).\n", strerror(errno));
		return -1;
	}
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd_error) < 0) {
		log_printf(LOG_ERROR, "system_block(): unable to setup error pipe (%s).\n", strerror(errno));
		return -1;
	}

	// handle signals
	signal(SIGPIPE, SIG_IGN);

        args[0] = "sh";
	args[1] = "-c";
	va_start(ap,cmd);
	vasprintf(&args[2],cmd,ap);
	va_end(ap);
	args[3] = NULL;
	// fork child process
	wait_pid = fork();
	if (wait_pid == 0) {
		// pipe in/output through socket to parent
		dup2(fd_send[0], STDIN_FILENO);
		close(fd_send[0]);
		close(fd_send[1]);
		dup2(fd_recv[0], STDOUT_FILENO);
		close(fd_recv[0]);
		close(fd_recv[1]);
		dup2(fd_error[0], STDERR_FILENO);
		close(fd_error[0]);
		close(fd_error[1]);
#ifdef MOD_NETCTL
		mod_netctl_close_all_clients();
#endif
		// spawn child
		execvp(args[0], args);
		// never reached if exec was ok
		exit(-1);
	} else if (wait_pid== -1) {
		free(args[2]);
		close(fd_send[0]);
		close(fd_send[1]);
		close(fd_recv[0]);
		close(fd_recv[1]);
		close(fd_error[0]);
		close(fd_error[1]);
		return -1;
	}


	// parent continues here

	log_printf(LOG_DEBUG, "system_bloc(): child spawned pid %d, piped on fd %d, %d and %d\n", wait_pid, fd_send[1], fd_recv[1], fd_error[1]);
	log_printf(LOG_DEBUG, "CMD: %s\n",args[2]);
	free(args[2]);
	close(fd_send[0]);
	if(std_in) {
		dup2(std_in,fd_send[1]);
	} else {
		close(fd_send[1]);
	}
	close(fd_recv[0]);
	if(std_out) {
		*std_out = fd_recv[1];
	} else {
		close(fd_recv[1]);
	}
		
	close(fd_error[0]);
	if(std_error) {
		*std_error = fd_error[1];
	} else {
		close(fd_error[1]);
	}
	waitpid(wait_pid,NULL,0);

	return return_status;
}
/*************************************************************************
 * EOF
 */
