/***************************************************************************
                          account.cpp  -  description
                             -------------------
    begin                : Sun Jan 5 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "account.h"

#include <qdir.h>
#include <qregexp.h>

#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <kmdcodec.h>
#include <kstandarddirs.h>

#include "kmessdebug.h"

#ifdef KMESSDEBUG_ACCOUNT
  #define KMESSDEBUG_ACCOUNT_GENERAL
  #define KMESSDEBUG_ACCOUNT_DIRTY
#endif

// The constructor
Account::Account()
 : autoreplyMessage_( i18n("I'm away from my computer") ),
   chatFormat_( i18n("{time} {name} says:\n{message}\n") ),
   contactFontColor_("#000000"),
   dirty_(false),
   emailCommand_("kmail"),
   emailSupported_(true),  // Assume true (for new accounts), until the msn server proves the opposite
   fontColor_("#000000"),
   friendlyName_( i18n("Your name") ),
   handle_( i18n("you@hotmail.com") ),
   idleTime_(5),
   notificationDuration_(10),
   saveChats_(false),
   savedChatDirectoryStructure_( 0 ),
   showAllowedContacts_(false),
   showContactsByGroup_(true),
   showEmail_(true),
   showOfflineContacts_(true),
   showOfflineNotification_(true),
   showOtherFolders_(true),
   showRemovedContacts_(false),
   showSidebar_(true),
   startInvisible_(false),
   useAutologin_(false),
   useContactFont_(false),
   useEmoticons_(true),
   useFontEffects_(true),
   useHotmail_(true),
   useIdleTimer_(true),
   useNotifier_(true),

   useProxy_(false),
   proxyType_(TYPE_HTTP),
   proxyPort_(80),

   showImage_(true),
   customImage_(false),
   msnObject_(0)
{
  QDir homeDir;

  font_.setFamily("Arial");
  font_.setBold(false);
  font_.setItalic(false);
  font_.setUnderline(false);
  font_.setPointSize(12);

  contactFont_.setFamily("Arial");
  contactFont_.setBold(false);
  contactFont_.setItalic(false);
  contactFont_.setUnderline(false);

  homeDir = QDir::home();
  saveChatPath_ = homeDir.absPath();
}



// The destructor
Account::~Account()
{
  if ( msnObject_ )
    delete msnObject_;

#ifdef KMESSDEBUG_ACCOUNT_GENERAL
  kdDebug() << "DESTROYED Account " << handle_ << endl;
#endif
}



// Copy an account
void Account::copyAccount( const Account *account )
{
  setLoginInformation( account->getHandle(), account->getFriendlyName(), account->getPassword() );
  setShowAllowedContacts( account->getShowAllowedContacts() );
  setShowContactsByGroup( account->getShowContactsByGroup() );
  setShowOfflineContacts( account->getShowOfflineContacts() );
  setShowRemovedContacts( account->getShowRemovedContacts() );

  setAlertsInformation      ( account->getUseIdleTimer(),
                              account->getIdleTime(),
                              account->getUseNotifier(),
                              account->getNotificationDuration(),
                              account->getShowOfflineNotification()       );
  setChatInformation        ( account->getUseContactFont(),
                              account->getUseEmoticons(),
                              account->getUseFontEffects(),
                              account->getChatFormat()                    );
  setChatLoggingInformation ( account->getSaveChats(),
                              account->getSaveChatPath(),
                              account->getSavedChatDirectoryStructure()   );
  setEmailInformation       ( account->getUseHotmail(),
                              account->getEmailCommand(),
                              account->getShowEmail(),
                              account->getShowOtherFolders()              );
  setEmailSupported         ( account->getEmailSupported()                );
  setFontInformation        ( account->getFont(),
                              account->getFontColor(),
                              account->getContactFont(),
                              account->getContactFontColor()              );
  setProxyInformation       ( account->getUseProxy(),
                              (int)account->getProxyType(),
                              account->getProxyServer(),
                              account->getProxyPort(),
                              account->getProxyUID(),
                              account->getProxyPassword()                 );

  setUseAutologin           ( account->getUseAutologin()                  );
  setAutoreplyMessage       ( account->getAutoreplyMessage()              );
  setShowSidebar            ( account->getShowSidebar()                   );
  setStartInvisible         ( account->getStartInvisible()                );

  setShowImage              ( account->getShowImage()                     );
  setCustomImage            ( account->getCustomImage()                   );

  updateMsnObject();
}



// Copy an account's user-interface, not settings, changable properties
void Account::copyAccountUISettings( const Account *account )
{
  setShowAllowedContacts( account->getShowAllowedContacts() );
  setShowOfflineContacts( account->getShowOfflineContacts() );
  setShowRemovedContacts( account->getShowRemovedContacts() );
  setShowContactsByGroup( account->getShowContactsByGroup() );
  setAutoreplyMessage   ( account->getAutoreplyMessage()    );
  setShowSidebar        ( account->getShowSidebar()         );
}



// Read the auto-reply message.
const QString& Account::getAutoreplyMessage() const
{
  return autoreplyMessage_;
}



// Return the chat format string.
const QString& Account::getChatFormat() const
{
  return chatFormat_;
}



// Return the font used for contact messages, if forced to.
const QFont& Account::getContactFont() const
{
  return contactFont_;
}



// Return the color of the forced contact font.
const QString& Account::getContactFontColor() const
{
  return contactFontColor_;
}



// Read the email command used when not using hotmail
const QString& Account::getEmailCommand() const
{
  return emailCommand_;
}


// Return whether email notifications are supported
bool Account::getEmailSupported() const
{
  return emailSupported_;
}


// Return the user message font.
const QFont& Account::getFont() const
{
  return font_;
}



// Return the color of the user message font.
const QString& Account::getFontColor() const
{
  return fontColor_;
}



// Return the user's friendlyName
QString Account::getFriendlyName() const
{
  return friendlyName_;
}



// Return the user's handle
QString Account::getHandle() const
{
  return handle_;
}



// Return the length of time waited before the user goes idle.
const int& Account::getIdleTime() const
{
  return idleTime_;
}



// Return the time popup windows are shown.
const int& Account::getNotificationDuration() const
{
  return notificationDuration_;
}



// Return the user's password
QString Account::getPassword() const
{
  return password_;
}



// Return the proxy password
const QString& Account::getProxyPassword() const
{
  return proxyPassword_;
}



// Return the proxy port
const int& Account::getProxyPort()  const
{
  return proxyPort_;
}



// Return the proxy server IP
const QString& Account::getProxyServer() const
{
  return proxyServer_;
}



// Return the proxy server type
const int& Account::getProxyType() const
{
  return (int)proxyType_;
}



// Return the user's proxy user ID
const QString& Account::getProxyUID() const
{
  return proxyUID_;
}



// Return whether or not to save chats.
bool Account::getSaveChats() const
{
  return saveChats_;
}



// Return the base directory to which chats are saved.
const QString& Account::getSaveChatPath() const
{
  return saveChatPath_;
}



// Return the code of the structure used to save chats.
const int& Account::getSavedChatDirectoryStructure() const
{
  return savedChatDirectoryStructure_;
}



// Read whether or not allowed contacts are shown.
bool Account::getShowAllowedContacts() const
{
  return showAllowedContacts_;
}



// Read whether contacts are shown by group or by online/offline.
bool Account::getShowContactsByGroup() const
{
  return showContactsByGroup_;
}



// Read whether the email information should be shown in the main view.
bool Account::getShowEmail() const
{
  return showEmail_;
}



// Read whether offline contacts should be shown.
bool Account::getShowOfflineContacts() const
{
  return showOfflineContacts_;
}



// Read property of bool showOfflineNotification_.
bool Account::getShowOfflineNotification() const
{
  return showOfflineNotification_;
}



// Read property of bool showOtherFolders.
bool Account::getShowOtherFolders() const
{
  return showOtherFolders_;
}



// Read whether or not removed (reverse) contacts are shown.
bool Account::getShowRemovedContacts() const
{
  return showRemovedContacts_;
}



// Read whether or not the sidebar should be shown in chats
bool Account::getShowSidebar() const
{
  return showSidebar_;
}



// Return whether or not to start in "hidden" mode
bool Account::getStartInvisible() const
{
  return startInvisible_;
}



// Return whether or not to autologin with this profile.
bool Account::getUseAutologin() const
{
  return useAutologin_;
}



// Return whether or not to show contact messages in the stored font.
bool Account::getUseContactFont() const
{
  return useContactFont_;
}



// Return whether or not to show emoticons in chats.
bool Account::getUseEmoticons() const
{
  return useEmoticons_;
}



// Return whether or not to use the bold/italic/underline effects.
bool Account::getUseFontEffects() const
{
  return useFontEffects_;
}



// Read whether the user uses hotmail to read email or a set command
bool Account::getUseHotmail() const
{
  return useHotmail_;
}



// Return whether the idle timer should be used.
bool Account::getUseIdleTimer() const
{
  return useIdleTimer_;
}



// Return whether or not to show popup notifications.
bool Account::getUseNotifier() const
{
  return useNotifier_;
}



// Return whether or not to use a proxy.
bool Account::getUseProxy() const
{
  return useProxy_;
}



// Returns whether or not to send our display image
bool Account::getShowImage() const
{
  return showImage_;
}



// Returns whether or not to use a custom image
bool Account::getCustomImage() const
{
  return customImage_;
}


// Return the path of the custom display image.
const QString Account::getCustomImagePath() const
{
  QString safeHandle = getHandle().lower().replace( QRegExp( "[^a-z0-9_.@]" ), "-" );
  return ::locateLocal("data", "kmess/msnpicture-" + safeHandle + ".png");
}


// Return the path to the *current* display image
const QString Account::getImagePath() const
{
  KStandardDirs *dirs   = KGlobal::dirs();
  QString imagePath;

  // return image path if user has selected to show it else return QString::null
  if ( showImage_ )
  {
    if ( customImage_ )
    {
      // if the user has selected a custom image it will be located (usually) in
      // ~/.kde/share/apps/kmess/msnpicture-handle.png
      imagePath = getCustomImagePath();

      // If the file can't be found, fallback to the default image.
      if(QFile::exists(imagePath))
      {
        return imagePath;
      }
    }

    // if the user wants the default image it will be located (usually) in
    // /usr/share/apps/kmess/pics/kmesspic.png
    imagePath = dirs->findResource( "data", "kmess/pics/kmesspic.png"  );
    return imagePath;
  }
  else
  {
    return QString::null;
  }
}


// Returns the string representation of the MsnObject
const QString Account::getMsnObjectString() const
{
  if ( msnObject_ )
    return msnObject_->objectString();
  else
    return "";
}



// Read in account properties
void Account::readProperties(KConfig *config, QString handle)
{
  handle_ = handle;
  QDir    homeDir;
  QString defaultPath;

  homeDir = QDir::home();
  defaultPath = homeDir.absPath();

  // Go the the profile's group
  config->setGroup( "Profile_" + handle_ );

  autoreplyMessage_            = config->readEntry    ( "autoreplymessage",            i18n("I'm away from my computer") );
  chatFormat_                  = config->readEntry    ( "chatFormat",                  i18n("{time} {name} says:\n{message}\n") );
  contactFont_.setFamily       ( config->readEntry    ( "contactfontfamily",           "Arial"                         ) );
  contactFont_.setBold         ( config->readBoolEntry( "contactfontbold",             false                           ) );
  contactFont_.setItalic       ( config->readBoolEntry( "contactfontitalic",           false                           ) );
  contactFont_.setUnderline    ( config->readBoolEntry( "contactfontunderline",        false                           ) );
  contactFont_.setPointSize    ( config->readNumEntry ( "contactfontpointsize",        10                              ) );
  contactFontColor_            = config->readEntry    ( "contactfontcolor" ,           "#000000"                         );
  emailCommand_                = config->readEntry    ( "emailCommand",                "kmail"                           );
  font_.setFamily              ( config->readEntry    ( "fontfamily",                  "Arial"                         ) );
  font_.setBold                ( config->readBoolEntry( "fontbold",                    false                           ) );
  font_.setItalic              ( config->readBoolEntry( "fontitalic",                  false                           ) );
  font_.setUnderline           ( config->readBoolEntry( "fontunderline",               false                           ) );
  font_.setPointSize           ( config->readNumEntry ( "fontpointsize",               12                              ) );
  fontColor_                   = config->readEntry    ( "fontColor" ,                  "#000000"                         );
  friendlyName_                = config->readEntry    ( "friendlyName" ,               QString::null                     );
  idleTime_                    = config->readNumEntry ( "idleTime",                    5                                 );
  notificationDuration_        = config->readNumEntry ( "notification",                5                                 );
  password_                    = config->readEntry    ( "password" ,                   QString::null                     );
  saveChatPath_                = config->readEntry    ( "saveChatPath",                defaultPath                       );
  saveChats_                   = config->readBoolEntry( "saveChats",                   false                             );
  savedChatDirectoryStructure_ = config->readNumEntry ( "savedChatDirectoryStructure", 0                                 );
  showAllowedContacts_         = config->readBoolEntry( "showAllowed",                 false                             );
  showContactsByGroup_         = config->readBoolEntry( "showByGroup",                 true                              );
  showEmail_                   = config->readBoolEntry( "showEmail",                   true                              );
  showOfflineContacts_         = config->readBoolEntry( "showOfflineContacts",         true                              );
  showOfflineNotification_     = config->readBoolEntry( "showOfflineNotification",     true                              );
  showOtherFolders_            = config->readBoolEntry( "showOtherFolders",            true                              );
  showRemovedContacts_         = config->readBoolEntry( "showRemoved",                 false                             );
  showSidebar_                 = config->readBoolEntry( "showSidebar",                 true                              );
  startInvisible_              = config->readBoolEntry( "startInvisible",              false                             );
  useAutologin_                = config->readBoolEntry( "useAutologin",                false                             );
  useContactFont_              = config->readBoolEntry( "useContactFont",              false                             );
  useEmoticons_                = config->readBoolEntry( "useEmoticons",                true                              );
  useFontEffects_              = config->readBoolEntry( "useFontEffects",              true                              );
  useHotmail_                  = config->readBoolEntry( "useHotmail",                  true                              );
  useIdleTimer_                = config->readBoolEntry( "useIdleTimer",                true                              );
  useNotifier_                 = config->readBoolEntry( "useNotifier",                 true                              );

  proxyPassword_               = config->readEntry    ( "proxyPassword",               ""                                );
  proxyPort_                   = config->readNumEntry ( "proxyPort",                   1080                              );
  proxyServer_                 = config->readEntry    ( "proxyServer",                 ""                                );
  proxyType_                   = (PROXYTYPE)config->readNumEntry ( "proxyType",        2                                 );//TYPE_SOCKS5
  proxyUID_                    = config->readEntry    ( "proxyUID",                    ""                                );
  useProxy_                    = config->readBoolEntry( "useProxy",                    false                             );

  showImage_                   = config->readBoolEntry( "showImage",                   true                              );
  customImage_                 = config->readBoolEntry( "customImage",                 false                             );

  updateMsnObject();

  // After reading options from file, consider "dirty" to still be false
  dirty_ = false;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Resetting 'dirty' to false." << endl;
#endif
}



// Save account properties
void Account::saveProperties(KConfig *config)
{
  if ( dirty_ )
  {
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account " << handle_ << " is 'dirty' so save properties." << endl;
#endif
    // Go the the profile's group
    config->setGroup( "Profile_" + handle_ );

    config->writeEntry( "autoreplymessage",            autoreplyMessage_            );
    config->writeEntry( "chatFormat",                  chatFormat_                  );
    config->writeEntry( "contactfontfamily",           contactFont_.family()        );
    config->writeEntry( "contactfontbold",             contactFont_.bold()          );
    config->writeEntry( "contactfontitalic",           contactFont_.italic()        );
    config->writeEntry( "contactfontunderline",        contactFont_.underline()     );
    config->writeEntry( "contactfontpointsize",        contactFont_.pointSize()     );
    config->writeEntry( "contactfontcolor",            contactFontColor_            );
    config->writeEntry( "emailCommand",                emailCommand_                );
    config->writeEntry( "fontfamily",                  font_.family()               );
    config->writeEntry( "fontbold",                    font_.bold()                 );
    config->writeEntry( "fontitalic",                  font_.italic()               );
    config->writeEntry( "fontunderline",               font_.underline()            );
    config->writeEntry( "fontpointsize",               font_.pointSize()            );
    config->writeEntry( "fontColor",                   fontColor_                   );
    config->writeEntry( "friendlyName",                friendlyName_                );
    config->writeEntry( "idleTime",                    idleTime_                    );
    config->writeEntry( "notification",                notificationDuration_        );
    config->writeEntry( "password",                    password_                    );
    config->writeEntry( "saveChatPath",                saveChatPath_                );
    config->writeEntry( "saveChats",                   saveChats_                   );
    config->writeEntry( "savedChatDirectoryStructure", savedChatDirectoryStructure_ );
    config->writeEntry( "showAllowed",                 showAllowedContacts_         );
    config->writeEntry( "showByGroup",                 showContactsByGroup_         );
    config->writeEntry( "showEmail",                   showEmail_                   );
    config->writeEntry( "showOfflineContacts",         showOfflineContacts_         );
    config->writeEntry( "showOfflineNotification",     showOfflineNotification_     );
    config->writeEntry( "showOtherFolders",            showOtherFolders_            );
    config->writeEntry( "showRemoved",                 showRemovedContacts_         );
    config->writeEntry( "showSidebar",                 showSidebar_                 );
    config->writeEntry( "startInvisible",              startInvisible_              );
    config->writeEntry( "useAutologin",                useAutologin_                );
    config->writeEntry( "useContactFont",              useContactFont_              );
    config->writeEntry( "useEmoticons",                useEmoticons_                );
    config->writeEntry( "useFontEffects",              useFontEffects_              );
    config->writeEntry( "useHotmail",                  useHotmail_                  );
    config->writeEntry( "useIdleTimer",                useIdleTimer_                );
    config->writeEntry( "useNotifier",                 useNotifier_                 );

    config->writeEntry( "proxyPassword",               proxyPassword_               );
    config->writeEntry( "proxyPort",                   proxyPort_                   );
    config->writeEntry( "proxyServer",                 proxyServer_                 );
    config->writeEntry( "proxyType",              (int)proxyType_                   );
    config->writeEntry( "proxyUID",                    proxyUID_                    );
    config->writeEntry( "useProxy",                    useProxy_                    );

    config->writeEntry( "showImage",                   showImage_                   );
    config->writeEntry( "customImage",                 customImage_                 );

#ifdef KMESSTEST
    ASSERT( autoreplyMessage_            == config->readEntry    ( "autoreplymessage",            i18n("I'm away from my computer") ) );
    ASSERT( contactFont_.family()        == config->readEntry    ( "contactfontfamily",           "Arial"                           ) );
    ASSERT( contactFont_.bold()          == config->readBoolEntry( "contactfontbold",             false                             ) );
    ASSERT( contactFont_.italic()        == config->readBoolEntry( "contactfontitalic",           false                             ) );
    ASSERT( contactFont_.underline()     == config->readBoolEntry( "contactfontunderline",        false                             ) );
    ASSERT( contactFont_.pointSize()     == config->readNumEntry ( "contactfontpointsize",        10                                ) );
    ASSERT( contactFontColor_            == config->readEntry    ( "contactfontcolor" ,           "#000000"                         ) );
    ASSERT( emailCommand_                == config->readEntry    ( "emailCommand",                "kmail"                           ) );
    ASSERT( font_.family()               == config->readEntry    ( "fontfamily",                  "Arial"                           ) );
    ASSERT( font_.bold()                 == config->readBoolEntry( "fontbold",                    false                             ) );
    ASSERT( font_.italic()               == config->readBoolEntry( "fontitalic",                  false                             ) );
    ASSERT( font_.underline()            == config->readBoolEntry( "fontunderline",               false                             ) );
    ASSERT( font_.pointSize()            == config->readNumEntry ( "fontpointsize",               12                                ) );
    ASSERT( fontColor_                   == config->readEntry    ( "fontColor" ,                  "#000000"                         ) );
    ASSERT( friendlyName_                == config->readEntry    ( "friendlyName" ,               QString::null                     ) );
    ASSERT( idleTime_                    == config->readNumEntry ( "idleTime",                    5                                 ) );
    ASSERT( notificationDuration_        == config->readNumEntry ( "notification",                5                                 ) );
    ASSERT( password_                    == config->readEntry    ( "password" ,                   QString::null                     ) );
    ASSERT( saveChatPath_                == config->readEntry    ( "saveChatPath",                ""                                ) );
    ASSERT( saveChats_                   == config->readBoolEntry( "saveChats",                   false                             ) );
    ASSERT( savedChatDirectoryStructure_ == config->readNumEntry ( "savedChatDirectoryStructure", 0                                 ) );
    ASSERT( showAllowedContacts_         == config->readBoolEntry( "showAllowed",                 false                             ) );
    ASSERT( showContactsByGroup_         == config->readBoolEntry( "showByGroup",                 true                              ) );
    ASSERT( showEmail_                   == config->readBoolEntry( "showEmail",                   true                              ) );
    ASSERT( showOfflineNotification_     == config->readBoolEntry( "showOfflineNotification",     true                              ) );
    ASSERT( showOtherFolders_            == config->readBoolEntry( "showOtherFolders",            true                              ) );
    ASSERT( showRemovedContacts_         == config->readBoolEntry( "showRemoved",                 false                             ) );
    ASSERT( showSidebar_                 == config->readBoolEntry( "showSidebar",                 true                              ) );
    ASSERT( useAutologin_                == config->readBoolEntry( "useAutologin",                false                             ) );
    ASSERT( useContactFont_              == config->readBoolEntry( "useContactFont",              false                             ) );
    ASSERT( useEmoticons_                == config->readBoolEntry( "useEmoticons",                true                              ) );
    ASSERT( useHotmail_                  == config->readBoolEntry( "useHotmail",                  true                              ) );
    ASSERT( useIdleTimer_                == config->readBoolEntry( "useIdleTimer",                true                              ) );
    ASSERT( useNotifier_                 == config->readBoolEntry( "useNotifier",                 true                              ) );
#endif
  }
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  else
  {
    kdDebug() << "Account " << handle_ << " is clean so DO NOT save properties." << endl;
  }
#endif
}



// Set alert information
void Account::setAlertsInformation( bool useIdleTimer, int idleTime, bool useNotifier, int notificationDuration, bool showOfflineNotification)
{
  useIdleTimer_ = useIdleTimer;
  idleTime_ = idleTime;
  useNotifier_ = useNotifier;
  notificationDuration_ = notificationDuration;
  showOfflineNotification_ = showOfflineNotification;

  emit changedTimerSettings();
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setAlertsInformation." << endl;
#endif
}



// Set the auto-reply message.
void Account::setAutoreplyMessage( const QString& autoreplyMessage)
{
  if ( autoreplyMessage_ != autoreplyMessage )
  {
    autoreplyMessage_ = autoreplyMessage;
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account Setting 'dirty' to true - setAutoreplyMessage." << endl;
#endif
  }
}



// Set chat information
void Account::setChatInformation( bool useContactFont, bool useEmoticons, bool useFontEffects, const QString& chatFormat )
{
  useContactFont_ = useContactFont;
  useEmoticons_ = useEmoticons;
  useFontEffects_ = useFontEffects;
  chatFormat_ = chatFormat;
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setChatInformation." << endl;
#endif
}



// Set chat logging information
void Account::setChatLoggingInformation( bool saveChats, const QString& saveChatPath, const int& directoryStructure )
{
  saveChats_ = saveChats;
  saveChatPath_ = saveChatPath;
  savedChatDirectoryStructure_ = directoryStructure;
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setChatLoggingInformation." << endl;
#endif
}



// Set email information
void Account::setEmailInformation( bool useHotmail, const QString& emailCommand, bool showEmail, bool showOtherFolders )
{
  useHotmail_ = useHotmail;
  emailCommand_ = emailCommand;
  showEmail_ = showEmail;
  showOtherFolders_ = showOtherFolders;

  emit changedEmailDisplaySettings();
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setEmailInformation." << endl;
#endif
}



// Update the email-supported state (protected method)
void Account::setEmailSupported(bool emailSupported)
{
  emailSupported_ = emailSupported; 
}



// Set the font
void Account::setFont( const QFont& font )
{
  font_ = font;
  emit changedFontSettings();
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setFont." << endl;
#endif
}



// Set the font color
void Account::setFontColor( const QString& fontColor )
{
  fontColor_ = fontColor;
  emit changedFontSettings();
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setFontColor." << endl;
#endif
}




// Set the fonts and font colors
void Account::setFontInformation( const QFont& font, const QString &fontColor, const QFont& contactFont, const QString &contactFontColor)
{
  font_ = font;
  fontColor_ = fontColor;
  contactFont_ = contactFont;
  contactFontColor_ = contactFontColor;
  emit changedFontSettings();
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setFontInformation." << endl;
#endif
}



// Change the friendly name
void Account::setFriendlyName(const QString& newName)
{
  if ( ( newName != "" ) && ( newName != friendlyName_ ) )
  {
    friendlyName_ = newName;
    emit changedFriendlyName();
    dirty_ = true;
  }
}



// Set basic account data
void Account::setLoginInformation( QString handle, QString friendlyName, QString password )
{
#ifdef KMESSTEST
  ASSERT( handle.contains("@") );
  ASSERT( handle != "" );
#endif
  handle_ = handle;
  friendlyName_ = friendlyName;
  emit changedFriendlyName();
  if ( password != "" )
  {
    password_ = password;
  }
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setLoginInformation." << endl;
#endif
}



// Set proxy information
void Account::setProxyInformation( bool useproxy, const int& proxytype,const QString& proxyserver, const int& proxyport,const QString& proxyuid, const QString& password )
{
  useProxy_      = useproxy;
  proxyServer_   = proxyserver;
  proxyPort_     = proxyport;
  proxyUID_      = proxyuid;
  proxyPassword_ = password;
  proxyType_     = (PROXYTYPE)proxytype;
  dirty_         = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setProxyInformation." << endl;
#endif
}



// Set whether or not allowed contacts are shown.
void Account::setShowAllowedContacts( bool showAllowedContacts )
{
  showAllowedContacts_ = showAllowedContacts;
  emit changedDisplaySettings();
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setShowAllowedContacts." << endl;
#endif
}




// Set whether contacts are shown by group or by online/offline.
void Account::setShowContactsByGroup( bool showContactsByGroup )
{
  if ( showContactsByGroup != showContactsByGroup_ )
  {
    showContactsByGroup_ = showContactsByGroup;
    emit changedViewMode();
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account Setting 'dirty' to true - setShowContactsByGroup." << endl;
#endif
  }
}



// Set whether offline contacts should be shown.
void Account::setShowOfflineContacts( bool showOfflineContacts )
{
  showOfflineContacts_ = showOfflineContacts;
  emit changedDisplaySettings();
  dirty_ = true;
}



// Set whether or not removed (reverse) contacts are shown.
void Account::setShowRemovedContacts( bool showRemovedContacts )
{
  showRemovedContacts_ = showRemovedContacts;
  emit changedDisplaySettings();
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setShowRemovedContacts." << endl;
#endif
}



// Set whether or not the sidebar should be shown in chats
void Account::setShowSidebar( bool showSidebar )
{
  if ( showSidebar_ != showSidebar )
  {
    showSidebar_ = showSidebar;
    dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
    kdDebug() << "Account Setting 'dirty' to true - setShowSidebar." << endl;
#endif
  }
}



// Set whether or not to start in "hidden" mode
void Account::setStartInvisible( bool startInvisible )
{
  startInvisible_ = startInvisible;
}



// Set whether or not to login with this account.
void Account::setUseAutologin( bool useAutologin)
{
  useAutologin_ = useAutologin;
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setUseAutologin." << endl;
#endif
}



// Set whether or not to show emoticons in chats.
void Account::setUseEmoticons( bool useEmoticons)
{
  useEmoticons_ = useEmoticons;
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account Setting 'dirty' to true - setUseEmoticons." << endl;
#endif
}



// Set whether or not to send our display image
void Account::setShowImage( bool showImage )
{
  showImage_ = showImage;
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account: Setting 'dirty' to true - setShowImage." << endl;
#endif
}


void Account::setCustomImage( bool customImage )
{
  customImage_ = customImage;
  dirty_ = true;
#ifdef KMESSDEBUG_ACCOUNT_DIRTY
  kdDebug() << "Account: Setting 'dirty' to true - setCustomImage." << endl;
#endif
}



// Creates the MsnObject only if the user wants to show a display picture
void Account::updateMsnObject()
{
  QString oldMsnString;
  QString newMsnString;

  if( msnObject_ )
  {
    oldMsnString = msnObject_->objectString();
    delete msnObject_;
    msnObject_ = 0;
  }

#ifdef KMESSDEBUG_ACCOUNT_GENERAL
  kdDebug() << "Creating MSN object. Show Image: " << showImage_ << endl;
#endif

  if ( showImage_ )
  {
#ifdef KMESSDEBUG_ACCOUNT_GENERAL
    kdDebug() << "Image source: " << getImagePath() << endl;
#endif

    QFile iFile( getImagePath() );
    if ( iFile.open( IO_ReadOnly ) )
    {
      QByteArray data = iFile.readAll();
      iFile.close();

      msnObject_   = new MsnObject( handle_, "TFR2C.tmp", "", MsnObject::DISPLAYPIC, data );
      newMsnString = msnObject_->objectString();
    }
    else
    {
#ifdef KMESSDEBUG_ACCOUNT_GENERAL
      kdDebug() << "Image could not be loaded: " << getImagePath() << endl;
#endif

      msnObject_ = 0;
    }
  }

  // Detect if there was a change.
  if(oldMsnString != newMsnString)
  {
#ifdef KMESSDEBUG_ACCOUNT_GENERAL
    kdDebug() << "MsnObject: " << newMsnString << endl;
#endif
    emit changedMsnObject();
  }
}


#include "account.moc"
