#
# Generates C code to serialize/deserialize structs as a binary stream of data
#

#
# Serialized data will have the following format:
# 
# <root_struct>::= <name_string><values>
# <values>::= <value>...<value>
# <value>::= 
#    <int_value>
#    <length><string>
#    <count><value>...<value>
#
# Order of elements is the order that appears in the struct.

from generator import *




def get_header(orig_file):
    text="""/* Automatically generated by autocode.py */

#include "%s"
#include "myx_ser_aux_functions.h"

#include <glib.h>

"""%orig_file
    return text


class C_SerializerCommon:
    def __init__(self, group):
        self._group= group
        self._indentation="  "
        
    def indent(self, s, depth=1):
        return self._indentation*depth+s

    def get_entity_name(self, variable):
        return variable["name"]
    
    def make_subst_dict(self, variable, for_array, extra_dict={}):
        d={
        "group_name":self._group["name"],
        "entity":self.get_entity_name(variable),
        "content":["obj->%(name)s","obj->%(name)s[i]"][for_array!=0]%variable
        }
        d.update(extra_dict)
        return d

    def generate_code(self):
        code= self.generate()
        head= "\n".join(self.get_head())+"\n"
        body= code+"\n"
        return head, body


    
class C_SerializerGenerator(OutputGenerator, C_SerializerCommon):
    def __init__(self, group, others):
        OutputGenerator.__init__(self, group, others)
        C_SerializerCommon.__init__(self, group)

    def process_int_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        return ['if ((r=ser_int(fd, %(content)s)) < 0) return r;'%d]

    def process_str_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        return ['if ((r=ser_string(fd, %(content)s)) < 0) return r;'%d]

    def process_struct_variable(self, variable, depth, for_array=0):
        d= {
        "group_name":self._group["name"],
        "entity":self.find_struct_by_type(variable["type"])["entity"],
        "content":["obj->%(name)s","obj->%(name)s+i"][for_array]%variable
        }
        if "ref:parent" not in variable["options"]:
            lines= [
            'if ((r=ser_%(group_name)s_%(entity)s(fd, %(content)s)) < 0) return r;'%d
            ]
        else:
            lines=[]
        return lines

    def process_custom_variable(self, variable, depth, for_array=0):
        raise NotImplementedError
        return []

    def process_variable(self, variable, depth, for_array=0):
        if self.is_enum(variable["type"]):
            lines= self.process_int_variable(variable, 1, for_array)
        else:
            lines= OutputGenerator.process_variable(self, variable, 0, for_array)
        ll= []
        for l in lines:
            ll.append(self.indent(l, depth))
        return ll

    def process_array(self, count, array, depth):
        lines=[
        self.indent("if ((r=ser_int(fd, obj->%(name)s)) < 0) return r;"%count, depth),
        self.indent("for (i= 0; i < obj->%(name)s; i++)"%count, depth),
        self.indent("{", depth)
        ]
        lines+= OutputGenerator.process_array(self, count, array, depth+1)
        lines+=[
        self.indent("}", depth)
        ]
        return lines


    def process_struct(self, struct, depth=0):
        d= {
        "group_name":self._group["name"],
        "entity":struct["entity"],
        "type":struct["typedef"]
        }

        static=""
        if "root" not in struct["options"]:
            static= "static "

        lines=[]
        lines+=[
        static+"int ser_%(group_name)s_%(entity)s(MYX_SFD *fd, %(type)s *obj)" % d,
        "{",
        self.indent("unsigned int i;", depth+1),
        self.indent("int r;", depth+1)
        ]
        
        if "root" in struct["options"]:
            lines+= [self.indent('if ((r= ser_string(fd, "%(entity)s")) < 0) return r;'%d)]

        aa=OutputGenerator.process_struct(self, struct, depth)
        lines+= aa
        lines+=[
        self.indent("return 0;"),
        "}"
        ]

        self._head.append(lines[0]+";")

        return "\n".join(lines)




class C_UnserializerGenerator(InputGenerator,C_SerializerCommon):
    def __init__(self, group, others):
        InputGenerator.__init__(self, group, others)
        C_SerializerCommon.__init__(self, group)

        self._root_function= []
        self._aux_functions= []


    def process_int_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        lines= ['if ((r= unser_int(fd, &(%(content)s))) < 0) return r;'%d]
        return lines
    
    def process_str_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        lines= ['if ((r= unser_string(fd, &(%(content)s))) < 0) return r;'%d]
        return lines

    def get_refparent_variable(self, struct):
        for s in struct["elements"]:
            if "ref:parent" in s["options"]:
                return s["name"]
        return None

    def process_struct_variable(self, variable, depth, for_array=0):
        type_struct=self.find_struct_by_type(variable["type"])
        d= {
        "group_name":self._group["name"],
        "entity":type_struct["entity"],
        "content":["obj->%(name)s","obj->%(name)s+i"][for_array]%variable
        }
        if "ref:parent" not in variable["options"]:
            lines= [
            'if ((r= unser_%(group_name)s_%(entity)s(fd, %(content)s)) < 0) return r;'%d
            ]
        else:
            lines= []
            
        # [*] check if this struct contains a variable that's a reference 
        # to this
        var= self.get_refparent_variable(type_struct)
        if var:
            d["refvar"]= var
            lines+=[
            "(%(content)s)->%(refvar)s= obj;"%d
            ]
            
        return lines

    def process_custom_variable(self, variable, depth, for_array=0):
        raise NotImplementedError
        return []

    def process_variable(self, variable, depth, for_array=0):
        if self.is_enum(variable["type"]):
            lines= self.process_int_variable(variable, 1, for_array)
        else:
            lines= InputGenerator.process_variable(self, variable, 0, for_array)

        if lines:
            ll= []
            for l in lines:
                ll.append(self.indent(l, depth))
            return ll
        else:
            return []

    def process_array(self, count, array, depth):
        d=array.copy()
        d["count_name"]= count["name"]
        d["base_type"]= self.array_variable_to_plain(array)["type"]
        lines=[
        self.indent("if ((r= unser_int(fd, &(obj->%(name)s))) < 0) return r;"%count, depth),
        self.indent("obj->%(name)s= g_new0(%(base_type)s,obj->%(count_name)s);"%d, depth),
        self.indent("for (i= 0; i < obj->%(count_name)s; i++)"%d, depth),
        self.indent("{", depth)
        ]
        lines+=InputGenerator.process_array(self, count, array, depth+1)
        lines+=[self.indent("}", depth)]
        return lines

    def process_struct(self, struct, depth=0): # override
        d= {
        "group_name":self._group["name"],
        "entity":struct["entity"],
        "type":struct["typedef"]
        }

        static=""
        if "root" not in struct["options"]:
            static= "static "

        lines=[]
        lines+=[
        static+"int unser_%(group_name)s_%(entity)s(MYX_SFD *fd, %(type)s *obj)" % d,
        "{",
        self.indent("int r;", depth+1)
        ]
        self._head.append(lines[0]+";")

        arrays, variables= self.split_arrays(struct)

        if arrays:
            lines.append(self.indent("unsigned int i;", depth+1))
        
        prop_flag=0
        for elem in variables:
            lines+= self.process_variable(elem, depth+1)
        for count,elem in arrays:
            lines+= self.process_array(count, elem, depth+1)

        lines+=[
        self.indent("return 0;", depth+1),
        "}"
        ]

        tmp= "\n".join(lines)

        return tmp
