/* quest.cc - Command line search tool using Xapian::QueryParser.
 *
 * Copyright (C) 2004,2005,2006,2007,2008,2009 Olly Betts
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include <config.h>

#include <xapian.h>

#include <stdlib.h>

#include <iostream>

#include "gnu_getopt.h"

using namespace std;

#define PROG_NAME "quest"
#define PROG_DESC "Xapian command line search tool"

// Stopwords:
static const char * sw[] = {
    "a", "about", "an", "and", "are", "as", "at",
    "be", "by",
    "en",
    "for", "from",
    "how",
    "i", "in", "is", "it",
    "of", "on", "or",
    "that", "the", "this", "to",
    "was", "what", "when", "where", "which", "who", "why", "will", "with"
};

static void show_usage() {
    cout << "Usage: "PROG_NAME" [OPTIONS] 'QUERY'\n"
"NB: QUERY should be quoted to protect it from the shell.\n\n"
"Options:\n"
"  -d, --db=DIRECTORY  database to search (multiple databases may be specified)\n"
"  -m, --msize=MSIZE   maximum number of matches to return\n"
"  -s, --stemmer=LANG  set the stemming language, the default is 'english'\n"
"                      (pass 'none' to disable stemming)\n"
"  -h, --help          display this help and exit\n"
"  -v, --version       output version information and exit\n";
}

int
main(int argc, char **argv)
{
    const char * opts = "d:m:s:hv";
    static const struct option long_opts[] = {
	{ "db",		required_argument, 0, 'd' },
	{ "msize",	required_argument, 0, 'm' },
	{ "stemmer",	required_argument, 0, 's' },
	{ "help",	no_argument, 0, 'h' },
	{ "version",	no_argument, 0, 'v' },
	{ NULL,		0, 0, 0}
    };

    Xapian::SimpleStopper mystopper(sw, sw + sizeof(sw) / sizeof(sw[0]));
    Xapian::Stem stemmer("english");
    int msize = 10;

    bool have_database = false;

    try {
	Xapian::Database db;

	int c;
	while ((c = gnu_getopt_long(argc, argv, opts, long_opts, 0)) != -1) {
	    switch (c) {
		case 'm':
		    msize = atoi(optarg);
		    break;
		case 'd':
		    db.add_database(Xapian::Database(optarg));
		    have_database = true;
		    break;
		case 's':
		    try {
			stemmer = Xapian::Stem(optarg);
		    } catch (const Xapian::InvalidArgumentError &) {
			cerr << "Unknown stemming language '" << optarg
			     << "'.\n"
				"Available language names are: "
			     << Xapian::Stem::get_available_languages() << endl;
			exit(1);
		    }
		    break;
		case 'v':
		    cout << PROG_NAME" - "PACKAGE_STRING << endl;
		    exit(0);
		case 'h':
		    cout << PROG_NAME" - "PROG_DESC"\n\n";
		    show_usage();
		    exit(0);
		case ':': // missing parameter
		case '?': // unknown option
		    show_usage();
		    exit(1);
	    }
	}

	if (argc - optind != 1) {
	    show_usage();
	    exit(1);
	}

	Xapian::QueryParser parser;
	parser.set_database(db);
	parser.set_default_op(Xapian::Query::OP_OR);
	parser.set_stemmer(stemmer);
	parser.set_stemming_strategy(Xapian::QueryParser::STEM_SOME);
	parser.set_stopper(&mystopper);

	Xapian::Query query = parser.parse_query(argv[optind]);
	cout << "Query: " << query.get_description() << endl;

	if (!have_database) {
	    cout << "No database specified so not running the query." << endl;
	    exit(0);
	}

	Xapian::Enquire enquire(db);
	enquire.set_query(query);

	Xapian::MSet mset = enquire.get_mset(0, msize);

	cout << "MSet:" << endl;
	for (Xapian::MSetIterator i = mset.begin(); i != mset.end(); i++) {
	    Xapian::Document doc = i.get_document();
	    string data = doc.get_data();
	    cout << *i << " [" << i.get_percent() << "%]\n" << data << "\n";
	}
	cout << flush;
    } catch (const Xapian::QueryParserError & e) {
	cout << "Couldn't parse query: " << e.get_msg() << endl;
	exit(1);
    } catch (const Xapian::Error & err) {
	cout << err.get_description() << endl;
	exit(1);
    }
}
