/*
 *  Copyright (C) 2008-2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "solar_panel.h"

#define COMP_(x)	solar_panel_ ## x

#define POWER	0.04 /* W */

#define FREQ	128

struct cpssp {
	/* Config */
	double power;

	/* Ports */
	struct sig_std_logic *port_current;

	/* State */
	double output;
	double remaining;

	unsigned long long time;
};

static void
COMP_(light_set)(
	void *_cpssp,
	unsigned int x,
	unsigned int y,
	uint8_t r,
	uint8_t g,
	uint8_t b
)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->output = (r + g + b) * cpssp->power / (255 + 255 + 255);
}

static void
COMP_(event)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->remaining += cpssp->output / FREQ;
	if (cpssp->remaining < 0) {
		sig_std_logic_set(cpssp->port_current, cpssp, SIG_STD_LOGIC_0);
	} else {
		unsigned int mV;
		unsigned int mA;

		mV = 2000;
		if (cpssp->output / mV < 1000.0) {
			mA = (int) (cpssp->output / mV);
		} else {
			mA = 1000;
		}
		sig_std_logic_set(cpssp->port_current, cpssp, SIG_COMB(mV, mA));

		cpssp->remaining -= sig_std_logic_supply(cpssp->port_current, cpssp);
		if (0 < cpssp->remaining) {
			cpssp->remaining = 0;
		}
	}

	cpssp->time += TIME_HZ / FREQ;
	time_call_at(cpssp->time, COMP_(event), cpssp);
}

void *
COMP_(create)(
	const char *name,
	const char *power,
        struct sig_manage *port_manage,
	struct sig_opt_rgb *port_light,
	struct sig_std_logic *port_current
)
{
	static const struct sig_opt_rgb_funcs light_funcs = {
		.pixel_set = COMP_(light_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	if (power) {
		cpssp->power = atof(power) * 1000.0;
	} else {
		cpssp->power = POWER;
	}

	/* Out */
	cpssp->port_current = port_current;
	sig_std_logic_connect_out(port_current, cpssp, SIG_STD_LOGIC_0);

	/* In */
	sig_opt_rgb_connect(port_light, cpssp, &light_funcs);

	/* Time */
	cpssp->time = 0;
	time_call_at(cpssp->time, COMP_(event), cpssp);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
