/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <errno.h>
#include <inttypes.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue.h"
#include "system_node_tree.h"
#include "string_tools.h"

#include "inspector.h"

volatile int insp_stopped = 0;
inspector_context *private_inspector_ctx;

#define MAX_BREAKPOINTS	128
#define MAX_CONDITIONS 128
#define LITERAL_SIZE (MAX_CONDITIONS * sizeof(uint64_t))

typedef int(*inspector_test_condition_t)(void*,void*);

typedef struct {
	inspector_test_condition_t test_cbs[MAX_CONDITIONS];
	void *as[MAX_CONDITIONS];
	void *bs[MAX_CONDITIONS];
	system_node *used_nodes[MAX_CONDITIONS*2];
	system_node_field *used_fields[MAX_CONDITIONS*2];
	uint16_t n_used_nodes;
	uint16_t n_used_fields;
	uint8_t literal_data[LITERAL_SIZE];
	uint8_t n_conditions;
} breakpoint_data;

typedef struct {
	breakpoint_data data;
	char *text;
	uint32_t id;
	uint32_t count;
	uint8_t enabled;
} counter_breakpoint;

typedef struct {
	breakpoint_data data;
	char *text;
	uint32_t id;
	uint8_t enabled;
} trigger_breakpoint;

typedef enum {
	SUCCESS = 0,
	GENERIC_ERROR,
	SYNTAX_ERROR,
	NO_SUCH_COUNTER,
	NO_SUCH_FIELD,
	ILLEGAL_COMPARATOR,
	ILLEGAL_CONJUNCTION,
	LIT_COMPARE_LIT,
	SIGN_COMPARE_UNSIGN,
	BAD_ALLOC_LITERAL,
	LITAREAL_OVERFLOW,
	INVALID_COMMAND,
} insp_result;

static const char* error_msg(insp_result error)
{
	switch (error) {
	case SUCCESS:
		return "success";
	case GENERIC_ERROR:
		return "Generic error";
	case SYNTAX_ERROR:
		return "Syntax error";
	case NO_SUCH_COUNTER:
		return "No such counter";
	case NO_SUCH_FIELD:
		return "No such field";
	case ILLEGAL_COMPARATOR:
		return "Illegal comparator";
	case ILLEGAL_CONJUNCTION:
		return "Illegal conjunction";
	case LIT_COMPARE_LIT:
		return "Comparing two literals is not allowed";
	case SIGN_COMPARE_UNSIGN:
		return "Comparison of signed and unsigned is not allowed";
	case BAD_ALLOC_LITERAL:
		return "No more space for literals left";
	case LITAREAL_OVERFLOW:
		return "Literal overflows compared type";
	default:
		break;
	}
	return "";
}

static inline int evaluate_counter(counter_breakpoint *bp, void *field_address)
{
	int ret = 1;
	int field_is_used = 0;
	unsigned i;
	if (!bp->enabled) {
		return 0;
	}
	for (i=0; i<bp->data.n_conditions; i++) {
		field_is_used |= bp->data.as[i] == field_address;
		field_is_used |= bp->data.bs[i] == field_address;
		ret &= bp->data.test_cbs[i](bp->data.as[i], bp->data.bs[i]);
		/* maybe faster to implement short cut semantic here ? */
	}
	ret &= field_is_used;
	if (ret) {
		bp->count++;
	}
	return ret;
}

static inline int evaluate_trigger(trigger_breakpoint *bp)
{
	int ret = 1;
	unsigned i;
	if (!bp->enabled) {
		return 0;
	}
	for (i=0; i<bp->data.n_conditions; i++) {
		ret &= bp->data.test_cbs[i](bp->data.as[i], bp->data.bs[i]);
		/* maybe faster to implement short cut semantic here ? */
	}
	return ret;
}

struct inspector_context_t {
	console_interface console;
	tree_node *current_node;
	uint32_t n_cbreakpoints;
	uint32_t n_tbreakpoints;
	counter_breakpoint cbreakpoints[MAX_BREAKPOINTS];
	trigger_breakpoint tbreakpoints[MAX_BREAKPOINTS];
	uint32_t next_cbp_id;
	uint32_t next_tbp_id;
};

typedef enum {
	INSP_CMD_NONE = 0,
	INSP_CMD_HELP = 1,
	INSP_CMD_PAUSE = 2,
	INSP_CMD_CONTINUE = 3,
	INSP_CMD_LIST = 4,
	INSP_CMD_CD = 5,
	INSP_CMD_STATE = 6,
	INSP_CMD_BREAK = 7,
	INSP_CMD_BREAK_SHOW = 8,
	INSP_CMD_BREAK_TRIGGER = 9,
	INSP_CMD_BREAK_COUNT = 10,
	INSP_CMD_BREAK_REMOVE = 11,
	INSP_CMD_BREAK_ENABLE = 12,
	INSP_CMD_BREAK_DISABLE = 13,
	INSP_CMD_BREAK_HELP = 14,
	INSP_CMD_STEP = 15,
	INSP_CMD_STEP_AND_STATE = 16,
} insp_cmd_id_t;


typedef struct {
	const char* string;
	int len;
	insp_cmd_id_t id;
} insp_command_t;

static insp_command_t commands[] = {
        {"",		0,	INSP_CMD_NONE},
        {"h",		1,	INSP_CMD_HELP},
        {"help",	4,	INSP_CMD_HELP},
        {"p",		1,	INSP_CMD_PAUSE},
        {"pause",	5,	INSP_CMD_PAUSE},
        {"c",		1,	INSP_CMD_CONTINUE},
        {"continue",	8,	INSP_CMD_CONTINUE},
        {"l",		1,	INSP_CMD_LIST},
        {"ls",		2,	INSP_CMD_LIST},
        {"list",	4,	INSP_CMD_LIST},
        {"cd",		2,	INSP_CMD_CD},
        {"state",	5,	INSP_CMD_STATE},
        {"b",		1,	INSP_CMD_BREAK},
        {"break",	5,	INSP_CMD_BREAK},

        {"show",	4,	INSP_CMD_BREAK_SHOW},
        {"count",	5,	INSP_CMD_BREAK_COUNT},
        {"trigger",	7,	INSP_CMD_BREAK_TRIGGER},
        {"del",		3,	INSP_CMD_BREAK_REMOVE},
        {"delete",	6,	INSP_CMD_BREAK_REMOVE},
        {"enable",	6,	INSP_CMD_BREAK_ENABLE},
        {"disable",	7,	INSP_CMD_BREAK_DISABLE},
        {"help",	4,	INSP_CMD_BREAK_HELP},

        {"s",		1,	INSP_CMD_STEP},
        {"step",	4,	INSP_CMD_STEP},
        {"ss",		2,	INSP_CMD_STEP_AND_STATE},
};

const char *INSP_USAGE_STRING = ""
                "Unknown command.\n"
                "Type h or help for a command list.\n";
const char *INSP_HELP_STRING = ""
                               "The following commands are available:\n"
                               "h|help\t\tPrint this help message.\n"
                               "p|pause\t\tPause simulation.\n"
                               "c|continue\tContinue simulation.\n"
                               "l|ls|list\t\tList child devices\n"
                               "cd\t\t\tChange current inspection device\n"
                               "state\t\tPrint state of current device\n"
                               "b|break\t\tManage breakpoints\n"
                               "";

const char *INSP_BREAK_HELP_STRING = ""
                                     "break help\t\tPrint this help message\n"
                                     "Subcommands:\n"
                                     "break show\t\tList all breakpoints\n"
                                     "break trigger\t\tAdd trigger breakpoint\n"
                                     "break count\t\tAdd counting breakpoint\n"
                                     "break del|delete\tRemove breakpoint\n"
                                     "break enable\t\tEnable breakpoint\n"
                                     "break disable\t\tDisable breakpoint\n"
                                     "\n"
                                     "Examples:\n"
                                     "\tbreak trigger eax != 4 && eip >= 0x8000\n"
                                     "\tbreak count ebx == ecx && sf == 1\n"
                                     "\tbreak trigger #0 > 10\n"
                                     "\tbreak disable bp1\n"
                                     "\tbreak disable #0\n"
                                     "";

static insp_cmd_id_t get_cmd_id(const char *string, int len)
{
	int i;
	const char *cmd_end = str_nchr(string, ' ', len);

	if (len == -1) {
		len = strlen(string);
	}
	if (cmd_end != NULL) {
		len = cmd_end - string;
	}
	for (i=0; i<sizeof(commands)/sizeof(insp_command_t); i++) {
		insp_command_t *cmd = &commands[i];
		if (len == cmd->len && strncmp(string, cmd->string, cmd->len) == 0) {
			/* found command entry */
			return cmd->id;
		}
	}
	return INSP_CMD_NONE;
}



static int is_used_by_breakpoint(inspector_context *ctx, void *addr)
{
	unsigned i;

	for (i = 0; i<ctx->n_cbreakpoints; i++) {
		unsigned j;
		counter_breakpoint *cbp = &ctx->cbreakpoints[i];
		for (j = 0; j<cbp->data.n_conditions; j++) {
			if (cbp->data.as[j] == addr || cbp->data.bs[j] == addr){
				return 1;
			}
		}
	}

	for (i = 0; i<ctx->n_tbreakpoints; i++) {
		unsigned j;
		trigger_breakpoint *tbp = &ctx->tbreakpoints[i];
		for (j = 0; j<tbp->data.n_conditions; j++) {
			if (tbp->data.as[j] == addr || tbp->data.bs[j] == addr){
				return 1;
			}
		}
	}
	return 0;
}

static void update_used_in_breakpoints(inspector_context *ctx)
{
	unsigned i;

	for (i = 0; i < ctx->n_cbreakpoints; i++) {
		unsigned j;
		counter_breakpoint *cbp = &ctx->cbreakpoints[i];
		if (!cbp->enabled) {
			continue;
		}
		for (j = 0; j<cbp->data.n_used_nodes; j++) {
			system_node *node = cbp->data.used_nodes[j];
			node->has_active_breakpoints = 1;
			if (node->set_inspection_cb != NULL) {
				node->set_inspection_cb(node->opaque, 1);
			}
		}
		for (j = 0; j<cbp->data.n_used_fields; j++) {
			cbp->data.used_fields[j]->is_used_in_breakpoints = 1;
		}
	}

	for (i = 0; i < ctx->n_tbreakpoints; i++) {
		unsigned j;
		trigger_breakpoint *tbp = &ctx->tbreakpoints[i];
		if (!tbp->enabled) {
			continue;
		}
		for (j = 0; j<tbp->data.n_used_nodes; j++) {
			system_node *node = tbp->data.used_nodes[j];
			node->has_active_breakpoints = 1;
			if (node->set_inspection_cb != NULL) {
				node->set_inspection_cb(node->opaque, 1);
			}
		}
		for (j = 0; j<tbp->data.n_used_fields; j++) {
			tbp->data.used_fields[j]->is_used_in_breakpoints = 1;
		}
	}
}

static counter_breakpoint*
find_counter(inspector_context *ctx, uint32_t id, uint32_t *index)
{
	unsigned i;

	for (i = 0; i<ctx->n_cbreakpoints; i++) {
		counter_breakpoint *cbp = &ctx->cbreakpoints[i];
		if (cbp->id == id) {
			if (index != NULL) {
				*index = i;
			}
			return cbp;
		}
	}
	return NULL;
}

static trigger_breakpoint*
find_trigger(inspector_context *ctx, uint32_t id, uint32_t *index)
{
	unsigned i;

	for (i = 0; i<ctx->n_tbreakpoints; i++) {
		trigger_breakpoint *tbp = &ctx->tbreakpoints[i];
		if (tbp->id == id) {
			if (index != NULL) {
				*index = i;
			}
			return tbp;
		}
	}
	return NULL;
}

typedef enum {
	STATE_BEGIN,
	STATE_COMPARATOR,
	STATE_CONJUNCTION,
	STATE_PARAM_A,
	STATE_PARAM_B,
} cmd_state_t;

typedef enum {
	INVALID,
	LITERAL,
	FIELD,
	COUNTER,
} param_type_t;

#define COMPARATOR_EQ 0
#define COMPARATOR_NEQ 1
#define COMPARATOR_LT 2
#define COMPARATOR_GT 3
#define COMPARATOR_LE 4
#define COMPARATOR_GE 5

static const data_type_t counter_type = DATA_TYPE_UINT32_T;


#include "lib/inspector_test_functions.inc"


static inline int is_digit(char c) {
	return '0' <= c && c <= '9';
}
static inline int is_letter(char c) {
	return ('A' <= c && c <= 'Z') || ('a' <= c && c <= 'z');
}
static inline int is_comparator(char c) {
	return c == '=' || c == '<' || c == '>' || c == '!';
}
static inline int is_c_variable(char c) {
	return is_letter(c) || is_digit(c) || c == '_';
}
static inline int is_conjunction(char c) {
	return c == '&';
}
static inline int is_hex(char c) {
	return is_digit(c) || ('a' <= c && c <= 'f') || ('A' <= c && c <= 'F');
}
static inline int is_x(char c) {
	return c == 'x' || c == 'X';
}

static int is_counter(const char *str, int len)
{
	if (len == -1) {
		len = strlen(str);
	}
	if (len < 2) {
		return 0;
	}
	if (str[0] != '#' || !is_digit(str[1])) {
		return 0;
	}
	len--;
	str++;
	while (len > 0) {
		if (!is_digit(*str)) {
			return 0;
		}
		len--;
		str++;
	}
	return 1;
}

static int is_literal(const char *str, int len)
{
	if (len == -1) {
		len = strlen(str);
	}
	if (len == 0) {
		return 0;
	}
	if (len == 1 && is_digit(str[0])) {
		return 1;
	}
	if (str[0] == '-' || (is_digit(str[0]) && is_digit(str[1]))) {
		/* decimal */
		len--;
		str++;
		while (len > 0) {
			if (!is_digit(*str)) {
				return 0;
			}
			len--;
			str++;
		}
		return 1;
	} else {
		/* hex */
		if (len < 3) {
			return 0;
		}
		if (str[0] != '0' || !is_x(str[1])) {
			return 0;
		}
		if (!is_hex(str[2])) {
			return 0;
		}
		len -= 3;
		str += 3;
		while (len > 0) {
			if (!is_hex(*str)) {
				return 0;
			}
			len--;
			str++;
		}
		return 1;
	}
}

static insp_result parse_ul(const char *str, int len, unsigned long *val)
{
	int old_errno = errno;
	char buf[(len == -1 ? strlen(str) : len) + 1];
	long long tmp;

	if (len == -1) {
		len = strlen(str);
	}
	if (len == 0) {
		return SYNTAX_ERROR;
	}

	strncpy(buf, str, len);
	buf[len] = '\0';

	if (!is_literal(str, len)) {
		return SYNTAX_ERROR;
	}

	errno = 0;
	tmp = strtoll(buf, NULL, 0);

	if (0 > tmp || tmp > ULONG_MAX || errno != 0) {
		errno = old_errno;
		return SYNTAX_ERROR;
	} else {
		errno = old_errno;
		*val = tmp;
		return SUCCESS;
	}
}

static insp_result
parse_literal(const char *str, int len, void *target, int size, int sign)
{
	insp_result ret = SUCCESS;
	int old_errno = errno;
	char buf[(len == -1 ? strlen(str) : len) + 1];

	if (len == -1) {
		len = strlen(str);
	}
	if (!is_literal(str, len)) {
		ret = SYNTAX_ERROR;
		goto fail;
	}

	strncpy(buf, str, len);
	buf[len] = '\0';
	errno = 0;

	if (sign) {
		long long int tmp;
		tmp = strtoll(buf, NULL, 10);
		if (errno != 0) {
			ret = SYNTAX_ERROR;
			goto fail;
		}
		switch (size) {
		case 1: {
			if (tmp > INT8_MAX || tmp < INT8_MIN) {
				ret = LITAREAL_OVERFLOW;
				goto fail;
			}
			int8_t tmp2 = tmp;
			memcpy(target, &tmp2, size);
			break;
		}
		case 2: {
			if (tmp > INT16_MAX || tmp < INT16_MIN) {
				ret = LITAREAL_OVERFLOW;
				goto fail;
			}
			int16_t tmp2 = tmp;
			memcpy(target, &tmp2, size);
			break;
		}
		case 4: {
			if (tmp > INT32_MAX || tmp < INT32_MIN) {
				ret = LITAREAL_OVERFLOW;
				goto fail;
			}
			int32_t tmp2 = tmp;
			memcpy(target, &tmp2, size);
			break;
		}
		case 8: {
			if (tmp > INT64_MAX || tmp < INT64_MIN) {
				ret = LITAREAL_OVERFLOW;
				goto fail;
			}
			int64_t tmp2 = tmp;
			memcpy(target, &tmp2, size);
			break;
		}
		default:
			assert(0);
		}
	} else {
		assert(str[0] != '-');
		unsigned long long int tmp;
		tmp = strtoull(buf, NULL, 0);
		if (errno != 0) {
			ret = SYNTAX_ERROR;
			goto fail;
		}
		switch (size) {
		case 1: {
			if (tmp > UINT8_MAX) {
				ret = LITAREAL_OVERFLOW;
				goto fail;
			}
			uint8_t tmp2 = tmp;
			memcpy(target, &tmp2, size);
			break;
		}
		case 2: {
			if (tmp > UINT16_MAX) {
				ret = LITAREAL_OVERFLOW;
				goto fail;
			}
			uint16_t tmp2 = tmp;
			memcpy(target, &tmp2, size);
			break;
		}
		case 4: {
			if (tmp > UINT32_MAX) {
				ret = LITAREAL_OVERFLOW;
				goto fail;
			}
			uint32_t tmp2 = tmp;
			memcpy(target, &tmp2, size);
			break;
		}
		case 8: {
			if (tmp > UINT64_MAX) {
				ret = LITAREAL_OVERFLOW;
				goto fail;
			}
			uint64_t tmp2 = tmp;
			memcpy(target, &tmp2, size);
			break;
		}
		default:
			assert(0);
		}
	}
fail:
	errno = old_errno;
	return ret;
}

static param_type_t param_get_type(inspector_context *ctx,
                                   const char *begin,
                                   const char *end)
{
	int len = end - begin;
	if (len < 1) {
		return INVALID;
	}
	if (is_counter(begin, len)) {
		return COUNTER;
	}
	if (is_literal(begin, len)) {
		return LITERAL;
	}
	if (begin[0] == '/' || is_letter(begin[0])) {
		return FIELD;
	}
	return INVALID;
}

static int parse_comparator(const char *str, int len)
{
	if (len == -1) {
		len = strlen(str);
	}
	if (len == 1) {
		if (str[0] == '<') {
			return COMPARATOR_LT;
		} else if (str[0] == '>') {
			return COMPARATOR_GT;
		}
	} else if (len == 2) {
		if (str[0] == '=' && str[1] == '=') {
			return COMPARATOR_EQ;
		} else if (str[0] == '!' && str[1] == '=') {
			return COMPARATOR_NEQ;
		} else if (str[0] == '<' && str[1] == '=') {
			return COMPARATOR_LE;
		} else if (str[0] == '>' && str[1] == '=') {
			return COMPARATOR_GE;
		}
	}
	return -1;
}

static void* counter_address(inspector_context *ctx, unsigned long id)
{
	unsigned i;

	if (id < 0) {
		return NULL;
	}

	for (i = 0; i<ctx->n_cbreakpoints; i++) {
		counter_breakpoint *cbp = &ctx->cbreakpoints[i];
		if (cbp->id == id) {
			return &cbp->count;
		}
	}
	return NULL;
}

static insp_result
parse_breakpoint3(inspector_context *ctx, const char *cmd,
                  int len, breakpoint_data *bp)
{
	uint32_t literal_offset = 0;
	insp_result valid = SYNTAX_ERROR;
	cmd_state_t state = STATE_PARAM_A;
	const char *current = cmd;

	const char *param_a_begin = NULL;
	const char *param_a_end = NULL;
	const char *param_b_begin = NULL;
	const char *param_b_end = NULL;

	const char *comparator_begin = NULL;
	const char *comparator_end = NULL;

	bp->n_conditions = 0;

	while (*cmd != '\0') {
		valid = SYNTAX_ERROR;
		char c = *cmd;

		if (state == STATE_PARAM_A && is_comparator(c)) {
			param_a_begin = current;
			param_a_end = cmd;
			current = cmd;
			state = STATE_COMPARATOR;
		} else if (state == STATE_COMPARATOR && !is_comparator(c)) {
			comparator_begin = current;
			comparator_end = cmd;
			current = cmd;
			state = STATE_PARAM_B;
		} else if (state == STATE_PARAM_B && is_conjunction(c)) {
			param_b_begin = current;
			param_b_end = cmd;
			current = cmd;
			state = STATE_CONJUNCTION;
		} else if (state == STATE_CONJUNCTION && !is_conjunction(c)) {
			/* is valid conjunction? */
			if (cmd-current != 2
			    || current[0] != '&'
			    || current[1] != '&') {
				return ILLEGAL_CONJUNCTION;
			}
			current = cmd;
			state = STATE_PARAM_A;
		}

		if (state == STATE_PARAM_B && cmd[1] == '\0') {
			param_b_begin = current;
			param_b_end = cmd + 1;
			current = cmd;
		}

		if (param_b_begin != NULL) {
			/*parse param b*/
			data_type_t data_type_a = DATA_TYPE_UINT8_T;
			data_type_t data_type_b = DATA_TYPE_UINT8_T;
			param_type_t param_type_a;
			param_type_t param_type_b;
			int signed_a = 0;
			int signed_b = 0;
			int unsigned_a = 0;
			int unsigned_b = 0;
			system_node_field *field_a = NULL;
			system_node_field *field_b = NULL;
			system_node *node_a = NULL;
			system_node *node_b = NULL;
			int comparator;

			current = cmd;
			state = STATE_CONJUNCTION;

			param_type_a = param_get_type(ctx,
			                              param_a_begin,
			                              param_a_end);
			param_type_b = param_get_type(ctx,
			                              param_b_begin,
			                              param_b_end);
			if (param_type_a == INVALID
			    || param_type_b == INVALID) {
				return SYNTAX_ERROR;
			}
			if (param_type_a == LITERAL
			    && param_type_b == LITERAL) {
				/* comparing two literals is not allowed */
				return LIT_COMPARE_LIT;
			}

			/* signed check */
			if (param_type_a == COUNTER) {
				unsigned_a = 1;
			} else if (param_type_a == FIELD) {
				field_a = system_find_field(tree_get_data(ctx->current_node), param_a_begin, param_a_end-param_a_begin, &node_a);
				if (field_a == NULL || node_a == NULL) {
					return NO_SUCH_FIELD;
				}
				signed_a = is_signed(field_a->type);
				unsigned_a = !signed_a;
			} else if (param_type_a == LITERAL) {
				if (param_a_begin[0] == '-') {
					signed_a = 1;
				}
			}

			if (param_type_b == COUNTER) {
				unsigned_b = 1;
			} else if (param_type_b == FIELD) {
				field_b = system_find_field(tree_get_data(ctx->current_node), param_b_begin, param_b_end-param_b_begin, &node_b);
				if (field_b == NULL || node_b == NULL) {
					return NO_SUCH_FIELD;
				}
				signed_b = is_signed(field_b->type);
				unsigned_b = !signed_b;
			} else if (param_type_b == LITERAL) {
				if (param_b_begin[0] == '-') {
					signed_b = 1;
				}
			}

			if ((signed_a && unsigned_b)
			    || (unsigned_a && signed_b)) {
				return SIGN_COMPARE_UNSIGN;
			}

			assert(param_type_a == COUNTER || param_type_a == FIELD || param_type_a == LITERAL);
			assert(param_type_b == COUNTER || param_type_b == FIELD || param_type_b == LITERAL);
			if (param_type_a == COUNTER) {
				data_type_a = counter_type;
			} else if (param_type_a == FIELD) {
				data_type_a = field_a->type;
			}
			if (param_type_b == COUNTER) {
				data_type_b = counter_type;
			} else if (param_type_b == FIELD) {
				data_type_b = field_b->type;
			}

			if (param_type_a == LITERAL) {
				data_type_a = data_type_b;
			}
			if (param_type_b == LITERAL) {
				data_type_b = data_type_a;
			}

			comparator = parse_comparator(comparator_begin,
			                              comparator_end
			                              - comparator_begin);
			if (comparator < 0) {
				return ILLEGAL_COMPARATOR;
			}

			if (bp != NULL) {
				uint32_t cond = bp->n_conditions;
				void *a = NULL;
				void *b = NULL;

				if (param_type_a == COUNTER) {
					unsigned long cnt = 0;
					insp_result error = parse_ul(param_a_begin +1,
					                     param_a_end
					                     - param_a_begin -1,
					                     &cnt);
					if (error != SUCCESS) {
						return error;
					}
					a = counter_address(ctx, cnt);
					if (a == NULL) {
						return NO_SUCH_COUNTER;
					}
				} else if (param_type_a == FIELD) {
					a = (uint8_t*)node_a->opaque + field_a->offset;
				} else if (param_type_a == LITERAL) {
					int size = get_word_size(data_type_b);
					insp_result error = SUCCESS;
					if (literal_offset + size > LITERAL_SIZE) {
						return BAD_ALLOC_LITERAL;
					}
					error = parse_literal(param_a_begin,
					                      param_a_end-param_a_begin,
					                      bp->literal_data + literal_offset,
					                      size, signed_a | signed_b);
					if (error != SUCCESS) {
						return error;
					}
					a = bp->literal_data + literal_offset;
					literal_offset += size;
				}

				if (param_type_b == COUNTER) {
					unsigned long cnt = 0;
					insp_result error = parse_ul(param_b_begin +1,
					                     param_b_end
					                     - param_b_begin -1,
					                     &cnt);
					if (error != SUCCESS) {
						return error;
					}
					b = counter_address(ctx, cnt);
					if (b == NULL) {
						return NO_SUCH_COUNTER;
					}
				} else if (param_type_b == FIELD) {
					b = (uint8_t*)node_b->opaque + field_b->offset;
				} else if (param_type_b == LITERAL) {
					int size = get_word_size(data_type_b);
					insp_result error = SUCCESS;
					if (literal_offset + size > LITERAL_SIZE) {
						return BAD_ALLOC_LITERAL;
					}
					error = parse_literal(param_b_begin,
					                      param_b_end-param_b_begin,
					                      bp->literal_data + literal_offset,
					                      size, signed_a | signed_b);
					if (error != SUCCESS) {
						return error;
					}
					b = bp->literal_data + literal_offset;
					literal_offset += size;
				}

				bp->as[cond] = a;
				bp->bs[cond] = b;
				bp->test_cbs[cond] = get_compare_func(
				                        data_type_a,
				                        data_type_b,
				                        comparator);
				bp->n_conditions++;

				if (node_a != NULL) {
					unsigned i = bp->n_used_nodes++;
					bp->used_nodes[i] = node_a;
				}
				if (field_a != NULL) {
					unsigned i = bp->n_used_fields++;
					bp->used_fields[i] = field_a;
				}
				if (node_b != NULL) {
					unsigned i = bp->n_used_nodes++;
					bp->used_nodes[i] = node_b;
				}
				if (field_b != NULL) {
					unsigned i = bp->n_used_fields++;
					bp->used_fields[i] = field_b;
				}
			}

			param_a_begin = NULL;
			param_a_end = NULL;
			param_b_begin = NULL;
			param_b_end = NULL;
			comparator_begin = NULL;
			comparator_end = NULL;
			valid = SUCCESS;
		}

		cmd++;
	}
	return valid;
}

static void console_insert_text_default(void *ctx, const char *text, int len) {}
static const char* console_get_prompt_text_default(void *ctx) {
	return ">";
}
static void console_pop_from_prompt_default(void *ctx) {}
static void console_push_to_prompt_default(void *ctx, const char *name, size_t len) {}

static void console_init(console_interface *console)
{
	/* TODO nullify callbacks*/
	console->ctx = NULL;
	console->insert_text_cb = &console_insert_text_default;
	console->get_prompt_text_cb = &console_get_prompt_text_default;
	console->pop_from_prompt_cb = &console_pop_from_prompt_default;
	console->push_to_prompt_cb = &console_push_to_prompt_default;
}

inspector_context* inspector_context_new(void)
{
	inspector_context *ctx;
	ctx = (inspector_context *)malloc(sizeof(inspector_context));
	if (ctx == NULL) {
		faum_log(FAUM_LOG_FATAL, "gui_gtk", __func__,
		        "Cannot alloc key_event_context, aborting.\n");
		assert(0);
	}
	console_init(&ctx->console);
	ctx->current_node = system_get_node_tree();
	assert(ctx->current_node != NULL);
	ctx->n_cbreakpoints = 0;
	ctx->n_tbreakpoints = 0;
	ctx->next_cbp_id = 0;
	ctx->next_tbp_id = 0;

	init_function_array();

	return ctx;
}

void inspector_set_console(inspector_context *ctx, console_interface *ci)
{
	memcpy(&ctx->console, ci, sizeof(console_interface));
}

console_interface* inspector_get_console(inspector_context *ctx)
{
	return &ctx->console;
}


static const char* inspector_sanatize_cmd(const char *string, int len)
{
	if (len == -1) {
		len = strlen(string);
	}
	while (len > 0 && string[0] == ' ') {
		string++;
		len--;
	}
	return string;
}


static int
format_breakpoint_argument(const char *cmd_in, int in_len, char *cmd_out)
{
	int out_len = 0;
	int i = 0;
	if (in_len == -1) {
		in_len = strlen(cmd_in);
	}
	for (i=0; i<in_len; i++) {
		char c = cmd_in[i];
		switch (c) {
		case ' ':
		case '\t':
			break;
		default:
			cmd_out[out_len++] = c;
		}
	}
	cmd_out[out_len] = '\0';
	return out_len;
}


#define BP_TYPE_INVALID -1
#define BP_TYPE_COUNTER 0
#define BP_TYPE_TRIGGER 1

#define PARSE_BP_INDEX(cmd_, id_, type_) do {\
	type_ = BP_TYPE_INVALID;\
	const char *cmd_end = str_nchr((cmd_) + 1, ' ', -1);\
	if (cmd_end == NULL) {\
	        console->insert_text_cb(console->ctx, "invalid argument\n", -1);\
	        break;\
	}\
	const char *arg = inspector_sanatize_cmd(cmd_end, -1);\
	if (0 == strncmp(arg, "#", 1)) {\
	        int error = parse_ul(arg+1, -1, &id_);\
	        if (error != SUCCESS) {\
	                console->insert_text_cb(console->ctx, "invalid argument\n", -1);\
	                break;\
	        }\
	        type_ = BP_TYPE_COUNTER;\
	} else if (0 == strncmp(arg, "bp", 2)) {\
	        int error = parse_ul(arg+2, -1, &id_);\
	        if (error != SUCCESS) {\
	                console->insert_text_cb(console->ctx, "invalid argument\n", -1);\
	                break;\
	        }\
	        type_ = BP_TYPE_TRIGGER;\
	} else {\
	        console->insert_text_cb(console->ctx, "invalid argument\n", -1);\
	        break;\
	}\
} while(0)

static insp_result
inspector_handle_break(inspector_context *ctx, const char *cmd, int len) {
	insp_cmd_id_t cmd_id;
	console_interface *console;

	cmd_id = get_cmd_id(cmd, len);
	console = inspector_get_console(ctx);

	switch(cmd_id) {
	case INSP_CMD_BREAK_SHOW: {
		unsigned i;

		console->insert_text_cb(console->ctx, "counters:\n", -1);
		for (i = 0; i<ctx->n_cbreakpoints; i++) {
			counter_breakpoint *cbp = &ctx->cbreakpoints[i];
			char buf[4096];
			buf[0] = '\0';
			snprintf(buf, 4096, "#%lu=%u\t%s%s\n",
			         (unsigned long)cbp->id, cbp->count,
			         cbp->enabled ? "" : "(disabled)\t", cbp->text);
			console->insert_text_cb(console->ctx, buf, -1);
		}

		console->insert_text_cb(console->ctx, "breakpoints:\n", -1);
		for (i = 0; i<ctx->n_tbreakpoints; i++) {
			trigger_breakpoint *tbp = &ctx->tbreakpoints[i];
			char buf[4096];
			buf[0] = '\0';
			snprintf(buf, 4096, "bp%lu:\t%s%s\n",
			         (unsigned long)tbp->id,
			         tbp->enabled ? "" : "(disabled)\t", tbp->text);
			console->insert_text_cb(console->ctx, buf, -1);
		}
		break;
	}
	case INSP_CMD_BREAK_TRIGGER: {
		trigger_breakpoint *tbp = NULL;
		int cmd_len = 0;
		insp_result parse_error = SUCCESS;
		int formatted_len = 0;
		if (ctx->n_tbreakpoints >= MAX_BREAKPOINTS) {
			console->insert_text_cb(console->ctx, "Maximum number of trigger breakpoints reached\n", -1);
			break;
		}
		const char *cmd_end = str_nchr(cmd + 1, ' ', -1);
		if (cmd_end == NULL) {
			console->insert_text_cb(console->ctx, "invalid argument\n", -1);\
			break;
		}
		/* format cmd string */
		cmd_len = strlen(cmd_end);
		char cmd_formatted[cmd_len+1];
		formatted_len = format_breakpoint_argument(cmd_end,
		                                           cmd_len,
		                                           cmd_formatted);

		/* parse string */
		tbp = &ctx->tbreakpoints[ctx->n_tbreakpoints];
		parse_error = parse_breakpoint3(ctx, cmd_formatted,
		                                formatted_len, &tbp->data);

		if (parse_error != SUCCESS) {
			console->insert_text_cb(console->ctx, error_msg(parse_error), -1);
			console->insert_text_cb(console->ctx, "\n", -1);
			return parse_error;
		}
		tbp->enabled = 1;
		/* copy string */
		tbp->text = strdup(cmd_formatted);
		tbp->id = ctx->next_tbp_id++;
		ctx->n_tbreakpoints++;

		update_used_in_breakpoints(ctx);
		break;
	}
	case INSP_CMD_BREAK_COUNT: {
		counter_breakpoint *cbp = NULL;
		int cmd_len = 0;
		insp_result parse_error = SUCCESS;
		int formatted_len = 0;
		if (ctx->n_cbreakpoints >= MAX_BREAKPOINTS) {
			console->insert_text_cb(console->ctx, "Maximum number of counter breakpoints reached\n", -1);
			break;
		}
		const char *cmd_end = str_nchr(cmd + 1, ' ', -1);
		if (cmd_end == NULL) {
			console->insert_text_cb(console->ctx, "invalid argument\n", -1);\
			break;
		}
		/* format cmd string */
		cmd_len = strlen(cmd_end);
		char cmd_formatted[cmd_len+1];
		formatted_len = format_breakpoint_argument(cmd_end,
		                                           cmd_len,
		                                           cmd_formatted);
		/* parse string */
		cbp = &ctx->cbreakpoints[ctx->n_cbreakpoints];
		parse_error = parse_breakpoint3(ctx, cmd_formatted,
		                                formatted_len, &cbp->data);
		if (parse_error != SUCCESS) {
			console->insert_text_cb(console->ctx, error_msg(parse_error), -1);
			console->insert_text_cb(console->ctx, "\n", -1);
			return parse_error;
		}
		cbp->enabled = 1;
		/* copy string */
		cbp->text = strdup(cmd_formatted);
		cbp->id = ctx->next_cbp_id++;
		cbp->count = 0;
		ctx->n_cbreakpoints++;

		update_used_in_breakpoints(ctx);
		break;
	}
	case INSP_CMD_BREAK_REMOVE: {
		unsigned long id;
		int type;

		PARSE_BP_INDEX(cmd, id, type);
		if (type == BP_TYPE_COUNTER) {
			unsigned i;
			uint32_t index = 0;
			counter_breakpoint *cbp = find_counter(ctx, id, &index);

			if (cbp == NULL) {
				console->insert_text_cb(console->ctx, "No such breakpoint.\n", -1);
				break;
			}

			if (is_used_by_breakpoint(ctx, &cbp->count)) {
				console->insert_text_cb(console->ctx, "This counter is used by other breakpoints or counters and cannot be deleted.\n", -1);
				break;

			}

			free(cbp->text);
			cbp->text = NULL;

			for (i = 0; i<cbp->data.n_used_nodes; i++) {
				system_node *node = cbp->data.used_nodes[i];
				node->has_active_breakpoints = 0;
				if (node->set_inspection_cb != NULL) {
					node->set_inspection_cb(node->opaque, 0);
				}
			}
			for (i = 0; i<cbp->data.n_used_fields; i++) {
				cbp->data.used_fields[i]->is_used_in_breakpoints = 0;
			}

			if (index != ctx->n_cbreakpoints - 1) {
				ctx->cbreakpoints[index] = ctx->cbreakpoints[ctx->n_cbreakpoints-1];
			}
			ctx->n_cbreakpoints--;
		} else if (type == BP_TYPE_TRIGGER) {
			unsigned i;
			uint32_t index = 0;
			trigger_breakpoint *tbp = find_trigger(ctx, id, &index);

			if (tbp == NULL) {
				console->insert_text_cb(console->ctx, "No such breakpoint.\n", -1);
				break;
			}

			free(tbp->text);
			tbp->text = NULL;

			for (i = 0; i<tbp->data.n_used_nodes; i++) {
				system_node *node = tbp->data.used_nodes[i];
				node->has_active_breakpoints = 0;
				if (node->set_inspection_cb != NULL) {
					node->set_inspection_cb(node->opaque, 0);
				}
			}
			for (i = 0; i<tbp->data.n_used_fields; i++) {
				tbp->data.used_fields[i]->is_used_in_breakpoints = 0;
			}

			if (index != ctx->n_tbreakpoints - 1) {
				ctx->tbreakpoints[index] = ctx->tbreakpoints[ctx->n_tbreakpoints-1];
			}
			ctx->n_tbreakpoints--;
		}

		update_used_in_breakpoints(ctx);
		console->insert_text_cb(console->ctx, "breakpoint deleted\n", -1);
		break;
	}
	case INSP_CMD_BREAK_ENABLE: {
		unsigned long id;
		int type;

		PARSE_BP_INDEX(cmd, id, type);
		if (type == BP_TYPE_COUNTER) {
			uint32_t index = 0;
			counter_breakpoint *cbp = find_counter(ctx, id, &index);

			if (cbp == NULL) {
				console->insert_text_cb(console->ctx, "No such breakpoint.\n", -1);
				break;
			}

			cbp->enabled = 1;
		} else if (type == BP_TYPE_TRIGGER) {
			uint32_t index = 0;
			trigger_breakpoint *tbp = find_trigger(ctx, id, &index);

			if (tbp == NULL) {
				console->insert_text_cb(console->ctx, "No such breakpoint.\n", -1);
				break;
			}

			tbp->enabled = 1;
		}

		update_used_in_breakpoints(ctx);
		console->insert_text_cb(console->ctx, "breakpoint enabled\n", -1);
		break;
	}
	case INSP_CMD_BREAK_DISABLE: {
		unsigned long id;
		int type;

		PARSE_BP_INDEX(cmd, id, type);
		if (type == BP_TYPE_COUNTER) {
			unsigned i;
			uint32_t index = 0;
			counter_breakpoint *cbp = find_counter(ctx, id, &index);

			if (cbp == NULL) {
				console->insert_text_cb(console->ctx, "No such breakpoint.\n", -1);
				break;
			}

			for (i = 0; i<cbp->data.n_used_nodes; i++) {
				system_node *node = cbp->data.used_nodes[i];
				node->has_active_breakpoints = 0;
				if (node->set_inspection_cb != NULL) {
					node->set_inspection_cb(node->opaque, 0);
				}
			}
			for (i = 0; i<cbp->data.n_used_fields; i++) {
				cbp->data.used_fields[i]->is_used_in_breakpoints = 0;
			}

			cbp->enabled = 0;
		} else if (type == BP_TYPE_TRIGGER) {
			unsigned i;
			uint32_t index = 0;
			trigger_breakpoint *tbp = find_trigger(ctx, id, &index);

			if (tbp == NULL) {
				console->insert_text_cb(console->ctx, "No such breakpoint.\n", -1);
				break;
			}

			for (i = 0; i<tbp->data.n_used_nodes; i++) {
				system_node *node = tbp->data.used_nodes[i];
				node->has_active_breakpoints = 0;
				if (node->set_inspection_cb != NULL) {
					node->set_inspection_cb(node->opaque, 0);
				}
			}
			for (i = 0; i<tbp->data.n_used_fields; i++) {
				tbp->data.used_fields[i]->is_used_in_breakpoints = 0;
			}

			tbp->enabled = 0;
		}

		update_used_in_breakpoints(ctx);
		console->insert_text_cb(console->ctx, "breakpoint disabled\n", -1);
		break;
	}
	case INSP_CMD_BREAK_HELP:
	default:
		console->insert_text_cb(console->ctx, INSP_BREAK_HELP_STRING, -1);
		break;
	}
	return SUCCESS;
}

static void
print_field(console_interface *console, system_node_field *field, void *opaque)
{
	char buf[4096];
	int off = 0;
	uint8_t *base_ptr = opaque;

	buf[0] = '\0';
	off += snprintf(buf + off, sizeof(buf)-off-1, "%s: ", field->name);

	switch (field->type) {
	case DATA_TYPE_UINT8_T:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%" PRIu8,
		                *((uint8_t*)(base_ptr+field->offset)));
		break;
	case DATA_TYPE_UINT16_T:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%" PRIu16,
		                *((uint16_t*)(base_ptr+field->offset)));
		break;
	case DATA_TYPE_UINT32_T:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%" PRIu32,
		                *((uint32_t*)(base_ptr+field->offset)));
		break;
	case DATA_TYPE_UINT64_T:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%" PRIu64,
		                *((uint64_t*)(base_ptr+field->offset)));
		break;
	case DATA_TYPE_INT8_T:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%" PRId8,
		                *((int8_t*)(base_ptr+field->offset)));
		break;
	case DATA_TYPE_INT16_T:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%" PRId16,
		                *((int16_t*)(base_ptr+field->offset)));
		break;
	case DATA_TYPE_INT32_T:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%" PRId32,
		                *((int32_t*)(base_ptr+field->offset)));
		break;
	case DATA_TYPE_INT64_T:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%" PRId64,
		                *((int64_t*)(base_ptr+field->offset)));
		break;
	case DATA_TYPE_INT:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%d",
		                *((int*)(base_ptr+field->offset)));
	case DATA_TYPE_UINT:
		off += snprintf(buf + off, sizeof(buf)-off-1, "%u",
		                *((unsigned int*)(base_ptr+field->offset)));
		break;
	default:
		off += snprintf(buf + off, sizeof(buf)-off-1, "(not implemented)");
		break;
	}

	off += snprintf(buf + off, sizeof(buf)-off-1, "\n");

	console->insert_text_cb(console->ctx, buf, -1);
}

void inspector_exec_command(inspector_context *ctx, const char* cmd_, int len)
{
	if (len == -1) {
		len = strlen(cmd_);
	}
	const char *cmd = inspector_sanatize_cmd(cmd_, len);
	insp_cmd_id_t cmd_id;
	console_interface *console;

	len -= cmd - cmd_;
	cmd_id = get_cmd_id(cmd, len);
	console = inspector_get_console(ctx);

	/* TODO: more commands */
	switch(cmd_id) {
	case INSP_CMD_NONE:
		/* no command */
		break;
	case INSP_CMD_HELP:
		console->insert_text_cb(console->ctx, INSP_HELP_STRING, -1);
		break;
	case INSP_CMD_PAUSE:
		insp_stopped = 1;
		break;
	case INSP_CMD_CONTINUE:
		insp_stopped = 0;
		break;
	case INSP_CMD_LIST: {
		lst_dnode *node_iter;
		lst *children;
		char buf[4096];
		size_t len = sizeof(buf);
		size_t off = 0;
		buf[len-1] = '\0';
		buf[off] = '\0';

		children = tree_get_children(ctx->current_node);

		lst_Foreach(children, node_iter) {
			tree_node *node = (tree_node *)node_iter->data;
			if (node == NULL) {
				continue;
			}
			system_node *sysnode = (system_node *)tree_get_data(node);
			if (sysnode == NULL) {
				continue;
			}
			off += snprintf(buf + off, len-off,
			                "node: %s\n", sysnode->name);
		}

		console->insert_text_cb(console->ctx, buf, -1);
		break;
	}
	case INSP_CMD_STATE: {
		int i;
		system_node *current = (system_node *)tree_get_data(ctx->current_node);

		for (i=0; i<current->num_fields; i++) {
			system_node_field *field = &current->fields[i];
			print_field(&ctx->console, field, current->opaque);
		}
		break;
	}
	case INSP_CMD_CD: {
		if (len < 4) {
			console->insert_text_cb(console->ctx, "cd: invalid arg\n", -1);
			break;
		}
		const char *arg = inspector_sanatize_cmd(cmd + 2, len - 2);
		int arg_len = len - (arg - cmd);
		if (arg_len == 2 && 0 == strncmp(arg, "..", arg_len)) {
			/* go up */
			tree_node *parent = tree_get_parent(ctx->current_node);
			if (parent != NULL) {
				ctx->current_node = parent;
				console->pop_from_prompt_cb(console->ctx);
			}
		} else {
			/* search for the child with arg as name */
			const char *arg_end = str_nchr(arg, ' ', arg_len);
			lst_dnode *node_iter;
			lst *children;

			if (arg_end != NULL) {
				arg_len = arg_end - arg;
			}

			children = tree_get_children(ctx->current_node);

			lst_Foreach(children, node_iter) {
				tree_node *node = (tree_node *)node_iter->data;
				if (node == NULL) {
					continue;
				}
				system_node *sysnode = (system_node *)tree_get_data(node);
				if (sysnode == NULL) {
					continue;
				}
				if (arg_len == strlen(sysnode->name)
				    && 0 == strncmp(arg, sysnode->name, arg_len)) {
					/* found child node */
					ctx->current_node = node;
					console->push_to_prompt_cb(console->ctx, sysnode->name, strlen(sysnode->name));
					break;
				}
			}
		}
		break;
	}
	case INSP_CMD_BREAK: {
		const char *cmd_end = str_nchr(cmd, ' ', len);
		if (cmd_end == NULL) {
			console->insert_text_cb(console->ctx, INSP_BREAK_HELP_STRING, -1);
			break;
		}
		const char *sub_cmd = inspector_sanatize_cmd(cmd_end, -1);
		inspector_handle_break(ctx, sub_cmd, -1);
		break;
	}
	case INSP_CMD_STEP: {
		system_node *current;

		if (!insp_stopped) {
			console->insert_text_cb(console->ctx,
			                        "step: FAUMachine is running\n",
			                        -1);
			break;
		}

		current = (system_node *)tree_get_data(ctx->current_node);
		if (current == NULL) {
			console->insert_text_cb(console->ctx, "Error: Invalid current component. Should not happen!\n", -1);
			break;
		}
		if (current->single_step_cb != NULL) {
			current->single_step_cb(current->opaque);
		} else {
			console->insert_text_cb(console->ctx, "Error: No step possible in this context.\n", -1);
			break;
		}
		break;
	}
	case INSP_CMD_STEP_AND_STATE: {
		inspector_exec_command(ctx, "step", -1);
		inspector_exec_command(ctx, "state", -1);
		console->insert_text_cb(console->ctx, "\n", -1);
		break;
	}
	default:
		console->insert_text_cb(console->ctx, INSP_USAGE_STRING, -1);
		break;
	}
}

void
inspector_evaluate_breaks(inspector_context *ctx,
                          system_node *node,
                          system_node_field *field)
{
	unsigned i;
	int breakpoint_reached = 0;

	/* evaluate all counter breakpoints */
	for (i=0; i<ctx->n_cbreakpoints; i++) {
		evaluate_counter(&ctx->cbreakpoints[i],
		                 (uint8_t*)node->opaque + field->offset);
	}
	/* evaluate all regular breakpoints */
	for (i=0; i<ctx->n_tbreakpoints; i++) {
		breakpoint_reached |= evaluate_trigger(&ctx->tbreakpoints[i]);
	}
	if (breakpoint_reached && !insp_stopped) {
		console_interface *console = inspector_get_console(ctx);
		const char *prompt = console->get_prompt_text_cb(console->ctx);
		console->insert_text_cb(console->ctx, "breakpoint reached\n", -1);
		console->insert_text_cb(console->ctx, prompt, -1);
		insp_stopped = 1;
		sched_abort_processes();
	}
}
