------------------------------------------------------------------------------
--                                  G P S                                   --
--                                                                          --
--                     Copyright (C) 2001-2014, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

--  UI extensions to GPS.Kernel.Modules
--
--  Contextual menus
--  ================
--
--   Here is a description of the sequence of events used to display contextual
--   menus in GPS:
--      - Each object that should have a contextual menu calls
--        Register_Contextual_Menu. The kernel will automatically setup
--        appropriate gtk callbacks.
--      - Whenever the user presses the right mouse button, the kernel will ask
--        the object to report the context in which the event occured (name of
--        selected file, selected project,...).
--      - Each of the registered module then has the opportunity to add entries
--        in the contextual menu, based on this context.
--      - The menu is displayed, and the callback for the selected menu item
--        will be called as usual.
--      - The menu is automatically destroyed, and the context freed, when the
--        action has finished executing.
--
--  Registering features
--  ====================
--
--   The behavior of the kernel and GPS itself can be modified extensively by
--   the modules, through a set of Register_* subprograms. This includes:
--      - Inserting new widgets in the MDI (either at startup or upon user
--        request)
--      - Adding new menus and toolbar icons
--      - Adding new contextual menu and new entries in existing menus
--      - Changing the default behavior of several standard functions, like
--        file edition, help file display, ... through Mime callbacks
--      - Adding new search contexts (see find_utils.ads in the vsearch module)
--      - Adding new predefined regular expressions in the search dialog
--      - Changing the way the current desktop is saved to disk and restored
--      - Changing what is displayed in tooltips in the editors
--      - Adding new attributes to projects, and the corresponding pages in the
--        project creation wizard or the project properties dialog.
--      - Adding new user-modifiable preferences (see gps-preferences.ads)
--      - Adding new supported languages (see language_handlers-gps.ads)
--        and the corresponding cross-referencing subprograms (same file)
--      - Each module can register new commands for the shell interpreter
--      - Adding key handlers, which have priority over other shortcuts

with GNAT.Strings;
with Gdk.Event;
with Glib.Action;
with Glib.Object;
with Glib.Values;
with Gtk.Application;
with Gtk.Handlers;
with Gtk.Menu;
with Gtk.Menu_Bar;
with Gtk.Menu_Item;
with Gtk.Target_List;
with Gtk.Toolbar;
with Gtk.Widget;
with Commands;             use Commands;
with Commands.Interactive; use Commands.Interactive;
with Interfaces.C.Strings;
with GPS.Kernel.Actions;   use GPS.Kernel.Actions;
with XML_Utils;

package GPS.Kernel.Modules.UI is

   function Get_Current_Module
     (Kernel : access Kernel_Handle_Record'Class) return Module_ID;
   --  Return the module the currently selected MDI child belongs to.
   --  null might be returned if there is either no selected child or GPS
   --  couldn't find its module

   -----------
   -- Types --
   -----------
   --  See also the types defined in gps-kernel.ads

   type GPS_Contextual_Menu_Record is new Gtk.Menu.Gtk_Menu_Record with record
      Kernel : access Kernel_Handle_Record'Class;
   end record;
   type GPS_Contextual_Menu is access all GPS_Contextual_Menu_Record'Class;

   package Context_Callback is new Gtk.Handlers.User_Callback
     (Glib.Object.GObject_Record, Selection_Context);

   --------------
   -- GActions --
   --------------
   --  This type is used to create a link between gtk+ actions (GAction) which
   --  are used for menus in a GtkApplication, and the actions defined by GPS,
   --  which provide more advanced features (multi-key bindings, automatic
   --  filters, icons,...)

   function New_G_Action
     (Kernel : not null access Kernel_Handle_Record'Class;
      Action : String) return Glib.Action.Gaction;
   --  Creates a new GAction that will execute the given GPS action

   ----------------------
   -- Contextual menus --
   ----------------------

   type Context_Factory is access procedure
     (Context      : in out Selection_Context;
      Kernel       : access Kernel_Handle_Record'Class;
      Event_Widget : access Gtk.Widget.Gtk_Widget_Record'Class;
      Object       : access Glib.Object.GObject_Record'Class;
      Event        : Gdk.Event.Gdk_Event;
      Menu         : Gtk.Menu.Gtk_Menu);
   --  This function should set the context associated with the contextual
   --  menu, when the mouse event Event happened on Widget.
   --  The mouse event occured in Event_Widget, and the contextual menu was
   --  registered for Object
   --  The object should also add its default entries into the menu, so that
   --  they always appear first in the menu. Note that the module will not be
   --  asked in the second step whether new entries should be added.
   --
   --  If null is returned, no contextual menu will be displayed.
   --
   --  The kernel is automatically set in the context.

   procedure Register_Contextual_Menu
     (Kernel          : access Kernel_Handle_Record'Class;
      Event_On_Widget : access Gtk.Widget.Gtk_Widget_Record'Class;
      Object          : access Glib.Object.GObject_Record'Class;
      ID              : Module_ID;
      Context_Func    : Context_Factory);
   --  Register that Widget should be associated with a contextual menu.
   --  Whenever a right-button click happens inside Event_On_Widget, then the
   --  following will happen:
   --     - the kernel detects the event, and creates an empty menu.
   --     - it asks Object, through Context_Func, the exact context for the
   --       menu (selected file, ....)
   --     - it then asks each of the registered modules whether it wants to
   --       add new items to the menu, and let it do so (through the
   --       Contextual_Menu_Handler provided in Register_Module)s
   --     - it then displays the menu
   --     - it finally cleans up the memory when the menu is hidden

   type Contextual_Menu_Label_Creator_Record is abstract tagged null record;
   type Contextual_Menu_Label_Creator is
     access all Contextual_Menu_Label_Creator_Record'Class;
   function Get_Label
     (Creator   : access Contextual_Menu_Label_Creator_Record;
      Context   : Selection_Context) return String is abstract;
   --  Create the name to use for a contextual menu.
   --  If this function returns the empty string, the menu will be filtered out

   type Custom_Expansion is access function
     (Context : Selection_Context) return String;
   --  Provide the custom expansion for %C when expanding a label. If the
   --  empty string is returned, the contextual entry will not be displayed.

   function Substitute_Label
     (Text    : String;
      Context : Selection_Context;
      Custom  : Custom_Expansion := null) return String;
   --  Substitute patterns like %e, %p,.. in Text.
   --  If some of the patterns could not be substituted, this function returns
   --  an empty string (so that the associated contextual menu does not appear)

   Default_Contextual_Group : constant := 0;
   procedure Register_Contextual_Menu
     (Kernel      : access Kernel_Handle_Record'Class;
      Name        : String;
      Action      : Action_Record_Access;
      Label       : String := "";
      Custom      : Custom_Expansion := null;
      Stock_Image : String := "";
      Ref_Item    : String := "";
      Add_Before  : Boolean := True;
      Group       : Integer := Default_Contextual_Group);
   --  Register a new contextual menu entry to be displayed.
   --  This menu will only be shown when the filter associated with the Action
   --  matches. The name used in the menu will be Label (or Name if label isn't
   --  specified), interpreted with the usual parameter substitution:
   --     %f => current file basename
   --     %d => current directory
   --     %p => current project name
   --     %l => current line
   --     %c => current columns
   --     %a => current category
   --     %e => current entity name
   --     %i => current importing project
   --     %s => current single-line selection (nothing if multiples lines are
   --           selected)
   --     %S => selection or expression info or var name
   --     %C => value returned by Custom (the menu will not appear if this
   --           returns the empty string or Custom is undefined)
   --  The label might contain a path to indicate submenus.
   --  Image will be added to the left of the contextual menu entry.
   --  Ref_Item is the name of another contextual menu (not a label), relative
   --  to which the menu should be placed. There is no garantee that the new
   --  entry will appear just before or just after that item, in particular if
   --  other entries had the same requirement.
   --
   --  Separators:
   --  If Action is null or the name of the menu item starts with '-' (for
   --  instance if Name = "/submenu/-my item") then a separator will be added
   --  to the contextual menu instead. It is added in a submenu if Label is not
   --  the empty string. It is good policy to specify a Ref_Item for a
   --  separator, since the separator will automatically be hidden if the
   --  Ref_Item itself is hidden
   --
   --  Groups:
   --  Group indicates the group of the entry. If Ref_Item is specified, this
   --  parameter is ignored. Otherwise, it groups items so that all items of
   --  the same group appear before all items with a greater group number.

   procedure Register_Contextual_Menu
     (Kernel      : access Kernel_Handle_Record'Class;
      Name        : String;
      Action      : Action_Record_Access;
      Label       : access Contextual_Menu_Label_Creator_Record'Class;
      Stock_Image : String := "";
      Ref_Item    : String := "";
      Add_Before  : Boolean := True;
      Group       : Integer := Default_Contextual_Group);
   --  Same as above, except the label of the menu is computed dynamically

   procedure Register_Contextual_Menu
     (Kernel            : access Kernel_Handle_Record'Class;
      Name              : String;
      Action            : Commands.Interactive.Interactive_Command_Access;
      Filter            : access Action_Filter_Record'Class := null;
      Enable_Filter     : access Action_Filter_Record'Class := null;
      Label             : access Contextual_Menu_Label_Creator_Record'Class;
      Stock_Image       : String := "";
      Ref_Item          : String := "";
      Add_Before        : Boolean := True;
      Group             : Integer := Default_Contextual_Group);
   --  Same as above, except the action to execute is defined internally.
   --  When the command is executed, the Context.Context field will be set to
   --  the current selection context, and Context.Event to the event that
   --  triggered the menu.
   --  Action doesn't need to Push_State/Pop_State, nor handle unexpected
   --  exceptions, since this is already done by its caller. This keeps the
   --  code shorter.
   --  Filter will act on the menu's visibility.
   --  Enable_Filter will act on its sensitivity.

   procedure Register_Contextual_Menu
     (Kernel            : access Kernel_Handle_Record'Class;
      Name              : String;
      Action         : Commands.Interactive.Interactive_Command_Access := null;
      Filter            : access Action_Filter_Record'Class := null;
      Enable_Filter     : access Action_Filter_Record'Class := null;
      Label             : String := "";
      Custom            : Custom_Expansion := null;
      Stock_Image       : String := "";
      Ref_Item          : String := "";
      Add_Before        : Boolean := True;
      Group             : Integer := Default_Contextual_Group);
   --  Same as above, but the menu title is a string where %p, %f,... are
   --  substituted.
   --  A separator is inserted if Action is null and the Filter matches.

   type Submenu_Factory_Record is abstract tagged null record;
   type Submenu_Factory is access all Submenu_Factory_Record'Class;
   procedure Append_To_Menu
     (Factory : access Submenu_Factory_Record;
      Object  : access Glib.Object.GObject_Record'Class;
      Context : Selection_Context;
      Menu    : access Gtk.Menu.Gtk_Menu_Record'Class) is abstract;
   --  Object is the object on which the contextual menu is displayed.
   --  New entries should be appended to Menu.

   procedure Register_Contextual_Submenu
     (Kernel            : access Kernel_Handle_Record'Class;
      Name              : String;
      Label             : String := "";
      Filter            : access Action_Filter_Record'Class := null;
      Enable_Filter     : access Action_Filter_Record'Class := null;
      Submenu           : Submenu_Factory := null;
      Ref_Item          : String := "";
      Add_Before        : Boolean := True;
      Group             : Integer := Default_Contextual_Group);
   --  Register a new submenu. Its contents can be computed dynamically by
   --  providing a Submenu callback. This can be left to null if all entries
   --  are added through Register_Contextual_Menu (in which case the call to
   --  Register_Contextual_Submenu can be used to position the parent menu
   --  where appropriate.
   --  Submenu is passed the submenu created for the item, so it doesn't need
   --  to create the submenu itself.

   function Emphasize (Name : String) return String;
   --  Parts of a contextual menu entry can be emphasized (name of entities
   --  for instance). This procedure should be used in this case, to provide
   --  consistent font and color for all emphasized words

   procedure Set_Contextual_Menu_Visible
     (Kernel  : access Kernel_Handle_Record'Class;
      Name    : String;
      Visible : Boolean);
   --  This procedure can be used to toggle the visibility of contextual menus.
   --  When a contextual menu was set as invisible, it will no longer appear.

   procedure Set_Contextual_Menu_Sensitivity
     (Kernel    : access Kernel_Handle_Record'Class;
      Name      : String;
      Sensitive : Boolean);
   --  Control whether the contextual menu entry is sensitive (ie "grayed-out")

   function Get_Registered_Contextual_Menus
     (Kernel : access Kernel_Handle_Record'Class)
      return GNAT.Strings.String_List_Access;
   --  Return the list of registered contextual menus. The returned array must
   --  be freed by the caller.

   procedure Create_Contextual_Menu
     (Kernel  : Kernel_Handle;
      Object  : Glib.Object.GObject;
      Context : Selection_Context;
      Menu    : in out Gtk.Menu.Gtk_Menu);
   --  Creates a menu from context and object.
   --  The Gtk_Menu must be created before calling this procedure.

   --------------
   -- Tooltips --
   --------------

   function Compute_Tooltip
     (Kernel  : access Kernel_Handle_Record'Class;
      Context : Selection_Context) return Gtk.Widget.Gtk_Widget;
   --  Given a context, pointing to e.g an entity, the kernel will ask
   --  each of the registered modules whether it wants to display a tooltip.
   --  The first module to return non-null will stop the process.
   --  If no module wants to display a tooltip, returns null.

   -----------
   -- Menus --
   -----------

   procedure Install_Menus
     (Kernel    : not null access Kernel_Handle_Record'Class;
      App       : not null access Gtk.Application.Gtk_Application_Record'Class;
      Description : GNATCOLL.VFS.Virtual_File;
      Menubar   : out Gtk.Menu_Bar.Gtk_Menu_Bar);
   --  Load an XML description of the menubar, and create it.

   procedure Start_Monitoring_Menus
     (Kernel      : not null access Kernel_Handle_Record'Class);
   --  Start monitoring the context changes to update the menu sensitivity.

   procedure Update_Menus_And_Buttons
     (Kernel  : not null access Kernel_Handle_Record'Class;
      Context : GPS.Kernel.Selection_Context := No_Context);
   --  Recompute the visibility and sensitivity of menus and toolbar buttons.
   --  This computation is asynchronous so that it doesn't block the user.

   procedure Register_Menu
     (Kernel      : access Kernel_Handle_Record'Class;
      Parent_Path : String;
      Item        : access Gtk.Menu_Item.Gtk_Menu_Item_Record'Class := null;
      Ref_Item    : String := "";
      Add_Before  : Boolean := True;
      Filter      : Action_Filter  := null;
      Menubar     : access Gtk.Menu_Bar.Gtk_Menu_Bar_Record'Class := null);
   --  Add new menu items to the menu bar, as a child of Parent_Path.
   --
   --  Parent_Path should have a form like "/main_main/submenu". Underscores
   --  are not used for mnemonics, and will be present in the final menu.
   --  Use String_Utils.Strip_Single_Underscores if needed.
   --
   --  Menus will be created if they don't exist.
   --  This is considered as an absolute path, as if it always started with
   --  a '/'.
   --
   --  Menubar will default to the main window's menubar.
   --
   --  Item might be null, in which case only the parent menu items are
   --  created, and Add_Before applies to the deepest one instead of Item.
   --
   --  The new item is inserted either:
   --    - before Ref_Item if the latter is not the empty string and Add_Before
   --      is true
   --    - after Ref_Item if the latter is not the empty string and Add_Before
   --      is false
   --    - at the end of the menu
   --
   --  To register a separator, do the following:
   --      Mitem : Gtk_Menu_Item;
   --      Gtk_New (Mitem);
   --      Register_Menu (Kernel, "/Parent_Path", Mitem);
   --
   --  The menu item will be active if Filter matches.

   procedure Register_Menu
     (Kernel        : not null access Kernel_Handle_Record'Class;
      Path          : String;
      Action        : String;
      Ref_Item      : String := "";
      Add_Before    : Boolean := True;
      Use_Mnemonics : Boolean := True);
   function Register_Menu
     (Kernel        : not null access Kernel_Handle_Record'Class;
      Path          : String;
      Action        : String;
      Ref_Item      : String := "";
      Add_Before    : Boolean := True;
      Optional      : Boolean := False;
      Use_Mnemonics : Boolean := True;
      Menubar       : access Gtk.Menu_Bar.Gtk_Menu_Bar_Record'Class := null)
      return Gtk.Menu_Item.Gtk_Menu_Item;
   --  Append a menu binding a GPS action. The action need not exist when the
   --  menu is created (but the menu will always be greyd out if the action
   --  does not exist).
   --  Accel_Key, Accel_Mods are looked up from the action.
   --  Filter is looked up from the action.
   --  The image is also looked up from the action.
   --
   --  When a menu is optional, it is hidden if its action does not exist.
   --  Otherwise, the menu is simply greyed out, but the menu is still visible.

   function Find_Menu_Item
     (Kernel : access Kernel_Handle_Record'Class;
      Path   : String) return Gtk.Menu_Item.Gtk_Menu_Item;
   --  Given an absolute path (see Register_Menu) for a menu item, return
   --  the underlying gtk menu item. Useful in particular to check or change
   --  the state of a menu item. Path is case insensitive

   procedure Update_Shortcut_Display
     (Kernel : access Kernel_Handle_Record'Class;
      Action : String);
   --  Update the shortcut for all menus associated with the action

   procedure Register_MDI_Menu
     (Kernel     : Kernel_Handle;
      Item_Name  : String;
      Accel_Path : String);
   --  Register in the GPS menu machinery the menus created by the GtkAda MDI

   procedure Execute_Menu
     (Kernel    : Kernel_Handle;
      Menu_Name : String);
   --  Execute immediately a menu. Menu_Name is the full path to the menu.

   ---------------------
   -- Toolbar buttons --
   ---------------------

   procedure Register_Button
     (Kernel   : not null access Kernel_Handle_Record'Class;
      Action   : String;
      Stock_Id : String := "";
      Toolbar  : access Gtk.Toolbar.Gtk_Toolbar_Record'Class := null;
      Position : Glib.Gint := -1;
      Hide     : Boolean := False);
   --  Register a button based on an action.
   --  The action need not be registered yet.
   --  Stock_Id overrides the action's default image, if specified.
   --  The button will be grayed out automatically whenever the action's filter
   --  indicate the action is not available in the current context.
   --
   --  If Hide is true, the button is hidden when the action does not apply
   --  to the context, instead of being grayed out.
   --
   --  The toolbar defaults to the global toolbar in GPS.
   --  The position can be computed with Get_Toolbar_Separator_Position.

   function Get_Toolbar_Section
     (Kernel  : not null access Kernel_Handle_Record'Class;
      Toolbar : access Gtk.Toolbar.Gtk_Toolbar_Record'Class := null;
      Section : String;
      Last    : Boolean := True) return Glib.Gint;
   --  Return the first or last item position in a given section of the
   --  toolbar. A toolbar starts on the first item after the separator with
   --  the name of the section, and ends just before the next separator.

   function Create_Toolbar
     (Kernel  : not null access Kernel_Handle_Record'Class;
      Id      : String)
      return Gtk.Toolbar.Gtk_Toolbar;
   --  Create a new toolbar with the given id.
   --  Its contents is read from the XML file in Install_Menus. Any button
   --  registered for it later on will be dynamically added to the toolbar.

   -------------------------
   -- Drag'n'drop support --
   -------------------------

   My_Target_Url    : constant Guint := 0;
   Target_Table_Url : constant Gtk.Target_List.Target_Entry_Array :=
     (1 => (Interfaces.C.Strings.New_String ("text/uri-list"),
            0, My_Target_Url));

   procedure Drag_Data_Received
     (Object : access Glib.Object.GObject_Record'Class;
      Args   : Glib.Values.GValues;
      Kernel : GPS.Kernel.Kernel_Handle);
   --  Handle text/uri-list drop events by loading the corresponding projects
   --  or files. Assume the selection data contains a string representing a LF
   --  or CR/LF separated list of files.

   -------------
   -- Markers --
   -------------

   function Create_Marker
     (Kernel : access Kernel_Handle_Record'Class;
      Load   : XML_Utils.Node_Ptr := null) return Location_Marker;
   --  Create a marker for the current module at the current location.
   --  Load is an XML node created through a call to Save
   --  (for a Location_Marker and is used to restore a marker from a previous
   --  session.
   --  null is returned if no Location_Marker could be created.

private

   type Action_Proxy is abstract tagged record
      Kernel   : access Kernel_Handle_Record'Class;
      Action   : GNAT.Strings.String_Access;

      Optional : Boolean;
      --  If True and the action is not found, the widget will be hidden.

      Hide     : Boolean;
      --  If true, the widget is hidden when the filter does not match.

      Looked_Up : access Action_Record;
      --  A field that must be used only to compare the current action with the
      --  one we previously looked up. Do not use to access the action itself,
      --  since this might be a dangling pointer if the action was
      --  unregistered. Use Lookup_Action instead.
   end record;
   --  Data required for all widgets that encapsulate an action.
   --  A widget must never store a direct Action_Record_Access, since the
   --  action might be unregistered at any point.
   --  This type also provides support for setting various properties of the
   --  widget based on the contents of the action.

   procedure Set_Active
     (Self   : in out Action_Proxy;
      Active : Boolean;
      Object : not null access Glib.Object.GObject_Record'Class) is null;
   --  Called whenever we recompute the status (enabled/disabled) of the
   --  action in the background.

end GPS.Kernel.Modules.UI;
