/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "FPSCounter.h"
#include "TimeFunction.h"

#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>

namespace libgltf
{

FPSCounter::FPSCounter(ShaderProgram* pShaderProgram)
    : pFont(new Font())
    , uiFPSProgram(0)
    , mLastTime(0.0)
    , mFrames(0)
    , mFPS(0)
{
    pFont->loadFont();
    init(pShaderProgram);
}

FPSCounter::~FPSCounter()
{
    if (NULL != pFont)
    {
        pFont->deleteFont();
        delete pFont;
    }
    glDeleteProgram(uiFPSProgram);
}

bool FPSCounter::loadFPSShader(ShaderProgram* pShaderProgram)
{
    uiFPSProgram = glCreateProgram();
    pShaderProgram->loadShader(uiFPSProgram, VERTSHADER, strlen(VERTSHADER),
                               GL_VERTEX_SHADER);
    pShaderProgram->loadShader(uiFPSProgram, FRAGSHADER, strlen(FRAGSHADER),
                               GL_FRAGMENT_SHADER);
    pFont->setShaderProgram(uiFPSProgram);
    return true;
}

void FPSCounter::init(ShaderProgram* pShaderProgram)
{
    loadFPSShader(pShaderProgram);
}

void FPSCounter::timeStamp()
{
    if( mLastTime <= 0.0001 )
    {
        mLastTime = libgltf::time::getCurrentTime();
    }
}

void FPSCounter::printFPS(glTFViewport* pViewport)
{
    glUseProgram(uiFPSProgram);
    glDisable(GL_DEPTH_TEST);
    GLint iLoc = glGetUniformLocation(uiFPSProgram, "projMatrix");
    const glm::mat4 aMat = glm::ortho(0.0f, float(pViewport->width), 0.0f,
                                      float(pViewport->height));
    glUniformMatrix4fv(iLoc, 1, false, glm::value_ptr(aMat));
    glm::vec4 lightVector = glm::vec4(0.0f, 1.0f, 1.0f, 1.0f);
    iLoc = glGetUniformLocation(uiFPSProgram, "vColor");
    glUniform4fv(iLoc, 1, glm::value_ptr(lightVector));

    double mCurrentTime = libgltf::time::getCurrentTime();
    double fDiff = libgltf::time::diffTime(mCurrentTime,mLastTime);

    if (fDiff >= 1.0)
    {
        mFPS = mFrames;
        mFrames = 0;
        mLastTime = mCurrentTime;
    }
    else
    {
        ++mFrames;
    }

    pFont->printDecimalInt(mFPS, pViewport->width - 40, 10, 15);

    glEnable(GL_DEPTH_TEST);
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
