/***************************** LICENSE START ***********************************

 Copyright 2019 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef COLOURWIDGET_H
#define COLOURWIDGET_H

#include "MvQPalette.h"

#include <QPen>
#include <QWidget>
#include <QComboBox>

class QLabel;
class QLineEdit;
class QPushButton;
class QSlider;
class QSpinBox;
class QTabWidget;
class QToolButton;

struct MvQColourItem
{
    MvQColourItem(QColor col, QString name) :
        col_(col),
        name_(name) {}
    QColor col_;
    QString name_;
};

class MvQColourGrid : public QWidget, public PaletteScanner
{
    Q_OBJECT

public:
    MvQColourGrid(int minSize, QWidget* parent = 0);
    void next(const string&, QColor, bool);

signals:
    void selected(QColor);

protected slots:
    void slotContextMenu(const QPoint& pos);

protected:
    void paintEvent(QPaintEvent* event);
    void resizeEvent(QResizeEvent* event);
    void mousePressEvent(QMouseEvent* event);
    void mouseMoveEvent(QMouseEvent* event);
    void createGrid();
    int index(QPoint);
    void sort();

    int minSize_;
    QList<MvQColourItem> items_;
    int gridSize_;
    int cellSize_;
    int cellsPerRow_;
    QPixmap pix_;
};


class MvQColourWheel : public QWidget
{
    Q_OBJECT

public:
    MvQColourWheel(int minSize, QWidget* parent = 0);
    void initColour(QColor);

public slots:
    void slotSetColour(int, int, int);

signals:
    void colourSelected(QColor colour);
    void hslSelected(int, int, int);
    void dragFinished();

protected:
    enum DragType
    {
        NoDrag,
        RingDrag,
        TriangleDrag
    };

    void paintEvent(QPaintEvent* event);
    void resizeEvent(QResizeEvent* event);
    void mousePressEvent(QMouseEvent* event);
    void mouseMoveEvent(QMouseEvent* event);
    void mouseReleaseEvent(QMouseEvent* event);

    void createRing();
    void createTriangle();
    void renderHueSelector(QPainter*);
    void renderLgtSatSelector(QPainter*);
    DragType checkPoint(QPoint);
    void dragRingSelector(QPoint);
    void dragTriangleSelector(QPoint);
    void setHue(int);
    void setLgtSat(int, int);
    bool lgtSatToPos(QPointF&);
    bool posToLgtSat(QPointF, int&, int&);
    bool projectPosToLgtSat(QPointF, int&, int&);
    int saturation(int, int);
    int chroma(int, int);

private:
    int minSize_;
    QPixmap ringPix_;
    QPixmap triPix_;
    int ringWidth_;
    int outerRadius_;
    int innerRadius_;
    int margin_;
    int hue_;
    int sat_;
    int lgt_;
    int lgtSatSelectorRadius_;
    QPen huePenBlack_;
    QPen huePenWhite_;
    QPoint centre_;
    DragType dragType_;
    QPoint lgtSatPos_;
    QPoint lgtSatOffset_;
    QPoint dragPos_;
};

class MvQColourSelectionWidget : public QWidget
{
    Q_OBJECT

public:
    explicit MvQColourSelectionWidget(bool tightMode = false, QWidget* parent = 0);
    void setColour(QColor);
    QColor currentColour() { return currentColour_; }

protected slots:
    void slotGridSelected(QColor);
    void slotWheelChanged(int, int, int);
    void slotWheelChanged(QColor);
    void slotRgbChanged(int);
    void slotHslChanged(int);
    void slotAlphaChanged(int alpha);
    void slotAlphaSliderChanged(int alpha);
    void slotHtmlChanged(QString);
    void slotMacroChanged(QString);

signals:
    void colourChanged(int, int, int);
    void colourSelected(QColor);
    void dragFinished();

private:
    void setCurrentColour(QColor);
    QString macroToString(QColor col);
    QColor macroToColour(QString name);

    bool tightMode_;
    QTabWidget* tab_;
    MvQColourWheel* wheel_;
    MvQColourGrid* grid_;

    QPushButton* okPb_;
    QObject* parentObject_;
    QColor currentColour_;

    QList<QSpinBox*> rgbSpin_;
    QList<QSpinBox*> hslSpin_;
    QLineEdit* htmlEdit_;
    QLineEdit* macroEdit_;
    QSlider* alphaSlider_;
    QSpinBox* alphaSpin_;

    bool ignoreRgbSpin_;
    bool ignoreHslSpin_;
    bool ignoreAlphaSpin_;
    bool ignoreAlphaSlider_;
    bool ignoreHtmlEdit_;
    bool ignoreMacroEdit_;
};

class MvQColourCombo : public QComboBox, public PaletteScanner
{
    Q_OBJECT

public:
    MvQColourCombo(QWidget *parent = nullptr);
    ~MvQColourCombo() {}

    QString currentValue() { return QString(); }
    void addValue(QString) {}

    //void refresh(const std::vector<string>&);

public slots:
    void changeSelection(QString);
    //void slotHelperEditConfirmed();

protected slots:
    void slotCurrentChanged(int);
    void slotCopyColour();
    void slotPasteColour();

signals:
    void colourSelected(QString);

protected:
    void next(const string&, QColor, bool);
    void addColour(QColor, QString, QString, bool setCurrent = false);
    void setCustomColour(QColor, QString, QString, bool setCurrent = false);
    //void updateHelper();
    void setColour(int, QColor);

    QPixmap pix_;
    bool broadcastChange_;
    int customColIndex_;
};

class MvQColourWidget : public QWidget
{
    Q_OBJECT

public:
    MvQColourWidget(QWidget * parent);
    ~MvQColourWidget() {}

    void initColour(QString);
    void start() {}
    bool dialog() { return false; }
    QWidget* widget() { return selector_; }
    QString currentValue() const;

signals:
    void valueChanged(QString);

protected slots:
    void slotSelectorEdited(QColor);
    void slotComboEdited(QString);

private:
    MvQColourCombo* colCb_;
    MvQColourSelectionWidget* selector_;
    QString oriName_;
};

#endif // COLOURWIDGET_H
