local helpers = require('test.functional.helpers')(after_each)
local Screen = require('test.functional.ui.screen')
local clear = helpers.clear
local feed, command, insert = helpers.feed, helpers.command, helpers.insert
local eq = helpers.eq
local meths = helpers.meths
local wait = helpers.wait


describe('ext_multigrid', function()
  local screen

  before_each(function()
    clear{args_rm={'--headless'}, args={'--cmd', 'set laststatus=2'}}
    screen = Screen.new(53,14)
    screen:attach({ext_multigrid=true})
    screen:set_default_attr_ids({
      [1] = {bold = true, foreground = Screen.colors.Blue1},
      [2] = {foreground = Screen.colors.Magenta},
      [3] = {foreground = Screen.colors.Brown, bold = true},
      [4] = {foreground = Screen.colors.SlateBlue},
      [5] = {bold = true, foreground = Screen.colors.SlateBlue},
      [6] = {foreground = Screen.colors.Cyan4},
      [7] = {bold = true},
      [8] = {underline = true, bold = true, foreground = Screen.colors.SlateBlue},
      [9] = {foreground = Screen.colors.SlateBlue, underline = true},
      [10] = {foreground = Screen.colors.Red},
      [11] = {bold = true, reverse = true},
      [12] = {reverse = true},
      [13] = {foreground = Screen.colors.DarkBlue, background = Screen.colors.LightGrey},
      [14] = {foreground = Screen.colors.Grey100, background = Screen.colors.Red},
      [15] = {bold = true, foreground = Screen.colors.SeaGreen4},
      [16] = {background = Screen.colors.LightGrey, underline = true},
      [17] = {background = Screen.colors.LightGrey, underline = true, bold = true, foreground = Screen.colors.Magenta},
      [18] = {bold = true, foreground = Screen.colors.Magenta},
      [19] = {foreground = Screen.colors.Brown},
    })
  end)

  after_each(function()
    screen:detach()
  end)

  it('default initial screen', function()
    screen:expect{grid=[[
    ## grid 1
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ]]}
  end)

  it('positions windows correctly', function()
    command('vsplit')
    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]], condition=function()
      eq({
        [2] = { win = {id=1000}, startrow = 0, startcol = 27, width = 26, height = 12 },
        [4] = { win = {id=1001}, startrow = 0, startcol =  0, width = 26, height = 12 }
      }, screen.win_position)
    end}
    command('wincmd l')
    command('split')
    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]{12:│}[5:--------------------------]|
      [4:--------------------------]{12:│}[5:--------------------------]|
      [4:--------------------------]{12:│}[5:--------------------------]|
      [4:--------------------------]{12:│}[5:--------------------------]|
      [4:--------------------------]{12:│}[5:--------------------------]|
      [4:--------------------------]{12:│}[5:--------------------------]|
      [4:--------------------------]{12:│}{11:[No Name]                 }|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      {12:[No Name]                  [No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]], condition=function()
      eq({
        [2] = { win = {id=1000}, startrow = 7, startcol = 27, width = 26, height =  5 },
        [4] = { win = {id=1001}, startrow = 0, startcol =  0, width = 26, height = 12 },
        [5] = { win = {id=1002}, startrow = 0, startcol = 27, width = 26, height =  6 }
      }, screen.win_position)
    end}
    command('wincmd h')
    command('q')
    screen:expect{grid=[[
    ## grid 1
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 5
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]], condition=function()
      eq({
        [2] = { win = {id=1000}, startrow = 7, startcol = 0, width = 53, height =  5 },
        [5] = { win = {id=1002}, startrow = 0, startcol = 0, width = 53, height =  6 }
      }, screen.win_position)
    end}
  end)

  describe('split', function ()
    describe('horizontally', function ()
      it('allocates grids', function ()
        command('sp')
        screen:expect([[
        ## grid 1
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                               |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ]])
      end)

      it('resizes grids', function ()
        command('sp')
        command('resize 8')
        screen:expect([[
        ## grid 1
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                               |
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ]])
      end)

      it('splits vertically', function()
        command('sp')
        command('vsp')
        command('vsp')
        screen:expect{grid=[[
        ## grid 1
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          {11:[No Name]            }{12:[No Name]        [No Name]      }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                               |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
                         |
          {1:~              }|
          {1:~              }|
          {1:~              }|
          {1:~              }|
          {1:~              }|
        ## grid 5
                          |
          {1:~               }|
          {1:~               }|
          {1:~               }|
          {1:~               }|
          {1:~               }|
        ## grid 6
          ^                    |
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
        ]]}
        insert('hello')
        screen:expect{grid=[[
        ## grid 1
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          [6:--------------------]{12:│}[5:----------------]{12:│}[4:---------------]|
          {11:[No Name] [+]        }{12:[No Name] [+]    [No Name] [+]  }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name] [+]                                        }|
          [3:-----------------------------------------------------]|
        ## grid 2
          hello                                                |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
          hello          |
          {1:~              }|
          {1:~              }|
          {1:~              }|
          {1:~              }|
          {1:~              }|
        ## grid 5
          hello           |
          {1:~               }|
          {1:~               }|
          {1:~               }|
          {1:~               }|
          {1:~               }|
        ## grid 6
          hell^o               |
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
        ]]}
      end)
      it('closes splits', function ()
        command('sp')
        screen:expect{grid=[[
        ## grid 1
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                               |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ]]}
        command('q')
        screen:expect{grid=[[
        ## grid 1
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ]]}
      end)
    end)

    describe('vertically', function ()
      it('allocates grids', function ()
        command('vsp')
        screen:expect{grid=[[
        ## grid 1
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          {11:[No Name]                  }{12:[No Name]                 }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                    |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 3
                                                               |
        ## grid 4
          ^                          |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ]]}
      end)
      it('resizes grids', function ()
        command('vsp')
        command('vertical resize 10')
        screen:expect{grid=[[
        ## grid 1
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          [4:----------]{12:│}[2:------------------------------------------]|
          {11:<No Name]  }{12:[No Name]                                 }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                    |
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
        ## grid 3
                                                               |
        ## grid 4
          ^          |
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
        ]]}
      end)
      it('splits horizontally', function ()
        command('vsp')
        command('sp')
        screen:expect{grid=[[
        ## grid 1
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          {11:[No Name]                 }{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          {12:[No Name]                  [No Name]                 }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                    |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 3
                                                               |
        ## grid 4
                                    |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 5
          ^                          |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ]]}
        insert('hello')
        screen:expect{grid=[[
        ## grid 1
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          [5:--------------------------]{12:│}[2:--------------------------]|
          {11:[No Name] [+]             }{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          {12:[No Name] [+]              [No Name] [+]             }|
          [3:-----------------------------------------------------]|
        ## grid 2
          hello                     |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 3
                                                               |
        ## grid 4
          hello                     |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 5
          hell^o                     |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ]]}
      end)
      it('closes splits', function ()
        command('vsp')
        screen:expect{grid=[[
        ## grid 1
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          [4:--------------------------]{12:│}[2:--------------------------]|
          {11:[No Name]                  }{12:[No Name]                 }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                    |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 3
                                                               |
        ## grid 4
          ^                          |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ]]}
        command('q')
        screen:expect{grid=[[
        ## grid 1
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ]]}
      end)
    end)
  end)

  describe('on resize', function ()
    it('rebuilds all grids', function ()
      screen:try_resize(25, 6)
      screen:expect{grid=[[
      ## grid 1
        [2:-------------------------]|
        [2:-------------------------]|
        [2:-------------------------]|
        [2:-------------------------]|
        {11:[No Name]                }|
        [3:-------------------------]|
      ## grid 2
        ^                         |
        {1:~                        }|
        {1:~                        }|
        {1:~                        }|
      ## grid 3
                                 |
      ]]}
    end)

    it('has minimum width/height values', function()
      screen:try_resize(1, 1)
      screen:expect{grid=[[
      ## grid 1
        [2:------------]|
        {11:[No Name]   }|
        [3:------------]|
      ## grid 2
        ^            |
      ## grid 3
                    |
      ]]}

      feed('<esc>:ls')
      screen:expect{grid=[[
      ## grid 1
        [2:------------]|
        {11:[No Name]   }|
        [3:------------]|
      ## grid 2
                    |
      ## grid 3
        :ls^         |
      ]]}
    end)
  end)

  describe('grid of smaller inner size', function()
    it('is rendered correctly', function()
      screen:try_resize_grid(2, 8, 5)

      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name]                                            }|
        [3:-----------------------------------------------------]|
      ## grid 2
        ^        |
        {1:~       }|
        {1:~       }|
        {1:~       }|
        {1:~       }|
      ## grid 3
                                                             |
      ]]}
    end)
  end)

  describe('grid of bigger inner size', function()
    it('is rendered correctly', function()
      screen:try_resize_grid(2, 80, 20)

      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name]                                            }|
        [3:-----------------------------------------------------]|
      ## grid 2
        ^                                                                                |
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
      ## grid 3
                                                             |
      ]]}
    end)
  end)


  describe('with resized grid', function()
    before_each(function()
      screen:try_resize_grid(2, 60, 20)
    end)
    it('gets written till grid width', function()
      insert(('a'):rep(60).."\n")

      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa|
        ^                                                            |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
    end)

    it('wraps with grid width', function()
      insert(('b'):rep(80).."\n")
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb|
        bbbbbbbbbbbbbbbbbbbb                                        |
        ^                                                            |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
    end)

    it('displays messages with default grid width', function()
      command('echomsg "this is a very very very very very very very very'..
        ' long message"')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name]                                            }|
        [3:-----------------------------------------------------]|
      ## grid 2
        ^                                                            |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
        this is a very very very...ry very very long message |
      ]]}
    end)

    it('creates folds with grid width', function()
      insert('this is a fold\nthis is inside fold\nthis is outside fold')
      feed('kzfgg')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        {13:^+--  2 lines: this is a fold································}|
        this is outside fold                                        |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
    end)
  end)

  it('multiline messages scroll over windows', function()
    command('sp')
    command('vsp')
    screen:expect{grid=[[
    ## grid 1
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    feed(":echoerr 'very' | echoerr 'much' | echoerr 'fail'<cr>")
    screen:expect{grid=[[
    ## grid 1
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
      {14:very}                                                 |
      {14:much}                                                 |
      {14:fail}                                                 |
      {15:Press ENTER or type command to continue}^              |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    feed('<cr>')
    screen:expect{grid=[[
    ## grid 1
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    command([[
      func! ErrMsg()
        for i in range(11)
          echoerr "error ".i
        endfor
      endfunc]])
    feed(":call ErrMsg()<cr>")
    screen:expect{grid=[[
    ## grid 1
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
      {14:Error detected while processing function ErrMsg:}     |
      {19:line    2:}                                           |
      {14:error 0}                                              |
      {14:error 1}                                              |
      {14:error 2}                                              |
      {14:error 3}                                              |
      {14:error 4}                                              |
      {14:error 5}                                              |
      {14:error 6}                                              |
      {14:error 7}                                              |
      {14:error 8}                                              |
      {14:error 9}                                              |
      {14:error 10}                                             |
      {15:Press ENTER or type command to continue}^              |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    feed("<c-c>")
    screen:expect{grid=[[
    ## grid 1
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      [5:--------------------------]{12:│}[4:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}
  end)

  it('handles switch tabs', function()
    command('vsp')
    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}


    command('tabnew')
    -- note the old grids aren't resized yet
    screen:expect{grid=[[
    ## grid 1
      {16: }{17:2}{16: [No Name] }{7: [No Name] }{12:                            }{16:X}|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('sp')
    screen:expect{grid=[[
    ## grid 1
      {16: }{17:2}{16: [No Name] }{7: }{18:2}{7: [No Name] }{12:                          }{16:X}|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 6
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('tabnext')
    screen:expect{grid=[[
    ## grid 1
      {7: }{18:2}{7: [No Name] }{16: }{17:2}{16: [No Name] }{12:                          }{16:X}|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5 (hidden)
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 6 (hidden)
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('tabnext')
    screen:expect{grid=[[
    ## grid 1
      {16: }{17:2}{16: [No Name] }{7: }{18:2}{7: [No Name] }{12:                          }{16:X}|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 6
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('tabclose')
    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      [4:--------------------------]{12:│}[2:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}
  end)

  it('supports mouse', function()
    insert('some text\nto be clicked')
    screen:expect{grid=[[
    ## grid 1
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicke^d                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ]]}

    meths.input_mouse('left', 'press', '', 2, 0, 5)
    screen:expect{grid=[[
    ## grid 1
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some ^text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ]]}

    feed(':new<cr>')
    insert('Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo')

    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing el|
      it, sed do eiusm^o                                    |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    meths.input_mouse('left', 'press', '', 2, 1, 6)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be ^clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing el|
      it, sed do eiusmo                                    |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    meths.input_mouse('left', 'press', '', 4, 1, 4)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing el|
      it, ^sed do eiusmo                                    |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    screen:try_resize_grid(4, 80, 2)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, ^sed do eiusmo          |
      {1:~                                                                               }|
    ]]}

    meths.input_mouse('left', 'press', '', 4, 0, 64)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do ^eiusmo          |
      {1:~                                                                               }|
    ]]}

    meths.input_mouse('left', 'press', '', 1,6, 20)
    -- TODO(bfredl): "batching" input_mouse is formally not supported yet.
    -- Normally it should work fine in async context when nvim is not blocked,
    -- but add a wait be sure.
    wait()
    meths.input_mouse('left', 'drag', '', 1, 4, 20)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do ^eiusmo          |
      {1:~                                                                               }|
    ]]}

    feed('<c-w><c-w><c-w>v')
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [5:--------------------------]{12:│}[2:--------------------------]|
      [5:--------------------------]{12:│}[2:--------------------------]|
      [5:--------------------------]{12:│}[2:--------------------------]|
      [5:--------------------------]{12:│}[2:--------------------------]|
      [5:--------------------------]{12:│}[2:--------------------------]|
      [5:--------------------------]{12:│}[2:--------------------------]|
      [5:--------------------------]{12:│}[2:--------------------------]|
      {11:[No Name] [+]              }{12:[No Name] [+]             }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                 |
      to be clicked             |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo          |
      {1:~                                                                               }|
    ## grid 5
      some text                 |
      to be ^clicked             |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    meths.input_mouse('left', 'press', '', 1,8, 26)
    wait()
    meths.input_mouse('left', 'drag', '', 1, 6, 30)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [5:------------------------------]{12:│}[2:----------------------]|
      [5:------------------------------]{12:│}[2:----------------------]|
      [5:------------------------------]{12:│}[2:----------------------]|
      [5:------------------------------]{12:│}[2:----------------------]|
      [5:------------------------------]{12:│}[2:----------------------]|
      [5:------------------------------]{12:│}[2:----------------------]|
      [5:------------------------------]{12:│}[2:----------------------]|
      {11:[No Name] [+]                  }{12:[No Name] [+]         }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo          |
      {1:~                                                                               }|
    ## grid 5
      some text                     |
      to be ^clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
  end)
end)
