/*
 * h323t38.cxx
 *
 * H.323 T.38 logical channel establishment
 *
 * Open H323 Library
 *
 * Copyright (c) 1998-2000 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: h323t38.cxx,v $
 * Revision 1.6  2001/11/09 05:39:54  craigs
 * Added initial T.38 support thanks to Adam Lazur
 *
 * Revision 1.5  2001/09/12 07:48:05  robertj
 * Fixed various problems with tracing.
 *
 * Revision 1.4  2001/08/06 03:08:57  robertj
 * Fission of h323.h to h323ep.h & h323con.h, h323.h now just includes files.
 *
 * Revision 1.3  2001/07/24 02:26:24  robertj
 * Added UDP, dual TCP and single TCP modes to T.38 capability.
 *
 * Revision 1.2  2001/07/19 10:48:20  robertj
 * Fixed bandwidth
 *
 * Revision 1.1  2001/07/17 04:44:32  robertj
 * Partial implementation of T.120 and T.38 logical channels.
 *
 */

#ifdef __GNUC__
#pragma implementation "h323t38.h"
#endif


#include <ptlib.h>
#include "h323t38.h"
#include "h323con.h"
#include "h245.h"
#include "t38proto.h"
#include "t38.h"


#define new PNEW


/////////////////////////////////////////////////////////////////////////////

H323_T38Capability::H323_T38Capability(TransportMode m)
{
  mode = m;
}


PObject::Comparison H323_T38Capability::Compare(const PObject & obj) const
{
  Comparison result = H323DataCapability::Compare(obj);
  if (result != EqualTo)
    return result;

  PAssert(obj.IsDescendant(H323_T38Capability::Class()), PInvalidCast);
  const H323_T38Capability & other = (const H323_T38Capability &)obj;

  if (mode < other.mode)
    return LessThan;

  if (mode > other.mode)
    return GreaterThan;

  return EqualTo;
}


PObject * H323_T38Capability::Clone() const
{
  return new H323_T38Capability(*this);
}


unsigned H323_T38Capability::GetSubType() const
{
  PTRACE(1, "H323_T38Capability::GetSubType()");
  return H245_DataApplicationCapability_application::e_t38fax;
}


PString H323_T38Capability::GetFormatName() const
{
  static const char * const modes[NumTransportModes] = {
    "UDP", "TCP2", "TCP"
  };
  return PString("T.38-") + modes[mode];
}


H323Channel * H323_T38Capability::CreateChannel(H323Connection & connection,
                                                H323Channel::Directions direction,
                                                unsigned int SessionID,
                               const H245_H2250LogicalChannelParameters *params) const
{
    H323Channel *channel;
    PTRACE(1, "H323_T38Capability::CreateChannel()\tSessionID=" << SessionID << " direction=" << direction);

    RTP_Session * session;
    if (params != NULL) {
        PTRACE(2, "H323_T38Capability::CreateChannel()\tparams != NULL SessionID=" << params->m_sessionID );
        // XXX this breaks things
        // session = connection.UseSession(params->m_sessionID, params->m_mediaControlChannel);
    } else {
        PTRACE(2, "H323_T38Capability::CreateChannel()\tparams = NULL");
        /* Make a fake transport address from the connection so things get
         * initialized with the transport type (IP, IPx, multicast, etc) */
        H245_TransportAddress addr;
        connection.GetControlChannel().SetUpTransportPDU(addr, 0);
        PTRACE(1, "H323_T38Capability::CreateChannel()\tSessionID=" << SessionID <<" addr = "<< addr);
        session = connection.UseSession(SessionID, addr);
    }

    channel = new H323_T38Channel(connection, *this, direction);
#if 0
    if (direction == H323Channel::IsReceiver) {
        channel = new H323_T38Channel(connection, *this, direction);
    } else {
        channel = connection.FindChannel(SessionID, TRUE);
        if (channel == NULL)
            PTRACE(1, "H323_T38Capability::CreateChannel()\tERROR: Incoming channel not found!");
    }
#endif
    return channel;
}

BOOL H323_T38Capability::OnSendingPDU(H245_DataApplicationCapability & cap) const
{
    PTRACE(1, "H323_T38Capability::OnSendingPDU");
    cap.m_maxBitRate = 144; // 14.4kbps
    cap.m_application.SetTag(H245_DataApplicationCapability_application::e_t38fax);
    H245_DataApplicationCapability_application_t38fax & fax = cap.m_application;
    H245_DataProtocolCapability & proto = fax.m_t38FaxProtocol;
    H245_T38FaxProfile & profile = fax.m_t38FaxProfile;

    if (mode == e_UDP) {
        proto.SetTag(H245_DataProtocolCapability::e_udp);
        profile.IncludeOptionalField(H245_T38FaxProfile::e_t38FaxUdpOptions);
        profile.m_t38FaxUdpOptions.m_t38FaxMaxBuffer = 200;
        profile.m_t38FaxUdpOptions.m_t38FaxMaxDatagram = 72;
        profile.m_t38FaxUdpOptions.m_t38FaxUdpEC.SetTag(H245_T38FaxUdpOptions_t38FaxUdpEC::e_t38UDPRedundancy);
    } else {
        proto.SetTag(H245_DataProtocolCapability::e_tcp);
        profile.IncludeOptionalField(H245_T38FaxProfile::e_t38FaxTcpOptions);
        profile.m_t38FaxTcpOptions.m_t38TCPBidirectionalMode = mode == e_SingleTCP;
    }
    return TRUE;
}
#if 0
    H245_NonStandardParameter &param = cap.m_application;
    param.m_data = "T38FaxUDP";
    param.m_nonStandardIdentifier.SetTag(H245_CapabilityIdentifier::e_h221NonStandard);
    H245_NonStandardIdentifier_h221NonStandard &h221id = param.m_nonStandardIdentifier;
    h221id.m_t35CountryCode =  181;
    h221id.m_t35Extension = 0;
    h221id.m_manufacturerCode = 18;

    return TRUE;
}
#endif

BOOL H323_T38Capability::OnReceivedPDU(const H245_DataApplicationCapability & cap)
{
  PTRACE(1, "H323_T38Capability::OnRecievedPDU");
  if (cap.m_application.GetTag() != H245_DataApplicationCapability_application::e_t38fax)
    return FALSE;

  const H245_DataApplicationCapability_application_t38fax & fax = cap.m_application;
  const H245_DataProtocolCapability & proto = fax.m_t38FaxProtocol;

  if (proto.GetTag() == H245_DataProtocolCapability::e_udp)
    mode = e_UDP;
  else {
    const H245_T38FaxProfile & profile = fax.m_t38FaxProfile;
    if (profile.m_t38FaxTcpOptions.m_t38TCPBidirectionalMode)
      mode = e_SingleTCP;
    else
      mode = e_DualTCP;
  }

  return TRUE;
}


////H323_T38Channel//////////////////////////////////////////////////////////

H323_T38Channel::H323_T38Channel(H323Connection & connection,
                        const H323_T38Capability & capability,
                        H323Channel::Directions dir)
        : H323DataChannel(connection, capability, dir)
{
    PTRACE(1, "H323_T38Channel::H323_T38Channel()");

    H323TransportAddress addr;
    PIPSocket::Address ip;
    WORD port;

    addr = connection.GetControlChannel().GetLocalAddress();
    addr.GetIpAndPort(ip, port);

    separateReverseChannel = capability.GetTransportMode() != H323_T38Capability::e_SingleTCP;
    usesTCP = capability.GetTransportMode() != H323_T38Capability::e_UDP;

    if (!usesTCP) {
        transport =  new H323TransportUDP(connection.GetEndPoint(), ip, port - 1);
        t38handler = new OpalT38Protocol();
    } else {
        PTRACE(1, "T.38 " << capability << " not supported!");
    }
}

H323_T38Channel::~H323_T38Channel()
{
    delete t38handler;
}

void H323_T38Channel::Receive()
{
    if (terminating)
        return;
    PTRACE(1, "H323_T38Channel::Receive()\tstarting");

    if (t38handler == NULL) {
        PTRACE(1, "H323_T38Channel::Receive()\tNo proto handler");
    } else {
        t38handler->Answer(*transport);
//      connection.CloseLogicalChannelNumber(number);
    }
    PTRACE(1, "H323_T38Channel::Receive()\tterminated");
}

void H323_T38Channel::Transmit()
{
    if (terminating)
        return;
    PTRACE(1, "H323_T38Channel::Transmit()\tstarting");
    if (t38handler == NULL) 
        PTRACE(1, "H323_T38Channel::Transmit()\tNo proto handler");
    else
        t38handler->Originate(*transport);
//      connection.CloseLogicalChannelNumber(number);
    PTRACE(1, "H323_T38Channel::Transmit()\tterminating");
}

BOOL H323_T38Channel::OnReceivedPDU(const H245_OpenLogicalChannel & open, unsigned & errorCode)
{
    PTRACE(1, "H323_T38Channel::OnReceivedPDU()");

    PTRACE(4, open);
    number = H323ChannelNumber(open.m_forwardLogicalChannelNumber, TRUE);

    if (open.m_forwardLogicalChannelParameters.m_multiplexParameters.GetTag() !=
        H245_OpenLogicalChannel_forwardLogicalChannelParameters_multiplexParameters::e_h2250LogicalChannelParameters) {
        PTRACE(1, "ERROR: Received PDU has no network info!!");
        return FALSE;
    }
    const H245_H2250LogicalChannelParameters & param = open.m_forwardLogicalChannelParameters.m_multiplexParameters;
    if (param.HasOptionalField(H245_H2250LogicalChannelParameters::e_mediaControlChannel)) {
        PIPSocket::Address ip;
        WORD port;
        H323TransportAddress addr = param.m_mediaControlChannel;

        addr.GetIpAndPort(ip, port, "udp");
        port--;

        H323TransportAddress addr2(ip,port);
        if (transport->SetRemoteAddress(addr2) == FALSE) {
            PTRACE(1, "Setting remote address failed!!");
        }
        PTRACE(4, "Remote address set to " << ip << ":" << port);
    }
    return TRUE;
}

void H323_T38Channel::OnSendOpenAck(const H245_OpenLogicalChannel & open, H245_OpenLogicalChannelAck & ack) const
{
    PTRACE(1, "H323_T38Channel::OnSendOpenAck()");

    ack.IncludeOptionalField(H245_OpenLogicalChannelAck::e_forwardMultiplexAckParameters);
    ack.m_forwardMultiplexAckParameters.SetTag(
                    H245_OpenLogicalChannelAck_forwardMultiplexAckParameters::e_h2250LogicalChannelAckParameters);

    H245_H2250LogicalChannelAckParameters &param = ack.m_forwardMultiplexAckParameters;

    param.IncludeOptionalField(H245_H2250LogicalChannelAckParameters::e_mediaChannel);
    param.IncludeOptionalField(H245_H2250LogicalChannelAckParameters::e_mediaControlChannel);
    param.IncludeOptionalField(H245_H2250LogicalChannelAckParameters::e_sessionID);

    H323TransportAddress addr = transport->GetLocalAddress();
    PIPSocket::Address ip;
    WORD port;

    addr.GetIpAndPort(ip, port, "udp");
    transport->SetUpTransportPDU(param.m_mediaChannel, port);

    addr = connection.GetControlChannel().GetLocalAddress();
    addr.GetIpAndPort(ip, port);
    connection.GetControlChannel().SetUpTransportPDU(param.m_mediaControlChannel, port);

    /* test to see if we're supposed to have a separate rev channel */
    if (separateReverseChannel) {
        /* if we are, then test to see if there already is one open */
        if (!connection.FindChannel(GetSessionID(), FALSE)) {
            /* and if there isn't... try to open one */
            PTRACE(1,"H323_T38Channel::OnSendOpenAck()\tOpenLogicalChannel SessionId=" << (unsigned) param.m_sessionID);
            if (connection.OpenLogicalChannel(GetCapability(), (unsigned) param.m_sessionID, H323Channel::IsTransmitter) == FALSE) {
                PTRACE(1, "H323_T38Channel::OnSendOpenAck()\tERROR: Opening outgoing channel failed!!!");
            }
        } else {
            PTRACE(1, "H323_T38Channel::OnSendOpenAck()\treverse channel already open, not trying to open a new one");
        }
    }
#if 0
    if () {
        H323Capability *cap = connection.GetEndPoint().FindCapability(H323Capability::e_Data,
                                    H245_DataApplicationCapability_application::e_t38fax );

        if (connection.OpenLogicalChannel(*cap, (unsigned) param.m_sessionID, H323Channel::IsTransmitter) == FALSE)
            PTRACE(1, "ERROR: Opening outgoing channel failed!!!");
    }
#endif
    PTRACE(3, ack);
}

BOOL H323_T38Channel::CreateTransport()
{
  PTRACE(1, "H323_T38Channel::CreateTransport()");
  if (transport == NULL) {
      transport = connection.GetControlChannel().GetLocalAddress().CreateTransport(connection.GetEndPoint());
  }
  return transport != NULL;
}

BOOL H323_T38Channel::CreateListener()
{
  PTRACE(1, "H323_T38Channel::CreateListener()");
  if (listener != NULL)
    return TRUE;

  if (usesTCP)
    return H323DataChannel::CreateListener();

  return CreateTransport();
}

#if 0
H323_T38Channel::H323_T38Channel(H323Connection & conn,
                                   const H323_T38Capability & cap,
                                   Directions dir)
  : H323DataChannel(conn, cap, dir)
{
  separateReverseChannel = cap.GetTransportMode() != H323_T38Capability::e_SingleTCP;
  usesTCP = cap.GetTransportMode() != H323_T38Capability::e_UDP;
}


void H323_T38Channel::Receive()
{
  PTRACE(2, "H323T38\tReceive thread started.");

  if (t38handler != NULL) {
    if (listener != NULL)
      transport = listener->Accept(30000);  // 30 second wait for connect back

    if (transport != NULL)
      t38handler->Answer(*transport);
    else {
      PTRACE(0, "H323T38\tNo transport, aborting thread.");
    }
  }
  else {
    PTRACE(1, "H323T38\tNo protocol handler, aborting thread.");
  }

  connection.CloseLogicalChannelNumber(number);

  PTRACE(2, "H323T38\tReceive thread ended");
}


void H323_T38Channel::Transmit()
{
  PTRACE(2, "H323T38\tTransmit thread started.");

  if (t38handler != NULL) {
    if (transport != NULL)
      t38handler->Originate(*transport);
    else {
      PTRACE(1, "H323T38\tNo transport, aborting thread.");
    }
  }
  else {
    PTRACE(1, "H323T38\tNo protocol handler, aborting thread.");
  }

  connection.CloseLogicalChannelNumber(number);

  PTRACE(2, "H323T38\tTransmit thread ended");
}


BOOL H323_T38Channel::CreateListener()
{
  if (listener != NULL)
    return TRUE;

  if (usesTCP)
    return H323DataChannel::CreateListener();

  return CreateTransport();
}

BOOL H323_T38Channel::CreateTransport()
{
  if (transport != NULL)
    return TRUE;

  if (usesTCP)
    return H323DataChannel::CreateTransport();

  PIPSocket::Address ip;
  WORD port;
  if (!connection.GetControlChannel().GetLocalAddress().GetIpAndPort(ip, port)) {
    PTRACE(2, "H323T38\tTrying to use UDP when base transport is not TCP/IP");
    PIPSocket::GetHostAddress(ip);
  }

  transport = new H323TransportUDP(connection.GetEndPoint(), ip);
  return TRUE;
}
#endif

/////////////////////////////////////////////////////////////////////////////
