/* Seahorse
 *
 * Copyright (C) 1999, 2000 Anthony Mulcahy
 *   
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>

#include "gpg-interface.h"
#include "seahorse.h"
#include "seahorsesig.h"
#include "support.h"

static GnomeUIInfo help_menu_uiinfo[] =
{
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_HELP ("seahorse"),
  GNOMEUIINFO_END
};

gchar *seahorse_armor_arg = NULL;
gchar *seahorse_clearsign_arg = NULL;
gchar *seahorse_decrypt_arg = NULL;
gchar *seahorse_detachsign_arg = NULL;
gchar *seahorse_encrypt_arg = NULL;
gchar *seahorse_file_arg = NULL;
gchar *seahorse_outfile_arg = NULL;
gchar *seahorse_recipient_arg = NULL;
gchar *seahorse_sign_arg = NULL;
gchar *seahorse_textmode_arg = NULL;
gchar *seahorse_verify_arg = NULL;
extern char	*signencrypt_filename;
extern char	*signencrypt_outfilename;
extern char	*signencrypt_username;
extern char	*verifydecrypt_filename;
extern char	*verifydecrypt_outfilename;
extern char	*comment;
extern char	*default_cipher;
extern int	ascii_arm;
extern int	datasrc_active_btn;
extern int	enable_comment;
extern int	output_to_file;
extern int	textmode;
extern int	signencrypt_active_btn;
extern int	verifydecrypt_active_btn;

static int	seahorse_height_arg = 400;
static int	seahorse_width_arg = 600;
static int	xpos_arg = 20;
static int	ypos_arg = 40;


static struct poptOption options[] =
{
    {
        "width",
        'w',
        POPT_ARG_INT,
        &seahorse_width_arg,
        0,
        N_("set the width of the window"),
        NULL
    },{
        "height",
        'h',
        POPT_ARG_INT,
        &seahorse_height_arg,
        0,
        N_("set the height of the window"),
        NULL
    },{
        "xpos",
        '\0',
        POPT_ARG_INT,
        &xpos_arg,
        0,
        N_("set the horizontal position of the window"),
        NULL
    },{
        "ypos",
        '\0',
        POPT_ARG_INT,
        &ypos_arg ,
        0,
        N_("set the vertical position of the window"),
        NULL
    },{
        "sign",
        's',
        POPT_ARG_NONE,
        &seahorse_sign_arg,
        0,
        N_("make a signature"),
        NULL
    },{
        "clearsign",
        '\0',
        POPT_ARG_NONE,
        &seahorse_clearsign_arg,
        0,
        N_("make a clear text signature"),
        NULL
    },{
        "detach-sign",
        'b',
        POPT_ARG_NONE,
        &seahorse_detachsign_arg,
        0,
        N_("make a detached signature"),
        NULL
    },{
        "encrypt",
        'e',
        POPT_ARG_NONE,
        &seahorse_encrypt_arg,
        0,
        N_("encrypt"),
        NULL
    },{
        "decrypt",
        'd',
        POPT_ARG_NONE,
        &seahorse_decrypt_arg,
        0,
        N_("decrypt"),
        NULL
    },{
        "verify",
        '\0',
        POPT_ARG_NONE,
        &seahorse_verify_arg,
        0,
        N_("verify a signature"),
        NULL
    },{
        "armor",
        'a',
        POPT_ARG_NONE,
        &seahorse_armor_arg,
        0,
        N_("create ascii armored output"),
        NULL
    },{
        "textmode",
        't',
        POPT_ARG_NONE,
        &seahorse_textmode_arg,
        0,
        N_("canonical textmode output"),
        NULL
    },{
        "recipient",
        'r',
        POPT_ARG_STRING,
        &seahorse_recipient_arg,
        0,
        N_("sign/encrypt for NAME"),
        N_("NAME(or EMAIL-ADDRESS)")
    },{
        "file",
        'f',
        POPT_ARG_STRING,
        &seahorse_file_arg,
        0,
        N_("use file FILE instead of stdin"),
        N_("FILE")
    },{
        "output-file",
        'o',
        POPT_ARG_STRING,
        &seahorse_outfile_arg,
        0,
        N_("write output to file OUTFILE (using - will write to stdout)"),
        N_("OUTFILE")
    },
    {NULL, '\0', 0, NULL, 0, NULL, NULL}
};


int islocaladdr(const char *hostname)
{
    int sock;
    struct sockaddr_in saddr;
    struct hostent *he;

    if( hostname==NULL )
        return 0;
    sock=socket(AF_INET,SOCK_DGRAM,0);
    if( sock<0 )
        return 0;
    if( ((he=gethostbyname(hostname))==NULL) || (he->h_addr_list[0]==NULL) )
    {
        close(sock);
        return 0;
    }
    saddr.sin_family=AF_INET;
    saddr.sin_port=0;
    memcpy(&saddr.sin_addr,he->h_addr_list[0],he->h_length);
    if( bind(sock,(struct sockaddr *)&saddr,sizeof(saddr))<0 )
    {
        close(sock);
        return 0;
    }
    close(sock);
    return 1;
}

void security_check()
{
    const gchar *display;
    gchar *p, *q;
    size_t len;
    int is_local;
    GtkWidget *log_textbox;
    XHostAddress *xha;
    int nhosts_return;
    Bool state_return;

    is_local=1;
    if( (display=gdk_get_display())==NULL )
        is_local=0;
    else
    {
        if( (p=strchr(display,':'))==NULL )
            is_local=0;
        else if( p>display ) 
        {
            len=p-display;
            if( (q=malloc(len+1))==NULL )
                is_local=0;
            else
            {
                memcpy(q,display,len);
                q[len]=0;
                if( strcmp(q,"unix")!=0 )
                {
                    if( !islocaladdr(q) )
                        is_local=0;
                }
                free(q);
            }
        }
    }
    log_textbox = get_widget (main_win, "log_textbox");
    if( !is_local )
    {
        gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL,
                        _("WARNING: Seahorse  has detected that  your display is  not directly connected  to the\n\
computer running Seahorse.  If you have not taken measures to protect this connection\n\
then eavesdroppers on your network connection can capture screen updates and keyboard\n\
input.  Do not read or enter sensitive information using an unprotected connection!"), -1);
        return;
    }
    if( (xha=XListHosts(GDK_DISPLAY(), &nhosts_return, &state_return))!=NULL )
    {
        if( !state_return )
        {
            gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL,
                            _("WARNING: Seahorse has detected that your  X access control is disabled. All local and\n\
remote users  can capture  your keyboard input  and screen  content.  Do not  read or\n\
enter sensitive information in this configuration!  See man xhost for details."), -1);
            XFree(xha);
            return;
        }
        if( nhosts_return>0 )
        {
            gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL,
                            _("WARNING: Seahorse  has detected that  your X access  control list is not  empty.  Any\n\
user from a host on that list can capture your keyboard input and screen content.  Do\n\
not read  or enter sensitive  information in this  configuration!  See man  xhost for\n\
details."), -1);
            XFree(xha);
            return;
        }
        
        XFree(xha);
    }
}

int
prepare_decrypt_stdin() {
        if (seahorse_decrypt_arg!=NULL)
		verifydecrypt_active_btn = 1;
	else if (seahorse_verify_arg!=NULL)
		verifydecrypt_active_btn = 0;

	if (seahorse_file_arg != NULL) {
		datasrc_active_btn = 0;
		verifydecrypt_filename = strdup (seahorse_file_arg);
	} else read_from_file (STDIN_FILENO);

	if (seahorse_outfile_arg != NULL) {
		output_to_file = 1;
		verifydecrypt_outfilename = strdup (seahorse_outfile_arg);
	}
	on_decrypt_activate (NULL, NULL); 
	
	return 0;
}


int
prepare_encrypt_stdin() {
	if ((seahorse_sign_arg!=NULL) && (seahorse_encrypt_arg!=NULL))
		signencrypt_active_btn = 4;
	else if (seahorse_clearsign_arg!=NULL)
		signencrypt_active_btn = 0;
	else if (seahorse_detachsign_arg!=NULL)
		signencrypt_active_btn = 1;
	else if (seahorse_sign_arg!=NULL)
		signencrypt_active_btn = 2;
        else if( seahorse_encrypt_arg!=NULL )
		signencrypt_active_btn = 3;

	if (seahorse_file_arg != NULL) {
		datasrc_active_btn = 0;
		signencrypt_filename = strdup (seahorse_file_arg);
	} else {
		read_from_file (STDIN_FILENO);
	}
	if (seahorse_outfile_arg != NULL) {
		output_to_file = 1;
		signencrypt_outfilename = strdup (seahorse_outfile_arg);
	}
	if (seahorse_recipient_arg != NULL) {
		signencrypt_username = strdup (seahorse_recipient_arg);
	}
	if (seahorse_armor_arg != NULL)
		ascii_arm = 1;
	else ascii_arm = 0;
	if (seahorse_textmode_arg != NULL)
		textmode = 1;
	else textmode = 0;

	on_encrypt_activate (NULL, NULL); 

	return 0;
}


int main(int argc, char *argv[])
{
    char *gnupg_dir;
    GtkWidget *data_textbox, *encrypt_button, *decrypt_button;

    static GtkTargetEntry drag_types[] =
    {
        { "text/uri-list", 0, TARGET_URI_LIST },
    };
    static gint n_drag_types = sizeof(drag_types)/sizeof(drag_types[0]);

    bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
    textdomain(PACKAGE);

    gnome_init_with_popt_table("seahorse", VERSION, argc, argv, options, 0, NULL );

    main_win = create_main_win();
    security_check();
    data_textbox = get_widget (main_win,
			"data_textbox");
    gtk_drag_dest_set (data_textbox,
                       GTK_DEST_DEFAULT_MOTION |
                       GTK_DEST_DEFAULT_HIGHLIGHT |
                       GTK_DEST_DEFAULT_DROP,
                       drag_types, n_drag_types,
                       GDK_ACTION_COPY);
    encrypt_button = get_widget (main_win,
			"button11");
    gtk_drag_dest_set (encrypt_button,
                       GTK_DEST_DEFAULT_MOTION |
                       GTK_DEST_DEFAULT_HIGHLIGHT |
                       GTK_DEST_DEFAULT_DROP,
                       drag_types, n_drag_types,
                       GDK_ACTION_COPY);
    decrypt_button = get_widget (main_win,
			"button12");
    gtk_drag_dest_set (decrypt_button,
                       GTK_DEST_DEFAULT_MOTION |
                       GTK_DEST_DEFAULT_HIGHLIGHT |
                       GTK_DEST_DEFAULT_DROP,
                       drag_types, n_drag_types,
                       GDK_ACTION_COPY);

    /* Glade doesn't support the GNOMEUIINFO_HELP macro at the
     * moment, so we need to add this manually.
     */
    gnome_app_insert_menus (GNOME_APP (main_win), "_Help/_About...", help_menu_uiinfo);
    gtk_widget_set_usize (main_win, seahorse_width_arg, 
		    seahorse_height_arg);
    gtk_widget_set_uposition (main_win, xpos_arg, ypos_arg);

    gnupg_dir = gpg_directory();
    if( gnupg_dir == NULL) {
	genkey_druid = create_genkey_druid();
	gtk_widget_show(genkey_druid);
    } else {
	gtk_widget_show(main_win);
        free(gnupg_dir);

	load_config ();
	

        if((seahorse_clearsign_arg!=NULL) ||(seahorse_detachsign_arg!=NULL) || (seahorse_sign_arg!=NULL) ) {
            if( prepare_encrypt_stdin()!=0 )
                return 1;
        } else if((seahorse_decrypt_arg!=NULL) ||(seahorse_verify_arg!=NULL)) {
            if( prepare_decrypt_stdin()!=0 )
                return 1;
        } else if( seahorse_encrypt_arg!=NULL )
        {
            if( prepare_encrypt_stdin()!=0 )
                return 1;
        }
        
    }

    
    gtk_main();
    return 0;
}
