/*
 * Worldvisions Tunnel Vision Software:
 *   Copyright (C) 1998 Worldvisions Computer Technology, Inc.
 * 
 * Classes for establishing a network tunnel on top of any stream and
 * managing encrypted communications over the tunnel.  See wvtunnel.h.
 */
#include "wvtunnel.h"
#include "wviproute.h"
#include "wvver.h"
#include <net/ethernet.h>
#include <time.h>
#include <assert.h>

static ether_header real_ethhdr;
#define ETHER_EXTRA 16

WvTunnelList WvTunnel::all_tunnels;

// if you change this, remember to update the enum in wvtunnel.h!
char *WvTunnel::cmdstr[] = {
    "HELLO", "OK", "ERROR", "!!", "X", "help", "quit",
    "XOR", "RSA", "blowfish", "ihave", "password", "vpn",
    NULL
};


//////////////////////////// WvTunnelListener



WvTunnelListener::WvTunnelListener(const WvIPPortAddr &_listenport,
				   WvStreamList *_list, WvConf &_cfg)
		: WvTCPListener(_listenport), cfg(_cfg)
{
    list = _list;
}


void WvTunnelListener::execute()
{
    if (list)
	list->append(new WvTunnel(accept(), true, cfg), true);
}



/////////////////////////// WvTunnel



WvTunnel::WvTunnel(WvStream *_tunnel, bool server_mode, WvConf &_cfg)
		: WvProtoStream(&listp,
				new WvLog(WvString("Tunnel#%s", tapinit()))),
		  log(logp->split(WvLog::Info)),
		  cfg(_cfg), ifc(WvString("tap%s", tunlnum))
{
    listp = (WvStream *)&streamlist;
    
    all_tunnels.append(this, false);
    
    if (!cfg.get("Tunnel Vision", "RSA Private Key", NULL))
    {
	log("Generating RSA key...");
	WvRSAKey k(1024);

	cfg.set("Tunnel Vision", "RSA Private Key", k.private_str());
	cfg.set("Tunnel Vision", "RSA Public Key", k.public_str());
	cfg.flush();
	log("Done.\n");
    }
	
    streamlist.append(tunnel = _tunnel, false);
    
    trusted = vpn_started = false;
    inpkt_size = 0;
    etunnel = tunnel;
    last_receive = time(NULL);
    
    real_ethhdr.ether_dhost[0] = 0xFE;
    real_ethhdr.ether_dhost[1] = 0xFD;
    real_ethhdr.ether_shost[0] = 0xFE;
    real_ethhdr.ether_shost[1] = 0xFD;
    real_ethhdr.ether_type = htons(ETHERTYPE_IP);
    
    if (!tunnel->isok())
    {
	log(WvLog::Error, "tunnel: %s\n", tunnel->errstr());
	return;
    }
    
    if (!tap)
    {
	WvString s("No available ethertap (/dev/tap*) devices!\n");
	if (tunnel->select(0, false, true))
	    tunnel->print("ERROR %s", s);
	log(WvLog::Error, s);
	return;
    }
    
    if (!tap->isok())
    {
	log(WvLog::Error, "tap%s: %s\n", tunlnum, tap->errstr());
	return;
    }
    
    ifc.up(false);
    
    if (server_mode)
	switch_state(ServerReady);
    else
	switch_state(Greeting);
}


WvTunnel::~WvTunnel()
{
    all_tunnels.unlink(this);
    
    log("Closing tunnel.\n");
    if (geterr())
	log(WvLog::Error, "Error was: %s\n", errstr());
    
    if (etunnel && etunnel != tunnel)
	delete etunnel;
    if (tunnel)
	delete tunnel;
    if (tap)
	delete tap;
    ifc.up(false);
    
    if (added_gwroute != WvIPNet())
    {
	WvInterface i(added_ifc);
	i.delroute(added_gwroute, 99);
    }

    if (added_route != WvIPNet())
    {
	WvInterface i(added_ifc);
	i.delroute(added_route, 99);
    }
}


void WvTunnel::close()
{
    if (etunnel && etunnel != tunnel)
	delete etunnel;
    if (tunnel)
	delete tunnel;
    if (tap)
	delete tap;
    etunnel = tunnel = NULL;
    tap = NULL;
}


int WvTunnel::tapinit()
{
    for (tunlnum = 0; tunlnum < MAX_VTUNNELS; tunlnum++)
    {
	tap = new WvFile(WvString("/dev/tap%s", tunlnum), O_RDWR|O_EXCL);
	if (tap->isok())
	    break;
	else
	{
	    delete tap;
	    tap = NULL;
	}
    }
    
    return tunlnum;
}


void WvTunnel::execute()
{
    if (state == VPN_Running)
	vpn_mode();
    else
    {
	if (!etunnel) return;
	
	Token *t1 = tokline(etunnel->getline(0));
	if (t1)
	{
	    do_state(*t1);
	    delete t1;
	}
    }
}


void WvTunnel::switch_state(int newstate)
{
    switch ((State)newstate)
    {
    case ServerReady:
	log(WvLog::Info,
	    "Tunnel server connected to %s.\n", *tunnel->src());
	tunnel->print("HELLO Welcome to Tunnel Vision ("
		      TUNNELV_VER_STRING ")\n");
	break;
	
    case TrustedServerReady:
	break;
	
    case Greeting:
	log(WvLog::Info,
	    "Tunnel client connecting to %s.\n", *tunnel->src());
	break;

    case RSA:
	{
	    WvRSAKey prvkey(WvString(cfg.get("Tunnel Vision",
			      "RSA Private Key", "")).edit(), true);
	    etunnel->print("rsa %s\n", prvkey.public_str());
	}
	break;
    
    case Blowfish:
	WvRandomStream().read(keybuf, sizeof(keybuf));
	etunnel->print("blowfish %s\n", sizeof(keybuf) * 8);
	etunnel->write(keybuf, sizeof(keybuf));
	break;
    
    case IHAVE1:
    case IHAVE2:
	generate_ihave();
	break;
    
    case Password:
	etunnel->print("password %s\n",
		  cfg.get("Tunnel Vision", "Magic Password", "DUNNO"));
	break;
	
    case VPN:
	etunnel->print("vpn\n");
	break;

    case VPN_Running:
	log("Starting to exchange packets.\n");
	vpn_started = true;
	ifc.up(true);
	streamlist.append(tap, false);
	break;
	
    case None:
	tunnel->close();
	break;
    }
    
    state = newstate;
}


void WvTunnel::do_state(Token &t1)
{
    int cmd = tokanal(t1, cmdstr);
    
    if (cmd < 0)
    {
	etunnel->print("ERROR Unknown command \"%s\"\n", t1.data);
	return;
    }
    
    
    // preprocess some commands which are common to all states in both the
    // client and the server.
    if (cmd == cTrivia)
	return; // always ignore trivia
    else if (cmd == cError  &&  state != IHAVE1)
	switch_state(None); // fatal: fall to bottom of function
    else if (cmd == cRandomize)
    {
	log(WvLog::Debug,
	    "Received %s bytes of connection randomizer.\n",
	    tokbuf.used() - 1);
	return;
    }
    
    
    // now handle remaining commands depending on the system state.
    if (state == ServerReady || state == TrustedServerReady)
	do_server(cmd);
    else
	do_client(cmd);
}


void WvTunnel::do_client(int cmd)
{
    WvString next;

    switch ((State)state)
    {
    case ServerReady:
    case TrustedServerReady:
	assert(0);
	break;
	
    case Greeting:
	if (cmd != cHello) break; // fatal
	switch_state(RSA);
	return;

    case RSA:
	{
	    if (cmd != cOK) break; // fatal
	    
	    // switch to RSA
	    next = next_token_str();
	    WvRSAKey pubkey(next.edit(), false);
	    WvRSAKey prvkey(WvString(cfg.get("Tunnel Vision",
				    "RSA Private Key", "")).edit(), true);
	    newcrypt(new WvRSAStream(tunnel, prvkey, pubkey));
	    
	    // and ask for Blowfish
	    switch_state(Blowfish);
	    return;
	}
    
    case Blowfish:
	if (cmd != cOK) break; // fatal
	newcrypt(new WvBlowfishStream(tunnel, keybuf, sizeof(keybuf)));
	switch_state(IHAVE1);
	return;
    
    case IHAVE1:
    case IHAVE2:
	if (cmd == cOK)
	    switch_state(VPN);
	else if (cmd == cError  &&  state == IHAVE1)
	    switch_state(Password);
	else if (cmd == cIHave)
	{
	    next = token_remaining();
	    process_ihave(next.edit());
	}
	else
	    break; // fatal
	return;
    
    case Password:
	if (cmd != cOK) break; // fatal
	switch_state(IHAVE2);
	return;
	
    case VPN:
	if (cmd != cOK) break; // fatal
	switch_state(VPN_Running);
	return;
	
    case VPN_Running:
    case None:
	break;
    }
    
    log(WvLog::Error, "Fatal error while negotiating with the remote.\n");
    switch_state(None);
}


void WvTunnel::do_server(int cmd)
{
    WvString next;
    const char *pass;
    int bytes;
    unsigned char *keybuf;
    
    // if we get this far, cmd is (supposedly) guaranteed to have a correct
    // value.
    switch (cmd)
    {
    case cHelp:
	etunnel->print("OK Known commands: xor, rsa, blowfish, "
		  "password, ihave, vpn, quit\n");
	return;
	
    case cQuit:
	etunnel->print("OK Goodbye.\n");
	switch_state(None);
	return;
	
    case cXOR:
	next = next_token_str();
	etunnel->print("OK Switching to XOR %s encryption.\n", atoi(next));
	newcrypt(new WvXORStream(tunnel, atoi(next)));
	break;
	
    case cRSA:
	{
	    next = next_token_str();
	    WvRSAKey pubkey(next.edit(), false);
	    WvRSAKey prvkey(WvString(cfg.get("Tunnel Vision",
					     "RSA Private Key", "")).edit(),
			    true);
	    remote_rsa = pubkey.public_str();
	    remote_rsa.unique();
	    if (cfg.get("Tunnel Vision Keys", remote_rsa, false))
	    {
		etunnel->print("!! I recognize your public key.\n");
		switch_state(TrustedServerReady);
	    }
	    etunnel->print("OK %s\n", prvkey.public_str());
	    newcrypt(new WvRSAStream(tunnel, prvkey, pubkey));
	}
	break;
	
    case cBlowfish:
	next = next_token_str();
	bytes = atoi(next) / 8;
	
	if (bytes < 4)
	{
	    etunnel->print("ERROR Key needs at least 32 bits.\n");
	}
	else
	{
	    keybuf = new unsigned char[bytes];
	    
	    etunnel->queuemin(bytes);
	    etunnel->select(5000); // FIXME
	    etunnel->read(keybuf, bytes);
	    etunnel->queuemin(0);
	    etunnel->print("OK Switching to %s-bit blowfish "
			   "encryption.\n", bytes * 8);
	    newcrypt(new WvBlowfishStream(tunnel, keybuf, bytes));
	    
	    delete keybuf;
	}
	break;
	
    case cPassword:
	next = next_token_str();
	pass = cfg.get("Tunnel Vision", "Magic Password", NULL);
	if (pass && !strcmp(next, pass))
	{
	    etunnel->print("OK I'll trust you from now on.\n");
	    switch_state(TrustedServerReady);
	    
	    if (remote_rsa)
	    {
		cfg.set("Tunnel Vision Keys", remote_rsa, true);
		cfg.flush();
	    }
	}
	else
	{
	    state = ServerReady; // no need to reinitialize the state
	    etunnel->print("ERROR Nope.\n");
	    
	    if (remote_rsa)
	    {
		cfg.set("Tunnel Vision Keys", remote_rsa, false);
		cfg.flush();
	    }
	}
	break;
	
    case cIHave:
	if (state == TrustedServerReady)
	{
	    next = token_remaining();
	    generate_ihave();
	    process_ihave(next.edit());
	}
	else
	    etunnel->print("ERROR I don't trust you.\n");
	break;
	
    case cVPN:
	if (state == TrustedServerReady)
	{
	    etunnel->print("OK Let's do it!\n");
	    switch_state(VPN_Running);
	}
	else
	    etunnel->print("ERROR I don't trust you.\n");
	break;
    }
}


void WvTunnel::vpn_mode()
{
    char buf[WVCRYPTO_BUFSIZE + ETHER_EXTRA];
    size_t len;
    __u16 *outpkt_size;
    
    while (select(0))
    {
	if (etunnel->select(0))
	{
	    last_receive = time(NULL);
	    
	    if (!inpkt_size)
	    {
		etunnel->queuemin(2);
		
		// queuemin() call may mean that data is no longer ready!
		// we must select() again to make sure.
		if (etunnel->select(0))
		{
		    len = etunnel->read(&inpkt_size, sizeof(inpkt_size));
		    if (len)
		    {
			inpkt_size = ntohs(inpkt_size);
			
			if (inpkt_size < WVCRYPTO_BUFSIZE)
			    etunnel->queuemin(inpkt_size);
			else
			    etunnel->queuemin(WVCRYPTO_BUFSIZE);
		    }
		}
	    }
	    
	    if (inpkt_size && etunnel->select(0))
	    {
		len = etunnel->read(buf + ETHER_EXTRA, inpkt_size);
		if (len)
		{
		    inpkt_size = 0;
		    etunnel->queuemin(0);
		    
		    memcpy(buf + ETHER_EXTRA - sizeof(ether_header),
			   &real_ethhdr, sizeof(ether_header));
		    if (tap->select(0, false, true))
			tap->write(buf, len + ETHER_EXTRA);
		    log(WvLog::Debug4,
			  "%s-byte packet: Remote -> Local.\n",
			  len);
		}
	    }
	}
	
	if (tap->select(0))
	{
	    len = tap->read(buf, sizeof(buf));
	    if (len > 0)
	    {
		len -= ETHER_EXTRA;
		outpkt_size = ((__u16 *)(buf + ETHER_EXTRA)) - 1;
		*outpkt_size = htons(len);
		if (etunnel->select(0, false, true))
		    etunnel->write(outpkt_size, sizeof(*outpkt_size) + len);

		log(WvLog::Debug4, "%s-byte packet: Local -> Remote.\n", len);
	    }
	}
    }
}


void WvTunnel::newcrypt(WvStream *_etunnel)
{
    unsigned char buf[72];
    size_t count;
    
    if (etunnel && etunnel != tunnel)
    {
	streamlist.unlink(etunnel);
	delete etunnel;
    }

    if (_etunnel)
    {
	etunnel = _etunnel;
	streamlist.append(etunnel, false);
    }
    else
	etunnel = tunnel;
    
    /*
     * initialize the stream with some randomness, to lessen the probability
     * of "known plaintext" attacks.
     */
    WvRandomStream().read(buf, sizeof(buf));
    for (count = 0; count < sizeof(buf); count++)
	while (buf[count] < 32) buf[count] += 7;
    
    count = sizeof(buf) - 1;
    count -= buf[count] / 4;
    buf[count] = 0;

    etunnel->print("X %s\n", (char *)buf);
}


bool WvTunnel::idle_timeout() const
{
    return last_receive + 10*60 < time(NULL);
}
    
    
bool WvTunnel::isok() const
{
    return tap && tunnel && tap->isok() && tunnel->isok() && !idle_timeout();
}


int WvTunnel::geterr() const
{
    if (tunlnum < 0)
	return -1;
    else if (tunnel && !tunnel->isok())
	return tunnel->geterr();
    else if (tap)
	return tap->geterr();
    else
	return 0;
}


const char *WvTunnel::errstr() const
{
    if (tunlnum < 0)
	return "No available ethertap (/dev/tap*) devices";
    else if (!tunnel->isok())
	return tunnel->errstr();
    else
	return tap->errstr();
}


const WvAddr *WvTunnel::src() const
{
    return tunnel ? tunnel->src() : NULL;
}


void WvTunnel::process_ihave(char *line)
{
    char *endptr, *maskptr;
    WvIPAddr remote(tunnel->src() ? *(WvIPPortAddr *)tunnel->src() : WvIPAddr());
    WvIPNet net;
    WvInterfaceDict id;
    
    ifc.up(true);
    
    // make sure that any new IHAVE entries do not conflict with the route
    // to our tunnel vision partner: add an explicit route to him through
    // the right interface.
    WvIPRouteList rl;
    rl.get_kernel();
    if (tunnel->src())
    {
	WvIPRoute *whichroute = rl.find(remote);
	if (whichroute)
	{
	    WvInterface i(whichroute->ifc);

	    if (whichroute->gateway == WvIPAddr())
	    {
		if (!i.addroute(remote, 99))
		{
		    added_ifc = whichroute->ifc;
		    added_route = remote;
		}
	    }
	    else // gateway does exist
	    {
		if (!i.addroute(whichroute->gateway, 99))
		{
		    added_ifc = whichroute->ifc;
		    added_route = whichroute->gateway;
		}
	    
		if (!i.addroute(remote, whichroute->gateway, 99))
		{
		    added_ifc = whichroute->ifc;
		    added_gwroute = remote;
		}
	    }
	}
    }
    

    line = trim_string(line);
    while (line && line[0])
    {
	endptr = strchr(line, ' ');
	if (endptr)
	    *endptr++ = 0;
	
	maskptr = strchr(line, '/');
	if (maskptr)
	{
	    *maskptr++ = 0;
	    net = WvIPNet(WvIPAddr(line), WvIPAddr(maskptr));
	}
	else
	    net = WvIPNet(WvIPAddr(line));
	
	if ((net.bits() < 32 || net.base() != remote)
	    && net != WvIPNet() && !id.on_local_net(net))
	{
	    ifc.addroute(net, 99);
	    etunnel->print("!! Routing %s to you through the tunnel.\n", net);
	    remote_nets.append(new WvIPNet(net), true);
	}
		       
	line = endptr ? trim_string(endptr) : NULL;
    }

#if 0    
    // FIXME!  Oh BOY, FIXME!!!!!
    // this appears to be caused by a kernel bug.  Downing the ethertap
    // interface will cause the kernel to stop answering for its local
    // ppp address unless we set it up as an IP alias... sometimes.
    etunnel->print("!! Fixing my own address...\n");
    WvInterface xx("ppp0"), aa("lo:dual");
    xx.rescan();
    aa.setipaddr(xx.ipaddr());
    aa.up(false);
    aa.setipaddr(xx.ipaddr());
    aa.up(true);
#endif
    etunnel->print("OK I've configured my routing table.\n");
}


void WvTunnel::generate_ihave()
{
    const char *stored = cfg.get("Tunnel Vision", "Local Nets", NULL);
    WvIPAddr lastaddr;
    bool did_set = false;
    
    if (stored)
    {
	ifc.setipaddr(WvIPNet(stored, 32));
	etunnel->print("IHAVE %s\n", stored);
    }
    else
    {
	WvInterfaceDict id;
	WvInterfaceDict::Iter i(id);
	
	etunnel->write("IHAVE ", 6);
	
	for (i.rewind(); i.next(); )
	{
	    WvInterface &ii = i;
	    
	    if (!ii.valid
		|| ii.hwaddr().encap() == WvEncap::Loopback
		|| WvIPAddr(ii.ipaddr()) == WvIPAddr()
		|| !strncmp(ii.name, "tap", 3)
		|| !ii.isup())
		  continue;
	    
	    etunnel->print("%s ", ii.ipaddr());
	    
	    if (ii.hwaddr().encap() == WvEncap::Ethernet && !did_set)
	    {
		lastaddr = ii.ipaddr();
		did_set = true;
	    }
	}
	
	ifc.setipaddr(WvIPNet(lastaddr, 32));
	etunnel->print("\n");
    }
}
