// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/views/relaunch_notification/relaunch_notification_controller_platform_impl_chromeos.h"

#include "chrome/browser/ui/ash/system_tray_client.h"
#include "chrome/browser/ui/views/relaunch_notification/relaunch_notification_metrics.h"
#include "chrome/browser/ui/views/relaunch_notification/relaunch_recommended_timer.h"
#include "chrome/browser/ui/views/relaunch_notification/relaunch_required_timer.h"
#include "chrome/grit/chromium_strings.h"
#include "chrome/grit/generated_resources.h"
#include "ui/base/l10n/l10n_util.h"

RelaunchNotificationControllerPlatformImpl::
    RelaunchNotificationControllerPlatformImpl() = default;

RelaunchNotificationControllerPlatformImpl::
    ~RelaunchNotificationControllerPlatformImpl() = default;

void RelaunchNotificationControllerPlatformImpl::NotifyRelaunchRecommended(
    base::Time detection_time) {
  if (!relaunch_recommended_timer_) {
    relaunch_recommended_timer_ = std::make_unique<RelaunchRecommendedTimer>(
        detection_time,
        base::BindRepeating(&RelaunchNotificationControllerPlatformImpl::
                                RefreshRelaunchRecommendedTitle,
                            base::Unretained(this)));

    relaunch_notification::RecordRecommendedShowResult(
        relaunch_notification::ShowResult::kShown);
  }

  RefreshRelaunchRecommendedTitle();
}

void RelaunchNotificationControllerPlatformImpl::NotifyRelaunchRequired(
    base::Time deadline) {
  if (!relaunch_required_timer_) {
    relaunch_required_timer_ = std::make_unique<RelaunchRequiredTimer>(
        deadline,
        base::BindRepeating(&RelaunchNotificationControllerPlatformImpl::
                                RefreshRelaunchRequiredTitle,
                            base::Unretained(this)));

    relaunch_notification::RecordRequiredShowResult(
        relaunch_notification::ShowResult::kShown);
  }

  RefreshRelaunchRequiredTitle();
}

void RelaunchNotificationControllerPlatformImpl::CloseRelaunchNotification() {
  SystemTrayClient::Get()->SetUpdateNotificationState(
      ash::mojom::NotificationStyle::DEFAULT, base::string16(),
      base::string16());

  relaunch_recommended_timer_.reset();
  relaunch_required_timer_.reset();
}

void RelaunchNotificationControllerPlatformImpl::SetDeadline(
    base::Time deadline) {
  relaunch_required_timer_->SetDeadline(deadline);
}

void RelaunchNotificationControllerPlatformImpl::
    RefreshRelaunchRecommendedTitle() {
  SystemTrayClient::Get()->SetUpdateNotificationState(
      ash::mojom::NotificationStyle::ADMIN_RECOMMENDED,
      relaunch_recommended_timer_->GetWindowTitle(),
      l10n_util::GetStringUTF16(IDS_RELAUNCH_RECOMMENDED_BODY));
}

void RelaunchNotificationControllerPlatformImpl::
    RefreshRelaunchRequiredTitle() {
  SystemTrayClient::Get()->SetUpdateNotificationState(
      ash::mojom::NotificationStyle::ADMIN_REQUIRED,
      relaunch_required_timer_->GetWindowTitle(),
      l10n_util::GetStringUTF16(IDS_RELAUNCH_REQUIRED_BODY));
}

bool RelaunchNotificationControllerPlatformImpl::IsRequiredNotificationShown()
    const {
  return relaunch_required_timer_ != nullptr;
}
