// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_BOREALIS_BOREALIS_CONTEXT_MANAGER_H_
#define CHROME_BROWSER_CHROMEOS_BOREALIS_BOREALIS_CONTEXT_MANAGER_H_

#include <string>

#include "base/callback.h"
#include "chrome/browser/chromeos/borealis/borealis_metrics.h"
#include "chrome/browser/chromeos/borealis/infra/described.h"
#include "chrome/browser/chromeos/borealis/infra/expected.h"
#include "components/keyed_service/core/keyed_service.h"

namespace borealis {

class BorealisContext;

class BorealisContextManager : public KeyedService {
 public:
  // An attempt to launch borealis. If the launch succeeds, holds a reference to
  // the context created for that launch, otherwise holds an error.
  using ContextOrFailure =
      Expected<BorealisContext*, Described<BorealisStartupResult>>;

  // Convenience definition for the callback provided by clients wanting to
  // launch borealis.
  using ResultCallback = base::OnceCallback<void(ContextOrFailure)>;

  BorealisContextManager() = default;
  BorealisContextManager(const BorealisContextManager&) = delete;
  BorealisContextManager& operator=(const BorealisContextManager&) = delete;
  ~BorealisContextManager() override = default;

  // Starts the Borealis VM and/or runs the callback when it is running.
  virtual void StartBorealis(ResultCallback callback) = 0;

  // Returns true if the VM is currently running.
  virtual bool IsRunning() = 0;

  // Stop the current running state, re-initializing the context manager
  // to the state it was in prior to being started. All pending callbacks are
  // invoked with kCancelled result.
  virtual void ShutDownBorealis() = 0;
};

}  // namespace borealis

#endif  // CHROME_BROWSER_CHROMEOS_BOREALIS_BOREALIS_CONTEXT_MANAGER_H_
