// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/spellchecker/spell_check_host_chrome_impl.h"

#include "base/bind.h"
#include "base/no_destructor.h"
#include "base/strings/utf_string_conversions.h"
#include "build/build_config.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/spellchecker/spellcheck_custom_dictionary.h"
#include "chrome/browser/spellchecker/spellcheck_factory.h"
#include "chrome/browser/spellchecker/spellcheck_service.h"
#include "components/spellcheck/browser/spellcheck_host_metrics.h"
#include "components/spellcheck/browser/spellcheck_platform.h"
#include "components/spellcheck/common/spellcheck_features.h"
#include "components/spellcheck/spellcheck_buildflags.h"
#include "content/public/browser/browser_context.h"
#include "content/public/browser/browser_thread.h"
#include "content/public/browser/render_process_host.h"
#include "mojo/public/cpp/bindings/self_owned_receiver.h"

#if BUILDFLAG(USE_BROWSER_SPELLCHECKER) && BUILDFLAG(ENABLE_SPELLING_SERVICE)
#include "chrome/browser/spellchecker/spelling_request.h"
#endif

#if defined(OS_CHROMEOS)
#include "chromeos/constants/chromeos_features.h"
#endif

namespace {

SpellCheckHostChromeImpl::Binder& GetSpellCheckHostBinderOverride() {
  static base::NoDestructor<SpellCheckHostChromeImpl::Binder> binder;
  return *binder;
}

}  // namespace

SpellCheckHostChromeImpl::SpellCheckHostChromeImpl(int render_process_id)
    : render_process_id_(render_process_id) {}

SpellCheckHostChromeImpl::~SpellCheckHostChromeImpl() = default;

// static
void SpellCheckHostChromeImpl::Create(
    int render_process_id,
    mojo::PendingReceiver<spellcheck::mojom::SpellCheckHost> receiver) {
  auto& binder = GetSpellCheckHostBinderOverride();
  if (binder) {
    binder.Run(render_process_id, std::move(receiver));
    return;
  }

  mojo::MakeSelfOwnedReceiver(
      std::make_unique<SpellCheckHostChromeImpl>(render_process_id),
      std::move(receiver));
}

// static
void SpellCheckHostChromeImpl::OverrideBinderForTesting(Binder binder) {
  GetSpellCheckHostBinderOverride() = std::move(binder);
}

void SpellCheckHostChromeImpl::RequestDictionary() {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  // The renderer has requested that we initialize its spellchecker. This
  // generally should only be called once per session, as after the first
  // call, future renderers will be passed the initialization information
  // on startup (or when the dictionary changes in some way).
  SpellcheckService* spellcheck = GetSpellcheckService();
  if (!spellcheck)
    return;  // Teardown.

  // The spellchecker initialization already started and finished; just
  // send it to the renderer.
  auto* host = content::RenderProcessHost::FromID(render_process_id_);
  if (host)
    spellcheck->InitForRenderer(host);

  // TODO(rlp): Ensure that we do not initialize the hunspell dictionary
  // more than once if we get requests from different renderers.
}

void SpellCheckHostChromeImpl::NotifyChecked(const base::string16& word,
                                             bool misspelled) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  SpellcheckService* spellcheck = GetSpellcheckService();
  if (!spellcheck)
    return;  // Teardown.
  if (spellcheck->GetMetrics())
    spellcheck->GetMetrics()->RecordCheckedWordStats(word, misspelled);
}

#if BUILDFLAG(USE_RENDERER_SPELLCHECKER)
void SpellCheckHostChromeImpl::CallSpellingService(
    const base::string16& text,
    CallSpellingServiceCallback callback) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  if (text.empty()) {
    std::move(callback).Run(false, std::vector<SpellCheckResult>());
    mojo::ReportBadMessage("Requested spelling service with empty text");
    return;
  }

  // Checks the user profile and sends a JSON-RPC request to the Spelling
  // service if a user enables the "Use enhanced spell check" option. When
  // a response is received (including an error) from the remote Spelling
  // service, calls CallSpellingServiceDone.
  auto* host = content::RenderProcessHost::FromID(render_process_id_);
  if (!host) {
    std::move(callback).Run(false, std::vector<SpellCheckResult>());
    return;
  }
  client_.RequestTextCheck(
      host->GetBrowserContext(), SpellingServiceClient::SPELLCHECK, text,
      base::BindOnce(&SpellCheckHostChromeImpl::CallSpellingServiceDone,
                     weak_factory_.GetWeakPtr(), std::move(callback)));
}

void SpellCheckHostChromeImpl::CallSpellingServiceDone(
    CallSpellingServiceCallback callback,
    bool success,
    const base::string16& text,
    const std::vector<SpellCheckResult>& service_results) const {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  SpellcheckService* spellcheck = GetSpellcheckService();
  if (!spellcheck) {  // Teardown.
    std::move(callback).Run(false, std::vector<SpellCheckResult>());
    return;
  }

  std::vector<SpellCheckResult> results = FilterCustomWordResults(
      base::UTF16ToUTF8(text), *spellcheck->GetCustomDictionary(),
      service_results);

#if defined(OS_CHROMEOS)
  if (base::FeatureList::IsEnabled(chromeos::features::kOnDeviceGrammarCheck) &&
      results.empty()) {
    auto* host = content::RenderProcessHost::FromID(render_process_id_);
    if (!host) {
      std::move(callback).Run(false, std::vector<SpellCheckResult>());
      return;
    }
    grammar_client_.RequestTextCheck(
        Profile::FromBrowserContext(host->GetBrowserContext()), text,
        base::BindOnce(&SpellCheckHostChromeImpl::CallGrammarServiceDone,
                       weak_factory_.GetWeakPtr(), std::move(callback)));
    return;
  }
#endif

  std::move(callback).Run(success, results);
}

#if defined(OS_CHROMEOS)
void SpellCheckHostChromeImpl::CallGrammarServiceDone(
    CallSpellingServiceCallback callback,
    bool success,
    const std::vector<SpellCheckResult>& results) const {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);
  SpellcheckService* spellcheck = GetSpellcheckService();
  if (!spellcheck) {  // Teardown.
    std::move(callback).Run(false, std::vector<SpellCheckResult>());
    return;
  }
  std::move(callback).Run(success, results);
}
#endif

// static
std::vector<SpellCheckResult> SpellCheckHostChromeImpl::FilterCustomWordResults(
    const std::string& text,
    const SpellcheckCustomDictionary& custom_dictionary,
    const std::vector<SpellCheckResult>& service_results) {
  std::vector<SpellCheckResult> results;
  for (const auto& result : service_results) {
    const std::string word = text.substr(result.location, result.length);
    if (!custom_dictionary.HasWord(word))
      results.push_back(result);
  }

  return results;
}
#endif  // BUILDFLAG(USE_RENDERER_SPELLCHECKER)

#if BUILDFLAG(USE_BROWSER_SPELLCHECKER) && BUILDFLAG(ENABLE_SPELLING_SERVICE)
void SpellCheckHostChromeImpl::CheckSpelling(const base::string16& word,
                                             int route_id,
                                             CheckSpellingCallback callback) {
  bool correct = spellcheck_platform::CheckSpelling(word, route_id);
  std::move(callback).Run(correct);
}

void SpellCheckHostChromeImpl::FillSuggestionList(
    const base::string16& word,
    FillSuggestionListCallback callback) {
  std::vector<base::string16> suggestions;
  spellcheck_platform::FillSuggestionList(word, &suggestions);
  std::move(callback).Run(suggestions);
}

void SpellCheckHostChromeImpl::RequestTextCheck(
    const base::string16& text,
    int route_id,
    RequestTextCheckCallback callback) {
  DCHECK(!text.empty());
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  // Initialize the spellcheck service if needed. The service will send the
  // language code for text breaking to the renderer. (Text breaking is required
  // for the context menu to show spelling suggestions.) Initialization must
  // happen on UI thread.
  SpellcheckService* spellcheck = GetSpellcheckService();

  if (!spellcheck) {  // Teardown.
    std::move(callback).Run({});
    return;
  }

  // OK to store unretained |this| in a |SpellingRequest| owned by |this|.
  requests_.insert(std::make_unique<SpellingRequest>(
      spellcheck->platform_spell_checker(), &client_, text, render_process_id_,
      route_id, std::move(callback),
      base::BindOnce(&SpellCheckHostChromeImpl::OnRequestFinished,
                     base::Unretained(this))));
}

#if defined(OS_WIN)
void SpellCheckHostChromeImpl::GetPerLanguageSuggestions(
    const base::string16& word,
    GetPerLanguageSuggestionsCallback callback) {
  SpellcheckService* spellcheck = GetSpellcheckService();

  if (!spellcheck) {  // Teardown.
    std::move(callback).Run({});
    return;
  }

  spellcheck_platform::GetPerLanguageSuggestions(
      spellcheck->platform_spell_checker(), word, std::move(callback));
}

void SpellCheckHostChromeImpl::InitializeDictionaries(
    InitializeDictionariesCallback callback) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  if (base::FeatureList::IsEnabled(
          spellcheck::kWinDelaySpellcheckServiceInit)) {
    // Initialize the spellcheck service if needed. Initialization must
    // happen on UI thread.
    SpellcheckService* spellcheck = GetSpellcheckService();

    if (!spellcheck) {  // Teardown.
      std::move(callback).Run(/*dictionaries=*/{}, /*custom_words=*/{},
                              /*enable=*/false);
      return;
    }

    dictionaries_loaded_callback_ = std::move(callback);

    spellcheck->InitializeDictionaries(
        base::BindOnce(&SpellCheckHostChromeImpl::OnDictionariesInitialized,
                       weak_factory_.GetWeakPtr()));
    return;
  }

  NOTREACHED();
  std::move(callback).Run(/*dictionaries=*/{}, /*custom_words=*/{},
                          /*enable=*/false);
}

void SpellCheckHostChromeImpl::OnDictionariesInitialized() {
  DCHECK(dictionaries_loaded_callback_);
  SpellcheckService* spellcheck = GetSpellcheckService();

  if (!spellcheck) {  // Teardown.
    std::move(dictionaries_loaded_callback_)
        .Run(/*dictionaries=*/{}, /*custom_words=*/{},
             /*enable=*/false);
    return;
  }

  const bool enable = spellcheck->IsSpellcheckEnabled();

  std::vector<spellcheck::mojom::SpellCheckBDictLanguagePtr> dictionaries;
  std::vector<std::string> custom_words;
  if (enable) {
    for (const auto& hunspell_dictionary :
         spellcheck->GetHunspellDictionaries()) {
      dictionaries.push_back(spellcheck::mojom::SpellCheckBDictLanguage::New(
          hunspell_dictionary->GetDictionaryFile().Duplicate(),
          hunspell_dictionary->GetLanguage()));
    }

    SpellcheckCustomDictionary* custom_dictionary =
        spellcheck->GetCustomDictionary();
    custom_words.assign(custom_dictionary->GetWords().begin(),
                        custom_dictionary->GetWords().end());
  }

  std::move(dictionaries_loaded_callback_)
      .Run(std::move(dictionaries), custom_words, enable);
}
#endif  // defined(OS_WIN)

void SpellCheckHostChromeImpl::OnRequestFinished(SpellingRequest* request) {
  auto iterator = requests_.find(request);
  requests_.erase(iterator);
}

// static
void SpellCheckHostChromeImpl::CombineResultsForTesting(
    std::vector<SpellCheckResult>* remote_results,
    const std::vector<SpellCheckResult>& local_results) {
  SpellingRequest::CombineResults(remote_results, local_results);
}
#endif  //  BUILDFLAG(USE_BROWSER_SPELLCHECKER) &&
        //  BUILDFLAG(ENABLE_SPELLING_SERVICE)

#if defined(OS_MAC)
int SpellCheckHostChromeImpl::ToDocumentTag(int route_id) {
  if (!tag_map_.count(route_id))
    tag_map_[route_id] = spellcheck_platform::GetDocumentTag();
  return tag_map_[route_id];
}

// TODO(groby): We are currently not notified of retired tags. We need
// to track destruction of RenderViewHosts on the browser process side
// to update our mappings when a document goes away.
void SpellCheckHostChromeImpl::RetireDocumentTag(int route_id) {
  spellcheck_platform::CloseDocumentWithTag(ToDocumentTag(route_id));
  tag_map_.erase(route_id);
}
#endif  // defined(OS_MAC)

SpellcheckService* SpellCheckHostChromeImpl::GetSpellcheckService() const {
  auto* host = content::RenderProcessHost::FromID(render_process_id_);
  if (!host)
    return nullptr;
  return SpellcheckServiceFactory::GetForContext(host->GetBrowserContext());
}
