// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/nearby_sharing/bluetooth_advertising_interval_client.h"

#include <utility>

#include "base/callback_helpers.h"
#include "base/time/time.h"
#include "chrome/browser/nearby_sharing/logging/logging.h"
#include "device/bluetooth/bluetooth_adapter.h"

namespace {

constexpr base::TimeDelta kIntervalMin = base::TimeDelta::FromMilliseconds(100);
constexpr base::TimeDelta kIntervalMax = base::TimeDelta::FromMilliseconds(100);

// A value of 0 will restore the interval to the system default.
constexpr base::TimeDelta kDefaultIntervalMin =
    base::TimeDelta::FromMilliseconds(0);
constexpr base::TimeDelta kDefaultIntervalMax =
    base::TimeDelta::FromMilliseconds(0);

}  // namespace

BluetoothAdvertisingIntervalClient::BluetoothAdvertisingIntervalClient(
    scoped_refptr<device::BluetoothAdapter> adapter)
    : adapter_(adapter) {}

BluetoothAdvertisingIntervalClient::~BluetoothAdvertisingIntervalClient() {
  RestoreDefaultInterval();
}

void BluetoothAdvertisingIntervalClient::ReduceInterval() {
  adapter_->SetAdvertisingInterval(
      kIntervalMin, kIntervalMax, base::DoNothing(),
      base::BindOnce(
          &BluetoothAdvertisingIntervalClient::OnSetIntervalForAdvertisingError,
          weak_ptr_factory_.GetWeakPtr()));
}

void BluetoothAdvertisingIntervalClient::RestoreDefaultInterval() {
  adapter_->SetAdvertisingInterval(
      kDefaultIntervalMin, kDefaultIntervalMax, base::DoNothing(),
      base::BindOnce(
          &BluetoothAdvertisingIntervalClient::OnRestoreDefaultIntervalError,
          weak_ptr_factory_.GetWeakPtr()));
}

void BluetoothAdvertisingIntervalClient::OnSetIntervalForAdvertisingError(
    device::BluetoothAdvertisement::ErrorCode code) {
  NS_LOG(WARNING) << __func__
                  << ": SetAdvertisingInterval() failed with error code = "
                  << code;
}

void BluetoothAdvertisingIntervalClient::OnRestoreDefaultIntervalError(
    device::BluetoothAdvertisement::ErrorCode code) {
  NS_LOG(WARNING) << __func__
                  << ": SetAdvertisingInterval() failed with error code = "
                  << code;
}
