// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/share/qr_code_generation_request.h"

#include <jni.h>

#include "base/android/jni_string.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/browser/share/android/jni_headers/BitmapDownloadRequest_jni.h"
#include "content/public/browser/download_manager.h"
#include "content/public/browser/download_request_utils.h"
#include "third_party/skia/include/core/SkBitmap.h"
#include "ui/base/webui/web_ui_util.h"
#include "ui/gfx/android/java_bitmap.h"

using base::android::JavaParamRef;

static void JNI_BitmapDownloadRequest_DownloadBitmap(
    JNIEnv* env,
    const JavaParamRef<jstring>& j_filename,
    const JavaParamRef<jobject>& j_bitmap) {
  base::string16 filename(ConvertJavaStringToUTF16(env, j_filename));
  SkBitmap bitmap =
      gfx::CreateSkBitmapFromJavaBitmap(gfx::JavaBitmap(j_bitmap));

  const GURL data_url = GURL(webui::GetBitmapDataUrl(bitmap));

  content::DownloadManager* download_manager =
      content::BrowserContext::GetDownloadManager(
          ProfileManager::GetLastUsedProfile());
  net::NetworkTrafficAnnotationTag traffic_annotation =
      net::DefineNetworkTrafficAnnotation("download_bitmap", R"(
        semantics {
          sender: "Sharing Hub Android"
          description:
            "Download bitmap request sent by chrome browser share component on Android."
          trigger: "User clicks 'download' from Sharing Hub on Android."
          data: "Bitmap generated by Sharing Hub feature."
          destination: LOCAL
        }
        policy {
          cookies_allowed: NO
          setting:
            "Users cannot disable this feature."
          policy_exception_justification:
            "N/A"
        })");
  auto params = std::make_unique<download::DownloadUrlParameters>(
      data_url, traffic_annotation);
  params->set_suggested_name(filename);
  download_manager->DownloadUrl(std::move(params));
}
