/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "autoreload.h"
#include "glade.h"
#include "dialog.h"
#include "embed.h"

#include <stdlib.h>
#include <string.h>
#include <gdk/gdk.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkspinbutton.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-util.h>
#include <gtkmozembed.h>

typedef struct
{
	GaleonEmbed *embed;
	GtkWidget *dialog;
	GtkWidget *check;
	GtkWidget *spin;
	GtkWidget *optionmenu;
	GtkWidget *label;
} AutoreloadDialog;

/* function prototypes */
static gint autoreload_timeout_cb (GaleonEmbed *embed);
void autoreload_ok_clicked_cb (GtkWidget *widget, AutoreloadDialog *dialog);
void autoreload_cancel_clicked_cb (GtkWidget *widget, AutoreloadDialog *dialog);
void autoreload_dialog_destroy_cb (GtkWidget *widget, AutoreloadDialog *dialog);
void autoreload_enable_toggled_cb (GtkToggleButton *button,
				   AutoreloadDialog *dialog);
static gint get_int_from_optionmenu (GtkWidget *optionmenu);
static void set_sensitivity (AutoreloadDialog *dialog, gboolean sensitive);

/**
 * autoreload_show_dialog: Open autoreload dialog
 */
void
autoreload_show_dialog (GaleonEmbed *embed)
{
	AutoreloadDialog *dialog = g_new0 (AutoreloadDialog, 1);
	GladeXML *gxml;
	
	/* build dialog from glade file */
	gxml = glade_widget_new ("galeon.glade", 
				 "autoreload_dialog",
				 &(dialog->dialog), dialog);
	
	dialog->embed = embed;
	dialog->check =
		glade_xml_get_widget (gxml, "autoreload_check");
	dialog->spin =
		glade_xml_get_widget (gxml, "autoreload_spin");
	dialog->optionmenu =
		glade_xml_get_widget (gxml, "autoreload_optionmenu");
	dialog->label =
		glade_xml_get_widget (gxml, "autoreload_label");

	/* sync dialog & embed */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->check),
				      embed->autoreload_info->enabled);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->spin),
				   embed->autoreload_info->delay);
	gtk_option_menu_set_history (GTK_OPTION_MENU (dialog->optionmenu),
				     embed->autoreload_info->delay_unit);
	set_sensitivity (dialog, embed->autoreload_info->enabled);
	dialog_set_parent (dialog->dialog, embed->parent_window->wmain);

	gtk_object_unref (GTK_OBJECT (gxml));
}

/**
 * autoreload_ok_clicked_cb: Apply clicked in the autoreload dialog
 */
void
autoreload_ok_clicked_cb (GtkWidget *widget, AutoreloadDialog *dialog)
{
	AutoreloadInfo *info = dialog->embed->autoreload_info;

	/* sync embed & dialog */
	info->enabled = GTK_TOGGLE_BUTTON (dialog->check)->active;
	info->delay = gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON (dialog->spin));
	info->delay_unit =
		get_int_from_optionmenu (dialog->optionmenu);

	/* get rid of existing timeout */
	if (info->timeout_id)
	{
		gtk_timeout_remove (info->timeout_id);
	}

	/* add new timeout if requested */
	if (info->enabled)
	{
		gint unit = 0;

		switch (info->delay_unit)
		{
		case 0: /* seconds */
			unit = 1000;
			break;
		case 1: /* minutes */
			unit = 60 * 1000;
			break;
		case 2: /* hours */
			unit = 60 * 60 * 1000;
			break;
		default:
			g_warning ("Should never get here.\n");
			break;
		}

		info->timeout_id = gtk_timeout_add (info->delay * unit,
						    (GtkFunction)
						    	autoreload_timeout_cb,
						    dialog->embed);
	}
	else
	{
		info->timeout_id = 0;
	}

	/* get rid of dialog */
	gtk_widget_destroy (dialog->dialog);
}

/**
 * autoreload_cancel_clicked_cb: Close clicked in the autoreload dialog
 */
void
autoreload_cancel_clicked_cb (GtkWidget *widget, AutoreloadDialog *dialog)
{
	gtk_widget_destroy (dialog->dialog);
}

/**
 * autoreload_dialog_destroy_cb: Autoreload dialog destroyed
 */
void
autoreload_dialog_destroy_cb (GtkWidget *widget, AutoreloadDialog *dialog)
{
	g_free (dialog);
}

/**
 * autoreload_enable_toggled_cb: "Enable autoreload" toggled
 */
void
autoreload_enable_toggled_cb (GtkToggleButton *button, AutoreloadDialog *dialog)
{
	set_sensitivity (dialog, button->active);
}

/**
 * set_sensitivity: set widget sensitivity
 */
static void
set_sensitivity (AutoreloadDialog *dialog, gboolean sensitive)
{
	gtk_widget_set_sensitive (dialog->spin, sensitive);
	gtk_widget_set_sensitive (dialog->optionmenu, sensitive);
	gtk_widget_set_sensitive (dialog->label, sensitive);
}

/**
 * autoreload_timeout_cb: timeout callback, reload the embed periodically
 */
static gint
autoreload_timeout_cb (GaleonEmbed *embed)
{
	if (embed == NULL) return FALSE;
	if (!embed_exists (embed)) return FALSE;
	if (!embed->autoreload_info->enabled)
	{
		embed->autoreload_info->timeout_id = 0;
		return FALSE;
	}
	embed_reload (embed, GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
	return TRUE;
}

/**
 * get_int_from_optionmenu: get optionmenu selection as an integer
 */
static gint
get_int_from_optionmenu (GtkWidget *optionmenu)
{
	GtkWidget *menu;
	GList *list;
	gpointer item;
	gint index;

	/* extract the selection */
	menu = GTK_OPTION_MENU (optionmenu)->menu;
	list = GTK_MENU_SHELL (menu)->children;
	item = gtk_menu_get_active (GTK_MENU (menu));
	index = g_list_index (list, item);
	
	return index;
}
