/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <typeinfo>
#include <qcursor.h>
#include <qpixmap.h>
#include <qimage.h>

#include <kapplication.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <kdesktopfile.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmimetype.h>
#include <krun.h>
#include <kservicegroup.h>
#include <ksycoca.h>
#include <ksycocaentry.h>
#include <kservice.h>
#include <kurldrag.h>

#include "kicker.h"
#include "recentapps.h"
#include "service_mnu.h"
#include "service_mnu.moc"
#include "popupmenutitle.h"
#include "popupposition.h"
#include "panelbutton.h"

const int idStart = 4242;
const int kRecentMenusOffset = 1001;

// static object that stores "recent" history
static RecentlyLaunchedApps s_RecentApps;

PanelServiceMenu::PanelServiceMenu(const QString & label, const QString & relPath, QWidget * parent,
                                   const char * name, bool addmenumode)
    : KPanelMenu(label, parent, name), relPath_(relPath), clearOnClose_(false),addmenumode_(addmenumode)
{
    readConfig();
    subMenus.setAutoDelete(true);

    excludeNoDisplay_=true;

    connect(KSycoca::self(),  SIGNAL(databaseChanged()),  SLOT(slotClearOnClose()));
    connect(Kicker::kicker(), SIGNAL(configurationChanged()), this, SLOT(configChanged()) );
    connect(this, SIGNAL(aboutToHide()), this, SLOT(slotClose()));
}

PanelServiceMenu::~PanelServiceMenu()
{
}

void PanelServiceMenu::readConfig()
{
    KConfigGroup group( KGlobal::config(), "menus" );
    merge_              = group.readBoolEntry( "MergeKDEDirs",              TRUE );
    detailed_           = group.readBoolEntry( "DetailedMenuEntries",       TRUE );
    detailedNamesFirst_ = group.readBoolEntry( "DetailedEntriesNamesFirst", FALSE );
}

// create and fill "recent" section at first
void PanelServiceMenu::createRecentMenuItems()
{
    s_RecentApps.init();
    s_RecentApps.m_nNumMenuItems = 0;

    QStringList RecentApps;
    s_RecentApps.getRecentApps(RecentApps);

    if (RecentApps.count() > 0) {
        KConfig* config  = KGlobal::config();
        config->setGroup("menus");
        bool showTitles = config->readBoolEntry("ShowMenuTitles", true);
        bool bSeparator = !showTitles;
        int nId = idStart + kRecentMenusOffset;
        int nIndex = showTitles ? 1 : 0;

        for (QValueList<QString>::ConstIterator it = RecentApps.fromLast(); ; --it) {
            KService::Ptr s = KService::serviceByDesktopPath(*it);
            if (!s)
                s_RecentApps.removeItem(*it);
            else
            {
                if (!bSeparator) {
                    bSeparator = true;
                    int id = insertItem(new PopupMenuTitle(s_RecentApps.caption(), font()), idStart + kRecentMenusOffset -1, 0);
                    setItemEnabled( id, false );
                }
                insertMenuItem(s, nId++, nIndex);
                s_RecentApps.m_nNumMenuItems++;
            }

            if (it == RecentApps.begin())
                break;
        }

        if (!showTitles)
        {
            insertSeparator(s_RecentApps.m_nNumMenuItems);
        }
    }
}

// updates "recent" section of KMenu
void PanelServiceMenu::updateRecentMenuItems(KService::Ptr & service)
{
    QString strItem(service->desktopEntryPath());

    // don't add an item from root kmenu level
    if (!strItem.contains('/'))
        return;

    // add it into recent apps list
    s_RecentApps.appLaunched(strItem);
    s_RecentApps.save();

    s_RecentApps.m_bNeedToUpdate = true;
}

void PanelServiceMenu::updateRecent()
{
    if (!s_RecentApps.m_bNeedToUpdate)
        return;

    s_RecentApps.m_bNeedToUpdate = false;

    int nId = idStart + kRecentMenusOffset;

    KConfig* config  = KGlobal::config();
    config->setGroup("menus");
    bool showTitles = config->readBoolEntry("ShowMenuTitles", true);

    // remove previous items
    if (s_RecentApps.m_nNumMenuItems > 0)
    {
        // -1 --> menu title
        int i = showTitles ? -1 : 0;
        for (; i < s_RecentApps.m_nNumMenuItems; i++)
        {
            removeItem(nId + i);
            entryMap_.remove(nId + i);
        }
        s_RecentApps.m_nNumMenuItems = 0;
    }

    // insert new items
    QStringList RecentApps;
    s_RecentApps.getRecentApps(RecentApps);

    if (RecentApps.count() > 0)
    {
        bool bNeedSeparator = showTitles;
        for (QValueList<QString>::ConstIterator it = RecentApps.fromLast(); ; --it)
        {
            KService::Ptr s = KService::serviceByDesktopPath(*it);
            if (!s)
            {
                s_RecentApps.removeItem(*it);
            }
            else
            {
                if (bNeedSeparator)
                {
                    bNeedSeparator = false;
                    int id = insertItem(new PopupMenuTitle(s_RecentApps.caption(), font()), nId - 1, 0);
                    setItemEnabled( id, false );
                }
                insertMenuItem(s, nId++, 1);
                s_RecentApps.m_nNumMenuItems++;
            }

            if (it == RecentApps.begin())
                break;
        }

        if (!showTitles)
        {
            insertSeparator(s_RecentApps.m_nNumMenuItems);
        }
    }
}

void PanelServiceMenu::clearRecentMenuItems()
{
  s_RecentApps.clearRecentApps();
  s_RecentApps.save();
  s_RecentApps.m_bNeedToUpdate = true;
  updateRecent();
}


void PanelServiceMenu::setExcludeNoDisplay( bool flag )
{
  excludeNoDisplay_=flag;
}

// the initialization is split in initialize() and
// doInitialize() so that a subclass does not have to
// redo all the service parsing (see e.g. kicker/menuext/prefmenu)

void PanelServiceMenu::initialize()
{
    if (initialized()) return;

    setInitialized(true);
    entryMap_.clear();
    clear();
    subMenus.clear();
    doInitialize();
}

void PanelServiceMenu::doInitialize()
{

    // Set the startposition outside the panel, so there is no drag initiated
    // when we use drag and click to select items. A drag is only initiated when
    // you click to open the menu, and then press and drag an item.
    startPos_ = QPoint(-1,-1);

    // We ask KSycoca to give us all services (sorted).
    KServiceGroup::Ptr root = KServiceGroup::group(relPath_);

    if (!root || !root->isValid())
        return;

    KServiceGroup::List list = root->entries(true, excludeNoDisplay_, true, detailed_ && !detailedNamesFirst_);

    if (list.isEmpty()) {
        setItemEnabled(insertItem(i18n("No Entries")), false);
        return;
    }

    int id = idStart;

    if (addmenumode_) {
        int mid = insertItem(SmallIconSet("ok"), i18n("Add This Menu"), id++);
        entryMap_.insert(mid, static_cast<KSycocaEntry*>(root));

        if (list.count() > 0) {
            insertSeparator();
            id++;
        }
    }

    QStringList suppressGenericNames = root->suppressGenericNames();

    KServiceGroup::List::ConstIterator it = list.begin();
    for (; it != list.end(); ++it) {

        KSycocaEntry * e = *it;

        if (e->isType(KST_KServiceGroup)) {

            KServiceGroup::Ptr g(static_cast<KServiceGroup *>(e));
            QString groupCaption = g->caption();

            // Avoid adding empty groups.
            KServiceGroup::Ptr subMenuRoot = KServiceGroup::group(g->relPath());
            if (subMenuRoot->childCount() == 0)
                continue;

            // Ignore dotfiles.
            if ((g->name().at(0) == '.'))
                continue;

            // Item names may contain ampersands. To avoid them being converted
            // to accelerators, replace them with two ampersands.
            groupCaption.replace("&", "&&");

            PanelServiceMenu * m =
                newSubMenu(g->name(), g->relPath(), this, g->name().utf8());
            m->setCaption( groupCaption );

            QPixmap normal = KGlobal::instance()->iconLoader()->loadIcon(g->icon(), KIcon::Small,
                                                                 0, KIcon::DefaultState, 0L, true);
            QPixmap active = KGlobal::instance()->iconLoader()->loadIcon(g->icon(), KIcon::Small,
                                                                 0, KIcon::ActiveState, 0L, true);
            // make sure they are not larger than 20x20
            if (normal.width() > 20 || normal.height() > 20)
                normal.convertFromImage(normal.convertToImage().smoothScale(20,20));
            if (active.width() > 20 || active.height() > 20)
                active.convertFromImage(active.convertToImage().smoothScale(20,20));

            QIconSet iconset;
            iconset.setPixmap(normal, QIconSet::Small, QIconSet::Normal);
            iconset.setPixmap(active, QIconSet::Small, QIconSet::Active);

            int newId = insertItem(iconset, groupCaption, m, id++);
            entryMap_.insert(newId, static_cast<KSycocaEntry*>(g));
            // We have to delete the sub menu our selves! (See Qt docs.)
            subMenus.append(m);
        }
        else if (e->isType(KST_KService)) {
            KService::Ptr s(static_cast<KService *>(e));
            insertMenuItem(s, id++, -1, &suppressGenericNames);
        }
        else if (e->isType(KST_KServiceSeparator)) {
            insertSeparator();
        }
    }
#if 0
    // WABA: tear off handles don't work together with dynamically updated
    // menus. We can't update the menu while torn off, and we don't know
    // when it is torn off.
    if ( count() > 0  && !relPath_.isEmpty() )
      if (KGlobalSettings::insertTearOffHandle())
        insertTearOffHandle();
#endif
}

void PanelServiceMenu::configChanged()
{
    s_RecentApps.m_bNeedToUpdate = false;
    s_RecentApps.configChanged();
    readConfig();
    deinitialize();
}

void PanelServiceMenu::insertMenuItem(KService::Ptr & s, int nId, int nIndex/*= -1*/, const QStringList *suppressGenericNames /* = 0 */)
{
    QString serviceName = s->name();
    // add comment
    if ( detailed_ ) {
      QString comment = s->genericName();
      if ( !comment.isEmpty() ) {
        if ( detailedNamesFirst_ )
        {
          if (!suppressGenericNames || !suppressGenericNames->contains(s->untranslatedGenericName()))
            serviceName = QString( "%1 (%2)" ).arg( serviceName ).arg( comment );
        }
        else
          serviceName = QString( "%1 (%2)" ).arg( comment ).arg( serviceName );
      }
    }

    // restrict menu entries to a sane length
    if ( serviceName.length() > 60 ) {
      serviceName.truncate( 57 );
      serviceName += "...";
    }

    // check for NoDisplay
    if (s->noDisplay())
        return;

    // ignore dotfiles.
    if ((serviceName.at(0) == '.'))
        return;

    // item names may contain ampersands. To avoid them being converted
    // to accelerators, replace them with two ampersands.
    serviceName.replace("&", "&&");

    QPixmap normal = KGlobal::instance()->iconLoader()->loadIcon(s->icon(), KIcon::Small,
                                                                 0, KIcon::DefaultState, 0L, true);
    QPixmap active = KGlobal::instance()->iconLoader()->loadIcon(s->icon(), KIcon::Small,
                                                                 0, KIcon::ActiveState, 0L, true);
    // make sure they are not larger than 20x20
    if (normal.width() > 20 || normal.height() > 20)
        normal.convertFromImage(normal.convertToImage().smoothScale(20,20));
    if (active.width() > 20 || active.height() > 20)
        active.convertFromImage(active.convertToImage().smoothScale(20,20));

    QIconSet iconset;
    iconset.setPixmap(normal, QIconSet::Small, QIconSet::Normal);
    iconset.setPixmap(active, QIconSet::Small, QIconSet::Active);

    int newId = insertItem(iconset, serviceName, nId, nIndex);
    entryMap_.insert(newId, static_cast<KSycocaEntry*>(s));
}

void PanelServiceMenu::activateParent(const QString &child)
{
    PanelServiceMenu *parentMenu = dynamic_cast<PanelServiceMenu*>(parent());
    if (parentMenu)
    {
       parentMenu->activateParent(relPath_);
    }
    else
    {
       PanelPopupButton *kButton = Kicker::kicker()->KButton();
       if (kButton && (kButton->popup() == this))
       {
           adjustSize();
           popup( popupPosition( kButton->popupDirection(), this, kButton ) );
       }
       else
       {
           show();
       }
    }

    if (!child.isEmpty())
    {
        EntryMap::Iterator mapIt;
        for ( mapIt = entryMap_.begin(); mapIt != entryMap_.end(); ++mapIt )
        {
            KServiceGroup *g = dynamic_cast<KServiceGroup *>(static_cast<KSycocaEntry*>(mapIt.data()));

            // if the dynamic_cast fails, we are looking at a KService entry
            if (g && (g->relPath() == child))
            {
               activateItemAt(indexOf(mapIt.key()));
               return;
            }
        }
    }
}

bool PanelServiceMenu::highlightMenuItem( const QString &menuItemId )
{
    initialize();

    // Check menu itself
    EntryMap::Iterator mapIt;
    for ( mapIt = entryMap_.begin(); mapIt != entryMap_.end(); ++mapIt )
    {
       // Skip recent files menu
       if (mapIt.key() >= idStart + kRecentMenusOffset)
          continue;
       KService *s = dynamic_cast<KService *>(static_cast<KSycocaEntry*>(mapIt.data()));
       if (s && (s->menuId() == menuItemId))
       {
          activateParent(QString::null);
          int index = indexOf(mapIt.key());
          setActiveItem(index);

          // Warp mouse pointer to location of active item
          QRect r = itemGeometry(index);
          QCursor::setPos(mapToGlobal(QPoint(r.x()+ r.width() - 15, r.y() + r.height() - 5)));
          return true;
       }
    }

    QPtrListIterator<QPopupMenu> menuIt(subMenus);
    QPopupMenu *subMenu;
    for(;(subMenu = menuIt.current()); ++menuIt)
    {
       PanelServiceMenu *serviceMenu = dynamic_cast<PanelServiceMenu*>(subMenu);
       if (serviceMenu && serviceMenu->highlightMenuItem(menuItemId))
          return true;
    }
    return false;
}

void PanelServiceMenu::slotExec(int id)
{
    if (!entryMap_.contains(id)) return;

    KSycocaEntry * e = entryMap_[id];

    kapp->propagateSessionManager();

    KService::Ptr service = static_cast<KService *>(e);
    KApplication::startServiceByDesktopPath(service->desktopEntryPath(),
      QStringList(), 0, 0, 0, "", true);

    updateRecentMenuItems(service);
    startPos_ = QPoint(-1,-1);
}

void PanelServiceMenu::mousePressEvent(QMouseEvent * ev)
{
    startPos_ = ev->pos();
    KPanelMenu::mousePressEvent(ev);
}

void PanelServiceMenu::mouseReleaseEvent(QMouseEvent * ev)
{
    if (ev->button()==RightButton)
      return;

    KPanelMenu::mouseReleaseEvent(ev);
}

void PanelServiceMenu::mouseMoveEvent(QMouseEvent * ev)
{
    KPanelMenu::mouseMoveEvent(ev);

    if ( (ev->state() & LeftButton ) != LeftButton )
        return;

    QPoint p = ev->pos() - startPos_;
    if (p.manhattanLength() <= QApplication::startDragDistance() )
        return;

    int id = idAt(startPos_);

    // Don't drag items we didn't create.
    if (id < idStart)
        return;

    if (!entryMap_.contains(id)) {
        kdDebug(1210) << "Cannot find service with menu id " << id << endl;
        return;
    }

    KSycocaEntry * e = entryMap_[id];

    KService::Ptr service = static_cast<KService *>(e);

    QString filePath;

    QPixmap icon;

    switch (e->sycocaType()) {

        case KST_KService:
            icon     = static_cast<KService *>(e)->pixmap(KIcon::Small);
            filePath = static_cast<KService *>(e)->desktopEntryPath();
            break;

        case KST_KServiceGroup:
            icon = KGlobal::iconLoader()
                   ->loadIcon(static_cast<KServiceGroup *>(e)->icon(), KIcon::Small);
            filePath = static_cast<KServiceGroup *>(e)->relPath();
            break;

        default:
            return;
            break;
    }

    // If the path to the desktop file is relative, try to get the full
    // path from KStdDirs.

    QString path = (filePath[0] == '/') ? filePath : locate("apps", filePath);
    KURL url;
    url.setPath(path);
    KURLDrag *d = new KURLDrag(KURL::List(url), this);

    d->setPixmap(icon);
    d->dragCopy();
    //close();

    // Set the startposition outside the panel, so there is no drag initiated
    // when we use drag and click to select items. A drag is only initiated when
    // you click to open the menu, and then press and drag an item.
    startPos_ = QPoint(-1,-1);
}

PanelServiceMenu *PanelServiceMenu::newSubMenu(const QString & label, const QString & relPath,
                                               QWidget * parent, const char * name)
{
    return new PanelServiceMenu(label, relPath, parent, name);
}

void PanelServiceMenu::slotClearOnClose()
{
    if (!initialized()) return;

    if (!isVisible()){
        clearOnClose_ = false;
        slotClear();
    } else {
        clearOnClose_ = true;
    }
}

void PanelServiceMenu::slotClose()
{
    if (clearOnClose_) {
        clearOnClose_ = false;
        slotClear();
    }
}

void PanelServiceMenu::slotClear()
{
    if( isVisible()) {
        // QPopupMenu's aboutToHide() is emitted before the popup is really hidden,
        // and also before a click in the menu is handled, so do the clearing
        // only after that has been handled
        QTimer::singleShot( 100, this, SLOT( slotClear()));
        return;
    }
    entryMap_.clear();
    KPanelMenu::slotClear();
    subMenus.clear();
}

void PanelServiceMenu::selectFirstItem()
{
    setActiveItem(indexOf(idStart));
}
