/*
 * pptpgre.c
 *
 * originally by C. S. Ananian
 * Modified for PoPToP
 * then rewritten by Pat LoPresti (patl@cag.lcs.mit.edu)
 *
 * $Id: pptpgre.c,v 1.8 1999/12/24 01:03:44 patl Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef __linux__
#define _GNU_SOURCE 1		/* broken arpa/inet.h */
#endif

#if HAVE_SYSLOG_H
#include <syslog.h>
#else
#include "our_syslog.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

#include "pptpctrl.h"
#include "ppphdlc.h"
#include "pptpgre.h"
#include "pptpdefs.h"
#include "defaults.h"

#ifndef HAVE_STRERROR
#include "compat.h"
#endif

/* Reset the HDLC decoding engine. */
static void
reset_pty_to_gre (struct gre_state *gre)
{
  gre->pty_last_char = HDLC_FLAG;
  gre->pty_escape = 0;
  gre->fcs = PPPINITFCS16;
}

/* Create and return a fresh gre_state for a new connection. */
struct gre_state *
pptp_gre_init(u_int32_t call_id_pair, u_int16_t max_window_size,
              int pty_fd, struct in_addr *inetaddrs)
{
  struct sockaddr_in addr;
  struct gre_state *gre;
  int hdrincl, hdrincl_len;
  int flags;

  gre = malloc (sizeof (struct gre_state));

  if (gre == NULL) {
    syslog (LOG_ERR, "FATAL: malloc failed!");
    return NULL;
  }

  memset (gre, 0, sizeof (*gre));
  reset_pty_to_gre (gre);
  gre->seq_recv=-1;

  /* Open IP protocol socket */
  gre->gre_fd = socket(AF_INET, SOCK_RAW, PPTP_PROTO);
  if (gre->gre_fd < 0) {
    syslog(LOG_ERR, "GRE: socket() failed");
    return NULL;
  }

  memset (&addr, 0, sizeof(addr));

  addr.sin_family = AF_INET;
  addr.sin_addr = inetaddrs[0];
  addr.sin_port = 0;
  if (bind (gre->gre_fd, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
    syslog(LOG_ERR, "GRE: bind() failed: %s", strerror(errno));
    syslog(LOG_ERR, "GRE: continuing, but may not work if multi-homed");
  }

  addr.sin_family = AF_INET;
  addr.sin_addr = inetaddrs[1];
  addr.sin_port = 0;
  if (connect(gre->gre_fd, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
    syslog(LOG_ERR, "GRE: connect() failed: %s", strerror(errno));
    return NULL;
  }

  /* Mark GRE socket non-blocking */
  flags = fcntl (gre->gre_fd, F_GETFL, 0);

  if (flags < 0) {
    syslog (LOG_ERR, "Could not get GRE socket flags: %s",
            strerror (errno));
    flags = 0;
  }

  if (fcntl (gre->gre_fd, F_SETFL, flags|O_NONBLOCK) < 0) {
    syslog (LOG_ERR, "Could not set GRE socket flags: %s",
            strerror (errno));
  }

  /* Tell kernel we do not want headers with our IP packets.  I was
     hoping this would eliminate the need to strip them from received
     packets, but instead it appears to only affect transmission.
     Also, zero is the default.  Need to sort this out; FIXME.  */
  hdrincl_len = sizeof(hdrincl);
  if (getsockopt (gre->gre_fd, SOL_IP, IP_HDRINCL,
                  &hdrincl, &hdrincl_len) < 0) {
    syslog (LOG_ERR, "Cannot get IP_HDRINCL option");
  }
  else {
    /*syslog (LOG_INFO, "Got hdrincl value %d", hdrincl);*/
    hdrincl = 0;
    if (setsockopt (gre->gre_fd, SOL_IP, IP_HDRINCL,
                    &hdrincl, sizeof(hdrincl)) < 0) {
      syslog (LOG_ERR, "Couldn't set hdrincl to false");
    }
  }

  gre->pty_fd = pty_fd;
  gre->call_id_pair = call_id_pair;  /* network byte order */
  gre->xmit_max_window_size = max_window_size;
  /* FIXME: maybe this should be dynamic? */
  gre->xmit_window_size = max_window_size;
  return gre;
}

/* Return whether we are allowed to send more data packets. */
static int
window_is_open (struct gre_state *gre)
{
#if 0
  /* FIXME: We cannot do this until we have timeouts, because if we
     lose enough ACKs we will think the window is closed and remain
     throttled forever. */
  return (gre->seq_sent - gre->ack_recv < gre->xmit_window_size);
#else
  return 1;
#endif
}

/* Try to create an outgoing GRE packet from the pty input buffer. */
static void
hdlc_to_gre (struct gre_state *gre)
{
  /* As long as there are bytes to process, and we are still building
     the outgoing packet, and we have not flooded the other end
     already, proceed.
    
     It might seem more logical to check for window_is_open() in
     write_gre (just before we send the packet), but that doesn't work
     because we must be able to send ACKs even when the window is
     closed.  Hence, the check goes here instead of in write_gre. */

  while (gre->pty_bytes_processed < gre->pty_bytes_read
         && !gre->gre_packet_ready
         && window_is_open (gre)) {
    unsigned char c;

    c = gre->pty_input_buffer[gre->pty_bytes_processed++];

    if (c != HDLC_FLAG) {
      /* Ordinary character */

      if (gre->pty_escape) {
        /* Previous char was escape char, so frob this character */
        c ^= 0x20;
        gre->pty_escape = 0;
      }
      else if (c == HDLC_ESCAPE) {
        /* This char is escape char; remember to frob next character */
        gre->pty_escape = 1;
        continue;
      }

      /* Update checksum */
      gre->fcs = (gre->fcs >> 8) ^ fcstab[(gre->fcs ^ c) & 0xFF];

#if 0
      /* FIXME: Section 4 of RFC 2637 says, "The encapsulated PPP
         packets are essentially PPP data packets less any media
         specific framing elements.  No HDLC flags, bit insertion,
         controlc characters, or control character escapes are
         included."
         
         I interpreted this to mean no HDLC address and control fields
         should be included either, but packet dumps show that
         Microsoft didn't mean it that way.  Hence this code is
         #ifdef'd out and should probably just be removed.  - patl */

      /* Strip address/control field of 0xFF/0x03 per RFCs 1662 and
         2637 */
      if (gre->pty_last_char == HDLC_FLAG) {
        if (c == 0xFF) {
          gre->pty_last_char = 0xFF;
          continue;
        }
      }
      else if (gre->pty_last_char == 0xFF) {
        if (c == 0x03) {
          gre->pty_last_char = 0;
          continue;
        }
        else {
          syslog (LOG_ERR, "Weird; unrecognized control field %x", c);
        }
      }

#endif
      gre->pty_last_char = 0;
      gre->gre_packet.data[gre->gre_packet_size++] = c;
    }

    else if (gre->pty_last_char == HDLC_FLAG) {
      /* Got a flag right after a flag.  Two flags in a row is not
         even worth logging because it happens between every pair of
         packets. */
      continue;
    }

    else {
      /* Got flag; packet is done. */

      if (gre->gre_packet_size < 4 || gre->pty_escape) {
        /* Silently discard short or aborted (escape+flag) packet per
           RFC 1662 */
        if (pptpctrl_debug)
          syslog (LOG_INFO, "GRE: Short or aborted packet from pppd %d %x",
                  gre->gre_packet_size, gre->pty_escape);
      }

      else if (gre->fcs != PPPGOODFCS16) {
        syslog (LOG_ERR, "GRE: Bad checksum from pppd");
      }

      else {
        /* Packet is good.  Remove checksum field, then tag as ready
           to send. */
        gre->gre_packet_size -= sizeof(u_int16_t);
        gre->gre_packet_ready = 1;
      }

      /* Reset the translation engine */
      reset_pty_to_gre (gre);
    }
  }
}

/* Read from pppd into pty input buffer.  Return value of 0 means "try
   again later", 1 means read was successful, -1 means a fatal error
   happened. */
static int
read_pty (struct gre_state *gre)
{
  int result;

  if (gre->pty_bytes_processed != gre->pty_bytes_read) {
    /* Input buffer is not empty */
    return 0;
  }

  /* Input buffer is empty; reset it. */
  gre->pty_bytes_processed = gre->pty_bytes_read = 0;
  memset (gre->pty_input_buffer, 0x44, HDLC_PACKET_SIZE);

  result = read (gre->pty_fd, gre->pty_input_buffer,
                 HDLC_PACKET_SIZE);

  if (result > 0) {
    gre->pty_bytes_read = result;
    return 1;
  }

  if (result < 0) {
    if (errno == EAGAIN || errno == EINTR) {
      return 0;
    }

    syslog (LOG_ERR, "Error reading from pppd: %s", strerror (errno));
    return -1;
  }

  syslog (LOG_ERR, "EOF reading from pppd");
  return -1;
}

static void
maybe_make_ack (struct gre_state *gre)
{
  /* If we are not in the process of building an outgoing packet, but
     we need to send an ack, do so. */

  if (gre->gre_packet_size == 0
      && gre->seq_recv != gre->ack_sent) {
    gre->gre_packet_ready = 1;
  }
}

/* Send a GRE packet out, if appropriate. */
static int
write_gre (struct gre_state *gre)
{
  struct pptp_gre_header header;
  unsigned char *packet;
  int have_ack, have_payload;
  int result;
  int packet_len;

  if (!gre->gre_packet_ready) {
    return 0;
  }

  /* Build the header */
  header.flags = hton8(PPTP_GRE_FLAG_K);
  header.ver = hton8 (PPTP_GRE_VER);
  header.protocol = hton16 (PPTP_GRE_PROTO);
  header.payload_len = hton16 (gre->gre_packet_size);
  header.call_id = GET_VALUE (PNS, gre->call_id_pair);

  have_ack = (gre->ack_sent != gre->seq_recv);
  have_payload = (gre->gre_packet_size > 0);

  header.ver |= hton8 (have_ack ? PPTP_GRE_FLAG_A : 0);
  header.flags |= hton8 (have_payload ? PPTP_GRE_FLAG_S : 0);

  /* Put header in the right place, filling in optional sequence
     number and optional acknowledgement number. */

  if (have_ack && have_payload) {
      gre->gre_packet.head.both.header = header;
      gre->gre_packet.head.both.seq = hton32 (gre->seq_sent+1);
      gre->gre_packet.head.both.ack = hton32 (gre->seq_recv);
      packet = (unsigned char *)&gre->gre_packet.head.both.header;
    }
    else if (have_ack) {
      gre->gre_packet.head.one.header = header;
      gre->gre_packet.head.one.seq_or_ack = hton32 (gre->seq_recv);
      packet = (unsigned char *)&gre->gre_packet.head.one.header;
    }
    else if (have_payload) {
      gre->gre_packet.head.one.header = header;
      gre->gre_packet.head.one.seq_or_ack = hton32 (gre->seq_sent+1);
      packet = (unsigned char *)&gre->gre_packet.head.one.header;
    }
    else {
      syslog (LOG_ERR,
              "INTERNAL ERROR: must either have payload or ack!");
      gre->gre_packet_ready = 0;
      return 0;
    }

  packet_len = gre->gre_packet.data - packet + gre->gre_packet_size;

  result = write (gre->gre_fd, packet, packet_len);

  if (result >= 0) {
    if (result != packet_len) {
      syslog (LOG_ERR, "INTERNAL ERROR: short write on raw socket");
    }

    if (have_payload)
      gre->seq_sent = gre->seq_sent + 1;

    if (have_ack) {
      gre->ack_sent = gre->seq_recv;
    }

    gre->gre_packet_size = 0;
    gre->gre_packet_ready = 0;
    memset (&gre->gre_packet, 0x55, sizeof(gre->gre_packet));
    return 1;
  }

  if (errno == EAGAIN || errno == EINTR) {
    if (pptpctrl_debug)
      syslog (LOG_INFO, "write() to GRE wants retry: %s", strerror (errno));
    return 0;
  }

  syslog (LOG_ERR,
          "Error writing GRE packet: %s", strerror (errno));
  return -1;
}

int
do_pty_to_gre (struct gre_state *gre)
{
  int wrote_packet;

  /* Keep going as long as we can profitably continue */
  do {
    wrote_packet = 0;

    if (read_pty (gre) < 0)
      return -1;

    hdlc_to_gre (gre);

    maybe_make_ack (gre);
   
    switch (write_gre (gre)) {
    case -1:
      return -1;

    case 1:
      wrote_packet = 1;
      break;
    }
  } while (wrote_packet);

  return 0;
}

/* True if sequence number A is less than B.  Handles wrap-around
   correctly. */
static int
seq_less_than (u_int32_t a, u_int32_t b)
{
  return ((a - b) > 0x80000000UL);
}

/* Try to read a packet from the GRE socket. */
static int
read_gre (struct gre_state *gre)
{
  int status, offset=0;
  unsigned int i,j;
  int recv_time,timeout;
  unsigned char buffer[GRE_PACKET_SIZE + 64 /* ip header */];
  struct gre_rcv_packet *recv_packet;
  unsigned char *data;
  struct pptp_gre_header header;
  int has_payload, has_ack;

  /* Always do the read because we might be receiving an ACK */

  memset (buffer, 0x66, sizeof(buffer));
  status = read (gre->gre_fd, buffer, sizeof(buffer));

  if (status < 0) {
    if (errno == EAGAIN || errno == EINTR) {
      /*syslog (LOG_INFO, "GRE read() wants retry");*/
      return 0;
    }

    syslog (LOG_ERR, "GRE: read error: %s", strerror (errno));
    return -1;
  }

  else if (status == 0) {
    syslog (LOG_ERR, "INTERNAL ERROR: 0 bytes read!");
    return 0;
  }

  /* strip off IP header, if present */
  if ((buffer[0] & 0xF0) == 0x40) {
    offset = (buffer[0]&0x0F) * 4;
  }

  status -= offset;

  if (status <= 0) {
    syslog (LOG_ERR, "Weird: Nothing left after IP header");
    return 0;
  }

  recv_packet = (struct gre_rcv_packet *) (buffer+offset);
  header = recv_packet->header;
  recv_time = time(NULL);

  /* Validate the packet. */
  if ((ntoh8 (header.ver) & ~PPTP_GRE_FLAG_A) != PPTP_GRE_VER
      || (ntoh8 (header.flags) & ~PPTP_GRE_FLAG_S) != PPTP_GRE_FLAG_K
      || ntoh16 (header.protocol) != PPTP_GRE_PROTO) {
    syslog (LOG_ERR, "GRE: Bad packet flags %x ver %x proto %x",
            ntoh8 (header.flags), ntoh8 (header.ver),
            ntoh16 (header.protocol));
    return 0;
  }

  /* Check call ID */
  if (header.call_id != GET_VALUE (PAC, gre->call_id_pair)) {
    syslog(LOG_ERR, "GRE: Discarding for incorrect call");
    return 0;
  }

  has_payload = PPTP_GRE_IS_S (ntoh8 (header.flags));
  has_ack = PPTP_GRE_IS_A (ntoh8 (header.ver));

#if 0
  syslog (LOG_INFO, "Got packet %s%s",
          has_payload ? "payload " : "",
          has_ack ? "ack" : "");
#endif

  /* First things first: Remember any ack we get. */
  if (has_ack) {
    u_int32_t ack;

    ack = ntoh32 (has_payload
                  ? recv_packet->body.both.ack
                  : recv_packet->body.one.seq_or_ack);

#if 0
    syslog (LOG_INFO, "read_gre got ack %d  ack_recv %d",
            ack, gre->ack_recv);
#endif

    if (seq_less_than (gre->ack_recv, ack))
      gre->ack_recv = ack;
  }

  /* Next, deal with the payload. */
  if (has_payload) {
    u_int32_t seq;

    /* Keep 8 packets of buffer room for packet reordering */
    if (gre->gre_num_packets >= (PCKT_RECV_WINDOW_SIZE-PCKT_REORDER_SIZE)) {
      syslog (LOG_ERR, "GRE: window overflowed.  Dropping packet...");
      return 1;
    }

    if (has_ack) {
      seq = ntoh32 (recv_packet->body.both.seq);
      data = recv_packet->body.both.data;
    }
    else {
      seq = ntoh32 (recv_packet->body.one.seq_or_ack);
      data = recv_packet->body.one.data;
    }

    /* Check sequence number; discard if out of order */
    if (!seq_less_than (gre->seq_recv, seq)) {
      syslog (LOG_WARNING,
              "Discarding out-of-order packet %u, already have %u",
              seq, gre->seq_recv);
      return 0;
    }

    /* Check payload size vs. amount of data read */
    if (ntoh16 (header.payload_len) + (data - (unsigned char *)recv_packet)
        != status) {
      syslog (LOG_ERR, "Wrong size read; got %d head %d body %d",
              status, data - (unsigned char *)recv_packet,
              ntoh16 (header.payload_len));
      return 0;
    }

    i=seq - gre->seq_recv - 1;

    /* Timeout if there are no buffered packets waiting to go to pppd,
     *  the current packet is not the next packet in the sequence,
     *  and it has been 3 seconds size we last passed on a packet. */
    timeout=FALSE;
    if((gre->gre_num_packets == 0) && (i>0) && (recv_time > (gre->gre_last_recv_time+PCKT_REORDER_WAIT_TIME))) {
      /* Check for packets in the buffer */
      for(j=0 ; (j<PCKT_REORDER_SIZE) && (gre->window[(gre->gre_current_packet+j)%(PCKT_RECV_WINDOW_SIZE+1)].header.payload_len == 0) ; j++);
      /* If there is a buffered packet, trigger a timeout */
      if(j!=PCKT_REORDER_SIZE)
	timeout=TRUE;
    }

    /* If this packet is beyond the reorder buffer,
     * stop waiting for the missing packet */
    if ((i >= PCKT_REORDER_SIZE) || timeout) {
      if(gre->gre_num_packets > 0) {
	/* There are still packets in the queue, so we can't skip packets yet */
	if(pptpctrl_debug) {
	  syslog (LOG_INFO, "Dropping out-of-order packet; got %u after %u",
		  seq, gre->seq_recv);
	}
	return 0;
      }
        /* Stop waiting for the oldest packet */
        for(j=0 ; (j<i) && (j<PCKT_REORDER_SIZE) && (gre->window[(gre->gre_current_packet+j)%(PCKT_RECV_WINDOW_SIZE+1)].header.payload_len == 0) ; j++);

        /* Missing more than PCKT_REORDER_SIZE consecutive packets - just ignore them and process this new packet */
	if(pptpctrl_debug) {
	  if(timeout) {
	    syslog (LOG_INFO, "Packet reorder timeout waiting for %u",gre->seq_recv+1);
	  } else if(j==PCKT_REORDER_SIZE) {
	    syslog (LOG_INFO, "Missing %d consecutive packets; got %u after %u",
		    PCKT_REORDER_SIZE,seq, gre->seq_recv);
	  } else {
	    syslog (LOG_INFO, "Exceeded packet reorder buffer size; got %u after %u; skipping %d packets",
		    seq, gre->seq_recv,j);
	  }
	}
        if(j==PCKT_REORDER_SIZE) {
            memcpy(&gre->window[gre->gre_current_packet],buffer+offset,status);
#if 0
	    if(pptpctrl_debug) {
	      syslog(LOG_INFO,"Filling packet buffer slot %d with packet %u",gre->gre_current_packet,seq);
	    }
#endif
            gre->seq_recv = seq;
            gre->gre_num_packets++;
	    gre->gre_last_recv_time=recv_time;
            return 1;
        }

        /* Else skip forward to the oldest previously received packet */
        gre->gre_current_packet=(gre->gre_current_packet+j) % (PCKT_RECV_WINDOW_SIZE+1);
        gre->seq_recv += j;

        i-=j;
     }
     /* Buffer the latest packet if it isn't too far ahead, else just drop it */
     if(i < PCKT_REORDER_SIZE) {
        if(pptpctrl_debug && (i>0)) {
	      syslog (LOG_INFO, "Buffering out-of-order packet; got %u after %u",
        	      seq, gre->seq_recv);
        }
        memcpy(&gre->window[(gre->gre_current_packet+gre->gre_num_packets+i) % (PCKT_RECV_WINDOW_SIZE+1)],buffer+offset,status);
#if 0
	    if(pptpctrl_debug) {
	      syslog(LOG_INFO,"Filling packet buffer slot %d with packet %u (curr=%d,n=%d,i=%d)",(gre->gre_current_packet+gre->gre_num_packets+i) % (PCKT_RECV_WINDOW_SIZE+1),seq,gre->gre_current_packet,gre->gre_num_packets,i);
	    }
#endif
     } else if(pptpctrl_debug) {
          syslog (LOG_INFO, "Dropping out-of-order packet %u",seq);
     }

     /* Skip lost packets if we get PCKT_REORDER_SIZE consecutive following packets */

     if((gre->gre_num_packets==0) && (gre->window[gre->gre_current_packet].header.payload_len == 0)) {
       for(i=0,j=1;(i < (PCKT_REORDER_RESUME_SIZE)) && (j<PCKT_REORDER_SIZE);j++) {
	 if(gre->window[(gre->gre_current_packet+j) % (PCKT_RECV_WINDOW_SIZE+1)].header.payload_len != 0) {
	   i++;
	 } else {
	   i=0;
	 }
       }
       if(i >= (PCKT_REORDER_RESUME_SIZE)) {
	 /* Advance counters over the lost packet(s) */
	 for(j=1 ; (j<PCKT_REORDER_SIZE) && (gre->window[(gre->gre_current_packet+j)%(PCKT_RECV_WINDOW_SIZE+1)].header.payload_len == 0) ; j++);
	 if((pptpctrl_debug) && (gre->seq_recv+j > 0)) {
	   syslog (LOG_INFO, "Gave up waiting for %d lost packets beginning with %u",j,gre->seq_recv+1);
	 }
	 
	 gre->gre_current_packet=(gre->gre_current_packet+j) % (PCKT_RECV_WINDOW_SIZE+1);
	 gre->seq_recv += j;
       }
     }

     /* Add all consecutive available packets to the queue */
     for(j=0 ; (j<PCKT_REORDER_SIZE) && (gre->window[(gre->gre_current_packet+gre->gre_num_packets+j)%(PCKT_RECV_WINDOW_SIZE+1)].header.payload_len != 0) ; j++);
     /*
     if(pptpctrl_debug && (j>1)) {
       syslog(LOG_INFO,"Adding %d packets to the queue",j);
     }
     */
     gre->seq_recv += j;
     gre->gre_num_packets += j;
     if(j>0) {
       gre->gre_last_recv_time = recv_time;
       return 1;
     }
     return 0;

#if 0
    /* Dump start of packet. */
    syslog (LOG_INFO,
            "payload size %d data %02x %02x %02x %02x %02x %02x %02x",
            status, data[0], data[1], data[2], data[3], data[4],
            data[5], data[6]);
#endif
  }

  if (!has_payload && !has_ack)
    syslog (LOG_ERR, "Weird; no payload and no ack");

  return 0;
}

/* Push a character into the PTY output buffer.  Optionally perform
   standard escaping.  Compute and return new checksum value. */
static u_int16_t
gre_hdlc_put (struct gre_state *gre, unsigned char c,
               int escape, u_int16_t fcs)
{
  if ((c < 0x20 || c == HDLC_FLAG || c == HDLC_ESCAPE)
      && escape) {
    gre->pty_output_buffer[gre->pty_output_size++] = HDLC_ESCAPE;
    gre->pty_output_buffer[gre->pty_output_size++] = c^0x20;
  }
  else
    gre->pty_output_buffer[gre->pty_output_size++] = c;

  return (fcs >> 8) ^ fcstab[(fcs^c)&0xff];
}

static void
gre_to_hdlc (struct gre_state *gre)
{
  struct gre_rcv_packet *packet;
  struct pptp_gre_header header;
  char *data;
  u_int16_t fcs;
  int i;

  /* If we have no packets to process, or if the output buffer is
     non-empty, bail. */
  if (gre->gre_num_packets == 0
      || gre->pty_output_size != gre->pty_bytes_written)
  {
    return;
  }

  gre->pty_output_size = gre->pty_bytes_written = 0;
  memset (gre->pty_output_buffer, 0x22, HDLC_PACKET_SIZE);

  packet = &gre->window[gre->gre_current_packet];
  header = packet->header;
  if (header.protocol != ntoh16(PPTP_GRE_PROTO)) {
    syslog (LOG_ERR, "INTERNAL ERROR: Bad protocol %x in gre_to_hdlc",
            ntoh16(header.protocol));
  }

  data = (PPTP_GRE_IS_A (ntoh8 (header.ver))
          ? packet->body.both.data
          : packet->body.one.data);

  /* Encode standard header */
  gre_hdlc_put (gre, HDLC_FLAG, 0, 0);
  fcs = PPPINITFCS16;

  /* Then the data */
  for (i = 0 ; i < ntoh16 (header.payload_len) ; i++) {
    fcs = gre_hdlc_put (gre, data[i], 1, fcs);
  }

  /* Finally the FCS and trailing flag */
  fcs ^= 0xFFFF;
  gre_hdlc_put (gre, fcs&0xFF, 1, 0);
  gre_hdlc_put (gre, fcs>>8, 1, 0);

  gre_hdlc_put (gre, HDLC_FLAG, 0, 0);

  /* Indicate that we have consumed this GRE packet */
  gre->gre_num_packets -= 1;
  gre->gre_current_packet = ((gre->gre_current_packet + 1)
                             % (PCKT_RECV_WINDOW_SIZE + 1));

  /* Zero out the packet so the payload_len parameter is zero which indicates this slot is empty */
  memset (packet, 0x00, sizeof (*packet));
}

/* Send some stuff to the PTY.  Return 1 if we wrote something, 0 if
   we could not write anything just now, -1 if fatal error. */
static int
write_pty (struct gre_state *gre)
{
  int result;

  if (gre->pty_bytes_written == gre->pty_output_size) {
    /* Nothing to write */
    return 0;
  }

  result = write (gre->pty_fd,
                  gre->pty_output_buffer + gre->pty_bytes_written,
                  gre->pty_output_size - gre->pty_bytes_written);

  if (result < 0) {
    if (errno == EAGAIN || errno == EINTR) {
      if (pptpctrl_debug)
        syslog (LOG_INFO, "write() to PTY wants retry: %s",
                strerror (errno));
      return 0;
    }

    else
      syslog (LOG_ERR, "write to pty failed: %s", strerror (errno));

    return -1;
  }

  /* No error, so remember how much we wrote */
  gre->pty_bytes_written += result;

  return (result > 0);
}

int
do_gre_to_pty (struct gre_state *gre)
{
  int read_packet;

  do {
    read_packet = 0;

    switch (read_gre (gre)) {
    case -1:
      return -1;

    case 1:
      read_packet = 1;
    }

    gre_to_hdlc (gre);

    if (write_pty (gre) < 0)
      return -1;

  } while (read_packet);

  return 0;
}
