/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"

#include <string.h>
#include <sys/stat.h>
#include <dirent.h>
#include <utime.h>
#include <ctype.h>

#include <stdio.h>

#ifdef BUFFY_SIZE

/* Find the last message in the file.
 * upon success return 0.
 * If no message found - return -1
 */
int fseek_last_message (FILE *f)
{
    char s[8];

    memset (s, 0, 8);

    fseek (f, 1, 2);

    while (fseek (f, -2, 1) != -1) 
    {
      s[6] = s[5];
      s[5] = s[4];
      s[4] = s[3];
      s[3] = s[2];
      s[2] = s[1];
      s[1] = s[0];
      s[0] = getc (f);

      if (!strcmp ("\n\nFrom ", s))
      {
	fseek (f, 1, 1);
	return 0;
      } 
    }

    if (!strncmp ("From ", s, 5)) 
    {
	fseek (f, 0, 0);
	return (0);
    }

    return (-1);
}

/* Return 1 if the last message is new */
int test_last_status_new (FILE *f)
{
  HEADER *hdr;
  int result = 0;
  
  if (fseek_last_message (f) == -1)
    return (0);

  hdr = mutt_new_header();
  mutt_read_rfc822_header (f, hdr);
  if (!(hdr->read || hdr->old))
    result = 1;
  mutt_free_header(&hdr);

  return result;
}

int test_new_folder (const char *path)
{
  FILE *f;
  int rc = 0;
  int typ;

  typ = mx_get_magic (path);

  if (typ != M_MBOX && typ != M_MMDF)
    return 0;

  f = fopen (path, "rb");
  rc = test_last_status_new (f);
  fclose (f);

  return rc;
}

BUFFY *mutt_find_mailbox (const char *path)
{
  BUFFY *tmp = NULL;

  for (tmp = Incoming; tmp; tmp = tmp->next)
  {
    if (strcmp (tmp->path, path) == 0)
      break;
  }
  return tmp;
}

void mutt_update_mailbox (BUFFY *b)
{
  struct stat sb;

  if (!b)
    return;

  if (stat (b->path, &sb) == 0)
    b->size = (long) sb.st_size;
  else
    b->size = 0;
  return;
}
#endif  

int mutt_parse_mailboxes (const char *s, void *data, char *err, size_t errlen)
{
  BUFFY **tmp;
  char buf[_POSIX_PATH_MAX];
  const char *p;
  size_t len;
#ifdef BUFFY_SIZE
  struct stat sb;
#endif /* BUFFY_SIZE */

  do
  {
    if ((p = strpbrk (s, " \t\n")))
      len = p - s;
    else
      len = strlen (s);

    if (len > sizeof (buf) - 1)
      len = sizeof (buf) - 1;

    memcpy (buf, s, len);
    buf[len] = 0;

    mutt_expand_path (buf, sizeof (buf));
    tmp = &Incoming;
    while (*tmp) /* simple check to avoid duplicates */
    {
      if (strcmp (buf, (*tmp)->path) == 0)
	break;
      tmp = &((*tmp)->next);
    }

    if (!*tmp)
    {
      *tmp = (BUFFY *) safe_malloc (sizeof (BUFFY));
      (*tmp)->path = safe_strdup (buf);
      (*tmp)->next = NULL;
    }
      
    (*tmp)->new = 1;

#ifdef BUFFY_SIZE
    /* for buffy_size, it is important that if the folder is new (tested by
     * reading it), the size is set to 0 so that later when we check we see
     * that it increased .  without buffy_size we probably don't care.
     */
    if (stat ((*tmp)->path, &sb) == 0 && !test_new_folder ((*tmp)->path))
    {
      /* some systems out there don't have an off_t type */
      (*tmp)->size = (long) sb.st_size;
    }
    else
      (*tmp)->size = 0;
#endif /* BUFFY_SIZE */

    if (p)
      SKIPWS (p);
  }
  while ((s = p));

  return 0;
}

#ifdef BUFFY_SIZE
/* people use buffy_size on systems where modified time attributes are BADLY
 * broken. Ignore them.
 */
#define STAT_CHECK (sb.st_size > tmp->size)
#else
#define STAT_CHECK ((sb.st_mtime > sb.st_atime) || (sb.st_ctime == sb.st_mtime))
#endif /* BUFFY_SIZE */

int mutt_buffy_check (void)
{
  static time_t	last = 0; /* to avoid too many checks */
  static int ct = 0;
  BUFFY *tmp;
  int was_new = 0;
  struct stat sb;
  time_t t;
  struct dirent *de;
  DIR *dirp;
  char path[_POSIX_PATH_MAX];
  int typ, rc = 0;

  /* fastest return if there are no mailboxes */
  if (!Incoming)
    return 0;
  t = time (NULL);
  if ((t - last) < 3)
    return ct;
  last = t;
  ct = 0;
  for (tmp = Incoming; tmp; tmp = tmp->next)
  {
    was_new = tmp->new;
    tmp->new = 0;
    if ((rc = stat (tmp->path, &sb)) == 0)
    {
      if (!Context || !Context->path || strcmp (tmp->path, Context->path)) 
      {
	typ = mx_get_magic (tmp->path);

	switch (typ)
	{
	  case M_MBOX:
	  case M_MMDF:
	  
	    if (STAT_CHECK)
	    {
	      ct++;
	      tmp->new = 1;
	    }
#ifdef BUFFY_SIZE
	    else /* some other program has deleted mail from the folder */
	      tmp->size = (long) sb.st_size;
#endif /* BUFFY_SIZE */
	    break;

	  case M_MAILDIR:

	    snprintf (path, sizeof (path), "%s/new", tmp->path);
	    if ((dirp = opendir (path)) == NULL)
	      break;
	    while ((de = readdir (dirp)) != NULL)
	    {
	      if (*de->d_name != '.')
	      {
		/* one new message is enough */
		ct++;
		tmp->new = 1;
		break;
	      }
	    }
	    closedir (dirp);
	    break;

	  case M_MH:

	    if ((dirp = opendir (tmp->path)) == NULL)
	      break;
	    while ((de = readdir (dirp)) != NULL)
	    {
	      if (!mh_valid_message (de->d_name))
		continue;
	      snprintf (path, sizeof (path), "%s/%s", tmp->path, de->d_name);
	      if (stat (path, &sb) != -1 && sb.st_mtime > sb.st_atime)
	      {
		ct++;
		tmp->new = 1;
		break;
	      }
	    }
	    closedir (dirp);
	    break;
	}
      }
#ifdef BUFFY_SIZE
      else if (Context && Context->path)
	tmp->size = (long) sb.st_size;	/* update the size */
#endif /* BUFFY_SIZE */
    }
#ifdef BUFFY_SIZE
    else /* if (rc != 0) the folder was deleted */
      tmp->size = 0;
#endif /* BUFFY_SIZE */
    
    if (!was_new && tmp->new) 
    {
      strfcpy (path, tmp->path, sizeof (path));
      mutt_pretty_mailbox (path);
      mutt_message ("New message arrived in %s.", path);
    }
  }
  return (ct);
}

/*
 * mutt_buffy() -- incoming folders completion routine
 *
 * given a folder name, this routine gives the next incoming folder with new
 * new mail.
 */
void mutt_buffy (char *s)
{
  int count = 0;
  BUFFY	*tmp = Incoming;

  mutt_expand_path (s, _POSIX_PATH_MAX);
  switch (count = mutt_buffy_check ())
  {
    case 0 :
    
      s = '\0';
      break;
      
    case 1 :
    
      while (tmp->new == 0)
	tmp = tmp->next;
      strcpy (s, tmp->path);
      mutt_pretty_mailbox (s);
      break;
      
    default :
    
      FOREVER
      {
	if (strcmp (s, tmp->path) == 0)
	  count = 0;
	else if (count == 0 && tmp->new)
	  break;
	tmp = tmp->next;
	if (tmp == NULL)
	{
	  tmp = Incoming;
	  count = 0;
	}
      }
      strcpy (s, tmp->path);
      mutt_pretty_mailbox (s);
      break;
  }
}
