/* Copyright Abandoned 1996 TCX DataKonsult AB & Monty Program KB & Detron HB
   This file is public domain and comes with NO WARRANTY of any kind */

#ifdef _WIN32
#include <winsock.h>
#include <odbcinst.h>
#endif
#include <global.h>
#include <my_sys.h>
#include <m_string.h>
#include <m_ctype.h>
#include "mysql.h"
#include "mysql_version.h"
#include "errmsg.h"
#include <sys/stat.h>
#include <signal.h>
#ifdef	 HAVE_PWD_H
#include <pwd.h>
#endif
#if !defined(MSDOS) && !defined(__WIN32__)
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#endif
#ifdef HAVE_SYS_UN_H
#  include <sys/un.h>
#endif
#if defined(THREAD) && !defined(__WIN32__)
#include <my_pthread.h>				/* because of signal()	*/
#endif
#ifndef INADDR_NONE
#define INADDR_NONE	-1
#endif

static bool	mysql_client_init=0;
static MYSQL	*current_mysql;
uint		mysql_port=0;
string		mysql_unix_port=0;

#define CLIENT_CAPABILITIES	(CLIENT_LONG_PASSWORD)

#if defined(MSDOS) || defined(__WIN32__)
#define ERRNO WSAGetLastError()
#define perror(A)
#else
#define ERRNO errno
#define SOCKET_ERROR -1
#define closesocket(A) close(A)
#endif

static MYSQL_DATA *read_rows (MYSQL *mysql,MYSQL_FIELD *fields,
			      uint field_count);
static void end_server(MYSQL *mysql);
static void remember_connection(MYSQL *mysql);
static void read_user_name(char *name);
static void append_wild(char *to,char *end,const char *wild);


/*****************************************************************************
** read a packet from server. Give error message if socket was down
** or package is an error message
*****************************************************************************/

static uint
net_safe_read(MYSQL *mysql)
{
  NET *net= &mysql->net;
  uint len=0;

  if (net->fd < 0 || ((len=my_net_read(net)) == packet_error || len == 0))
  {
    DBUG_PRINT("error",("Wrong connection or packet. fd: %d  len: %d",
			net->fd,len));
    end_server(mysql);
    net->last_errno=CR_SERVER_LOST;
    strmov(net->last_error,ER(net->last_errno));
    return(packet_error);
  }
  if (net->buff[0] == 255)
  {
    if (len > 3)
    {
      char *pos=(char*) net->buff+1;
      if (mysql->protocol_version > 9)
      {						/* New client protocol */
	net->last_errno=uint2korr(pos);
	pos+=2;
      }
      else
	net->last_errno=CR_UNKNOWN_ERROR;
      (void) strmake(net->last_error,(char*) pos, sizeof(net->last_error)-1);
    }
    else
    {
      net->last_errno=CR_UNKNOWN_ERROR;
      (void) strmov(net->last_error,ER(net->last_errno));
    }
    DBUG_PRINT("error",("Got error: %d (%s)", net->last_errno,
			net->last_error));
    return(packet_error);
  }
  return len;
}


/* Get the length of next field. Change parameter to point at fieldstart */
static ulong
net_field_length(uchar **packet)
{
  reg1 uchar *pos= *packet;
  if (*pos < 251)
  {
    (*packet)++;
    return (ulong) *pos;
  }
  if (*pos == 251)
  {
    (*packet)++;
    return NULL_LENGTH;
  }
  if (*pos == 252)
  {
    (*packet)+=3;
    return uint2korr(pos+1);
  }
  if (*pos == 253)
  {
    (*packet)+=4;
    return uint3korr(pos+1);
  }
  (*packet)+=6;					/* Must be 254 when here */
  return uint4korr(pos+1);
}

static void free_rows(MYSQL_DATA *cur)
{
  if (cur)
  {
    sql_free(&cur->alloc);
    my_free((gptr) cur,MYF(0));
  }
}


static my_bool mysql_reconnect(MYSQL *mysql)
{
  MYSQL tmp_mysql;
  DBUG_ENTER("mysql_reconnect");
  
  if (!mysql->reconnect || !mysql->host_info)
    DBUG_RETURN(1);
  if (!mysql_real_connect(&tmp_mysql,mysql->host,mysql->user,mysql->passwd,
			  mysql->port,mysql->unix_socket,
			  mysql->client_flag))
    DBUG_RETURN(1);
  if (mysql->db && mysql_select_db(&tmp_mysql,mysql->db))
  {
    mysql_close(&tmp_mysql);
    DBUG_RETURN(1);
  }
  tmp_mysql.free_me=mysql->free_me;
  mysql->free_me=0;
  mysql_close(mysql);
  memcpy(mysql,&tmp_mysql,sizeof(tmp_mysql));
  net_clear(&mysql->net);
  mysql->affected_rows= (ulong) ~0L;
  DBUG_RETURN(0);
}


static int
simple_command(MYSQL *mysql,enum enum_server_command command, const char *arg,
	       uint length, my_bool skipp_check)
{
  NET *net= &mysql->net;

  if (mysql->net.fd < 0)
  {						/* Do reconnect if possible */
    if (mysql_reconnect(mysql))
    {
      net->last_errno=CR_SERVER_GONE_ERROR;
      strmov(net->last_error,ER(net->last_errno));
      return -1;
    }
  }
  if (mysql->status != MYSQL_STATUS_READY)
  {
    strmov(net->last_error,ER(mysql->net.last_errno=CR_COMMANDS_OUT_OF_SYNC));
    return -1;
  }

  mysql->net.last_error[0]=0;
  mysql->net.last_errno=0;
  mysql->info=0;
  mysql->affected_rows= (ulong) ~0L;
  remember_connection(mysql);
  net_clear(net);			/* Clear receive buffer */
  if (!arg)
    arg="";
  if (net_write_command(net,(uchar) command,arg,
			length ? length :strlen(arg)))
  {
    DBUG_PRINT("error",("Can't send command to server. Error: %d",errno));
    end_server(mysql);
    if (mysql_reconnect(mysql) ||
	net_write_command(net,(uchar) command,arg,
			  length ? length :strlen(arg)))
    {
      net->last_errno=CR_SERVER_GONE_ERROR;
      strmov(net->last_error,ER(net->last_errno));
      return -1;
    }
  }
  if (!skipp_check)
    return (net_safe_read(mysql) == packet_error ? -1 : 0);
  return 0;
}


static void free_old_query(MYSQL *mysql)
{
  DBUG_ENTER("free_old_query");
  if (mysql->fields)
    sql_free(&mysql->field_alloc);
  init_alloc_root(&mysql->field_alloc,8192);	/* Assume rowlength < 8192 */
  mysql->fields=0;
  mysql->field_count=0;				/* For API */
  DBUG_VOID_RETURN;
}

#define USERNAMELENGTH 16

#if !defined(MSDOS) && ! defined(VMS) && !defined(__WIN32__)
static void read_user_name(char *name)
{
  DBUG_ENTER("read_user_name");
  if (geteuid() == 0)
    (void) strmov(name,"root");		/* allow use of surun */
  else
  {
#ifdef HAVE_GETPWUID
    struct passwd *skr,*getpwuid();
    char *str,*getlogin();

    if ((str=getlogin()) == NULL)
      if ((skr=getpwuid(geteuid())) != NULL)
	str=skr->pw_name;
      else if (!(str=getenv("USER")) && !(str=getenv("LOGNAME")) &&
	       !(str=getenv("LOGIN")))
	str="UNKNOWN_USER";
    (void) strmake(name,str,USERNAMELENGTH);
#elif HAVE_CUSERID
    (void) cuserid(name);
#else
    strmov(name,"UNKNOWN_USER");
#endif
  }
  DBUG_VOID_RETURN;
}

#else /* If MSDOS || VMS */

static void read_user_name(char *name)
{
  strmov(name,"ODCB");	 /* ODBC will send user variable */
}

#endif

/*
** Expand wildcard to a sql string
*/

static void
append_wild(char *to, char *end, const char *wild)
{
  end-=5;					/* Some extra */
  if (wild && wild[0])
  {
    to=strmov(to," like '");
    while (*wild && to < end)
    {
      if (*wild == '\\' || *wild == '\'')
	*to++='\\';
      *to++= *wild++;
    }
    if (*wild)					/* Too small buffer */
      *to++='%';				/* Nicer this way */
    to[0]='\'';
    to[1]=0;
  }
}



/**************************************************************************
** Init debugging if MYSQL_DEBUG environment variable is found
**************************************************************************/

void mysql_debug(char *debug)
{
#ifndef DBUG_OFF
  char	*env;
  if (_db_on_)
    return;					/* Already using debugging */
  if (debug)
  {
    DEBUGGER_ON;
    DBUG_PUSH(debug);
  }
  else if ((env = getenv("MYSQL_DEBUG")))
  {
    DEBUGGER_ON;
    DBUG_PUSH(env);
#if !defined(_WINVER) && !defined(WINVER)
    puts("\n-------------------------------------------------------");
    puts("MYSQL_DEBUG found. libmysql started with the following:");
    puts(env);
    puts("-------------------------------------------------------\n");
#else
    {
      char buff[80];
      strmov(strmov(buff,"libmysql: "),env);
      MessageBox((HWND) 0,"Debugging variable MYSQL_DEBUG used",buff,MB_OK);
    }
#endif
  }
#endif
}


/**************************************************************************
** Store the server socket currently in use
** Used by pipe_handler if error on socket interrupt
**************************************************************************/

static void
remember_connection(MYSQL *mysql)
{
  current_mysql = mysql;
}

/**************************************************************************
** Close the server connection if we get a SIGPIPE
   ARGSUSED
**************************************************************************/

static sig_handler
pipe_sig_handle(int sig __attribute__((unused)))
{
  DBUG_PRINT("info",("Hit by signal %d",sig));
#ifdef NOT_USED
  end_server(current_mysql);
#endif
#ifdef DONT_REMEMBER_SIGNAL
  (void) signal(SIGPIPE,pipe_sig_handle);
#endif
}


/**************************************************************************
** Shut down connection
**************************************************************************/

static void
end_server(MYSQL *mysql)
{
  DBUG_ENTER("end_server");
  if (mysql->net.fd >= 0)
  {
    DBUG_PRINT("enter",("Socket: %d", mysql->net.fd));
    (void) shutdown(mysql->net.fd,2);
    (void) closesocket(mysql->net.fd);
    mysql->net.fd= -1;
    net_end(&mysql->net);
    free_old_query(mysql);
  }
  DBUG_VOID_RETURN;
}


void mysql_free_result(MYSQL_RES *result)
{
  DBUG_ENTER("mysql_free_result");
  DBUG_PRINT("enter",("mysql_res: %lx",result));
  if (result)
  {
    free_rows(result->data);
    if (result->fields)
      sql_free(&result->field_alloc);
    if (result->row)
      my_free((gptr) result->row,MYF(0));
    my_free((gptr) result,MYF(0));
  }
  DBUG_VOID_RETURN;
}


/***************************************************************************
** Change field rows to field structs
***************************************************************************/

static MYSQL_FIELD *
unpack_fields(MYSQL_DATA *data,MEM_ROOT *alloc,uint fields,bool default_value)
{
  MYSQL_ROWS	*row;
  MYSQL_FIELD	*field,*result;
  DBUG_ENTER("unpack_fields");

  field=result=(MYSQL_FIELD*) sql_alloc_root(alloc,sizeof(MYSQL_FIELD)*fields);
  if (!result)
    DBUG_RETURN(0);

  for (row=data->data; row ; row = row->next,field++)
  {
    field->table=  sql_strdup_root(alloc,(char*) row->data[0]);
    field->name=   sql_strdup_root(alloc,(char*) row->data[1]);
    field->length= (uint) uint3korr(row->data[2]);
    field->type=   (enum enum_field_types) (uchar) row->data[3][0];
    field->flags=   (uint) (uchar) row->data[4][0];
    field->decimals=(uint) (uchar) row->data[4][1];
    if (default_value && row->data[5])
      field->def=sql_strdup_root(alloc,(char*) row->data[5]);
    else
      field->def=0;
    field->max_length= 0;
  }
  free_rows(data);				/* Free old data */
  DBUG_RETURN(result);
}


/* Read all rows (fields or data) from server */

static MYSQL_DATA *read_rows(MYSQL *mysql,MYSQL_FIELD *mysql_fields,
			     uint fields)
{
  uint	field,pkt_len;
  ulong len;
  uchar *cp;
  char	*to;
  MYSQL_DATA *result;
  MYSQL_ROWS **prev_ptr,*cur;
  NET *net = &mysql->net;
  DBUG_ENTER("read_rows");

  if ((pkt_len=(uint) net_safe_read(mysql)) == packet_error)
    DBUG_RETURN(0);
  if (!(result=(MYSQL_DATA*) my_malloc(sizeof(MYSQL_DATA),
				       MYF(MY_WME | MY_ZEROFILL))))
  {
    net->last_errno=CR_OUT_OF_MEMORY;
    strmov(net->last_error,ER(net->last_errno));
    DBUG_RETURN(0);
  }
  init_alloc_root(&result->alloc,8192);		/* Assume rowlength < 8192 */
  result->alloc.min_malloc=sizeof(MYSQL_ROWS);
  prev_ptr= &result->data;
  result->rows=0;
  result->fields=fields;

  while (*(cp=net->buff) != 254 || pkt_len != 1)
  {
    result->rows++;
    if (!(cur= (MYSQL_ROWS*) sql_alloc_root(&result->alloc,
					    sizeof(MYSQL_ROWS))) ||
	!(cur->data= ((MYSQL_ROW)
		      sql_alloc_root(&result->alloc,
				     (fields+1)*sizeof(char *)+pkt_len))))
    {
      free_rows(result);
      net->last_errno=CR_OUT_OF_MEMORY;
      strmov(net->last_error,ER(net->last_errno));
      DBUG_RETURN(0);
    }
    *prev_ptr=cur;
    prev_ptr= &cur->next;
    to= (char*) (cur->data+fields+1);
    for (field=0 ; field < fields ; field++)
    {
      if ((len=net_field_length(&cp)) == NULL_LENGTH)
      {						/* null field */
	cur->data[field] = 0;
      }
      else
      {
	cur->data[field] = to;
	memcpy(to,(char*) cp,len); to[len]=0;
	to+=len+1;
	cp+=len;
	if (mysql_fields)
	{
	  if (mysql_fields[field].max_length < len)
	    mysql_fields[field].max_length=len;
	}
      }
    }
    cur->data[field]=to;			/* End of last field */
    if ((pkt_len=net_safe_read(mysql)) == packet_error)
    {
      free_rows(result);
      DBUG_RETURN(0);
    }
  }
  *prev_ptr=0;					/* last pointer is null */
  DBUG_PRINT("exit",("Got %d rows",result->rows));
  DBUG_RETURN(result);
}


/*
** Read one row. Uses packet buffer as storage for fields.
** When next package is read, the previous field values are destroyed
*/


static int
read_one_row(MYSQL *mysql,uint fields,MYSQL_ROW row, uint *lengths)
{
  uint field,pkt_len,len;
  uchar *pos,*prev_pos;

  if ((pkt_len=(uint) net_safe_read(mysql)) == packet_error)
    return -1;
  if (pkt_len == 1 && mysql->net.buff[0] == 254)
    return 1;				/* End of data */
  prev_pos= 0;				/* allowed to write at packet[-1] */
  pos=mysql->net.buff;
  for (field=0 ; field < fields ; field++)
  {
    if ((len=net_field_length(&pos)) == NULL_LENGTH)
    {						/* null field */
      row[field] = 0;
      *lengths++=0;
    }
    else
    {
      row[field] = (char*) pos;
      pos+=len;
      *lengths++=len;
    }
    if (prev_pos)
      *prev_pos=0;				/* Terminate prev field */
    prev_pos=pos;
  }
  row[field]=(char*) prev_pos+1;		/* End of last field */
  *prev_pos=0;					/* Terminate last field */
  return 0;
}


/**************************************************************************
** Connect to sql server
** If host == 0 then use localhost
**************************************************************************/

MYSQL *mysql_connect(MYSQL *mysql,const char *host,
		     const char *user, const char *passwd)
{
  return mysql_real_connect(mysql,host,user,passwd,0,NullS,0);
}


MYSQL *mysql_real_connect(MYSQL *mysql,const char *host, const char *user,
			  const char *passwd,
			  uint port, const char *unix_socket,uint client_flag)
{
  char		buff[100],scramble_buff[9],*end,*host_info;
  int		sock;
  ulong		ip_addr;
  struct	sockaddr_in sock_addr;
  NET		*net;
#ifndef MSDOS
  char		*env;
  int		opt;
  struct servent *serv_ptr;
#endif

#ifdef HAVE_SYS_UN_H
  struct	sockaddr_un UNIXaddr;
#endif
  DBUG_ENTER("mysql_connect");

#ifndef DONT_USE_MYSQL_PWD
  if (!passwd)
    passwd=getenv("MYSQL_PWD");  /* read Password from environment (haneke) */
#endif
  if (!mysql_client_init)
  {
    mysql_client_init=1;
    my_init();
    init_client_errs();
    if (!mysql_port)
    {
      mysql_port = MYSQL_PORT;
#ifndef MSDOS
      if ((serv_ptr = getservbyname("mysql", "tcp")))
	mysql_port = (uint) ntohs((ushort) serv_ptr->s_port);
      if ((env = getenv("MYSQL_TCP_PORT")))
	mysql_port =(uint) atoi(env);
#endif
    }
#ifndef MSDOS
    if (!mysql_unix_port)
    {
      mysql_unix_port = MYSQL_UNIX_ADDR;
      if ((env = getenv("MYSQL_UNIX_PORT")))
	mysql_unix_port = env;
    }
#endif
    mysql_debug(NullS);
  }

  DBUG_PRINT("enter",("host: %s",host ? host : "(Null)"));
  if (!mysql)
  {
    if (!(mysql=(MYSQL*) my_malloc(sizeof(*mysql),MYF(MY_WME | MY_ZEROFILL))))
      DBUG_RETURN(0);
    mysql->free_me=1;
  }
  else
    bzero(mysql,sizeof(*mysql));
  remember_connection(mysql);
  mysql->reconnect=1;			/* Reconnect as default */
  net= &mysql->net;
  net->fd= -1;				/* If something goes wrong */
  if (!host && !(host=getenv("MYSQL_HOST")) || !host[0])
    host=LOCAL_HOST;

  /*
  ** Grab a socket and connect it to the server
  */

#ifdef HAVE_SYS_UN_H
  if (!strcmp(host,LOCAL_HOST) && (unix_socket || mysql_unix_port))
  {
    if (!unix_socket)
      unix_socket=mysql_unix_port;
    host_info=ER(CR_LOCALHOST_CONNECTION);
    DBUG_PRINT("info",("Using UNIX sock '%s'",unix_socket));
    if ((sock = socket(AF_UNIX,SOCK_STREAM,0)) == SOCKET_ERROR)
    {
      net->last_errno=CR_SOCKET_CREATE_ERROR;
      strmov(net->last_error,ER(net->last_errno));
      goto error;
    }
    net->fd=sock;
    opt = 1;
    (void) setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *)&opt,
		      sizeof(opt));

    bzero(&UNIXaddr,sizeof(UNIXaddr));
    UNIXaddr.sun_family = AF_UNIX;
    strmov(UNIXaddr.sun_path, unix_socket);
    if (connect(sock,(struct sockaddr *) &UNIXaddr, sizeof(UNIXaddr)) <0)
    {
      DBUG_PRINT("error",("Got error %d on connect to local server",ERRNO));
      net->last_errno=CR_CONNECTION_ERROR;
      strmov(net->last_error,ER(net->last_errno));
      goto error;
    }
  }
  else
#endif
  {
    if (!port)
      port=mysql_port;
    sprintf(host_info=buff,ER(CR_TCP_CONNECTION),host);
    DBUG_PRINT("info",("Server name: '%s'.  TCP sock: %d", host,port));
    if ((sock = socket(AF_INET,SOCK_STREAM,0)) == SOCKET_ERROR)
    {
      net->last_errno=CR_IPSOCK_ERROR;
      strmov(net->last_error,ER(net->last_errno));
      goto error;
    }
    net->fd=sock;
#ifndef MSDOS
    opt = 1;
    (void) setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *)&opt,
		      sizeof(opt));
#endif
    bzero(&sock_addr,sizeof(sock_addr));
    sock_addr.sin_family = AF_INET;

    /*
    ** The server name may be a host name or IP address
    */

    if ((int) (ip_addr = inet_addr(host)) != (int) INADDR_NONE)
    {
      memcpy(&sock_addr.sin_addr,&ip_addr,sizeof(ip_addr));
    }
    else
#if defined(HAVE_GETHOSTBYNAME_R) && defined(_REENTRANT) && defined(THREAD)
    {
      int tmp_errno;
      struct hostent tmp_hostent,*hp;
      char buff2[2048];
      hp = gethostbyname_r(host,&tmp_hostent,buff2,sizeof(buff2),&tmp_errno);
      if (!hp)
      {
	net->last_errno=CR_UNKNOWN_HOST;
	sprintf(net->last_error, ER(CR_UNKNOWN_HOST), host, tmp_errno);
	goto error;
      }
      memcpy(&sock_addr.sin_addr,hp->h_addr, (size_t) hp->h_length);
    }
#else
    {
      struct hostent *hp;
      if (!(hp=gethostbyname(host)))
      {
	net->last_errno=CR_UNKNOWN_HOST;
	sprintf(net->last_error, ER(CR_UNKNOWN_HOST), host, errno);
	goto error;
      }
      memcpy(&sock_addr.sin_addr,hp->h_addr, (size_t) hp->h_length);
    }
#endif
    sock_addr.sin_port = (ushort) htons((u_short) port);
    if (connect(sock,(struct sockaddr *) &sock_addr, sizeof(sock_addr))<0)
    {
      DBUG_PRINT("error",("Got error %d on connect to '%s'",ERRNO,host));
      net->last_errno= CR_CONN_HOST_ERROR;
      sprintf(net->last_error ,ER(CR_CONN_HOST_ERROR), host, ERRNO);
      goto error;
    }
  }

#if defined(SIGPIPE)
  (void) signal(SIGPIPE,SIG_IGN);
#endif
  DBUG_PRINT("info",("Connection socket %d",sock));
  if (my_net_init(net,sock))
    goto error;
  /* Get version info */
  mysql->protocol_version= PROTOCOL_VERSION;	/* Assume this */
  if (net_safe_read(mysql) == packet_error)
    goto error;

  /* Check if version of protocoll matches current one */

  mysql->protocol_version= net->buff[0];
  DBUG_DUMP("packet",(char*) net->buff,10);
  DBUG_PRINT("info",("mysql protocol version %d, server=%d",
		     PROTOCOL_VERSION, mysql->protocol_version));
  if (mysql->protocol_version != PROTOCOL_VERSION &&
      mysql->protocol_version != PROTOCOL_VERSION-1)
  {
    net->last_errno= CR_VERSION_ERROR;
    sprintf(net->last_error, ER(CR_VERSION_ERROR), mysql->protocol_version,
	    PROTOCOL_VERSION);
    goto error;
  }
  end=strend((char*) net->buff+1);
  mysql->thread_id=uint4korr(end+1);
  strmov(scramble_buff,end+5);

  /* Save connection information */
  if (!user) user="";
  if (!passwd) passwd="";
  if (!my_multi_malloc(MYF(0),
		      &mysql->host_info,strlen(host_info)+1,
		      &mysql->host,strlen(host)+1,
		      &mysql->user,strlen(user)+1,
		      &mysql->passwd,strlen(passwd)+1,
		      &mysql->unix_socket,unix_socket ? strlen(unix_socket)+1
		      :1,
		      &mysql->server_version,(uint) (end - (char*) net->buff),
		      NullS))
  {
    strmov(net->last_error, ER(net->last_errno=CR_OUT_OF_MEMORY));
    goto error;
  }
  strmov(mysql->host_info,host_info);
  strmov(mysql->host,host);
  strmov(mysql->user,user);
  strmov(mysql->passwd,passwd);
  if (unix_socket)
    strmov(mysql->unix_socket,unix_socket);
  else
    mysql->unix_socket=0;
  strmov(mysql->server_version,(char*) net->buff+1);
  mysql->port=port;
  mysql->client_flag=client_flag;
  DBUG_PRINT("info",("Server version = '%s'",mysql->server_version));

  /* Send client information for access check */
  int2store(buff,CLIENT_CAPABILITIES);
  int3store(buff+2,max_allowed_packet);
  if (user && user[0])
    strmake(buff+5,user,32);
  else
    read_user_name((char*) buff+5);
  DBUG_PRINT("info",("user: %s",buff+5));
  end=scramble(strend(buff+5)+1, scramble_buff, passwd,
	       (my_bool) (mysql->protocol_version == 9));

  if (my_net_write(net,buff,(uint) (end-buff)) || net_flush(net) ||
      net_safe_read(mysql) == packet_error)
    goto error;
  DBUG_PRINT("exit",("Mysql handler: %lx",mysql));
  DBUG_RETURN(mysql);

error:
  DBUG_PRINT("error",("message: %u (%s)",net->last_errno,net->last_error));
  end_server(mysql);
  if (mysql->free_me)
    my_free((gptr) mysql,MYF(0));
  DBUG_RETURN(0);
}


/**************************************************************************
** Set current database
**************************************************************************/

int
mysql_select_db(MYSQL *mysql, const char *db)
{
  int error;
  DBUG_ENTER("mysql_select_db");
  DBUG_PRINT("enter",("db: '%s'",db));

  if ((error=simple_command(mysql,COM_INIT_DB,db,strlen(db),0)))
    DBUG_RETURN(error);
  my_free(mysql->db,MYF(MY_ALLOW_ZERO_PTR));
  mysql->db=my_strdup(db,MYF(MY_WME));
  DBUG_RETURN(0);
}


/*************************************************************************
** Send a QUIT to the server and close the connection
** If handle is alloced by mysql connect free it.
*************************************************************************/

void
mysql_close(MYSQL *mysql)
{
  DBUG_ENTER("mysql_close");
  if (mysql)					/* Some simple safety */
  {
    if (mysql->net.fd >= 0)
    {
      free_old_query(mysql);
      mysql->status=MYSQL_STATUS_READY;	/* Force command */
      simple_command(mysql,COM_QUIT,NullS,0,1);
      end_server(mysql);
    }
    my_free((gptr) mysql->host_info,MYF(MY_ALLOW_ZERO_PTR));
    my_free(mysql->db,MYF(MY_ALLOW_ZERO_PTR));
    mysql->host_info=mysql->db=0;
    if (mysql->free_me)
      my_free((gptr) mysql,MYF(0));
  }
  DBUG_VOID_RETURN;
}


/**************************************************************************
** Do a query. If query returned rows, free old rows.
** Read data by mysql_store_result or by repeat call of mysql_fetch_row
**************************************************************************/

int
mysql_query(MYSQL *mysql, const char *query)
{
  return mysql_real_query(mysql,query,strlen(query));
}


int
mysql_real_query(MYSQL *mysql, const char *query,uint length)
{
  uchar *pos;
  uint field_count;
  MYSQL_DATA *fields;
  DBUG_ENTER("mysql_real_query");
  DBUG_PRINT("enter",("handle: %lx",mysql));
  DBUG_PRINT("query",("Query = \"%s\"",query));

  if (simple_command(mysql,COM_QUERY,query,length,1) ||
      (length=net_safe_read(mysql)) == packet_error)
    DBUG_RETURN(-1);
  free_old_query(mysql);			/* Free old result */
  pos=(uchar*) mysql->net.buff;
  if ((field_count=(uint) net_field_length(&pos)) == 0)
  {
    mysql->affected_rows= net_field_length(&pos);
    mysql->insert_id=	  net_field_length(&pos);
    if (pos < mysql->net.buff+length && net_field_length(&pos))
      mysql->info=(char*) pos;
    DBUG_RETURN(0);
  }
  mysql->extra_info= net_field_length(&pos); /* Maybe number of rec */
  if (!(fields=read_rows(mysql,(MYSQL_FIELD*) 0,5)))
    DBUG_RETURN(-1);
  if (!(mysql->fields=unpack_fields(fields,&mysql->field_alloc,field_count,0)))
    DBUG_RETURN(-1);
  mysql->status=MYSQL_STATUS_GET_RESULT;
  mysql->field_count=field_count;
  DBUG_RETURN(0);
}


/**************************************************************************
** Alloc result struct for buffered results. All rows are read to buffer.
** mysql_data_seek may be used.
**************************************************************************/

MYSQL_RES *
mysql_store_result(MYSQL *mysql)
{
  MYSQL_RES *result;
  DBUG_ENTER("mysql_store_result");

  if (!mysql->fields)
    DBUG_RETURN(0);
  if (mysql->status != MYSQL_STATUS_GET_RESULT)
  {
    strmov(mysql->net.last_error,
	   ER(mysql->net.last_errno=CR_COMMANDS_OUT_OF_SYNC));
    DBUG_RETURN(0);
  }
  mysql->status=MYSQL_STATUS_READY;		/* server is ready */
  if (!(result=(MYSQL_RES*) my_malloc(sizeof(MYSQL_RES)+
				      sizeof(uint)*mysql->field_count,
				      MYF(MY_WME | MY_ZEROFILL))))
  {
    mysql->net.last_errno=CR_OUT_OF_MEMORY;
    strmov(mysql->net.last_error, ER(mysql->net.last_errno));
    DBUG_RETURN(0);
  }
  result->eof=1;				/* Marker for buffered */
  result->lengths=(uint*) (result+1);
  if (!(result->data=read_rows(mysql,mysql->fields,mysql->field_count)))
  {
    my_free((gptr) result,MYF(0));
    DBUG_RETURN(0);
  }
  mysql->affected_rows= result->row_count=result->data->rows;
  result->data_cursor=  result->data->data;
  result->fields=	mysql->fields;
  result->field_alloc=	mysql->field_alloc;
  result->field_count=	mysql->field_count;
  result->current_field=0;
  result->current_row=0;			/* Must do a fetch first */
  mysql->fields=0;				/* fields is now in result */
  DBUG_RETURN(result);				/* Data fetched */
}


/**************************************************************************
** Alloc struct for use with unbuffered reads. Data is fetched by domand
** when calling to mysql_fetch_row.
** mysql_data_seek is a noop.
**
** No other queries may be specified with the same MYSQL handle.
** There shouldn't be much processing per row because mysql server shouldn't
** have to wait for the client (and will not wait more than 30 sec/packet).
**************************************************************************/

MYSQL_RES *
mysql_use_result(MYSQL *mysql)
{
  MYSQL_RES *result;
  DBUG_ENTER("mysql_use_result");

  if (!mysql->fields)
    DBUG_RETURN(0);
  if (mysql->status != MYSQL_STATUS_GET_RESULT)
  {
    strmov(mysql->net.last_error,
	   ER(mysql->net.last_errno=CR_COMMANDS_OUT_OF_SYNC));
    DBUG_RETURN(0);
  }
  if (!(result=(MYSQL_RES*) my_malloc(sizeof(MYSQL_RES)+
				      sizeof(uint)*mysql->field_count,
				      MYF(MY_WME | MY_ZEROFILL))))
    DBUG_RETURN(0);
  result->lengths=(uint*) (result+1);
  if (!(result->row=(MYSQL_ROW)
	my_malloc(sizeof(result->row[0])*(mysql->field_count+1), MYF(MY_WME))))
  {					/* Ptrs: to one row */
    my_free((gptr) result,MYF(0));
    DBUG_RETURN(0);
  }
  result->fields=	mysql->fields;
  result->field_alloc=	mysql->field_alloc;
  result->field_count=	mysql->field_count;
  result->current_field=0;
  result->handle=	mysql;
  result->current_row=	0;
  mysql->fields=0;			/* fields is now in result */
  mysql->status=MYSQL_STATUS_USE_RESULT;
  DBUG_RETURN(result);			/* Data is read to be fetched */
}



/**************************************************************************
** Return next field of the query results
**************************************************************************/

MYSQL_FIELD *
mysql_fetch_field(MYSQL_RES *result)
{
  if (result->current_field >= result->field_count)
    return(NULL);
  return &result->fields[result->current_field++];
}


/**************************************************************************
**  Return next row of the query results
**************************************************************************/

MYSQL_ROW
mysql_fetch_row(MYSQL_RES *res)
{
  if (!res->data)
  {						/* Unbufferred fetch */
    int error;
    if (!res->eof)
    {
      if (!(error=read_one_row(res->handle,res->field_count,res->row,
			       res->lengths)))
      {
	res->row_count++;
	return (res->current_row=res->row);
      }
      else
      {
	res->eof=1;
	res->handle->status=MYSQL_STATUS_READY;
      }
    }
    return (MYSQL_ROW) NULL;
  }
  else
  {
    MYSQL_ROW tmp;
    if (!res->data_cursor)
      return (res->current_row=(MYSQL_ROW) NULL);
    tmp = res->data_cursor->data;
    res->data_cursor = res->data_cursor->next;
    return (res->current_row=tmp);
  }
}

/**************************************************************************
** Get column lengths of the current row
** If one uses mysql_use_result, res->lengths contains the length information,
** else the lengths are calculated from the offset between pointers.
**************************************************************************/

uint *
mysql_fetch_lengths(MYSQL_RES *res)
{
  uint *lengths,*prev_length;
  byte *start;
  MYSQL_ROW column,end;

  if (!(column=res->current_row))
    return 0;					/* Something is wrong */
  if (res->data)
  {
    start=0;
    prev_length=0;				/* Keep gcc happy */
    lengths=res->lengths;
    for (end=column+res->field_count+1 ; column != end ; column++,lengths++)
    {
      if (!*column)
      {
	*lengths=0;				/* Null */
	continue;
      }
      if (start)				/* Found end of prev string */
	*prev_length= (uint) (*column-start-1);
      start= *column;
      prev_length=lengths;
    }
  }
  return res->lengths;
}

/**************************************************************************
** Move to a specific row and column
**************************************************************************/

void
mysql_data_seek(MYSQL_RES *result, uint row)
{
  MYSQL_ROWS	*tmp=0;
  DBUG_PRINT("info",("mysql_data_seek(%d)",row));
  if (result->data)
    for (tmp=result->data->data; row-- && tmp ; tmp = tmp->next) ;
  result->current_row=0;
  result->data_cursor = tmp;
}

/*************************************************************************
** put the row or field cursor at the saved position.
** This dosen't restore any data. The next mysql_fetch_row or
** mysql_fetch_field will return the next row or field after the last used
*************************************************************************/

MYSQL_ROW_OFFSET
mysql_row_seek(MYSQL_RES *result, MYSQL_ROW_OFFSET row)
{
  MYSQL_ROW_OFFSET return_value=result->data_cursor;
  result->current_row= 0;
  result->data_cursor= row;
  return return_value;
}


MYSQL_FIELD_OFFSET
mysql_field_seek(MYSQL_RES *result, uint fieldnr)
{
  MYSQL_FIELD_OFFSET return_value=result->current_field;
  result->current_field=fieldnr;
  return return_value;
}

/*****************************************************************************
** List all databases
*****************************************************************************/

MYSQL_RES *
mysql_list_dbs(MYSQL *mysql, const char *wild)
{
  char buff[100];
  DBUG_ENTER("mysql_list_dbs");

  append_wild(strmov(buff,"show databases"),buff+sizeof(buff),wild);
  if (mysql_query(mysql,buff) < 0)
    DBUG_RETURN(0);
  DBUG_RETURN (mysql_store_result(mysql));
}


/*****************************************************************************
** List all tables in a database
** If wild is given then only the tables matching wild is returned
*****************************************************************************/

MYSQL_RES *
mysql_list_tables(MYSQL *mysql, const char *wild)
{
  char buff[100];
  DBUG_ENTER("mysql_list_tables");

  append_wild(strmov(buff,"show tables"),buff+sizeof(buff),wild);
  if (mysql_query(mysql,buff) < 0)
    DBUG_RETURN(0);
  DBUG_RETURN (mysql_store_result(mysql));
}


/**************************************************************************
** List all fields in a table
** If wild is given then only the fields matching wild is returned
** Instead of this use query:
** show fields in 'table' like "wild"
**************************************************************************/

MYSQL_RES *
mysql_list_fields(MYSQL *mysql, const char *table, const char *wild)
{
  MYSQL_RES *result;
  MYSQL_DATA *query;
  char	     buff[257],*end;
  DBUG_ENTER("mysql_list_fields");

  LINT_INIT(query);

  end=strmake(strmake(buff, table,128)+1,wild ? wild : "",128);
  if (simple_command(mysql,COM_FIELD_LIST,buff,(uint) (end-buff),1) ||
      !(query = read_rows(mysql,(MYSQL_FIELD*) 0,6)))
    DBUG_RETURN(NULL);

  free_old_query(mysql);
  if (!(result = (MYSQL_RES *) my_malloc(sizeof(MYSQL_RES),
					 MYF(MY_WME | MY_ZEROFILL))))
  {
    free_rows(query);
    DBUG_RETURN(NULL);
  }
  result->field_alloc=mysql->field_alloc;
  mysql->fields=0;
  result->field_count = query->rows;
  result->fields=	unpack_fields(query,&result->field_alloc,
				      result->field_count,1);
  result->eof=1;
  DBUG_RETURN(result);
}

/* List all running processes (threads) in server */

MYSQL_RES *
mysql_list_processes(MYSQL *mysql)
{
  MYSQL_DATA *fields;
  uint field_count;
  uchar *pos;
  DBUG_ENTER("mysql_list_processes");

  LINT_INIT(fields);
  if (simple_command(mysql,COM_PROCESS_INFO,0,0,0))
    DBUG_RETURN(0);
  free_old_query(mysql);
  pos=(uchar*) mysql->net.buff;
  field_count=(uint) net_field_length(&pos);
  if (!(fields = read_rows(mysql,(MYSQL_FIELD*) 0,5)))
    DBUG_RETURN(NULL);
  if (!(mysql->fields=unpack_fields(fields,&mysql->field_alloc,field_count,0)))
    DBUG_RETURN(0);
  mysql->status=MYSQL_STATUS_GET_RESULT;
  mysql->field_count=field_count;
  DBUG_RETURN(mysql_store_result(mysql));
}


int
mysql_create_db(MYSQL *mysql, const char *db)
{
  DBUG_ENTER("mysql_createdb");
  DBUG_PRINT("enter",("db: %s",db));
  DBUG_RETURN(simple_command(mysql,COM_CREATE_DB,db, (uint) strlen(db),0));
}


int
mysql_drop_db(MYSQL *mysql, const char *db)
{
  DBUG_ENTER("mysql_drop_db");
  DBUG_PRINT("enter",("db: %s",db));
  DBUG_RETURN(simple_command(mysql,COM_DROP_DB,db,(uint) strlen(db),0));
}


int
mysql_shutdown(MYSQL *mysql)
{
  DBUG_ENTER("mysql_shutdown");
  DBUG_RETURN(simple_command(mysql,COM_SHUTDOWN,0,0,0));
}


int
mysql_refresh(MYSQL *mysql,uint options)
{
  uchar bits[1];
  DBUG_ENTER("mysql_refresh");
  bits[0]= (uchar) options;
  DBUG_RETURN(simple_command(mysql,COM_REFRESH,(char*) bits,1,0));
}

int
mysql_kill(MYSQL *mysql,ulong pid)
{
  char buff[4];
  DBUG_ENTER("mysql_kill");
  int4store(buff,pid);
  DBUG_RETURN(simple_command(mysql,COM_PROCESS_KILL,buff,4,0));
}


int
mysql_dump_debug_info(MYSQL *mysql)
{
  DBUG_ENTER("mysql_dump_debug_info");
  DBUG_RETURN(simple_command(mysql,COM_DEBUG,0,0,0));
}

char *
mysql_stat(MYSQL *mysql)
{
  DBUG_ENTER("mysql_stat");
  if (simple_command(mysql,COM_STATISTICS,0,0,0))
    return mysql->net.last_error;
  if (!mysql->net.buff[0])
  {
    mysql->net.last_errno=CR_WRONG_HOST_INFO;
    strmov(mysql->net.last_error, ER(mysql->net.last_errno));
    return mysql->net.last_error;
  }
  DBUG_RETURN((char*) mysql->net.buff);
}


char *
mysql_get_server_info(MYSQL *mysql)
{
  return((char*) mysql->server_version);
}


char *
mysql_get_host_info(MYSQL *mysql)
{
  return(mysql->host_info);
}


uint
mysql_get_proto_info(MYSQL *mysql)
{
  return (mysql->protocol_version);
}

char *
mysql_get_client_info(void)
{
  return MYSQL_SERVER_VERSION;
}


/****************************************************************************
** Some support functions
****************************************************************************/

/*
** Add escape characters to a string (blob?) to make it suitable for a insert
** to should at least have place for length*2+1 chars
*/

uint mysql_escape_string(char *to,char *from,uint length)
{
  char *end;
  for (end=from+length; from != end ; from++)
  {
#ifdef USE_BIG5CODE
    if (from+1 != end && isbig5head(*from))
    {
      *to++= *from++;
      *to++= *from;
      continue;
    }
#endif
    switch (*from) {
    case 0:				/* Must be escaped for 'mysql' */
      *to++= '\\';
      *to++= '0';
      break;
    case '\n':				/* Must be escaped for logs */
      *to++= '\\';
      *to++= 'n';
      break;
    case '\r':
      *to++= '\\';
      *to++= 'r';
      break;
    case '\\':
      *to++= '\\';
      *to++= '\\';
      break;
    case '\'':
      *to++= '\'';
      *to++= '\'';
      break;
    default:
      *to++= *from;
    }
  }
  *to=0;
  return (uint) (to-from);
}
