/*
**	mysqldump.c  - Dump a tables contents and format to an ASCII file
**
**
** This software is provided "as is" without any expressed or implied warranty.
**
** The author's original notes follow :-
**
**		******************************************************
**		*						     *
**		* AUTHOR: Igor Romanenko (igor@frog.kiev.ua)	     *
**		* DATE:   December 3, 1994			     *
**		* WARRANTY: None, expressed, impressed, implied      *
**		*	    or other				     *
**		* STATUS: Public domain				     *
**		* Adapted and optimized for mysql by		     *
**		* Michael Widenius				     *
**		*						     *
**		******************************************************
*/

#define DUMP_VERSION "4.0"

#include <global.h>
#include <my_sys.h>
#include <m_string.h>
#include "mysql.h"
#include "mysql_version.h"
#include <getopt.h>

/* Exit codes */

#define EX_USAGE 1
#define EX_MYSQLERR 2
#define EX_CONSCHECK 3

/* index into 'show fields from table' */

#define SHOW_FIELDNAME	0
#define SHOW_TYPE	1
#define SHOW_NULL	2
#define SHOW_DEFAULT	4
#define SHOW_EXTRA	5

static bool	verbose = 0, tFlag = 0, cFlag = 0,dFlag = 0, quick=0,
		lock_tables=0, ignore_errors=0, flush_logs=0;
static MYSQL	mysql,*sock=0;
static char	insert_pat[12 * 1024],*password=0,*current_user=0,
		*current_host=0,*current_db=0,*path=0;
static int	first_error=0;
static struct option long_options[] =
{
  {"debug",		optional_argument,	0, '#'},
  {"force",		no_argument,		0, 'f'},
  {"help",		no_argument,		0, '?'},
  {"complete-insert",	no_argument,		0, 'c'},
  {"flush-logs",	no_argument,		0, 'F'},
  {"host",		required_argument,	0, 'h'},
  {"lock-tables",	no_argument,		0, 'l'},
  {"no-create-info",	no_argument,		0, 't'},
  {"no-data",		no_argument,		0, 'd'},
  {"password",		optional_argument,	0, 'p'},
  {"port",		required_argument,	0, 'P'},
  {"quick",		no_argument,		0, 'q'},
  {"set-variable",	required_argument,	0, 'O'},
  {"socket",		required_argument,	0, 'S'},
  {"tab",		required_argument,	0, 'T'},
#ifndef DONT_ALLOW_USER_CHANGE
  {"user",		required_argument,	0, 'u'},
#endif
  {"verbose",		no_argument,		0, 'v'},
  {"version",		no_argument,		0, 'V'},
  {0, 0, 0, 0}
};


CHANGEABLE_VAR changeable_vars[] = {
  { "net_buffer_length", (long*) &net_buffer_length,16384,4096,16384*1024L,
    MALLOC_OVERHEAD,1024},
  { 0, 0, 0, 0, 0, 0, 0}
};


static void safe_exit(int error);
static void write_heder(FILE *sql_file);


static void print_version(void)
{
  printf("%s  Ver %s Distrib %s, for %s (%s)\n",my_progname,DUMP_VERSION,
	 MYSQL_SERVER_VERSION,SYSTEM_TYPE,MACHINE_TYPE);
}


static void usage(void)
{
  print_version();
  puts("By Igor Romanenko & Monty & Jani. This software is in public Domain");
  puts("This software comes with ABSOLUTELY NO WARRANTY\n");
  puts("Dumping definition and data mysql database or table");
  printf("Usage: %s [OPTIONS] database [tables]\n",my_progname);
  printf("\n\
  -#, --debug=...       Output debug log. Often this is 'd:t:o,filename`\n\
  -?, --help		Displays this help and exits.\n\
  -c, --compleat-insert Use complete insert statements.\n\
  -F  --flush-logs	Flush logs file in server before starting dump\n\
  -f, --force		Continue even if we get an sql-error.\n\
  -h, --host=...	Connect to host.\n\
  -l, --lock-tables     Lock all tables for read.\n\
  -t, --no-create-info	Don't write table creation info.\n\
  -d, --no-data		No row information.\n\
  -O, --set-variable var=option\n\
			give a variable an value. --help lists variables\n\
  -p, --password[=...]	Password to use when connecting to server.\n\
			If password is not given it's asked from the tty.\n\
  -P, --port=...	Port number to use for connection.\n\
  -q, --quick		Don't buffer query, dump directly to stdout.\n\
  -S, --socket=...	Socket file to use for connection.\n\
  -T, --tab=...         Creates tab separated textfile for each table to\n\
	                given path. (creates .sql and .txt files)\n");
#ifndef DONT_ALLOW_USER_CHANGE
  printf("\
  -u, --user=#		User for login if not current user.\n");
#endif
  printf("\
  -v, --verbose		Print info about the various stages.\n\
  -V, --version		Output version information and exit.\n"
  );
}


static void write_heder(FILE *sql_file)
{
  fprintf(sql_file, "# MySQL dump %s\n#\n", DUMP_VERSION);
  fprintf(sql_file, "# Host: %s    Database: %s\n",
	  current_host ? current_host : "localhost", current_db);
  fputs("#--------------------------------------------------------\n",
	sql_file);
}


static int get_options(int *argc,char ***argv)
{
  int c,option_index;
  bool tty_password=0;

  set_all_changeable_vars(changeable_vars);
  while ((c=getopt_long(*argc,*argv,"#::p::h:u:O:P:S:T:cdflqtvV?I"
			,long_options, &option_index)) != EOF)
  {
    switch(c) {
    case 'f':
      ignore_errors=1;
      break;
    case 'F':
      flush_logs=1;
      break;
    case 'h':
      current_host=my_strdup(optarg,MYF(MY_WME));
      break;
#ifndef DONT_ALLOW_USER_CHANGE
    case 'u':
      current_user=optarg;
      break;
#endif
    case 'O':
      if (set_changeable_var(optarg, changeable_vars))
      {
	usage();
	return(1);
      }
      break;
    case 'p':
      if (optarg)
      {
	password=my_strdup(optarg,MYF(MY_FAE));
	while (*optarg) *optarg++= 'x';		/* Destroy argument */
      }
      else
	tty_password=1;
      break;
    case 'P':
      mysql_port= (unsigned int) atoi(optarg);
      break;
    case 'S':
      mysql_unix_port= optarg;
      break;
    case 'T':
      path= optarg;
      break;
    case '#':
      DBUG_PUSH(optarg ? optarg : "d:t:o");
      break;
    case 'c': cFlag=1; break;
    case 'd': dFlag=1; break;
    case 'l': lock_tables=1; break;
    case 'q': quick=1; break;
    case 't': tFlag=1;	break;
    case 'v': verbose=1; break;
    case 'V': print_version(); exit(0);
    default:
      fprintf(stderr,"Illegal option character '%c'\n",opterr);
      /* Fall throught */
    case 'I':
    case '?':
      usage();
      exit(0);
    }
  }
  (*argc)-=optind;
  (*argv)+=optind;
  if (*argc < 1)
  {
    usage();
    return 1;
  }
  current_db= *((*argv)++);
  (*argc)--;
  if (tty_password)
    password=get_tty_password(NullS);
  return(0);
}


/*
** DBerror -- prints mysql error message and exits the program.
*/

static void DBerror(MYSQL *mysql)
{
  my_printf_error(0,"mysql error: %s", MYF(0), mysql_error(mysql));
  safe_exit(EX_MYSQLERR);
}

static void safe_exit(int error)
{
  if (!first_error)
    first_error= error;
  if (ignore_errors)
    return;
  if (sock)
    mysql_close(sock);
  exit(error);
}
/*
** dbConnect -- connects to the host and selects DB.
**	      Also checks whether the tablename is a valid table name.
*/

void dbConnect(char *host, char *database,char *user,char *passwd)
{
  if (verbose)
  {
    fprintf(stderr, "Connecting to %s...\n", host ? host : "localhost");
  }
  if (!(sock = mysql_connect(&mysql,host,user,passwd)))
  {
    ignore_errors=0;	  /* NO RETURN FROM DBerror */
    DBerror(&mysql);
  }
  if ( verbose )
    fprintf(stderr, "Selecting data base %s...\n", database);
  if (mysql_select_db(sock, database) == -1)
  {
    ignore_errors=0;	  /* NO RETURN FROM DBerror */
    DBerror(&mysql);
  }
}



/*
** dbDisconnect -- disconnects from the host.
*/

void dbDisconnect(char *host)
{
  if (verbose)
    fprintf(stderr, "Disconnecting from %s...\n", host ? host : "localhost");
  mysql_close(sock);
}

static void unescape(FILE *file,char *pos,uint length)
{
  char *tmp=(char*) my_malloc(length*2+1, MYF(MY_WME));
  if (!tmp)
  {
    ignore_errors=0;				/* Fatal error */
    safe_exit(EX_MYSQLERR);			/* Force exit */
  }
  mysql_escape_string(tmp, pos, length);
  fputc('\'', file);
  fputs(tmp, file);
  fputc('\'', file);
  my_free(tmp, MYF(MY_WME));
}


/*
** getStructure -- retrievs database structure, prints out corresponding
**		   CREATE statement and fills out insert_pat.
** Return values:  number of fields in table, 0 if error
*/

uint getTableStructure(char *table)
{
  MYSQL_RES	*tableRes;
  MYSQL_ROW	row;
  uint		init=0, numFields;
  char		*strpos;
  FILE		*sql_file = stdout;

  if (verbose)
    fprintf(stderr, "Retrieving table structure for table %s...\n", table);

  sprintf(insert_pat,"show fields from %s",table);
  if (mysql_query(sock,insert_pat) || !(tableRes=mysql_store_result(sock)))
  {
    fprintf(stderr, "mysql error: Can't get info about table: '%s' (%s)\n",
	    table, mysql_error(sock));
    if (sql_file != stdout)
      my_fclose(sql_file, MYF(MY_WME));
    safe_exit(EX_MYSQLERR);
    return 0;
  }

  /* Make an sql-file, if path was given iow. option -T was given */
  if (!tFlag)
  {
    if (path)
    {
      char filename[FN_REFLEN], tmp_path[FN_REFLEN];
      strmov(tmp_path,path);
      convert_dirname(tmp_path);
      sql_file= my_fopen(fn_format(filename, table, tmp_path, ".sql", 4),
			 O_WRONLY, MYF(MY_WME));
      if (!sql_file) /* If file couldn't be opened */
      {
	safe_exit(EX_MYSQLERR);
	return 0;
      }
      write_heder(sql_file);
    }
    fprintf(sql_file, "\n#\n# Table structure for table '%s'\n#\n", table);
    fprintf(sql_file, "CREATE TABLE %s (\n", table);
  }
  if (cFlag)
    sprintf(insert_pat, "INSERT INTO %s (", table);
  else
    sprintf(insert_pat, "INSERT INTO %s VALUES (", table);

  strpos=strend(insert_pat);
  while ((row=mysql_fetch_row(tableRes)))
  {
    uint *lengths=mysql_fetch_lengths(tableRes);
    if (init)
    {
      if (!tFlag)
	fputs(",\n",sql_file);
      if (cFlag)
	strpos=strmov(strpos,", ");
    }
    init=1;
    if (cFlag)
      strpos=strmov(strpos,row[SHOW_FIELDNAME]);
    if (!tFlag)
    {
      fprintf(sql_file, "  %s %s", row[SHOW_FIELDNAME],row[SHOW_TYPE]);
      if (row[SHOW_DEFAULT])
      {
	fputs(" DEFAULT ", sql_file);
	unescape(sql_file,row[SHOW_DEFAULT],lengths[SHOW_DEFAULT]);
      }
      if (!row[SHOW_NULL][0])
	fputs(" NOT NULL", sql_file);
      if (row[SHOW_EXTRA][0])
	fprintf(sql_file, " %s",row[SHOW_EXTRA]);
    }
  }
  numFields = mysql_num_rows(tableRes);
  mysql_free_result(tableRes);
  if (!tFlag)
  {
    char buff[20+FN_REFLEN];
    uint keynr,primary_key;
    sprintf(buff,"show keys from %s",table);
    if (mysql_query(sock, buff))
    {
      fprintf(stderr, "mysql error: Can't get keys for table %s (%s)\n", table,
	      mysql_error(sock));
      if (sql_file != stdout)
	my_fclose(sql_file, MYF(MY_WME));
      safe_exit(EX_MYSQLERR);
      return 0;
    }

    tableRes=mysql_store_result(sock);
    /* Find first which key is primary key */
    keynr=0;
    primary_key=INT_MAX;
    while ((row=mysql_fetch_row(tableRes)))
    {
      if (atoi(row[3]) == 1)
      {
	keynr++;
	if (atoi(row[1]) == 0 && primary_key == INT_MAX)
	  primary_key=keynr;
	if (!strcmp(row[2],"PRIMARY"))
	{
	  primary_key=keynr;
	  break;
	}
      }
    }
    mysql_data_seek(tableRes,0);
    keynr=0;
    while ((row=mysql_fetch_row(tableRes)))
    {
      if (atoi(row[3]) == 1)
      {
	if (keynr++)
	  putc(')', sql_file);
	if (atoi(row[1]))			 /* Test if dupplicate key */
	  fprintf(sql_file, ",\n  KEY %s (",row[2]); /* Dupplicate allowed */
	else if (keynr == primary_key)
	  fputs(",\n  PRIMARY KEY (",sql_file); /* First UNIQUE is primary */
	else
	  fprintf(sql_file, ",\n  UNIQUE %s (",row[2]); /* UNIQUE key */
      }
      else
	putc(',', sql_file);
      fputs(row[4], sql_file);
      if (row[7])
	fprintf(sql_file, "(%s)",row[7]);			/* Sub key */
    }
    if (keynr)
      putc(')', sql_file);
    fputs("\n);\n", sql_file);
  }
  if (cFlag)
    strpos=strmov(strpos,") VALUES (");
  return(numFields);
}




/*
** dumpTable saves database contents as a series of INSERT statements.
*/

void dumpTable(uint numFields, char *table)
{
  char query[FN_REFLEN*2+48];
  MYSQL_RES	*res;
  MYSQL_FIELD	*field;
  MYSQL_ROW		row;
  uint		i;
  int		init = 1;

  if (verbose)
    fprintf(stderr, "Sending SELECT query...\n");
  if (path)
  {
    char filename[FN_REFLEN], tmp_path[FN_REFLEN];
    strmov(tmp_path, path);
    convert_dirname(tmp_path);
    my_load_path(tmp_path, tmp_path, NULL);
    fn_format(filename, table, tmp_path, ".txt", 4);
    my_delete(filename, MYF(0)); /* 'INTO OUTFILE' doesn't work, if
				    filename wasn't deleted */
    sprintf(query, "SELECT * INTO OUTFILE '%s' FROM %s", filename, table);
    if (mysql_query(sock, query))
    {
      DBerror(sock);
      return;
    }
  }
  else
  {
    printf("\n#\n# Dumping data for table '%s'\n#\n\n", table);
    sprintf(query, "SELECT * FROM %s", table);
    if (mysql_query(sock, query))
      {
	DBerror(sock);
	return;
      }
    if (quick)
      res=mysql_use_result(sock);
    else
      res=mysql_store_result(sock);
    if (!res)
      {
	DBerror(sock);
	return;
      }
    if (verbose)
      fprintf(stderr, "Retrieving rows...\n");
    if (mysql_num_fields(res) != numFields)
    {
      fprintf(stderr,"Error in field count!  Aborting.\n\n");
      safe_exit(EX_CONSCHECK);
      return;
    }

    while ((row=mysql_fetch_row(res)))
    {
      uint *lengths=mysql_fetch_lengths(res);

      fputs(insert_pat,stdout);
      init = 1;
      mysql_field_seek(res,0);
      for (i = 0; i < mysql_num_fields(res); i++)
      {
	if (!(field = mysql_fetch_field(res)))
	{
	  fprintf(stderr,"Not enough fields! Aborting\n");
	  safe_exit(EX_CONSCHECK);
	  return;
	}
	if (!init )
	  printf(",");
	else
	  init=0;
	if (row[i])
	{
	  if (!IS_NUM(field->type))
	    unescape(stdout, row[i], lengths[i]);
	  else
	    fputs(row[i],stdout);
	}
	else
	{
	  fputs("NULL",stdout);
	}
      }
      fputs(");\n",stdout);
    }
    if (!mysql_eof(res))
    {
      fprintf(stderr,"Unexpected error: %s",mysql_error(sock));
      safe_exit(EX_CONSCHECK);
      return;
    }
    mysql_free_result(res);
  }
}



char *getTableName(int reset)
{
  static MYSQL_RES *res = NULL;
  MYSQL_ROW		row;

  if (!res)
  {
    if (!(res = mysql_list_tables(sock,NullS)))
      return(NULL);
  }
  if ((row = mysql_fetch_row(res)))
  {
    return((char*) row[0]);
  }
  else
  {
    if (reset)
      mysql_data_seek(res,0);			/* We want to read again */
    else
      mysql_free_result(res);
    return(NULL);
  }
}


int main(int argc, char **argv)
{
  uint	numRows;
  MY_INIT(argv[0]);

  /*
  ** Check out the args
  */
  if (get_options(&argc,&argv))
    exit(EX_USAGE);
  if (!path)
    write_heder(stdout);
  dbConnect(current_host,current_db,current_user,password);
  if (argc)
  {
    if (lock_tables)
    {
      DYNAMIC_STRING query;
      int i;
      init_dynamic_string(&query, "LOCK TABLES ", 256, 1024);
      for (i=0 ; i < argc ; i++)
      {
	dynstr_append(&query, argv[i]);
	dynstr_append(&query, " READ,");
      }
      if (mysql_real_query(sock, query.str, query.length-1))
	DBerror(sock); /* We shall countinue here, if --force was given */
    }
    if (flush_logs)
    {
      if (mysql_refresh(sock, REFRESH_LOG))
	DBerror(sock); /* We shall countinue here, if --force was given */
    }
    for (; argc > 0 ; argc-- , argv++)
    {
      numRows = getTableStructure(*argv);
      if (!dFlag && numRows > 0)
	dumpTable(numRows,*argv);
    }
  }
  else
  {
    char *table;
    if (lock_tables)
    {
      DYNAMIC_STRING query;
      init_dynamic_string(&query, "LOCK TABLES ", 256, 1024);
      while ((table = getTableName(1)))
      {
	dynstr_append(&query, table);
	dynstr_append(&query, " READ,");
      }
      if (mysql_real_query(sock, query.str, query.length-1))
	DBerror(sock);	/* We shall countinue here, if --force was given */
    }
    while ((table = getTableName(0)))
    {
      numRows = getTableStructure(table);
      if (!dFlag && numRows > 0)
	dumpTable(numRows,table);
    }
  }
  dbDisconnect(current_host);
  puts("");
  my_free(password,MYF(MY_ALLOW_ZERO_PTR));
  my_end(0);
  exit(first_error);
}
