/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Read configuration file(s), and manage server/configuration
 * structures.
 * $Id: dirtree.c,v 1.10 1997/11/18 01:23:16 flood Exp $
 */

/* History:
 * 5/1/97 0.99.0pl2
 *  Used to be named "config.c", renamed to dirtree.c (directive
 *  tree) so as not to conflict with GNU autoconf's top-level config.h.
 */

#include "conf.h"

#include <sys/stat.h>
#include <stdarg.h>

#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif

xaset_t *servers = NULL;
server_rec *main_server = NULL;
int tcpBackLog = TUNABLE_DEFAULT_BACKLOG;
int SocketBindTight = FALSE;
char ServerType = SERVER_STANDALONE;
int TimeoutLogin = TUNABLE_TIMEOUTLOGIN;
int TimeoutIdle = TUNABLE_TIMEOUTIDLE;
int TimeoutNoXfer = TUNABLE_TIMEOUTNOXFER;

/* Used by find_config_* */
xaset_t *find_config_top = NULL;

static void _mergedown(xaset_t*,int);

/* Used only while reading configuration files */

struct {
  pool *tpool;
  array_header *sstack,*cstack;
  server_rec **curserver;
  config_rec **curconfig;
} conf;

char *get_word(char **cp)
{
  char *ret,*dst;
  char quote_mode = 0;

  if(!cp || !*cp || !**cp)
    return NULL;

  while(**cp && isspace(**cp)) (*cp)++;

  if(!**cp)
    return NULL;

  ret = dst = *cp;
  
  if(**cp == '\"') {
    quote_mode++;
    (*cp)++;
  }

  while(**cp && (quote_mode ? (**cp != '\"') : !isspace(**cp))) {
    if(**cp == '\\' && quote_mode) {
      /* escaped char */
      if(*((*cp)+1))
        *dst = *(++(*cp));
    }

    *dst++ = **cp;
    ++(*cp);
  }

  if(**cp) (*cp)++;
  *dst = '\0';

  return ret;
}

cmd_rec *get_config_cmd(pool *ppool, FILE *fp)
{
  char buf[256],*cp,*wrd;
  cmd_rec *newcmd;
  pool *newpool;
  array_header *tarr;
  int i;

  while(fgets(buf,sizeof(buf)-1,fp)) {
    i = strlen(buf);
    if(i && buf[i-1] == '\n')
      buf[i-1] = '\0';

    for(cp = buf; *cp && isspace(*cp); cp++) ;

    if(*cp == '#' || !*cp)		/* Comment or blank line */
      continue;

    /* Build a new pool for the command structure and array */
    newpool = make_sub_pool(ppool);
    newcmd = (cmd_rec*)pcalloc(newpool,sizeof(cmd_rec));
    newcmd->pool = newpool;
    tarr = make_array(newpool,4,sizeof(char**));

    /* Add each word to the array */
    while((wrd = get_word(&cp)) != NULL) {
      *((char**)push_array(tarr)) = pstrdup(newpool,wrd);
      newcmd->argc++;
    }

    *((char**)push_array(tarr)) = NULL;
    
    /* The array header's job is done, we can forget about it and
     * it will get purged when the command's pool is cleared
     */

    newcmd->argv = (char**)tarr->elts;

    /* Perform a fixup on configuration directives so that:
     * -argv[0]--  -argv[1]-- ----argv[2]-----
     * <Option     /etc/adir  /etc/anotherdir>
     *   .. becomes ..
     * -argv[0]--  -argv[1]-  ----argv[2]----
     * <Option>    /etc/adir  /etc/anotherdir
     */

    if(newcmd->argc && *(newcmd->argv[0]) == '<') {
      char *cp = newcmd->argv[newcmd->argc-1];

      if(*(cp + strlen(cp)-1) == '>' && newcmd->argc > 1) {
        if(!strcmp(cp,">")) {
          newcmd->argv[newcmd->argc-1] = NULL;
          newcmd->argc--;
        } else
          *(cp + strlen(cp)-1) = '\0';

        cp = newcmd->argv[0];
        if(*(cp + strlen(cp)-1) != '>')
          newcmd->argv[0] = pstrcat(newcmd->pool,cp,">",NULL);
      }
    }

        
    return newcmd;
  }

  return NULL;
}

void init_dyn_stacks(pool *p,config_rec *top)
{
  conf.sstack = make_array(p,1,sizeof(server_rec*));
  conf.curserver = (server_rec**)push_array(conf.sstack);
  *conf.curserver = main_server;
  conf.cstack = make_array(p,3,sizeof(config_rec*));
  conf.curconfig = (config_rec**)push_array(conf.cstack);
  *conf.curconfig = NULL;
  conf.curconfig = (config_rec**)push_array(conf.cstack);
  *conf.curconfig = top;
}

void init_conf_stacks()
{
  pool *pool = make_sub_pool(permanent_pool);

  conf.tpool = pool;
  conf.sstack = make_array(pool,1,sizeof(server_rec*));
  conf.curserver = (server_rec**)push_array(conf.sstack);
  *conf.curserver = main_server;
  conf.cstack = make_array(pool,10,sizeof(config_rec*));
  conf.curconfig = (config_rec**)push_array(conf.cstack);
  *conf.curconfig = NULL;
}

void free_dyn_stacks()
{
  bzero(&conf,sizeof(conf));
}

void free_conf_stacks()
{
  destroy_pool(conf.tpool);
  bzero(&conf,sizeof(conf));
}

/* Used by modules to start/end configuration sections */

server_rec *start_new_server(const char *addr)
{
  server_rec *s;
  pool *p;

  p = make_sub_pool(permanent_pool);

  s = (server_rec*)pcalloc(p,sizeof(server_rec));
  s->pool = p;
  
  /* Have to make sure it ends up on the end of the chain,
   * otherwise main_server becomes useless.
   */

  xaset_insert_end(servers,(xasetmember_t*)s);
  s->set = servers;
  if(addr)
    s->ServerAddress = pstrdup(s->pool,addr);

  conf.curserver = (server_rec**)push_array(conf.sstack);
  *conf.curserver = s;
  return s;
}

server_rec *end_new_server()
{
  if(!*conf.curserver)
    return NULL;

  if(conf.curserver == (server_rec**)conf.sstack->elts)
    return NULL; /* Disallow underflows */

  
  conf.curserver--;
  conf.sstack->nelts--;


  return *conf.curserver;
}

/* Starts a sub-configuration */
  
config_rec *start_sub_config(const char *name)
{
  config_rec *c,*parent = *conf.curconfig;
  pool *p;
  xaset_t **set;

  if(parent) {
    p = make_sub_pool(parent->pool);
    set = &parent->subset;
  } else {
    p = make_sub_pool((*conf.curserver)->pool);
    set = &(*conf.curserver)->conf;
  }

  c = (config_rec*)pcalloc(p,sizeof(config_rec));

  if(!*set)
    *set = xaset_create(p,NULL);

  xaset_insert(*set,(xasetmember_t*)c);
  
  c->pool = p;
  c->set = *set;
  c->parent = parent;
  if(name)
    c->name = pstrdup(p,name);

  if(parent && (parent->config_type == CONF_DYNDIR))
    c->flags |= CF_DYNAMIC;

  /* Now insert another level onto the stack */
  if(!*conf.curconfig)
    *conf.curconfig = c;
  else {
    conf.curconfig = (config_rec**)push_array(conf.cstack);
    *conf.curconfig = c;
  }

  return c;
}

/* Pop one level off the stack */
config_rec *end_sub_config()
{
  if(conf.curconfig == (config_rec**)conf.cstack->elts) {
    if(*conf.curconfig)
      *conf.curconfig = NULL;
    return NULL;
  }

  conf.curconfig--;
  conf.cstack->nelts--;

  return *conf.curconfig;
}

/* Adds a config_rec to the specified set */
config_rec *add_config_set(xaset_t **set,const char *name)
{
  pool *p;
  config_rec *c,*parent = NULL;

  if(!*set) {
    p = make_sub_pool(permanent_pool);
    *set = xaset_create(p,NULL);
  } else {
    if((*set)->xas_list)
      parent = ((config_rec*)((*set)->xas_list))->parent;
    p = (*set)->mempool;
  }

  c = (config_rec*)pcalloc(p,sizeof(config_rec));
  
  c->pool = p;
  c->set = *set;
  c->parent = parent;
  if(name)
    c->name = pstrdup(p,name);
  xaset_insert_end(*set,(xasetmember_t*)c);
  return c;
}

/* Adds a config_rec on the current "level" */
config_rec *add_config(const char *name)
{
  server_rec *s = *conf.curserver;
  config_rec *parent,*c = *conf.curconfig;
  pool *p;
  xaset_t **set;

  if(c) {
    parent = c;
    p = c->pool;
    set = &c->subset;
  } else {
    parent = NULL;
    
    if(!s->conf || !s->conf->xas_list)
      p = make_sub_pool(s->pool);
    else
      p = ((config_rec*)s->conf->xas_list)->pool;
    set = &s->conf;
  }

  if(!*set)
    *set = xaset_create(p,NULL);

  c = add_config_set(set,name);
  c->parent = parent;

  return c;
}

/* Per-directory configuration */

static int _strmatch(register char *s1, register char *s2)
{
  register int len = 0;

  while(*s1 && *s2 && *s1++ == *s2++)
    len++;

  return len;
}

static config_rec *_recur_match_path(pool *p,xaset_t *s, char *path)
{
  config_rec *c,*res;
  char *tmp;

  if(!s)
    return NULL;

  for(c = (config_rec*)s->xas_list; c; c=c->next)
    if(c->config_type == CONF_DIR) {
      if(!(tmp = dir_realpath(p,c->name)))
        tmp = c->name;

      if(c->argv[1]) {
        if(*(char*)(c->argv[1]) == '~')
          c->argv[1] = dir_canonical_path(c->pool,(char*)c->argv[1]);
        tmp = pdircat(p,(char*)c->argv[1],tmp,NULL);
      }

      if(!strcmp(tmp,path))
        return c;			/* Exact match */

      if(!strstr(tmp,"/*")) {
        if(*tmp && *(tmp+(strlen(tmp)-1)) == '/') {
          *(tmp+(strlen(tmp)-1)) = '\0';
          if(!strcmp(tmp,path))
            return c;
        }
        tmp = pstrcat(p,tmp,"/*",NULL);
      }

      if(fnmatch(tmp,path,FNM_PATHNAME) == 0) {
        if(c->subset) {
          res = _recur_match_path(p,c->subset,path);
          if(res)
            return res;
        }
        return c;
      }
    }

  return NULL;
}

config_rec *dir_match_path(pool *p, char *path)
{
  char *tmp,*apath;
  config_rec *res = NULL;

  tmp = pstrdup(p,path);
  if(*(tmp+strlen(tmp)-1) == '*')
    *(tmp+strlen(tmp)-1) = '\0';
  if(*(tmp+strlen(tmp)-1) == '/')
    *(tmp+strlen(tmp)-1) = '\0';

  if(session.anon_config) {
    res = _recur_match_path(p,session.anon_config->subset,tmp);
    if(!res) {
      apath = dir_canonical_path(p,session.anon_config->name);
      if(apath && !strncmp(apath,tmp,strlen(apath)))
        return NULL;
    }
  }

  if(!res)
    res = _recur_match_path(p,main_server->conf,tmp);

/*
  if(!res)
    res = ((session.anon_config) ? session.anon_config : (config_rec*)main_server->conf->xas_list);
*/

  return res;
}

int dir_get_param(pool *pp,char *path,char *param)
{
  char *fullpath,*tmp;
  pool *p;
  config_rec *c;

  p = make_sub_pool(pp);

  if(*path != '/')
    fullpath = pstrcat(p,session.cwd,"/",path,NULL);
  else
    fullpath = path;

  if((tmp = dir_realpath(p,fullpath)) != NULL)
    fullpath = tmp;

  if(session.anon_root)
    fullpath = pdircat(p,session.anon_root,fullpath,NULL);

  c = dir_match_path(p,fullpath);

  destroy_pool(p);

  if(c)
    return get_param_int(c->subset,param,FALSE);
  return -1;
}

static int _dir_check_op(pool *p,xaset_t *c,int op,
                         int uid,int gid,int mode)
{
  int i,res = 1,pcheck = 0;
  int *gidp = NULL,u,g;

  if(!c)
    return 1;				/* Default is to allow */
  if(uid == session.uid)
    pcheck |= (mode & S_IRWXU);

  if(session.gid == gid)
    pcheck |= (mode & S_IRWXG);
  else
    for(i = session.gids->nelts, gidp = (int*)session.gids->elts; i;
        i--, gidp++)
          if(*gidp == gid) { 
            pcheck |= (mode & S_IRWXG);
            break;
          }

  if(uid != session.uid && gid != session.gid && (!gidp || *gidp != gid))
    pcheck |= (mode & S_IRWXO);

  switch(op) {
  case OP_HIDE:
    u = get_param_int(c,"HideUser",FALSE);
    g = get_param_int(c,"HideGroup",FALSE);
    if((u != -1 && u == uid && u != session.uid) || 
       (g != -1 && g == gid && g != session.gid)) {
      res = 0;
      break;
    }

    if(get_param_int(c,"HideNoAccess",FALSE) == 1) {
      if(S_ISDIR(mode))
        res = pcheck &= (S_IXUSR|S_IXGRP|S_IXOTH);
      else
        res = pcheck &= (S_IRUSR|S_IRGRP|S_IROTH);
    }
    break;
  case OP_COMMAND:
    if(get_param_int(c,"AllowAll",FALSE) == 1)
      /* nop */;  
    else if(get_param_int(c,"DenyAll",FALSE) == 1)
      res = 0;
    break;
  }

  return res;
}

int dir_check_op_mode(pool *p,char *path,int op,
                       int uid,int gid,int mode)
{
  char *fullpath;
  xaset_t *c;
  config_rec *sc;
  int res;

  if(*path != '/')
    fullpath = pdircat(p,session.cwd,path,NULL);
  else
    fullpath = path;

  if(session.anon_root)
    fullpath = pdircat(p,session.anon_root,fullpath,NULL);
  
  c = CURRENT_CONF;
  sc = _recur_match_path(p,c,fullpath);

  if(sc)
    res = _dir_check_op(p,sc->subset,op,uid,gid,mode);
  else
    res = _dir_check_op(p,c,op,uid,gid,mode);

  return res;  
}

int dir_check_op(pool *pp,char *path,int op)
{
  struct stat sbuf;

  if(stat(path,&sbuf) == -1)
    return 1;

  return dir_check_op_mode(pp,path,op,sbuf.st_uid,sbuf.st_gid,sbuf.st_mode);
}

static int _check_ip_access(xaset_t *conf, char *name)
{
  char buf[1024];
  char *mask;
  char **argv;
  int argc; 
  config_rec *c = find_config(conf,CONF_PARAM,name,FALSE);

  while(c) {

    for(argc = c->argc, argv = (char**)c->argv; argc; argc--, argv++) {
      mask = buf;

      if(!strcasecmp(*argv,"ALL"))
        return TRUE;
      if(!strcasecmp(*argv,"NONE"))
        return FALSE;

      if(**argv == '.') {
        *mask++ = '*';
        strncpy(mask,*argv,sizeof(buf)-2);
      } else if(*(*argv+strlen(*argv)-1) == '.') {
        strncpy(mask,*argv,sizeof(buf)-2);
        buf[1023] = '\0';
        strcpy(&buf[strlen(buf)-1],"*");
      } else
        strncpy(mask,*argv,sizeof(buf)-1);

      buf[1023] = '\0';

      if(fnmatch(buf,session.c->remote_name,FNM_NOESCAPE) == 0 ||
         fnmatch(buf,inet_ntoa(*session.c->remote_ipaddr),FNM_NOESCAPE) == 0)
           return TRUE;
    }

    c = find_config_next(c->next,CONF_PARAM,name,FALSE);

  }

  return FALSE;
}

/* _check_limit returns 1 if allowed, 0 if implicitly allowed,
 * and -1 if implicitly denied and -2 if explicitly denied.
 */
   
static int _check_limit(config_rec *c)
{
  int order;

  /* First check for AllowAll/DenyAll */
  if(get_param_int(c->subset,"AllowAll",FALSE) == 1)
    return 1;
  if(get_param_int(c->subset,"DenyAll",FALSE) == 1)
    return -2;

  /* Next check IP deny/allow */
  if((order = get_param_int(c->subset,"Order",FALSE)) == -1)
    order = ORDER_ALLOWDENY;

  if(order == ORDER_DENYALLOW) {
    if(_check_ip_access(c->subset,"Deny"))
      return -2;
    if(_check_ip_access(c->subset,"Allow"))
      return 1;
  } else {
    if(_check_ip_access(c->subset,"Allow"))
      return 1;
    if(_check_ip_access(c->subset,"Deny"))
      return -2;
  }

  return (order == ORDER_DENYALLOW ? -1 : 0);
}

/* Note: if and == 1, the logic is short circuited so that the first
 * failure results in a FALSE return from the entire function, if and
 * == 0, an ORing operation is assumed and the function will return
 * TRUE if any <limit LOGIN> allows access.
 */

int login_check_limits(xaset_t *conf, int recurse, 
                       int and, int *found)
{
  int res = and;
  int rfound;
  config_rec *c;
  int argc;
  char **argv;

  *found = 0;

  if(!conf || !conf->xas_list)
    return TRUE;			/* default is to allow */

  /* First check top level */
  for(c = (config_rec*)conf->xas_list; c; c=c->next)
    if(c->config_type == CONF_LIMIT) {
      for(argc = c->argc, argv = (char**)c->argv; argc; argc--, argv++)
        if(!strcasecmp("LOGIN",*argv))
          break;

      if(argc) {
        if(and) {
          switch(_check_limit(c)) {
          case 1: res = (res && TRUE); (*found)++; break;
	  case -1:
          case -2: res = (res && FALSE); (*found)++; break;
          }
          if(!res)
            break;
        } else
          switch(_check_limit(c)) {
          case 1: res = TRUE;
	  case -1:
          case -2: (*found)++; break;
          }
      }
    }

  if( ((res && and) || (!res && !and)) && recurse ) {
    for(c = (config_rec*)conf->xas_list; c; c=c->next)
      if(c->config_type == CONF_ANON && c->subset && c->subset->xas_list) {
       if(and) {
         res = (res && login_check_limits(c->subset,recurse,
                                          and,&rfound));
         (*found) += rfound;
         if(!res)
           break;
       } else {
         int rres;

         rres = login_check_limits(c->subset,
                                  recurse,and,&rfound);
         if(rfound)
           res = (res || rres);
         (*found) += rfound;
         if(res)
           break;
       }
     }
  }

  if(!*found && !and)
    return TRUE;			/* Default is to allow */
  return res;
}

int dir_check_limits(config_rec *c, char *cmd, int hidden)
{
  /* Check limit directives */
  int res = 1;
  config_rec *lc;
  int i;

  errno = 0;

  while(c && (res == 1)) {
    if(c->subset)
      for(lc = (config_rec*)c->subset; lc && (res == 1); lc=lc->next)
        if(lc->config_type == CONF_LIMIT) {
          for(i = 0; i < lc->argc; i++)
            if(!strcmp(cmd,(char*)(lc->argv[i])))
              break;

          if(i == lc->argc)
            continue;
          /* Found a limit directive associated with the current
           * command
           */
          if(hidden && get_param_int(lc->subset,"IgnoreHidden",FALSE) == 1)
            { res = 0; errno = ENOENT; break; }
          switch(_check_limit(lc)) {
          case 1: res++; break;
          case -1:
          case -2: res = 0; break;
          default: continue;
          }

          break;
        }


    c = c->parent;
  }

  if(!res && !errno)
    errno = EACCES;

  return res;
}

void build_dyn_config(pool *p,char *_path)
{
  char *fullpath,*tmp,*path,*dynpath,*cp;
  struct stat sbuf;
  config_rec *d;
  FILE *fp;
  cmd_rec *cmd;
  xaset_t **set = NULL;
  int isfile,removed = 0;

  /* Switch through each directory, from "deepest" up looking for
   * new or updated .ftpaccess files
   */

  if(!_path)
    return;

  path = pstrdup(p,_path);

  while(path) {
    dynpath = pdircat(p,path,".ftpaccess",NULL);

    if(*path != '/')
      fullpath = pdircat(p,session.cwd,path,NULL);

    if((tmp = dir_realpath(p,path)))
      fullpath = tmp;

    if(session.anon_root)
      fullpath = pdircat(p,session.anon_root,path,NULL);
    else
      fullpath = path;

    isfile = stat(dynpath,&sbuf);

    d = dir_match_path(p,fullpath);

    if(!d && isfile != -1) {
      set = (session.anon_config ? &session.anon_config->subset :
             &main_server->conf);
      d = add_config_set(set,fullpath);
      d->config_type = CONF_DIR;
      d->argc = 1;
      d->argv = pcalloc(d->pool,2*sizeof(void*));
    } else if(d) {
      config_rec *newd,*dnext;

      if(isfile != -1 && strcmp(d->name,fullpath) != 0) {
        set = &d->subset;
        newd = add_config_set(set,fullpath);
        newd->config_type = CONF_DIR;
        newd->argc = 1;
        newd->argv = pcalloc(newd->pool,2*sizeof(void*));
	newd->parent = d;
        d = newd;
      } else if(d->subset && d->subset->xas_list &&
                strcmp(d->name,fullpath) == 0 && 
                (isfile == -1 || sbuf.st_ctime > (time_t)d->argv[0])) {
        set = (d->parent ? &d->parent->subset :
               &main_server->conf);

        /* remove all old dynamic entries */
        for(newd = (config_rec*)d->subset->xas_list; newd; newd=dnext) {
          dnext = newd->next;

          if(newd->flags & CF_DYNAMIC) {
            xaset_remove(d->subset,(xasetmember_t*)newd);
            removed++;
          }
        }

        if(!d->subset->xas_list) {
          destroy_pool(d->subset->mempool);
          d->subset = NULL;
          d->argv[0] = NULL;

	  /* If the file has been removed and no entries exist in this
           * dynamic entry, remove it completely
           */

          if(isfile == -1)
            xaset_remove(*set,(xasetmember_t*)d);
        }
      }
    }

    if(isfile != -1 && d && sbuf.st_ctime > (time_t)d->argv[0]) {
      /* File has been modified or not loaded yet */
      d->argv[0] = (void*)sbuf.st_ctime;

      fp = fopen(dynpath,"r");
      if(fp) {
        removed = 0;

        init_dyn_stacks(p,d);
        d->config_type = CONF_DYNDIR;

        while((cmd = get_config_cmd(p,fp)) != NULL) {
          if(cmd->argc) {
            conftable *c;
            char found = 0;
            char *errmsg;

            cmd->server = *conf.curserver;
            cmd->config = *conf.curconfig;
              
            for(c = m_conftable; c->directive; c++)
              if(!strcasecmp(c->directive,cmd->argv[0])) {
                ++found;
                if((errmsg = (char*)call_module(c->m,c->handler,cmd)) != NULL) {
                  log_pri(LOG_WARNING,"warning: %s",errmsg);
                }
		destroy_pool(cmd->tmp_pool);
              }

            if(!found)
              log_pri(LOG_WARNING,"warning: unknown configuration directive '%s'.",
                      cmd->argv[0]);

          }
 
          destroy_pool(cmd->pool);
        }

	log_debug(DEBUG5,"dynamic configuration added/updated for %s.",
                         fullpath);

        d->config_type = CONF_DIR;
        free_dyn_stacks();

        _mergedown(*set,TRUE);
        fclose(fp);
      }
    }

    if(isfile == -1 && removed && d && set) {
      log_debug(DEBUG5,"dynamic configuration removed for %s.",
                       fullpath);
      _mergedown(*set,FALSE);
    }

    cp = rindex(path,'/');
    if(cp && strcmp(path,"/") != 0)
      *cp = '\0';
    else
      path = NULL;
  }
}

/* dir_check() returns 1 if operation is allowed on current path,
 * or 0 if not.
 */

int dir_check(pool *pp, char *cmd, char *group, char *path)
{
  char *fullpath;
  config_rec *c;
  struct stat sbuf;
  pool *p;
  int res = 1,hidden = 0,_umask = 0;

  p = make_sub_pool(pp);

  fullpath = dir_realpath(p,path);

  if(!fullpath)
    fullpath = pdircat(p,session.cwd,path,NULL);
  else 
    path = fullpath;

  if(session.anon_root)
    fullpath = pdircat(p,session.anon_root,fullpath,NULL);

  /* Check and build all appropriate dynamic configuration entries */
  build_dyn_config(p,path);

  session.dir_config = c = dir_match_path(p,fullpath);

  if(!c && session.anon_config)
    c = session.anon_config;

  if((_umask = get_param_int(CURRENT_CONF,"Umask",FALSE)) == -1)
    _umask = 0;

  if((session.fsgid = get_param_int(CURRENT_CONF,"GroupOwner",FALSE)) == -1)
    session.fsgid = 0;			/* Don't try to set gid */

  if(stat(path,&sbuf) != -1) {
    hidden = !_dir_check_op(p,CURRENT_CONF,OP_HIDE,sbuf.st_uid,sbuf.st_gid,sbuf.st_mode);
    res = _dir_check_op(p,CURRENT_CONF,OP_COMMAND,sbuf.st_uid,sbuf.st_gid,sbuf.st_mode);
  }

  if(res) {
    res = dir_check_limits(c,cmd,hidden);

    /* If specifically allowed, res will be > 1 and we don't want to
     * check the command group limit
     */

    if(res == 1 && group)
      res = dir_check_limits(c,group,hidden);

    /* if still == 1, no explicit allow so check lowest priority "ALL" group */
    if(res == 1)
      res = dir_check_limits(c,"ALL",hidden);
  }

  if(res && _umask)
    umask(_umask);

  destroy_pool(p);
  return res;
}

/*
 * Move all the members (i.e. a "branch") of one config set to
 * a different parent.
 */

static void _reparent_all(config_rec *newparent,xaset_t *set)
{
  config_rec *c,*cnext;

  if(!newparent->subset)
    newparent->subset = xaset_create(newparent->pool,NULL);

  for(c = (config_rec*)set->xas_list; c; c = cnext) {
    cnext = c->next;
    xaset_remove(set,(xasetmember_t*)c);
    xaset_insert(newparent->subset,(xasetmember_t*)c);
    c->set = newparent->subset;
    c->parent = newparent;
  }
}

/* Recursively find the most appropriate place to move a CONF_DIR
 * directive to.
 */

static config_rec *_find_best_dir(xaset_t *set,char *path,int *matchlen)
{
  config_rec *c,*res = NULL,*rres;
  int len,imatchlen,tmatchlen;

  *matchlen = 0;

  if(!set || !set->xas_list)
    return NULL;

  for(c = (config_rec*)set->xas_list; c; c=c->next) {
    if(c->config_type == CONF_DIR) {
      if(!strcmp(c->name,path))
        continue;				/* Don't examine the current */
      len = strlen(c->name);
      while(len > 0 && (*(c->name+len-1) == '*' ||
                        *(c->name+len-1) == '/'))
        len--;

      if(!strncmp(c->name,path,len) &&
         len < strlen(path)) {
           rres = _find_best_dir(c->subset,path,&imatchlen);
           tmatchlen = _strmatch(path,c->name);
           if(!rres && tmatchlen > *matchlen) {
             res = c;
             *matchlen = tmatchlen;
           } else if(imatchlen > *matchlen) {
             res = rres;
             *matchlen = imatchlen;
           }     
         }
    }
  }

  return res;
}

/* Reorder all the CONF_DIR configuration sections, so that they are
 * in directory tree order
 */

static void _reorder_dirs(xaset_t *set)
{
  config_rec *c,*cnext,*newparent;
  int tmp;

  if(!set || !set->xas_list)
    return;

  for(c = (config_rec*)set->xas_list; c; c=cnext) {
    cnext = c->next;

    if(c->config_type == CONF_DIR) {
      /* If <Directory *> is used inside <Anonymous>, move all
       * the directives from '*' into the higher level
       */
      if(!strcmp(c->name,"*") && c->parent &&
         c->parent->config_type == CONF_ANON) {
        if(c->subset)
          _reparent_all(c->parent,c->subset);
        xaset_remove(c->parent->subset,(xasetmember_t*)c);
      } else {
        newparent = _find_best_dir(set,c->name,&tmp);
        if(newparent) {
          if(!newparent->subset)
            newparent->subset = xaset_create(newparent->pool,NULL);

          xaset_remove(c->set,(xasetmember_t*)c);
          xaset_insert(newparent->subset,(xasetmember_t*)c);
          c->set = newparent->subset;
          c->parent = newparent;
        }
      }
    }
  }

  /* Top level is now sorted, now we recursively sort all the sublevels
   */
  for(c = (config_rec*)set->xas_list; c; c=c->next)
    if(c->config_type == CONF_DIR || c->config_type == CONF_ANON)
      _reorder_dirs(c->subset);
}

void debug_dump_config(xaset_t *s,char *indent)
{
  config_rec *c;

  if(!indent)
    indent = "";

  for(c = (config_rec*)s->xas_list; c; c=c->next) {
    log_debug(DEBUG5,"%s%s",indent,c->name);
    if(c->subset)
      debug_dump_config(c->subset,pstrcat(permanent_pool,indent," ",NULL));
  }
}

static void _mergedown(xaset_t *s,int dynamic)
{
  config_rec *c,*dest,*newconf;
  int argc;
  void **argv,**sargv;
  
  if(!s || !s->xas_list)
    return;

  for(c = (config_rec*)s->xas_list; c; c=c->next)
    if(c->flags & CF_MERGEDOWN)
      for(dest = (config_rec*)s->xas_list; dest; dest=dest->next)
        if(dest->config_type == CONF_ANON ||
           dest->config_type == CONF_DIR) {
          /* If an option of the same name/type is found in the
           * next level down, it overrides, so we don't merge.
           */
          if(find_config(dest->subset,dest->config_type,
                         c->name,FALSE))
            continue;

          if(!dest->subset)
            dest->subset = xaset_create(dest->pool,NULL);

          newconf = add_config_set(&dest->subset,c->name);
          newconf->config_type = c->config_type;
          newconf->flags = c->flags | (dynamic ? CF_DYNAMIC : 0);
          newconf->argc = c->argc;
          newconf->argv = palloc(newconf->pool,(c->argc+1)*sizeof(void*));
          argv = newconf->argv; sargv = c->argv;
          argc = newconf->argc;
          while(argc--)
            *argv++ = *sargv++;
          *argv++ = NULL;
        }
          
  /* Top level merged, recursively merge lower levels */
  for(c = (config_rec*)s->xas_list; c; c=c->next)
    if(c->subset && (c->config_type == CONF_ANON ||
                     c->config_type == CONF_DIR))
      _mergedown(c->subset,dynamic);
}
    
void fixup_dirs(server_rec *s)
{
  if(!s || !s->conf)
    return;

  _reorder_dirs(s->conf);

  /* Merge mergeable configuration items down
   */

  _mergedown(s->conf,FALSE);

/*
  for(c = (config_rec*)s->conf->xas_list; c; c=c->next)
    if(c->config_type == CONF_ANON)
      _reorder_dirs(c->subset);
*/
  log_debug(DEBUG5,"");
  log_debug(DEBUG5,"Config for %s:",s->ServerName);
  debug_dump_config(s->conf,NULL);
}

config_rec *find_config_next(config_rec *c, int type, const char
                             *name, int recurse)
{
  config_rec *top = c;

  /* We do two searches (if recursing) so that we find the "highest"
   * level first.
   */

  if(!c)
    return NULL;

restart:

  for(; c; c=c->next)
    if((type == -1 || type == c->config_type) &&
       (!name || !strcmp(name,c->name)))
      return c;

  if(recurse) {
    do {
      config_rec *res = NULL;

      for(c = top; c; c=c->next) {
        if(c->subset && c->subset->xas_list)
          res = find_config_next((config_rec*)c->subset->xas_list,
                                 type,name,recurse+1);
          if(res)
            break;
        }

      if(!res && top->parent && recurse == 1 &&
         top->parent->next &&
         top->parent->set != find_config_top) {
        top = top->parent->next; c = top;
        goto restart;
      }
  
      return res;
    } while(1);
  }
  return NULL;
}
    
void find_config_set_top(config_rec *c)
{
  if(c && c->parent)
    find_config_top = c->parent->set;
  else
    find_config_top = NULL;
}


config_rec *find_config(xaset_t *set, int type, const char *name, int recurse)
{
  if(!set || !set->xas_list)
    return NULL;

  find_config_set_top((config_rec*)set->xas_list);

  return find_config_next((config_rec*)set->xas_list,type,name,recurse);
}

/* These next two functions return the first argument in a
 * CONF_PARAM configuration entry.  If more than one or all
 * parameters are needed, the caller will need to use find_config,
 * and iterate through the argv themselves.
 * _int returns -1 if the config name is not found, _ptr returns
 * NULL.
 */

long get_param_int(xaset_t *set,const char *name,int recurse)
{
  config_rec *c;

  if(!set)
    return -1;

  c = find_config(set,CONF_PARAM,name,recurse);

  if(c && c->argc)
    return (int)c->argv[0];

  return -1;  /* Parameters aren't allowed to contain neg. integers anyway */
}

void *get_param_ptr(xaset_t *set,const char *name,int recurse)
{
  config_rec *c;

  if(!set)
    return NULL;

  c = find_config(set,CONF_PARAM,name,recurse);

  if(c && c->argc)
    return c->argv[0];

  return NULL;
}

int remove_config(xaset_t *set, const char *name,int recurse)
{
  server_rec *s = (conf.curserver ? *conf.curserver : main_server);
  config_rec *c;
  int found = 0;
  xaset_t *fset;

  while((c = find_config(set,-1,name,recurse)) != NULL) {
    found++;

    fset = c->set;
    xaset_remove(fset,(xasetmember_t*)c);

    if(!fset->xas_list)
      if(c->parent && c->parent->subset == fset) {
        c->parent->subset = NULL;
        destroy_pool(fset->mempool);
      } else if(s->conf == fset) {
        s->conf = NULL;
        destroy_pool(fset->mempool);
      }
  }

  return found;
}
        
config_rec *add_config_param_set(xaset_t **set,const char *name,int num,...)
{
  config_rec *c = add_config_set(set,name);
  void **argv;
  va_list ap;

  if(c) {
    c->config_type = CONF_PARAM;
    c->argc = num;
    c->argv = pcalloc(c->pool,(num+1)*sizeof(void*));

    argv = c->argv;
    va_start(ap,num);

    while(num-- > 0)
      *argv++ = va_arg(ap,void*);


    va_end(ap);
  }

  return c;
}

config_rec *add_config_param_str(const char *name, int num, ...)
{
  config_rec *c = add_config(name);
  char *arg;
  void **argv;
  va_list ap;

  if(c) {
    c->config_type = CONF_PARAM;
    c->argc = num;
    c->argv = pcalloc(c->pool,(num+1) * sizeof(char*));

    argv = c->argv;
    va_start(ap,num);

    while(num-- > 0) {
      arg = va_arg(ap,char*);
      if(arg)
        *argv++ = pstrdup(c->pool,arg);
      else
        *argv++ = NULL;
    }

    va_end(ap);
  }

  return c;
}

config_rec *add_config_param(const char *name,int num,...)
{
  config_rec *c = add_config(name);
  void **argv;
  va_list ap;

  if(c) {
    c->config_type = CONF_PARAM;
    c->argc = num;
    c->argv = pcalloc(c->pool,(num+1) * sizeof(void*));
    
    argv = c->argv;
    va_start(ap,num);

    while(num-- > 0)
      *argv++ = va_arg(ap,void*);

    va_end(ap);
  }

  return c;
}

int parse_config_file(const char *fname)
{
  FILE *fp;
  cmd_rec *cmd;
  pool *tmp_pool = make_sub_pool(permanent_pool);
  char *errmsg;
 
  fp = pfopen(tmp_pool,fname,"r");

  if(!fp) { destroy_pool(tmp_pool); return -1; }
  
  while((cmd = get_config_cmd(tmp_pool,fp)) != NULL) {
    if(cmd->argc) {
      conftable *c;
      char found = 0;

      cmd->server = *conf.curserver;
      cmd->config = *conf.curconfig;

      for(c = m_conftable; c->directive; c++)
        if(!strcasecmp(c->directive,cmd->argv[0])) {
          ++found;
          if((errmsg = (char*)call_module(c->m,c->handler,cmd)) != NULL) {
            log_pri(LOG_ERR,"Fatal: %s",errmsg);
            exit(1);
          }

          destroy_pool(cmd->tmp_pool);
        }

       if(!found) {
         log_pri(LOG_ERR,"Fatal: unknown configuration directive '%s'.",
                 cmd->argv[0]);
         exit(1);
       }
    }

    destroy_pool(cmd->pool);
  }

  pfclose(tmp_pool,fp);
  destroy_pool(tmp_pool);
  return 0;
}

/* Go through each server configuration and complain if important
 * information is missing (post reading configuration files).
 * otherwise fill in defaults where applicable
 */

void fixup_servers()
{
  server_rec *s;

  for(s = (server_rec*)servers->xas_list; s; s=s->next) {
    if(!s->ServerPort)
      s->ServerPort = inet_getservport(s->pool,"ftp","tcp");
    if(!s->ServerAddress)
      s->ServerAddress = inet_gethostname(s->pool);
    if(!s->ServerAdmin)
      s->ServerAdmin = pstrcat(s->pool,"root@",s->ServerAddress,NULL);

    if(!s->tcp_rwin)
      s->tcp_rwin = TUNABLE_DEFAULT_RWIN;
    if(!s->tcp_swin)
      s->tcp_swin = TUNABLE_DEFAULT_SWIN;

    s->ipaddr = inet_getaddr(s->pool,s->ServerAddress);

    if(!s->ipaddr) {
      log_pri(LOG_ERR,"Fatal: unable to determine IP address of `%s'.",
                      s->ServerAddress);
      exit(1);
    }

    fixup_dirs(s);
  }

  clear_inet_pool();
}

void init_config()
{
  pool *pool = make_sub_pool(permanent_pool);

  servers = xaset_create(pool,NULL);

  pool = make_sub_pool(permanent_pool);
  main_server = (server_rec*)pcalloc(pool,sizeof(server_rec));
  xaset_insert(servers,(xasetmember_t*)main_server);

  main_server->pool = pool;
  main_server->set = servers;
}
