;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  memory.asm
;* Purpose: Memory management functions for DOS simulator
;* Entries: dos48, dos49, dos4A, freeall, checkmcb
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Include assembler macros:
;
include ..\..\headers\asm\macros.inc
include ..\..\headers\asm\layout.inc
include ..\..\headers\asm\memory.inc
include .\dospriv.inc


;
;**************************************************************************
;
; Data segment
;
data_start

		public	first_mcb	; exported to dosinit.asm

		extrn	curr_psp:word	; current PSP segment

first_mcb	dw	0		; pointer to first MCB

data_end


;
;**************************************************************************
;
; Start code segment.
;
text_start

	public	dos48, dos49, dos4A		; define entry poimts
	public	freeall
	public	checkmcb


;
;**************************************************************************
;
; Allocate memory block. This routine is very easy in that it allocates
; the first possible block it can find. This is not very economical, but
; sufficient for the DOS simulator.
; Note that this routine might also get called from outside a DOS interrupt,
; so it should not use any reference to the DOS stack structure!
; Input:  BX  -  number of paragraphs
; Output: AX  -  segment address or error code, if carry flag set
;         BX  -  maximum available number of paragraphs
;         Carry flag  -  set if error
; Registers changed: AX, BX
;
dos48		proc	near

	push	es
	push	dx
	xor	dx,dx			; size of largest available block
	mov	es,first_mcb		; start looking at first MCB
	call	checkmcb		; check the first MCB
	jc	short dos489

dos481: mov	ax,es:[mcb_owner]
	or	ax,ax			; is block available?
	jz	short dos483
dos484:	call	nextmcb			; no, jump to next MCB
	jnc	short dos481
	or	ax,ax
	jnz	short dos482		; serious error
	mov	bx,dx			; end of list reached, return largest
	mov	ax,ERR_NOMEM		; available memory block size
dos482:	stc				; return with error
	jmp	short dos489

dos483:	mov	ax,es:[mcb_size]
	cmp	bx,0FFFFh		; 0FFFFh size means: determine amount of
	je	short dos487		; free memory. never get that much.
	cmp	ax,bx			; check size of free memory block
	jae	short dos485
dos487:	cmp	ax,dx			; not sufficient, save in DX if greater
	jbe	short dos484		; than any preceding block
	mov	dx,ax
	jmp	short dos484		; continue with next memory block

dos485:	mov	dx,ax			; block is large enough, save it's old
	mov	es:[mcb_size],bx	; size and then set the new one
	mov	ax,curr_psp
	or	ax,ax			; is a process running already
	jnz	short dos486
	mov	ax,0FFFFh		; dummy PSP address
dos486:	mov	es:[mcb_owner],ax	; save owner of memory block
	sub	dx,bx			; compute size of remaining free block
	jbe	short dos488		; nothing left

	mov	ax,es
	add	ax,bx			; compute address of new MCB
	inc	ax
	dec	dx			; adjust size of remaining memory block
	mov	bl,MCB_MEM_ID
	xchg	bl,es:[mcb_id]		; remember old ID and set new ID
	push	ds
	mov	ds,ax
	mov	ds:[mcb_id],bl
	mov	ds:[mcb_size],dx	; setup new free memory block
	mov	ds:[mcb_owner],0
	pop	ds
	mov	bx,es:[mcb_size]	; restore BX register
	call	cleanup			; cleanup free memory blocks

dos488:	mov	ax,es			; return address of new memory block
	inc	ax			; to caller
	clc
dos489:	pop	dx
	pop	es
	ret

dos48		endp


;
;**************************************************************************
;
; Free a memory block.
; Note that this routine might also get called from outside a DOS interrupt,
; so it should not use any reference to the DOS stack structure!
; Input:  ES  -  segment of memory block to free
; Output: AX  -  error code if carry flag set
;         Carry flag  -  set if error
; Registers changed: AX
;
dos49		proc	near

	push	es
	push	dx
	push	ax
	mov	dx,es
	dec	dx			; convert memory block address into
	call	findmcb			; MCB address and find MCB
	jc	short dos497
	mov	es:[mcb_owner],0	; free memory block
	call	cleanup			; integrate the new free memory block
dos497:	pop	dx
	jc	short dos499		; on error don't restor AX register
	mov	ax,dx
dos499:	pop	dx
	pop	es
	ret

dos49		endp


;
;**************************************************************************
;
; Resize memory block.
; Note that this routine might also get called from outside a DOS interrupt,
; so it should not use any reference to the DOS stack structure!
; Input:  BX  -  new size of memory block in paragraphs
;         ES  -  segment of memory block
; Output: BX  -  maximum available memory space
;         AX  -  error code if carry flag set
;         Carry flag  -  set if error
; Registers changed: AX, BX
;
dos4A		proc	near

	push	es
	push	dx
	push	ax
	mov	dx,es
	dec	dx			; convert memory block address into
	call	findmcb			; MCB address and find MCB
	jc	short dos4A7

	mov	dx,es:[mcb_size]	; used at label dos4A4
	cmp	bx,dx
	je	short dos4A8		; nothing changes
	jb	short dos4A4		; make it smaller

; Make the memory block larger. To do this, first check if the next
; block has enough memory space available. Then include it into the
; current block.

	mov	dx,es
	call	nextmcb			; check next MCB
	jnc	short dos4A2
	or	ax,ax
	jnz	short dos4A7
dos4A1:	mov	ax,ERR_NOMEM		; when at the end of the memory,
	jmp	short dos4A7		; there is no more available

dos4A2: cmp	es:[mcb_owner],0	; is the next memory block free?
	jne	short dos4A1
	push	dx
	mov	ax,es:[mcb_size]	; load size of free memory block
	mov	dl,es:[mcb_id]		; load ID of free memory block
	pop	es
	add	ax,es:[mcb_size]	; compute size of both memory blocks
	inc	ax			; combined
	cmp	ax,bx			; is it enough
	jae	short dos4A3
	mov	bx,ax			; return maximum size to caller
	jmp	short dos4A1

dos4A3:	mov	es:[mcb_size],ax	; set size of new combined segment
	mov	es:[mcb_id],dl		; set ID of new segment
	cmp	ax,bx			; if the size is exactly what has
	je	short dos4A8		; been requested, we don't need to split
	mov	dx,ax

; Make the memory block smaller.

dos4A4:	sub	dx,bx			; compute size of remaining memory block
	dec	dx
	jz	short dos4A8		; new memory block is zero size
	mov	es:[mcb_size],bx	; set new size of memory block
	mov	ax,es
	add	ax,bx			; compute address of new MCB
	inc	ax
	mov	bl,MCB_MEM_ID
	xchg	bl,es:[mcb_id]		; remember old ID and set new ID
	push	ds
	mov	ds,ax
	mov	ds:[mcb_id],bl
	mov	ds:[mcb_size],dx	; setup new free memory block
	mov	ds:[mcb_owner],0
	pop	ds
	mov	bx,es:[mcb_size]	; restore BX register
	call	cleanup			; cleanup free memory blocks
	jnc	short dos4A8

; Termination with or without error.

dos4A7:	stc				; return with error
	pop	dx			; don't restore AX
	jmp	short dos4A9

dos4A8:	clc				; no error
	pop	ax
dos4A9:	pop	dx
	pop	es
	ret

dos4A		endp


;
;**************************************************************************
;
; Free all memory blocks of an owner. This routine is called internally
; by the process management code.
; Input:  DX  -  PSP segment of owner
; Output: AX  -  error code if carry flag set
;         CX  -  number of MCBs freed
;         Carry flag -  set if error
; Registers changed: AX, CX
;
freeall		proc	near

	push	es
	xor	cx,cx
	mov	es,first_mcb		; start at first MCB
	call	checkmcb		; check that first MCB
	jc	short free9

free1:	mov	ax,es:[mcb_owner]
	cmp	ax,dx			; found memory block of owner
	jne	short free2
	mov	es:[mcb_owner],0	; free current memory block
	inc	cx
free2:	call	nextmcb			; continue with next MCB
	jnc	short free1
	or	ax,ax			; reached end of MCB list
	jz	short free8
	stc
	jmp	short free9

free8:	call	cleanup			; cleanup free memory blocks
free9:	pop	es
	ret

freeall		endp


;
;**************************************************************************
;
; Find an MCB in the list.
; Input:  DX  -  MCB to search for
; Output: ES  -  segment of MCB
;         AX  -  error code
;         Carry flag  -  set if error
; Registers changed: AX, ES
;
findmcb		proc	near

	mov	es,first_mcb		; start at first MCB
	call	checkmcb		; check that first MCB
	jc	short findm3
findm1:	mov	ax,es
	cmp	ax,dx			; compare MCB addresses
	je	short findm3		; found it
	call	nextmcb			; proceed with next MCB
	jnc	short findm1
	or	ax,ax			; found an error?
	jnz	short findm2
	mov	ax,ERR_INVMEM		; just got to end of list
findm2:	stc
findm3:	ret

findmcb		endp


;
;**************************************************************************
;
; Cleanup MCB list by concatenating adjacent free MCB's.
; Input:  none
; Output: AX  -  error code
;         Carry flag  -  set if error
; Registers changed: AX
;
cleanup		proc	near

	push	bx
	push	dx
	push	es
	mov	es,first_mcb		; start at first MCB
	call	checkmcb		; check that first MCB
	jc	short clean9

clean1:	mov	ax,es:[mcb_owner]
	or	ax,ax			; concatenation possible if current
	jz	short clean2		; MCB is free
clean5:	call	nextmcb			; otherwise check next MCB if it's
	jnc	short clean1		; free
	jmp	short clean4		; handle error

clean2:	mov	dx,es
clean6:	call	nextmcb			; get next MCB
	jnc	short clean3
clean4:	or	ax,ax			; at end of list? this 'or' also
	jz	short clean9		; clears the carry flag
	stc				; return with error
	jmp	short clean9

clean3:	mov	ax,es:[mcb_owner]	; is next MCB free?
	or	ax,ax
	jnz	short clean5		; no, continue with next one
	mov	bl,es:[mcb_id]
	mov	ax,es:[mcb_size]
	inc	ax			; get total size of next block
	mov	es,dx
	add	es:[mcb_size],ax	; add size to preceding MCB
	mov	es:[mcb_id],bl		; copy ID from old to new MCB
	jmp	short clean6		; continue with the next MCB

clean9:	pop	es
	pop	dx
	pop	bx
	ret

cleanup		endp


;
;**************************************************************************
;
; Proceed to next MCB.
; Input:  ES  -  Segment of old MCB
; Output: ES  -  Segment of new MCB
;         AX  -  error code (0 if end of MCB list)
;         Carry Flag  -  set if error
; Registers changed: AX, ES
;
nextmcb		proc	near

	xor	ax,ax
	cmp	es:[mcb_id],MCB_END_ID
	je	short nextm2
	mov	ax,es
	add	ax,es:[mcb_size]	; add size to current MCB pointer
	jc	short nextm1		; should never happen
	inc	ax
	cmp	ax,LOWMEM		; check for upper limit
	jae	short nextm1
	cmp	ax,first_mcb		; check for lower limit
	jb	short nextm1
	mov	es,ax
	xor	ax,ax
	jmp	short checkmcb		; check for correct MCB

nextm1:	mov	ax,ERR_INVMCB		; return with error
nextm2:	stc
	ret

nextmcb		endp


;
;**************************************************************************
;
; Check if MCB is correct.
; Input:  ES  -  segment of MCB
; Output: AX  -  error code (unchanged if no error)
;         Carry flag  -  set if error
; Registers changed: AX
;
checkmcb	proc	near

	cmp	es:[mcb_id],MCB_MEM_ID
	je	short check1
	cmp	es:[mcb_id],MCB_END_ID
	je	short check1
	mov	ax,ERR_INVMCB
	stc				; error
check1:	ret

checkmcb	endp


;
;**************************************************************************
;
text_end

	end

