/*
 **************************************************************************
 *
 * Utility program to combine the bootrom image with the loader
 *
 * Module:  pass2.c
 * Purpose: Combine the bootrom image with the loader image
 * Entries: main
 *
 **************************************************************************
 *
 * Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#ifdef MSDOS
#include <io.h>
#endif
#include "../../headers/memory.h"
#include "../../headers/version.h"
#include "./utility.h"


#define BLKSIZE		512		/* Size of input block buffer */
#define ROMSIG		0xaa55		/* ROM signature */


char *progname;
char *romname = NULL;
char *ldrname = NULL;
char *outname = NULL;
unsigned short chksum = 0;
int infile;
int outfile;



/*
 * Write one block of data to the output file.
 */
int dowrite(unsigned char *buf, int bufsize)
{
  int j, ofs;

  if ((ofs = write(outfile, (char *)buf, bufsize)) < 0) {
	perror(outname);
	exit(1);
  }

  /* Compute the checksum */
  for (j = 0; j < ofs; j++)
	chksum += buf[j];
#ifdef DEBUG
  fprintf(stderr, "w:%d<%s>\n", ofs, outname);
#endif
  return ofs;
}



/*
 * Read one block of data either from the kernel image file or the
 * packet driver binary.
 */
int doread(unsigned char *buf, int bufsize, char *name)
{
  int ofs;

  if ((ofs = read(infile, (char *)buf, bufsize)) < 0) {
	perror(name);
	exit(1);
  }
#ifdef DEBUG
  fprintf(stderr, "r:%d<%s>\n", ofs, name);
#endif
  return ofs;
}



void usage(void)
{
  fprintf(stderr, "usage: %s -i<rom image> -l<loader image> [-o<out file>]\n", progname);
  exit(1);
}



void main(int argc, char **argv)
{
  unsigned long romlength;		/* Size of bootrom kernel image */
  unsigned long writecnt;		/* number of bytes in output file */
  unsigned char inbuf[BLKSIZE];		/* Input file buffer */
  int dorom;
  int i, len;
  char *cp;

  /* Determine our own name */
#ifdef MSDOS
  if ((cp = strrchr(argv[0], '\\')) == NULL)
#else
  if ((cp = strrchr(argv[0], '/')) == NULL)
#endif
	cp = argv[0];
  else
	cp++;
  progname = strdup (cp);

  /* Isolate the different command line options */
  for (i = 1; i < argc; i++) {
	if (*argv[i] != '-' || strlen(argv[i]) < 2)
		usage();
	 switch (argv[i][1]) {
		case 'i':  if (strlen(argv[i]) > 2) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           romname = strdup(cp);
		           break;
		case 'l':  if (strlen(argv[i]) > 2) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           ldrname = strdup(cp);
		           break;
		case 'o':  if (strlen(argv[i]) > 2) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           outname = strdup(cp);
		           break;
		default:   usage();
	}
  }

  /* Check if all necessary command line options are given */
  if (romname == NULL || ldrname == NULL)
	usage();

  /* Open the output file */
  if (outname == NULL) {
	outfile = fileno(stdout);
  } else {
#ifdef MSDOS
	if ((outfile = open(outname, O_CREAT | O_WRONLY | O_TRUNC | O_BINARY, 0666)) < 0) {
#else
	if ((outfile = open(outname, O_CREAT | O_WRONLY | O_TRUNC, 0666)) < 0) {
#endif
		perror(outname);
		exit(1);
	}
  }

  /*
   * First copy the loader image into the output file. By checking for the
   * ROM signature we can also determine wether we are going to build a ROM
   * or a floppy image.
   */
#ifdef MSDOS
  if ((infile = open(ldrname, O_RDONLY | O_BINARY)) < 0) {
#else
  if ((infile = open(ldrname, O_RDONLY)) < 0) {
#endif
	perror(ldrname);
	exit(1);
  }
  len = doread(inbuf, BLKSIZE, ldrname);
  dorom = ((USHORT)ttoh(*((USHORT *)(&inbuf[0]))) == ROMSIG);
  writecnt = 0;
  while (TRUE) {
	writecnt += dowrite(inbuf, len);
	if ((len = doread(inbuf, BLKSIZE, ldrname)) == 0) break;
  }
  close(infile);

  /* Next copy the kernel image into the output file */
#ifdef MSDOS
  if ((infile = open(romname, O_RDONLY | O_BINARY)) < 0) {
#else
  if ((infile = open(romname, O_RDONLY)) < 0) {
#endif
	perror(romname);
	exit(1);
  }
  len = doread(inbuf, BLKSIZE, romname);
  romlength = 0;
  while (TRUE) {
	romlength += dowrite(inbuf, len);
	if ((len = doread(inbuf, BLKSIZE, romname)) == 0) break;
  }
  writecnt += romlength;
  if (writecnt > 65534L) {
	fprintf(stderr, "Bootrom code cannot be >64kB\n");
	exit(1);
  }
  close(infile);

  /*
   * If we got a real ROM here, determine the next available ROM size and
   * fill the output file up to that amount with 0 bytes. The last byte
   * has to remain for the checksum. Then update the size of the physical
   * ROM at the beginning of the output file, and write the size of the
   * bootrom kernel image.
   */
  if (dorom) {
	unsigned short serno;
	unsigned long romsize = 8192;	/* Physical size of ROM, 8kB is min. */
	unsigned char romsizchar;

	/* Compute the smallest possible ROM size */
	for (; romsize < writecnt + 1; )
		romsize <<= 1;

	/* Compute the serial number from the checksum and the current time */
	serno = (unsigned short)(time(NULL) & 0xffff) + chksum;

	/* Compute the missing size values and the final checksum */
	romsizchar = romsize >> 9;	/* Divide by 512 */
	chksum += romsizchar;
	for (i = 0; i < sizeof(romsize); i++)
		chksum += ((unsigned char *)&romlength)[i];
	chksum += (unsigned char)(serno & 0xff) +
					(unsigned char)((serno >> 8) & 0xff);

	/* Copy 0 byte buffer and checksum to output file */
	len = BLKSIZE;
	memset(inbuf, 0, BLKSIZE);
	while (writecnt < romsize) {
		/* Check if this is the last block */
		if (romsize - writecnt <= len) {
			len = (int)(romsize - writecnt);
			inbuf[len - 1] = (unsigned char)
				(0 - (unsigned char)(chksum & 0xff));
		}
		writecnt += dowrite(inbuf, len);
	}

	/* Put the physical ROM length into the output file */
	if (lseek(outfile, (long)ROMLENOFS, 0) < 0) {
		perror(outname);
		exit(1);
	}
	(void)dowrite((unsigned char *)&romsizchar, 1);

	/* Put the serial number into the output file */
	if (lseek(outfile, (long)ROMSEROFS, 0) < 0) {
		perror(outname);
		exit(1);
	}
	*((USHORT *)(&inbuf[0])) = (USHORT)htot(serno);
	(void)dowrite(inbuf, 2);
  }

  /* The length of the kernel image has to go at the beginning of the file */
  if (lseek(outfile, (long)ROMSIZEOFS, 0) < 0) {
	perror(outname);
	exit(1);
  }
  *((USHORT *)(&inbuf[0])) = (USHORT)htot(romlength);
  (void)dowrite(inbuf, 2);

  close(outfile);
  exit(0);
}

