#include "cthugha.h"
#include "options.h"

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <dirent.h>
#include <string.h>
#include <ctype.h>

#include "sound.h"
#include "display.h"
#include "action.h"
#include "translate.h"
#include "information.h"
#include "display.h"
#include "sound.h"
#include "cd_player.h"
#include "keys.h"
#include "net_sound.h"
#ifdef CTH_X11
    #include "cthugha-X11.h"
#endif

FILE * ini_file = 0;			/* the currently open ini-file */
const char RDB[] = "X11 resource database";
const char * ini_files[] = {		/* search these files as ini-files */
    "/usr/local/lib/cthugha/",
    "$(HOME)/.",
    "./",
    extra_lib_path,
#ifdef CTH_X11
    RDB,			/* to load from X11 resource database */
#endif
    NULL
};
int nr_ini_files = sizeof(ini_files) / sizeof(char *);

#ifdef CTH_X11
   static XrmDatabase database;
#endif

/*
 *  open a new ini-file 
 */
int open_ini_file(const char * fileName, char * mode) {
    char fname[256];
    char filename[256];
    char * filenameL;
    char * var;
    char c;
    int i,j;

    if ( fileName == NULL)			/* check if filename given */
	return 1;
    if ( fileName[0] == '\0')
	return 1;
	
    close_ini_file();				/* close old one for sure */

#ifdef CTH_X11
    if ( fileName == RDB ) {
	printfv("Initializing X11 resource database\n");
	XrmInitialize();
	if( (database = XrmGetDatabase(cth_display)) == NULL)
	    return 1;
	return 0;
    }
#endif

    strncpy(filename, fileName, 255);	/* create copy to work with */
    strncat(filename, "cthugha.ini", 255);
    filenameL = filename;

    /* replace environment variables in path */
    for(i=0; filename[i] != '\0'; i++) {
	if( filename[i] == '$') {		/* use environment variable */
	    filename[i] = '\0';
	    if( filename[i+1] == '(') {		/*  of form $(VAR) */
		/* search end of VAR */
		for(j=i+2; (filename[j] != '\0') && (filename[j] != ')'); j++);

		filename[j] = '\0';

		var = filename + i + 2;

		/* get variable */
		var = getenv(var);

	    } else {				/*  of form $VAR */
		/* search end of VAR */
		for(j=i+1; (filename[j] != '\0') && (filename[j] != '/'); j++);
		c=filename[j];
		
		filename[j] = '\0';

		var = filename + i + 1;

		/* get variable */
		var = getenv(var);

		filename[j] = c;
		if ( c != '\0')
		    j--;
	    }

	    /* put $VAR and filename together */
	    strncpy( fname, filename, 255);	/* part before $VAR */
	    if ( var != NULL)			/* $VAR if found */
		strncat( fname, var, 255);
	    strncat(fname, filename+j+1, 255);	/* part after $VAR */
	    
	    filenameL = fname;	
	}
    }

    printfv("opening ini-file: %s\n", filenameL);

    /* open file */
    return ( (ini_file = fopen( filenameL, mode)) == NULL ) ? 1 : 0 ;
}

/*
 *  close the currently open ini-file
 */
int close_ini_file() {
    if ( ini_file == NULL)
	return 0;
    fclose(ini_file);
    ini_file = NULL;
    return 0;
}


/*
 * Scan from beginning of ini-file, till section is reached
 */
int scan_to_section(const char * section) {
    char str[256];
    char * end;
	
    /* go to top of ini-file */	
    rewind(ini_file);
	
    do {
	/* read one line from file */
	if( fgets( str, 256, ini_file) == NULL )  {
	    /* end of file or readerror and the section not found */
#ifdef DEBUG
	    printfe("Section %s not found in ini_file.\n", section);
#endif
	    return 1;
	}
		
	/* check '[' and search and check for ']' */
	if ( (str[0] != '[') || ((end = strchr(str, ']') ) == NULL) ){
	    str[0] = section[0] + 1;
	    continue;
	}
	*end = 0;

    } while( strcasecmp( str+1, section) );

    /* found the section */
    return 0;
}

/*
 * Scan in current section till entry is found
 */
char * scan_to_entry(const char * entry) {
    static char str[258];
    char * end = NULL;
    char * tmp;
    int i, pos;
    do {
	/* remember file-position */
	pos = ftell(ini_file);
		
	/* read one line from file */
	if( fgets( str, 256, ini_file) == NULL )  {
	    /* end of file or readerror and the section not found */
#ifdef DEBUG
	    printfv("Entry %s not found in ini_file.\n", entry);
#endif
	    return NULL;
	}

	if ( str[0] == '[') {
	    /* found next section before entry */
#ifdef DEBUG
	    printfv("Entry %s not found in ini_file.\n", entry); 
#endif	
	    /* bring file-pointer back before the '[' */
	    fseek(ini_file, pos, SEEK_SET);
	    return NULL;
	}
		
	if ( (str[0] == '#') || (str[0] == ';') )	/* comment -> ignore*/
	    continue;

	if ( (end = strchr( str, '=')) == NULL)	{	/* not an entry */
	    continue;
	}
		
	*end = 0;
	do { ++end; } while( isspace(*end) );		/* skip WS */
				
	for( i=0; str[i] != '\0'; i++)			/* remove WS */
	    if ( isspace(str[i]) ) {
		str[i] = '\0';
		break;
	    }
		
    } while( strcasecmp(entry,str) );
    /* found entry */

    /* bring the file-pointer back */
    fseek(ini_file, pos, SEEK_SET);		

    /* remove trailing \n */
    tmp = end;
    while( (*tmp != '\0') && (*tmp != '\n'))
	tmp ++;
    *tmp = '\0';;
		
    /* return found entry */
    return end;
}	

/* 
 *  getini_<type>
 *  Read in the currently open ini-file the <entry> in <section>.
 *  <value> is only changed if the entry is found.
 */
int getini_str(const char * section, const char * entry, char * value) {
    char * pos;
    char * pos2;
	
#ifdef CTH_X11
    if ( ini_file == NULL) {
	char * str_type;
	XrmValue Entry;
	char class[512], name[512];

	/* build up name and class */
	strcpy(class, "Cthugha.");	strcpy(name, "cthugha.");
	strcat(class, section);		strcat(name, section);
	strcat(class, ".");		strcat(name, ".");
	strcat(class, entry);		strcat(name, entry);

	if( XrmGetResource(database, name, class, &str_type, &Entry) ) {
	    while( isspace(*Entry.addr) )
	    	Entry.addr ++;
	    if ( *Entry.addr == '\0')
	    	return 1;
	    strcpy(value, Entry.addr);
	    return 0;
	}
	return 1;
    }
#endif

    if ( scan_to_section(section) )			/* find section */
	return 1;
    if ( (pos=scan_to_entry(entry)) == NULL)		/* find entry in sec.*/
	return 1;
    for(pos2=pos; *pos2 != '\0'; pos2++)		/* get only one word */
	if( isspace(*pos2) )
	    *pos2 = '\0';

    if ( *pos == '\0' )					/* check if empty */
	return 1;
		
    strcpy(value, pos);

    return 0;				
}
int getini_int(const char * section, const char * entry, int * value) {
    int tvalue;
    char * pos;
    char str[512];
	
    if ( getini_str(section, entry, str) )
	return 1;

    /* first try to read as number */
    tvalue = strtol(str, &pos, 0);
    if ( str == pos )
	/* not a number */
	return 1;

    *value = tvalue;

    return 0;				
}
int getini_yesno(const char * section, const char * entry, int * value) {
    char str[512];
	
    if ( getini_str(section, entry, str) )
	return 1;

    if ( ! strncasecmp("yes", str,3) )
	*value = 1;
    else if ( ! strncasecmp("on", str,2) )
	*value = 1;
    else if ( ! strncasecmp("1", str,1) )
	*value = 1;
    else if ( ! strncasecmp("no", str,2) )
	*value = 0;
    else if ( ! strncasecmp("off", str,3) )
	*value = 0;
    else if ( ! strncasecmp("0", str,1) )
	*value = 0;
    else {
	printfe("Illegal yes/no-value for %s in %s.\n", entry, section);
	return 1;
    }

    return 0;				
}

/*
 * Insert data with size bytes at the current position of the file
 */
int finsert(FILE * file, const void * data, int size) {
    /* I hope no errors occur during writing */
    char str[5000];
    int nr;
    int pos, pos2, d;
	
    if  ( size <= 0)
	return 0;

    /* remember the position to insert */
    pos = ftell(file);

    /* determine size of rest of file */
    fseek(file, 0, SEEK_END);		/* find end of file */
    pos2 = ftell(file);

    if ( pos2 == pos) {			/* special tratement for eof */
	fwrite(data, size, 1, file);			/* write new */
    }

    while( pos2 > pos) {		/* while there is something to shift */
	d = ( (pos2-pos) > 5000) ? 5000 : pos2-pos;	/* block to shift */
	fseek(file, -d, SEEK_CUR);			/* move back */
	nr = fread(str, 1, d, file);			/* read old */
	fseek(file, -nr, SEEK_CUR);			/* move back */
	fwrite(data, size, 1, file);			/* write new */
	fwrite(str, nr, 1, file);			/* write old */
	fseek(file, -size-nr, SEEK_CUR);		/* back to where we
							   started writing */
	pos2 -= d;
    }

    return 0;
}

/*
 * putinit_<type>
 * write a new value in the ini-file for a given entry in a section
 */
int putini_str(const char * section, const char * entry, char * value) {
    char str[512], str2[256];
    int pos, left;

    if ( scan_to_section(section) ) {
	/* section not found - create it - 
	   easy cause file-pointer is now at the end*/
	sprintf(str, "[%s]\n%s=%s\n", section, entry, value);
	fwrite(str, strlen(str), 1, ini_file);
	return 0;
    }
    if ( scan_to_entry(entry) == NULL) {
	/* section found - but no entry -
	   filepointer is before next section */
	sprintf(str, "%s=%s\n", entry, value);
		
	/* insert enough space for the entry and fill with it */
	finsert(ini_file, str, strlen(str));
		
	return 0;
    }
    /* entry-found - replace the exiting entry */
	
    /* create new entry-string */
    sprintf(str, "%s=%s\n", entry, value);

    /* determine size of current entry */
    pos = ftell(ini_file);
    fgets(str2, 256, ini_file);
    left = strlen(str) - strlen(str2);

    /* go back to write-position */
    fseek(ini_file, pos, SEEK_SET);

    if ( left >= 0) {				/* new entry is bigger */
	/* write first part of string */
	fwrite(str, strlen(str2), 1, ini_file);

	/* make space and fill with string */
	finsert(ini_file, str + strlen(str) - left, left);
    }
    if ( left < 0) {				/* new entry is smaller */
	/* write new entry */
	fwrite(str, strlen(str)-1, 1, ini_file);
		
	/* fill up with space */
	memset(str, ' ', 256);
	fwrite(str, -left, 1, ini_file);	
    }

    return 0;	
}
int putini_yesno(const char * section, const char * entry, int value) {
    return putini_str(section, entry, value ? "on" : "off");
}

int putini_int(const char * section, const char * entry, int value) {
    char str[256];
    sprintf(str, "%d", value);
    return putini_str(section, entry, str);
}
int putini_opt(const char * section, const char * entry, 
	       opt_data * in, int value) {
    if ( value >= 0) 
	return putini_str(section, entry, in[value].name);
    return 1;
}
	       

/* 
 *  Read settings from ini-file's
 */

/* some macros to save typing work */
#define S_CD	"Sound_and_CD"
#define STUP	"Startup"
#define CHG	"Change"
#define DIS	"Disable"
#define DISP	"Display_and_Buffer"
#define GEN	"General"
#define SERV	"Server"

int read_ini() {
    int version;
    int ini;
    int value=0;
#ifndef CTH_server
    char str[256];
#endif
	
    for(ini=0; ini < nr_ini_files; ini++) {
	
	/* try to open ini-file */
	if ( open_ini_file(ini_files[ini], "r") )
	    continue;

	/* General options */
	version = CTHUGHA_VERSION;
	getini_int(GEN, "version", &version);
	if ( version != CTHUGHA_VERSION)
	    printfe("Version of ini settings in '%s' differs from Cthugha-Version.\n"
		    "found %d instead of %d\n", ini_files[ini], version, 
		    CTHUGHA_VERSION);
	
	getini_yesno(GEN, "verbose",	&cthugha_verbose);
#ifndef CTH_server
	getini_yesno(GEN, "save",	&options_save);
	getini_yesno(GEN, "stretch",	&trans_stretch);
	getini_str  (GEN, "path",	extra_lib_path);
	getini_str  (GEN, "prt-file",	display_prt_file);
	getini_yesno(GEN, "esc",	&key_esc);
	getini_yesno(GEN, "pause",	&cthugha_pause);
	getini_yesno(GEN, "dbl-load",   &double_load);
#endif

	/* Sound and CD options */
	if( ! getini_int(S_CD, "line",	&value) )    do_param('L', value, 0);
	if( ! getini_int(S_CD, "mic",	&value) )    do_param('M', value, 0);
	if( ! getini_int(S_CD, "cd",	&value) )    do_param('C', value, 0);
	if( ! getini_int(S_CD, "volume",&value) )    do_param('V', value, 0);
#ifndef CTH_server
	if( ! getini_str(S_CD, "network",str) )	     do_param('N', 0, str);
#endif
	value = 0; getini_yesno(S_CD, "no-sund", &value);
	if( value) sound_source = SNDSRC_DEBUG;
	if( ! getini_int(S_CD, "track",	&value) )    do_param('c', value, 0);
	getini_int  (S_CD, "rate",	&sound_sample_rate);
	getini_yesno(S_CD, "stereo",	&sound_stereo);	
	getini_yesno(S_CD, "cd-stop",	&cd_stop_on_exit);
	getini_yesno(S_CD, "cd-random",	&cd_randomplay);
	getini_yesno(S_CD, "cd-loop",	&cd_loop);
	getini_yesno(S_CD, "cd-eject",	&cd_eject_on_end);
	getini_yesno(S_CD, "cd-autoplay",	&cd_first_track);
	if( cd_first_track == 0)	cd_first_track = -1;
	getini_yesno(S_CD, "snd-sync",  &sound_sync);

#ifndef CTH_server
	/* Startup options */
	getini_str(STUP, "flame",	flame_first);
	getini_str(STUP, "wave",	wave_first);		
	getini_str(STUP, "palette",	palette_first);
	getini_str(STUP, "display",	screen_first);
	getini_str(STUP, "translate",	translate_first);
	getini_str(STUP, "pcx",		pcx_first);
	getini_int(STUP, "table",	&table_first);
	getini_int(STUP, "massage",	&massage_first);
#endif

#ifndef CTH_server
	/* Change options */
	getini_int(CHG,	"time",		&sound_wait_min);
	getini_int(CHG,	"random",	&sound_wait_random);
	getini_int(CHG,	"quiet-time",	&sound_wait_quiet);
	getini_int(CHG, "silence-time",	&sound_quiet_change);
	getini_int(CHG,	"beat-level",	&sound_peaklevel);
	getini_int(CHG,	"beat-nr",	&sound_wait_beat);	
	getini_yesno(CHG,"lock",	&action_lock);
	if( ! getini_str(CHG, "quiet-file", str) )
	    do_param('q', 0, str);
	getini_yesno(CHG, "little",	&change_little);
#endif

#ifndef CTH_server
	/* Disable options */
	getini_yesno(DIS, "internal_pal",	&display_internal_pal);
	getini_yesno(DIS, "external_pal",	&display_external_pal);
	getini_yesno(DIS, "pcx",		&display_use_pcx); 
	getini_yesno(DIS, "fft",		&sound_use_fft);
	getini_yesno(DIS, "translate",		&use_translations);	
#endif
		

#ifndef CTH_server
	/* Display and Buffer options */
	if( ! getini_str(DISP, "disp-mode", str) )
	    do_param('D', atoi(str), str);
#ifdef CTH_console
	getini_yesno(DISP, "disp-direct",	&display_direct);
#endif
	getini_yesno(DISP, "sync",		&display_syncwait);
	if( ! getini_str(DISP, "buff-size", str) )
	    do_param('S', atoi(str), str);
	if( ! getini_yesno(DISP, "tile",	&value) )
	    display_tile_x = display_tile_y = value;
	getini_yesno(DISP, "tile-x",		&display_tile_x);
	getini_yesno(DISP, "tile-y",		&display_tile_y);
#ifdef CTH_X11
	getini_yesno(DISP, "mit-shm",		&display_mit_shm);
	getini_yesno(DISP, "on-root",		&display_on_root);
	getini_yesno(DISP, "private-cmap",	&display_private_cmap);
#endif
#endif

	/* Server options */
#if !defined(CTH_server) && !defined(CTH_SS)
	getini_yesno(SERV, "server",		&server);
#endif
#ifdef CTH_server
	getini_int(SERV, "srv-wait",		&srv_wait_time);
#else
	getini_int(SERV, "clt-port",		&CLT_PORT);
#endif
	getini_int(SERV, "srv-port",		&REQ_PORT);

	close_ini_file();
    }
    return 0;
}

#ifndef CTH_server
/* 
 * read the sections [waves] [flames] [displays] 
 *                   [palettes] [pcxs] and [translations] 
 */
int read_ini_usage() {
    int ini,i;

    for(ini=0; ini < nr_ini_files; ini++) {
	
	/* try to open ini-file */
	if ( open_ini_file(ini_files[ini], "r") )
	    continue;

	/* get usage */
	for(i=0; i < nr_waves; i++) 
	    getini_yesno("waves", waves[i].name, &(waves[i].use));

	for(i=0; i < nr_flames; i++)
	    getini_yesno("flames", flames[i].name, &(flames[i].use));

	for(i=0; i < nr_screens; i++)
	    getini_yesno("displays", screens[i].name, &(screens[i].use));

	for(i=0; i < nr_palettes; i++)
	    getini_yesno("palettes", palettes[i].name, &(palettes[i].use));

	for(i=0; i < nr_pcx; i++)
	    getini_yesno("pcxs", pcxs[i].name, &(pcxs[i].use));

	for(i=0; i < nr_translations; i++)
	    getini_yesno("translations", translations[i].name, 
			 &(translations[i].use));

	/* close the ini-file */
	close_ini_file();
    }
    return 0;
}

/*
 *  Write active selection of wave, flame, display to the "most special"  
 *  ini-file.
 */
int write_ini() {
    int i=0;

    write_ini_startup();			/* save startup-sectin */
	
    for(i=nr_ini_files -2; i >= 0; i--) {

	/* try to open ini-file */
	if ( open_ini_file(ini_files[i], "r+") )
	    continue;
			
	/* write waves, flames, displays, palettes, pcxs, translations sec. */
	for(i=0; i < nr_waves; i++)
	    putini_yesno("waves", waves[i].name, waves[i].use);

	for(i=0; i < nr_flames; i++)
	    putini_yesno("flames", flames[i].name, flames[i].use);

	for(i=0; i < nr_screens; i++)
	    putini_yesno("displays", screens[i].name, screens[i].use);

	for(i=0; i < nr_palettes; i++)
	    putini_yesno("palettes", palettes[i].name, palettes[i].use);

	for(i=0; i < nr_pcx; i++)
	    putini_yesno("pcxs", pcxs[i].name, pcxs[i].use);
		
	for(i=0; i < nr_translations; i++)
	    putini_yesno("translations", translations[i].name,
			 translations[i].use);

	close_ini_file();
	return 0;
    }	
    /* no ini-files found */
    return 1;
}	

/*
 * write only the startup-section
 */
int write_ini_startup() {
    int i=0;
	
    for(i=nr_ini_files -2; i >= 0; i--) {

	/* try to open ini-file */
	if ( open_ini_file(ini_files[i], "r+") )
	    continue;
			
	/* save current flame, wave, ... as new start-values */
	putini_opt(STUP, "flame", flames,
		   opt_number(flame, flames, nr_flames));
	putini_opt(STUP, "wave", waves,
		   opt_number(display_wave, waves, nr_waves));	
	putini_opt(STUP, "palette", palettes,
		   opt_number(active_palette, palettes, nr_palettes));
	putini_opt(STUP, "display", screens,
		   opt_number(update_screen, screens, nr_screens));
	putini_opt(STUP, "translate", translations,
		   opt_number(active_translation, translations, 
			      nr_translations));
	putini_opt(STUP, "pcx",	pcxs,
		   opt_number(active_pcx, pcxs, nr_pcx));
	putini_int(STUP, "table", active_table);
	putini_int(STUP, "massage", sound_massage_style);
	
	putini_int(CHG,	"time",		sound_wait_min);
	putini_int(CHG,	"random",	sound_wait_random);
	putini_int(CHG,	"beat-nr",	sound_wait_beat);	
	putini_int(CHG,	"beat-level",	sound_peaklevel);
	putini_int(CHG,	"quiet-time",	sound_wait_quiet);
	putini_int(CHG, "silence-time",	sound_quiet_change);
	putini_yesno(CHG,"lock",	action_lock);
	putini_yesno(CHG, "little",	change_little);

  
	close_ini_file();
	return 0;
    }	
    /* no ini-files found */
    return 1;
}

#endif


