/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** llerror.c
**
** error reporting procedures
*/

# include "lclintMacros.nf"
# include "llbasic.h"
# include "llmain.h"
# include "version.h"

static void printIndentMessage (/*@only@*/ cstring sc, int indent) 
   /*@modifies stderr@*/ ;
static int lclerrors = 0;
static int lastfileloclen = 10;
static /*@only@*/ cstring lastmsg = cstring_undefined;
static int mcount = 0;
static /*@only@*/ cstring saveOneMessage = cstring_undefined; 
static /*@only@*/ fileloc lastparseerror = fileloc_undefined;
static /*@only@*/ fileloc lastbug = fileloc_undefined;
static bool llgenerrorreal (/*@only@*/ cstring s, fileloc fl, bool iserror, bool indent)
                 /*@modifies stderr@*/ ;
static bool llgenerroraux (/*@only@*/ cstring s, fileloc fl, bool iserror, bool indent)
                 /*@modifies stderr@*/ ;
static void printError (/*@only@*/ cstring sc)
   /*@globals lastfileloclen @*/ 
   /*@modifies stderr @*/ ;
static void printMessage (/*@only@*/ cstring s)
   /*@modifies stderr@*/ ;

static /*@null@*/ char *
maxcp (/*@null@*/ /*@returned@*/ char *a, /*@null@*/ /*@returned@*/ char *b)
{
  if (a > b) return a;
  else return b;
}

static void
printBugReport (void)
{
  fprintf (stderr, "     *** Please report bug to %s ***\n",
	   LCLINT_MAINTAINER);
  /* don't exit (EXIT_FAILURE); */
}

static void
prepareMessage (void)
{
  if (context_isPreprocessing () 
      && context_getDebug (FLG_SHOWSCAN))
    {
      fprintf (stderr, " >\n");
    }
}

static void
closeMessage (void)
{
  if (context_isPreprocessing () 
      && context_getDebug (FLG_SHOWSCAN))
    {
      fprintf (stderr, "< more preprocessing .");      
    }
}

void
llmsg (/*@only@*/ cstring s)
{
  context_setNeednl ();
  prepareMessage ();
  printMessage (s);
  closeMessage ();
}

void
llmsgplain (/*@only@*/ cstring s)
{
  context_setNeednl ();
  prepareMessage ();
  printMessage (s);
  closeMessage ();
}

void flagWarning (cstring s)
{
  if (context_getFlag (FLG_WARNFLAGS))
    {
      showHerald ();
	  
      if (fileloc_isBuiltin (currentloc))
	{
	  llmsg (message ("Warning: %s", s));
	}
      else
	{
	  llgenmsg (message ("Warning: %s", s), currentloc);
	}
    }
}

static void
llgenhint (/*@only@*/ cstring s) /*@modifies stderr@*/
{
  context_setNeednl ();
  printIndentMessage (s, 2);
}  

void
llhint (cstring s)
{
  if (context_getFlag (FLG_HINTS) &&   
      !(context_inSuppressRegion () || context_inSuppressZone (currentloc)))
    {
      llgenhint (s);
    }
  else
    {
      cstring_free (s);
    }
}

static void
llshowhint (flagcode f)
{
  if (context_getFlag (FLG_HINTS))
    {
      if ((flagcode_numReported (f) == 0) || context_getFlag (FLG_FORCEHINTS))
	{
	  cstring desc = flagcodeHint (f);
	  
	  if (cstring_isDefined (desc))
	    {
	      llgenhint (cstring_copy (desc));
	    }
	}
    }
}


static void
llsuppresshint (char c, flagcode f)
{
  DPRINTF (("suppresshint: %s", flagcode_unparse (f)));

  if (context_getFlag (FLG_HINTS))
    {
      if ((flagcode_numReported (f) == 0) || context_getFlag (FLG_FORCEHINTS))
	{
	  cstring desc = flagcodeHint (f);
	  context_setNeednl ();
	  lastfileloclen = 8;
	  
	  if (cstring_isDefined (desc))
	    {
	      llgenhint (message ("%s (%h%s will suppress message)", desc, c, 
				  flagcode_name (f)));
	    }
	  else
	    {
	      llgenhint (message ("(%h%s will suppress message)", c, 
				  flagcode_name (f)));
	    }
	}
    }
}

static void
llnosuppresshint (flagcode f)
{
  DPRINTF (("no suppresshint: %s", flagcode_unparse (f)));

  if (context_getFlag (FLG_FORCEHINTS))
    {
      cstring desc = flagcodeHint (f);
      context_setNeednl ();
      lastfileloclen = 8;
      
      if (cstring_isDefined (desc))
	{
	  printError (message ("    %s", desc));
	}
    }
}

/*@constant int MAXSEARCH; @*/
# define MAXSEARCH 20

/*@constant int MINLINE; @*/
# define MINLINE 35

typedef /*@null@*/ /*@dependent@*/ char *nd_charp;

static void
mstring_split (/*@returned@*/ char **sp,
	       /*@out@*/ nd_charp *tp,
	       int maxline, /*@in@*/ int *indentchars)
{
  char *nl;
  char *t;
  char *s = *sp;

  *tp = NULL;

  if (maxline < MINLINELEN)
    {
      maxline = MINLINELEN;
    }

  if (*indentchars > 0)
    {
      DPRINTF (("indent: %s / %d", s, *indentchars));
      s = *sp = mstring_concatFree1 (mstring_spaces (*indentchars), s);
      DPRINTF (("==> :%s:", s));
    }

  nl = strchr (s, '\n');

  /*
  ** splitting:
  **     
  **    if there is a newline in first maxline characters, split there
  **    if line len is <= maxline, return no split
  **    if there is a ':' or ';' followed by ' ' in first maxline characters,
  **       split there unless the ' ' is followed by a '}', then 
  **       split after '}'
  **       of the ';' is inside quotation marks
  **    if there is a space or tab in last maxsearch characters, split there
  **    else, split at maxline
  **
  **    special code: slash [1-9] after a newline means indent the rest <n> chars
  **
  */

  if ((nl != NULL) && ((nl - s) < maxline))
    {
      *nl = '\0';
      t = nl + 1;

      if (*t == '\0') 
	{
	  return;
	}
      
      if (*t >= '\1' && *t <= '\7')
	{
	  *indentchars += (int) (*t - '\1') + 1;
	  t++;
	  DPRINTF (("set indent: %d / %s", *indentchars, t));
	}

      *tp = t;
      return;
    }
  else if (size_toInt (strlen (s)) < maxline)
    {
      return;
    }
  else
    {
      int i = 0;
      char savechar;
      char *lcolon, *lsemi, *splitat;

      splitat = NULL;

      t = s + maxline - 1; 
      savechar = *t;

      *t = '\0';
      lcolon = strrchr (s, ':');
      lsemi = strrchr (s, ';');
      *t = savechar;
      
      splitat = maxcp (lcolon, lsemi);
      
      if (splitat != NULL && ((int)(splitat - s) > MINLINE) 
	  && *(splitat + 1) == ' ' && *(splitat + 2) != '}')
	{
	  *(splitat + 1) = '\0';
	  t = splitat + 2;
	  *tp = t;
	  return;
	}
      
      while (*t != ' ' && *t != '\t' && i < MAXSEARCH)
	{
	  t--;
	  i++;
	}
      
      if (*t != ' ' && *t != '\t')
	{
	  t = mstring_copy (s + maxline);
	  *(s + maxline) = '\0';

	  if (*t == '\0')
	    {
	      sfree (t);
	      return;
	    }

	  mstring_markFree (t);
	  *tp = t;
	  return;
	}
      else
	{
	  *t = '\0';
	  t++;

	  if (*t == '\0') return;

	  /*
	  ** returns unqualified as only
	  */
  
	  *tp = t;
	  return;
	}
    }
}

static
void limitmessage (/*@only@*/ cstring s, fileloc loc)
{
  if (mcount > context_getLimit () + 1)
    {
      cstring_free (s);
    }
  else
    {
      cstring flstring = fileloc_unparse (loc);
      
      lastfileloclen = cstring_length (flstring);
      cstring_free (saveOneMessage); 
      saveOneMessage = message ("%q: %q", flstring, s);
    }
}

void cleanupMessages ()
{
  if (context_unlimitedMessages ())
    {
     ;
    }
  else
    {
      int unprinted = mcount - context_getLimit ();
      
      if (unprinted > 0)
	{
	  if (unprinted == 1 && cstring_isDefined (saveOneMessage))
	    {
	      prepareMessage ();
	      printError (saveOneMessage);
	      closeMessage ();
	      saveOneMessage = cstring_undefined;
	    }
	  else
	    {
	      if (cstring_isDefined (saveOneMessage))
		{
		  /* cstring_free (saveOneMessage); */
		  saveOneMessage = cstring_undefined;
		}
	      
	      fprintf (stderr, "%s: (%d more similar errors unprinted)\n",
		       cstring_toCharsSafe (fileloc_filename (currentloc)), 
		       mcount - context_getLimit ());
	    }
	}
    }

  mcount = 0;
}

void
llgenmsg (/*@only@*/ cstring s, fileloc fl)
{
  cstring flstring = fileloc_unparse (fl);
  lastfileloclen = cstring_length (flstring);

  prepareMessage ();
  (void) printError (message ("%q: %q", flstring, s));
  closeMessage ();
}

void
llgenindentmsg (/*@only@*/ cstring s, fileloc fl)
{
  cstring flstring = fileloc_unparse (fl);

  prepareMessage ();
  (void) printIndentMessage (message ("%q: %q", flstring, s), 3);
  closeMessage ();
}

void
llgenindentmsgnoloc (/*@only@*/ cstring s)
{
  prepareMessage ();
  (void) printIndentMessage (s, 3);
  closeMessage ();
}

bool
llgentypeerror (ctype t1, exprNode e1, ctype t2, exprNode e2,
		/*@only@*/ cstring s, fileloc fl)
{
  if (llgenerroraux (s, fl, TRUE, FALSE))
    {
      flagcode code;

      if ((exprNode_isCharLit (e1) && ctype_isInt (t2))
	  || (exprNode_isCharLit (e2) && ctype_isInt (t1)))
	{
	  code = FLG_CHARINTLITERAL;
	}
      else if ((exprNode_isNumLit (e1) && ctype_isReal (t2))
	       || (exprNode_isNumLit (e2) && ctype_isReal (t1)))
	{
	  code = FLG_NUMLITERAL;
	}
      else if ((ctype_isDirectBool (t1) && ctype_isInt (t2))
	       || (ctype_isInt (t1) && ctype_isDirectBool (t2)))
	{
	  code = FLG_BOOLINT;
	}
      else if (((ctype_isChar (t1) && !ctype_isInt (t1)) && ctype_isInt (t2))
	       || ((ctype_isInt (t1) && (ctype_isChar (t2) && !ctype_isInt (t2)))))
	{
	  code = FLG_CHARINT;
	}
      else if ((ctype_isInt (t1) && ctype_isInt (t2))
	       || (ctype_isChar (t1) && ctype_isChar (t2))
	       || (ctype_isDouble (t1) && ctype_isDouble (t2)))
	{
	  if (!bool_equal (ctype_isSigned (t1), ctype_isSigned (t2)))
	    {
	      code = FLG_IGNORESIGNS;
	    }
	  else 
	    {
	      code = FLG_IGNOREQUALS;
	    }
	}
      else if (ctype_isArbitraryIntegral (ctype_realType (t1)))
	{
	  if (ctype_isArbitraryIntegral (ctype_realType (t2)))
	    {
	      code = FLG_MATCHANYINTEGRAL;
	    }
	  else if (ctype_match (t2, ctype_ulint))
	    {
	      code = FLG_LONGUNSIGNEDINTEGRAL;
	    }
	  else if (ctype_match (t2, ctype_lint))
	    {
	      code = FLG_LONGINTEGRAL;
	    }
	  else if (ctype_isInt (t2))
	    {
	      code = FLG_MATCHANYINTEGRAL;
	    }
	  else
	    {
	      code = FLG_TYPE;
	    }
	}
      else if (ctype_isArbitraryIntegral (ctype_realType (t2)))
	{
	  if (ctype_isArbitraryIntegral (ctype_realType (t1)))
	    {
	      code = FLG_MATCHANYINTEGRAL;
	    }
	  else if (ctype_match (t1, ctype_ulint))
	    {
	      code = FLG_LONGUNSIGNEDINTEGRAL;
	    }
	  else if (ctype_match (t1, ctype_lint))
	    {
	      code = FLG_LONGINTEGRAL;
	    }
	  else if (ctype_isInt (t1))
	    {
	      code = FLG_MATCHANYINTEGRAL;
	    }
	  else
	    {
	      code = FLG_TYPE;
	    }
	}
      else if (ctype_isAbstract (t1) && !ctype_isAbstract (t2))
	{
	  uentry ue1 = usymtab_getTypeEntry (ctype_typeId (t1));
	  ctype ct = uentry_getType (ue1);

	  if (ctype_match (ct, t2))
	    {
	      if (context_getFlag (FLG_HINTS))
		{
		  llgenhint 
		    (message ("Underlying types match, but %s is an "
			      "abstract type that is not accessible here.",
			      ctype_unparse (t1)));
		}
	      code = INVALID_FLAG;
	    }
	  else
	    {
	      code = FLG_TYPE;
	    }
	}
      else if (ctype_isAbstract (t2) && !ctype_isAbstract (t1))
	{
	  uentry ue = usymtab_getTypeEntry (ctype_typeId (t2));
	  ctype ct = uentry_getType (ue);
	  
	  if (ctype_match (ct, t1))
	    {
	      if (context_getFlag (FLG_HINTS))
		{
		  llgenhint 
		    (message ("Underlying types match, but %s is an "
			      "abstract type that is not accessible here.",
			      ctype_unparse (t2)));
		}
	      
	      code = INVALID_FLAG;
	    }
	  else
	    {
	      code = FLG_TYPE;
	    }
	}
      else
	{
	  code = FLG_TYPE;
	}
  
      if (code != INVALID_FLAG)
	{
	  if (code != FLG_TYPE)
	    {
	      llshowhint (code);
	    }
	  else
	    {
	      llsuppresshint ('-', code);
	    }
	  
	  flagcode_recordError (code);
	}

      closeMessage ();
      return TRUE;
    }
  else
    {
      flagcode_recordSuppressed (FLG_TYPE);
      return FALSE;
    }
}

bool  
llgenerror (flagcode o, /*@only@*/ cstring s, fileloc fl)
{
  if (llgenerroraux (s, fl, TRUE, FALSE))
    {
      llnosuppresshint (o);
      flagcode_recordError (o);
      closeMessage ();
      return TRUE;
    }
  else
    {
      flagcode_recordSuppressed (o);
      return FALSE;
    }
}

bool
llgenhinterror (flagcode o, /*@only@*/ cstring s, /*@only@*/ cstring hint,
		fileloc fl)
{
  if (!context_suppressFlagMsg (o, fl))
    {
      if (llgenerroraux (s, fl, TRUE, FALSE))
	{
	  flagcode_recordError (o);
	  
	  if (context_getFlag (FLG_HINTS)) 
	    {
	      llgenhint (hint);
	    }
	  else
	    {
	      cstring_free (hint);
	    }

	  closeMessage ();
	  return TRUE;
	}

      cstring_free (hint);
    }
  else
    {
      cstring_free (hint);
      cstring_free (s);
    }

  flagcode_recordSuppressed (o);
  return FALSE;
}

static bool  
llrealerror (/*@only@*/ cstring s, fileloc fl)
{
  return (llgenerrorreal (s, fl, TRUE, FALSE));
}

static bool
llgenerroraux (/*@only@*/ cstring s, fileloc fl, bool iserror, bool indent)
{
  DPRINTF (("inSuppressZone: %s", s));
  DPRINTF (("generroraux: %s", s));

  if (context_inSuppressZone (fl))
    {
      cstring_free (s);
      return FALSE;
    }
  else
    {
      DPRINTF (("not in suppress zone!"));
    }

  return (llgenerrorreal (s, fl, iserror, indent));
}

void
llforceerror (flagcode code, /*@only@*/ cstring s, fileloc fl)
{
  flagcode_recordError (code);
  (void) llgenerrorreal (s, fl, TRUE, FALSE);
  closeMessage ();
}

static bool
llgenerrorreal (/*@only@*/ cstring s, fileloc fl, bool iserror, bool indent)
{
  cstring flstring;

  /* duplicate message (rescanning a header file */

  if (!messageLog_add (context_messageLog (), fl, s))
    {
      DPRINTF (("duplicate: %s", s));
      cstring_free (s);
      return FALSE;
    }

  if (iserror) context_hasError ();

  if (context_unlimitedMessages ())
    {
      ;
    }
  else
    {
      /*
       ** suppress excessive messages:
       **    check up to ':'
       **
       */

      char *sc = cstring_toCharsSafe (s);
      char *tmpmsg = strchr (sc, ':');

      if (tmpmsg == NULL)
	{
	  tmpmsg = sc; 
	}
      else
	{
	  char *savechar = tmpmsg;
	  *tmpmsg = '\0';
	  tmpmsg = sc; 
	  *savechar = ':';
	}
      
      if (cstring_equal (lastmsg, cstring_fromChars (tmpmsg)))
	{
	  mcount++;
	  if (mcount == (context_getLimit () + 1))
	    {
	      limitmessage (s, fl);
	      return FALSE;
	    }

	  if (mcount > (context_getLimit ())) 
	    {
	      cstring_free (s);
	      return FALSE;
	    }
	}
      else
	{
	  cleanupMessages ();
	  mcount = 0;
	  cstring_free (lastmsg); 
	  lastmsg = cstring_fromCharsNew (tmpmsg);
	}
    }

  if (context_hasAliasAnnote ())
    {
      char *sc = cstring_toCharsSafe (s);
      char *fcolon = strchr (sc, ':');
      cstring a = context_getAliasAnnote ();

      DPRINTF (("a: %s / s: %s / f: %s", a, sc, fcolon));

      if (fcolon == NULL)
	{
	  s = message ("%q (%q)", s, a);
	}
      else
	{
	  cstring afterColon;

	  *fcolon = '\0';
	  afterColon = cstring_fromCharsNew (fcolon + 1);

	  s = message ("%q (%q):%q", s, a, afterColon);
	}
      DPRINTF (("aliasAnnot: %s", s));
    }
 
  if (context_hasMessageAnnote ())
    {
      char *fcolon = strchr (cstring_toCharsSafe (s), ':');
      
      DPRINTF (("message annote: %s", s));
      
      if (fcolon == NULL)
	{
	  /*@-dependenttrans@*/ /* s becomes dependent for fcolon */
	  s = message ("%q (%q)", s, context_getMessageAnnote ());
	  /*@=dependenttrans@*/
	}
      else
	{
	  cstring afterColon;

	  *fcolon = '\0';
	  afterColon = cstring_fromCharsNew (fcolon + 1);

	  /*@-dependenttrans@*/ /* s becomes dependent for fcolon */
	  s = message ("%q (%q):%q", s, 
		       context_getMessageAnnote (), afterColon);
	  /*@=dependenttrans@*/
	}
     } 
  
  context_setNeednl ();
  prepareMessage ();

  if (context_showFunction ())
    {
      cstring fname = fileloc_unparseFilename (currentloc);

      if (context_inIterDef ())
	{
	  fprintf (stderr, "%s: (in iter %s)\n",
		   cstring_toCharsSafe (fname),
		   cstring_toCharsSafe (context_inFunctionName ()));
	}
      else if (context_inIterEnd ())
	{
	  fprintf (stderr, "%s: (in iter finalizer %s)\n",
		   cstring_toCharsSafe (fname),
		   cstring_toCharsSafe (context_inFunctionName ()));
	}
      else if (context_inMacro ())
	{
	  fprintf (stderr, "%s: (in macro %s)\n", cstring_toCharsSafe (fname),
		   cstring_toCharsSafe (context_inFunctionName ()));
	}
      else
	{
	  fprintf (stderr, "%s: (in function %s)\n",
		   cstring_toCharsSafe (fname),
		   cstring_toCharsSafe (context_inFunctionName ()));
	}
      
      cstring_free (fname);
      context_setShownFunction ();
    }
  
  flstring = fileloc_unparse (fl);
  
  lastfileloclen = cstring_length (flstring);

  if (indent)
    {
      printError (message ("   %q: %q", flstring, s));
    }
  else
    {
      printError (message ("%q: %q", flstring, s));
    }

  return TRUE;
}

/*
** printError
**
** error contains no '\n'
**    error fits in one line: print it
**    error fits in two lines with 3-space indent after fileloc: print it
**    split line with 5-space indent from left margin: print it
**    
*/

static
void printMessage (/*@only@*/ cstring s) 
{
  printIndentMessage (s, 0);
}

static
void printIndentMessage (/*@only@*/ cstring sc, int indent) 
{
  int maxlen = context_getLineLen (); 
  char *s = cstring_toCharsSafe (sc);

  DPRINTF (("pim: %s / %d", sc, indent));

  do 
    {
      char *t = NULL;
      char *st = s;

      mstring_split (&st, &t, maxlen, &indent);
      fprintf (stderr, "%s\n", st);
      llassert (t != s);
      s = t;
    } while (s != NULL) ;
  
  cstring_free (sc);
}

static 
void printError (/*@only@*/ cstring sc) 
{
  int maxlen = context_getLineLen (); 
  int nspaces = lastfileloclen + 5;
  int nextlen = maxlen - nspaces; 
  int len = cstring_length (sc);
  int indent = 0;
  char *s = cstring_toCharsSafe (sc);
  char *t = NULL;
  
  if (len < (maxlen + nextlen) && (strchr (s, '\n') == NULL))
    {
      mstring_split (&s, &t, maxlen, &indent);
  
      fprintf (stderr, "%s\n", s);

      if (t != NULL)
	{
	  len = mstring_length (t);
	  
	  if (len < (maxlen - 3) && (strchr (t, '\n') == NULL) 
	      && len > (nextlen - 1))
	    {
	      fprintf (stderr, "    %s\n", t);
	    }
	  else
	    {
	      char *spaces = (char *) dmalloc ((nspaces + 1) * sizeof (*spaces));
	      int i;
	      
	      for (i = 0; i < nspaces; i++) 
		{
		  spaces[i] = ' ';
		}

	      spaces[nspaces] = '\0';
	      
	      while (t != NULL)
		{
		  char *st = t;
		  mstring_split (&st, &t, nextlen, &indent);
		  fprintf (stderr, "%s%s\n", spaces, st);
		}

	      sfree (spaces); 
	    }
	}
    }
  else
    {
      if (len < (maxlen + maxlen - 1) && (strchr (s, '\n') != NULL))
	{
	  nspaces = ((maxlen + maxlen - 1) - len) / 2;
	  
	  if (nspaces < 1) nspaces = 1;

	  nextlen = maxlen - nspaces; 
	  
	  mstring_split (&s, &t, maxlen, &indent);
	  
	  fprintf (stderr, "%s\n", s);
	  
	  if (t != NULL)
	    {
	      char *spaces = (char *) dmalloc ((nspaces + 1) * sizeof (*spaces));
	      int i;

	      for (i = 0; i < nspaces; i++) 
		{
		  spaces[i] = ' ';
		}

	      spaces[nspaces] = '\0';
	  
	      while (t != NULL)
		{
		  char *st = t;

		  mstring_split (&st, &t, nextlen, &indent);
		  fprintf (stderr, "%s%s\n", spaces, st);
		}

	      sfree (spaces); 
	    }
	}
      else
	{
	  nspaces = 4;
	  nextlen = maxlen - nspaces; 
	  
	  mstring_split (&s, &t, maxlen, &indent);
	  
	  fprintf (stderr, "%s\n", s);
	  
	  if (t != NULL)
	    {
  	      char *spaces = (char *) dmalloc ((nspaces + 1) * sizeof (*spaces));
	      int i;

	      for (i = 0; i < nspaces; i++) 
		{
		  spaces[i] = ' ';
		}

	      spaces[nspaces] = '\0';
	  
	      while (t != NULL)
		{
		  char *st = t;
		  mstring_split (&st, &t, nextlen, &indent);
		  fprintf (stderr, "%s%s\n", spaces, st);
		}

	      sfree (spaces); 
	    }
	}
    }
  /* sfree (s); */
/*@-mustfree@*/
} /*< <<< */
/*@=mustfree@*/

void
llfatalbug (/*@only@*/ cstring s)
{
  prepareMessage ();
  printError (message ("%q: *** Fatal bug: %q",
		       fileloc_unparse (currentloc), s));
  printCodePoint ();
  printBugReport ();
  llexit (LLFAILURE);
}

void
lclfatalbug (char *msg)
{
  prepareMessage ();
  printError (message ("*** Fatal Bug: %s", cstring_fromChars (msg)));
  printCodePoint ();
  printBugReport ();
  llexit (LLFAILURE);
}

void
checkParseError (void)
{
  if (fileloc_withinLines (lastparseerror, currentloc, 10))
    {
      llfatalerror (message ("%q: Cannot recover from parse error.",
			     fileloc_unparse (currentloc)));
    }
}

void
llbug (/*@only@*/ cstring s)
{
  /*@unchecked@*/ static int numbugs = 0;

  prepareMessage ();

  if (fileloc_withinLines (lastparseerror, currentloc, 7))
    {
      llfatalerror (message ("%q: Cannot recover from parse error.",
			     fileloc_unparse (currentloc)));
    }

  printError (message ("%q: *** Internal Bug: %q", fileloc_unparse (currentloc), s));
  printCodePoint ();
  printBugReport ();

  numbugs++;

  if (numbugs > 5 && fileloc_withinLines (lastbug, currentloc, 2))
    {
      llfatalerror (message ("%q: Cannot recover from last bug.",
			     fileloc_unparse (currentloc)));
    }
  
  fprintf (stderr, "       (attempting to continue, results may be incorrect)\n");
  fileloc_free (lastbug);
  lastbug = fileloc_copy (currentloc);
  closeMessage ();	  
}

void
lclbug (/*@only@*/ cstring s)
{
  prepareMessage ();
  printError (message ("*** Internal Bug: %q", s)); 
  printCodePoint ();
  printBugReport ();
  fprintf (stderr, "       (attempting to continue, results may be incorrect)\n");
  closeMessage ();
}

void
llfatalerror (cstring s)
{
  prepareMessage ();
  printError (s);
  printError (cstring_makeLiteral ("*** Cannot continue."));
  llexit (LLFAILURE);
}

void
llfatalerrorLoc (/*@only@*/ cstring s)
{
  prepareMessage ();
  printError (message ("%q: %q", fileloc_unparse (currentloc), s));
  printError (cstring_makeLiteral ("*** Cannot continue."));
  llexit (LLFAILURE);
}

/*
** free's s!
*/

void
llgloberror (/*@only@*/ cstring s)
{
  if (context_inSuppressRegion ())
    {
      cstring_free (s);
    }
  else
    {
      context_setNeednl ();
      prepareMessage ();
      context_hasError ();
      flagcode_recordError (FLG_SPECIAL);
      printError (s);
      closeMessage ();
    }
}

bool
lclHadError (void)
{
  return (lclerrors > 0);
}

bool
lclHadNewError (void)
{
  static int lastcall = 0;

  if (lclerrors > lastcall)
    {
      lastcall = lclerrors;
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

int
lclNumberErrors (void)
{
  return (lclerrors);
}

void
lclerror (ltoken t, /*@only@*/ cstring msg)
{
  lclerrors++;
  
  if (ltoken_getCode (t) != NOTTOKEN)
    {
      cstring loc = ltoken_unparseLoc (t);

      lastfileloclen = cstring_length (loc);

      printError (message ("%q: %q", loc, msg));
    }
  else
    {
      printError (msg);
    }
}

void
lclplainerror (/*@only@*/ cstring msg)
{
  lclerrors++;
  
  printError (msg);
}

void
lclfatalerror (ltoken t, /*@only@*/ cstring msg)
{
  lclerror (t, msg);
  printError (cstring_makeLiteral ("*** Cannot continue"));
  llexit (LLFAILURE);
}

void
lclplainfatalerror (/*@only@*/ cstring msg)
{
  printError (message ("*** Cannot continue: %q", msg));
  llexit (LLFAILURE);
}

void
lclRedeclarationError (ltoken id)
{
  cstring s = ltoken_getRawString (id);
  
  DPRINTF (("token: [%ld] %s", id, ltoken_unparse (id)));

  if (usymtab_existsEither (s))
    {
      uentry le = usymtab_lookupEither (s);
      
      lclerror (id, message ("Respecification of %s", s));
      llgenindentmsg (message ("Previous specification of %q",
			       uentry_getName (le)),
		uentry_whereSpecified (le));
    }
  else
    {
      lclerror (id, message ("Identifier redeclared: %s", s));
    }
}
  
void ppllerror (/*@only@*/ cstring s)
{
  if (context_getDebug (FLG_SHOWSCAN))
    {
      if (context_inSuppressZone (currentloc))
	{
	  cstring_free (s);
	}
      else
	{
	  fprintf (stderr, " >\n");
	  llerror (FLG_PREPROC, s);
	  fprintf (stderr, "< more preprocessing .");      
	}
    }
  else
    {
      llerror (FLG_PREPROC, s);
    }
}

void ppllmsg (cstring s)
{
  if (context_getDebug (FLG_SHOWSCAN))
    {
      fprintf (stderr, " >\n");
      llmsg (s);
      fprintf (stderr, "< more preprocessing .");      
    }
  else
    {
      llmsg (s);
    }
}

void llparseerror (/*@only@*/ cstring s)
{
  if (fileloc_withinLines (lastparseerror, currentloc, 5))
    {
      cstring_free (s);
    }
  else
    {
      llerror (FLG_SYNTAX, s);

      fileloc_free (lastparseerror);
      lastparseerror = fileloc_copy (currentloc);
    }
}

bool lloptgenerror (flagcode o, /*@only@*/ cstring s, fileloc loc)
{
  if (llrealerror (s, loc))
    {
      llsuppresshint ('-', o);
      closeMessage ();
      flagcode_recordError (o);
      return TRUE;
    }

  flagcode_recordSuppressed (o);
  return FALSE;
}

bool optgenerror2 (flagcode f1, flagcode f2, /*@only@*/ cstring s, fileloc loc)
{
  if (context_suppressFlagMsg (f1, loc))
    {
      flagcode_recordSuppressed (f1);
      cstring_free (s);
    }
  else
    {
      if (context_suppressFlagMsg (f2, loc))
	{
	  flagcode_recordSuppressed (f2);
	  cstring_free (s);
	}
      else
	{
	  if (llrealerror (s, loc))
	    {
	      llsuppresshint ('-', f2);
	      flagcode_recordError (f2);
	      closeMessage ();
	      return TRUE;
	    }

	  flagcode_recordSuppressed (f2);
	}
    }
  return FALSE;
}

bool optgenerror2n (flagcode f1, flagcode f2, /*@only@*/ cstring s, fileloc loc)
{
  DPRINTF (("optgenerror: %s", s));
  
  if (context_suppressFlagMsg (f1, loc))
    {
      flagcode_recordSuppressed (f1);
      cstring_free (s);
    }
  else
    {
      if (context_suppressNotFlagMsg (f2, loc))
	{
	  flagcode_recordSuppressed (f2);
	  cstring_free (s);
	}
      else
	{
	  if (llrealerror (s, loc))
	    {
	      llsuppresshint ('+', f2);
	      flagcode_recordError (f2);
	      closeMessage ();
	      return TRUE;
	    }

	  flagcode_recordSuppressed (f2);
	}
    }
  return FALSE;
}

bool llnoptgenerror (flagcode o, /*@only@*/ cstring s, fileloc loc)
{
  if (llrealerror (s, loc))
    {
      llsuppresshint ('+', o);
      flagcode_recordError (o);
      closeMessage ();
      return TRUE;
    }
  
  flagcode_recordSuppressed (o);
  return FALSE;
}
      




