/* 
	      efaxlib.c - utility routines for efax
		     Copyright 1995 Ed Casas
*/

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h> 
#include <time.h>

#include "efaxlib.h"

#define MAXTSTAMP 80		/* maximum length of a time stamp */

#ifndef SEEK_SET
#define SEEK_SET 0
#endif

char *verb = "ewin" ;		
char *argv0 = "" ;

int optind = 1 ;
char *optarg ;

extern t4tab wtab [ ( 64 + 27 + 13 ) + 1 ] ; /* T.4 coding tables */
extern t4tab btab [ ( 64 + 27 + 13 ) + 1 ] ;

char *iformatstr[] = { "0text", "1pbm", "2fax", 0 } ;
char *oformatstr[] = { "1pbm" , "2fax", "3pcl", "4ps",  "5pgm", 0 } ;
     

/* Print time stamp. */

void tstamp ( int dotime )
{
  static time_t last = 0, now ;
  char *fmt = 0, tbuf [ MAXTSTAMP ] ;

  now = time ( 0 ) ;

  if ( dotime || ( now - last >   0 ) ) fmt = " (%M:%S)" ;
  if (             now - last > 600 )   fmt = " %c" ;

  if ( fmt ) {
    strftime ( tbuf, MAXTSTAMP, fmt, localtime( &now ) ) ;
    fputs ( tbuf , LOGF ) ;
    last = now ; 
  }
}


/* For systems without strerror(3) */

char *strerror( int i )       
{
  extern int sys_nerr;
  extern char *sys_errlist[];
  return ( i >= 0 && i < sys_nerr ) ? sys_errlist[i] : "Unknown Error" ;
}


/* Return string corresponding to character c. */

char *cname ( u_char c ) 
{
#define CNAMEFMT "<0x%02x>"
#define CNAMELEN 6+1
  static char *cname [ 256 ] = {		/* character names */
  "<NUL>","<SOH>","<STX>","<ETX>", "<EOT>","<ENQ>","<ACK>","<BEL>",
  "<BS>", "<HT>", "<LF>", "<VT>",  "<FF>", "<CR>", "<SO>", "<SI>", 
  "<DLE>","<XON>","<DC2>","<XOFF>","<DC4>","<NAK>","<SYN>","<ETB>",
  "<CAN>","<EM>", "<SUB>","<ESC>", "<FS>", "<GS>", "<RS>", "<US>" } ;
  static char names[ (127-32)*2 + 129*(CNAMELEN) ], *p=names ;
  static int i=0 ;
    
  if ( ! i ) 
    for ( i=32 ; i<256 ; i++ ) {
      sprintf ( cname [ i ] = p , i<127 ? "%c" : CNAMEFMT , i ) ;
      p += strlen ( p ) + 1 ;
    }

  return cname [ c ] ;
} 

/* Print a message with a variable number of printf()-type arguments if the
   first character appears in the global verb[ose] string.  Other leading
   characters and digits do additional actions: + allows the message to be
   continued on the same line, '-' buffers the message instead of printing
   it, E, and W expand into strings, S prints the error message for the
   most recent system error, 0-4 set the return value, a space is skipped
   and ends prefix.  Returns 0 if no prefix digit. */

int msg ( const char *fmt, ... ) 
{ 
  int err=0, dolf=1, flush=1, dotime=0 ;
  char *ps="", *pe="", *pw="" ;
  const char *p ;
  static int atcol1=1 ;
  va_list ap ;
  va_start ( ap, fmt ) ;

  for ( p=fmt ; *p ; p++ ) {
    switch ( *p ) {
    case ' ': p++ ; goto print ;
    case 'A': break ;				   /* program args */
    case 'C': break ;				   /* commands/responses */
    case 'M': break ;				   /* modem dialogue */
    case 'N': break ;				   /* negotiation */
    case 'E': pw = "Error: " ; break ;
    case 'H': break ;				   /* HDLC frame data */
    case 'I': break ;				   /* information */
    case '+': dolf = 0 ; break ;
    case '-': flush = 0 ; break ;
    case 'R': break ;				   /* reception errors */
    case 'S': ps = strerror ( errno ) ; break ;
    case 'T': dotime = 1 ; break ;
    case 'W': pw = "Warning: " ; break ;
    case '0': case '1': case '2': case '3': case '4': case '5': 
      err = *p - '0' ; break ;
    default: goto print ;
    }
  }

  print:  
  if ( strchr ( verb , tolower ( *fmt ) ) ) {
    if ( atcol1 ) fprintf ( LOGF , "%s: %s%s" , argv0 , pe , pw ) ;
    vfprintf( LOGF, p , ap ) ;
    fputs ( ps , LOGF ) ;
    atcol1 = dolf ;
    if ( atcol1 ) {
      tstamp ( dotime ) ; 
      fputs ( "\n" , LOGF ) ;
    }
    if ( flush ) fflush ( LOGF ) ;
  }

  va_end ( ap ) ;
  return err ;
}


/* Simple (one option per argument) version of getopt(3). */

int nextopt( int argc, char **argv, char *args )
{
  char *a, *p ;

  if ( optind >= argc || *(a = argv[optind]) != '-' ) return -1 ;
  optind++ ;

  if ( ! *(a+1) || ( ( p = strchr ( args , *(a+1) ) ) == 0 ) )
    return msg ( "Eunknown option (%s)" , a ) , '?' ; 

  if ( *(p+1) != ':' ) optarg = 0 ;
  else
    if ( *(a+2) ) optarg = a+2 ;
    else
      if ( optind >= argc ) return msg ( "Eno argument for (%s)", a ) , '?' ;
      else optarg = argv [ optind++ ] ;
  return *(a+1) ;
}

/* Initialize state of encoder for variable-length code words. */

void newENCODER ( ENCODER *e )
{
  e->x = 0 ;
  e->shift = -8 ;
}


/* Store a code word `code' of length `bits' (<=24) in buffer pointed to by
   `buf'.  Bits that don't fit in complete bytes are saved between calls.
   To flush the remaining bits call the function with code=0 and bits=0.
   Returns pointer to next free element in output buffer.  Calling function
   must ensure at least bits/8 bytes are available in buffer.  */

u_char *putcode ( ENCODER *e, short code, short bits , u_char *buf )
{
  e->x = ( e->x << bits ) | code ;
  e->shift += bits ? bits : -e->shift ;

  while ( e->shift >= 0 ) {
    *buf++ = e->x >> e->shift ;
    e->shift -= 8 ;
  }

  return buf ;
}


/* Convert run lengths to 1-D T.4-codes.  First run is white.  Silently
   truncates run lengths that are too long. After using this function EOLs
   may need to be added and/or the putcode() buffer flushed.  Returns
   pointer to next free element in output buffer. */

u_char *runtocode ( ENCODER *e, short *runs, int nr, u_char *codes )
{
  u_char col = 0, *maxcodes = codes + MAXCODES ;
  t4tab *ctab = wtab, *p ;
  short rlen ;
  long x ;
  short shift ;

#define PUTCODE(p) { x = ( x << p->bits ) | p->code ;  shift += p->bits ; \
	while ( shift >= 0 ) { *codes++ = x >> shift ; shift -= 8 ; } }

  x = e->x ; shift = e->shift ;

  while ( nr-- > 0 ) {
    rlen = *runs++ ;
    if ( rlen > 63 ) {				/* make-up code */
      if ( rlen > MAXRUNLEN ) rlen = MAXRUNLEN ;
      p = ctab + 63 + ( rlen >> 6 ) ;
      if ( codes < maxcodes ) PUTCODE(p) ;
    }
    p = ctab + ( rlen & 0x3f ) ;		/* terminating code */
    if ( codes < maxcodes ) PUTCODE(p) ;
    ctab = ( col ^= 1 ) ? btab : wtab ;
  }  
  
  e->x = x ; e->shift = shift ;

  return codes ;
}


/* Pad/truncate run-length coded scan line 'runs' of 'nr' runs by 'pad'
   pixels (truncate if negative).  Returns the new number of runs. */

int xpad ( short *runs, int nr, int pad )
{
  if ( pad < 0 ) {		          /* truncate */
    while ( pad < 0 ) pad += ( nr <= 0 ? -pad : runs [ --nr ] ) ;
    runs [ nr++ ] = pad ;
  } else {				  /* pad with white */
    if ( nr & 1 ) runs [ nr - 1 ] += pad ;
    else runs [ nr++ ] = pad ;
  }
  return nr ;
}


/* Shift a run-length coded scan line right by s pixels (left if negative).
   If necessary, zero-length runs are created to avoid copying.  Returns
   the pixel width change (+/-). */

int xshift ( short *runs, int nr, int s )
{
  int i=0, n=0 ;
  if ( s < 0 ) {
    for ( i = 0 ; s < 0 && i < nr ; i++ ) { 
      s += runs [ i ] ;
      n -= runs [ i ] ;
      runs [ i ] = 0 ; 
    }
    i-- ;
  }
  if ( i < nr ) {
    runs [ i ] += s ;
    n += s ;
  }
  return n ;
}


/* Scale nr run lengths in buffer pointed to by p to scale image
   horizontally.  The scaling factor is xs/256. Returns new line width in
   pixels. */

int xscale ( short *p , int nr, int xs )
{
  int inlen=0, outlen=0 ;
  for ( ; nr-- > 0 ; p++ ) {
    inlen += *p ;
    *p = ( ( inlen * xs + 128 ) >> 8 ) - outlen ;
    outlen += *p ;
  }
  return outlen ;
}


/* Zero-terminated lists of run lengths for each byte. */

u_char byteruns [ 1408 + 1 ] = 
   "8071061106205120511105210530413041210411110411204220421104310440"
   "3140313103121103122031112031111103112103113032303221032111032120"
   "3320331103410350215021410213110213202121202121110212210212302111"
   "3021112102111111021111202112202112110211310211402240223102221102"
   "2220221120221111022121022130233023210231110231202420241102510260"
   "1160115101141101142011312011311101132101133011213011212101121111"
   "0112112011222011221101123101124011114011113101111211011112201111"
   "1120111111110111112101111130111230111221011121110111212011132011"
   "1311011141011150125012410123110123201221201221110122210122301211"
   "3012112101211111012111201212201212110121310121401340133101321101"
   "3220131120131111013121013130143014210141110141201520151101610170"
   "1701610151101520141201411101421014301313013121013111101311201322"
   "0132110133101340121401213101212110121220121112012111110121121012"
   "1130122301222101221110122120123201231101241012501115011141011131"
   "1011132011121201112111011122101112301111130111112101111111101111"
   "1120111122011112110111131011114011240112310112211011222011211201"
   "1211110112121011213011330113210113111011312011420114110115101160"
   "2602510241102420231202311102321023302213022121022111102211202222"
   "0222110223102240211402113102112110211220211112021111110211121021"
   "1130212302122102121110212120213202131102141021503503410331103320"
   "3212032111032210323031130311210311111031112031220312110313103140"
   "4404310421104220411204111104121041305305210511105120620611071080" ;

/* Convert byte-aligned bit-mapped n-byte scan line into array of run
   lengths.  Run length array must have *more* than 8*n elements.  First
   run is white.  Returns number of runs coded.  */

int bittorun ( u_char *bits, int n, short *runs )
{
  static u_char init=0, *rltab [ 256 ] ;
  register u_char *p, c, lastc = 0x00 ;
  short *runs0 = runs ;

  if ( ! init ) {		/* initialize pointer and run tables */
    int i = 0 ;
    for ( rltab[ 0 ] = p = byteruns ; *p ; p++ )
      if ( ! ( *p -= '0' ) && i < 255 ) 
	rltab [ ++i ] = p+1 ;
    init = 1 ;
  }

  *runs = 0 ;
  for ( ; n > 0 ; n-- ) {
    p = rltab [ c = *bits++ ] ;
    if ( ( lastc & 0x01 ) ? ! ( c & 0x80 ) : ( c & 0x80 ) )
      *(++runs) = *p++ ;		  /* new run */
    else 			  
      *runs += *p++ ;			  /* continue run */
    while ( *p ) 
      *(++runs) = *p++ ;
    lastc = c ;
  }

  return runs - runs0 + 1  ;
}


/* Bitwise-OR two run-length coded scan lines.  The run length
   vectors a and b are OR-ed into c.  If c is null, the result is
   placed in a.  The new image width is stored in pels if it is
   not null.  Returns the number of runs in the result.  */

int runor ( short *a, int na, short *b, int nb, short *c, int *pels )
{
  register short la, lb ;
  int ia, ib, ic, np=0 ;
  short tmp [ MAXRUNS ] ;

  if ( ! c ) c = tmp ;

  la = a [ ia = 0 ] ;
  lb = b [ ib = 0 ] ;
  c [ ic = 0 ] = 0 ;

  while ( 1 ) {
    if ( la <= lb ) {			  /* select shorter sub-run */
      if ( ( ( ia | ib ) ^ ic ) & 1 )	  /* OR of subruns same colour as c? */
	c [ ++ic ] = la ;		  /* no, new output run */
      else 
	c [ ic ] += la ;		  /* yes, add it */
      lb -= la ;			  /* align subruns */
      if ( ++ia >= na ) break ;		  /* done */
      la = a [ ia ] ;			  /* get new subrun */
    } else {				  /* same for line b ... */
      if ( ( ( ia | ib ) ^ ic ) & 1 ) 
	c [ ++ic ] = lb ;
      else 
	c [ ic ] += lb ;
      la -= lb ;
      if ( ++ib >= nb ) break ;
      lb = b [ ib ] ;
    }
  }

  if ( ia < na )
    while ( 1 ) {
      if ( ( ia ^ ic ) & 1 )	  
	c [ ++ic ] = la ;		  
      else 
	c [ ic ] += la ;		  
      if ( ++ia >= na ) break ;		  
      la = a [ ia ] ;			  
    } 
  else
    while ( 1 ) {
      if ( ( ib ^ ic ) & 1 ) 
	c [ ++ic ] = lb ;
      else 
	c [ ic ] += lb ;
      if ( ++ib >= nb ) break ;
      lb = b [ ib ] ;
    }

  if ( c == tmp ) for ( ia=0 ; ia <= ic ; ia++ ) np += a[ia] = c[ia] ;

  if ( pels ) *pels = np ;

  return ic + 1 ;
}  
  

/* Check a raw PBM file header and get bitmap dimensions.  Leaves stream
   positioned on first data byte.  Returns 0 or closes file and returns 2
   on errors. */

int openpbm ( FILE *f, int *w, int *h )
{
  int err = 0 ;

  if ( f && fscanf ( f , "P4 %d %d%*c", w, h ) != 2 ) {
    err = msg ("E2can't read (raw) PBM header" ) ;
    fclose ( f ) ;
    *w = *h = 0 ;
  } 
  
  return err ;
}


/* Read a PBM font and fill in font structure.  If the file name is null or
   there are errors, it initializes the font to the built-in font. Returns
   0 if OK, 2 on errors. */

int readfont ( char *fname, pbmfont *font )
{
  int err=0, i, j, n=0 ;

  if ( fname ) {

    FILE *f ;

    if ( ( f = fopen ( fname , "rb" ) ) == 0 )
      err = msg ("ES2can't open font file %s:\n", fname ) ;

    if ( ! err && openpbm ( f, & font->w, & font->h ) != 0 )
      err = msg ( "E2bad font file (%s) header", fname ) ;
    
    if ( ! err && ( n = fread ( font->buf, 1, MAXFONTBUF, f ) ) != 
	font->w * font->h / 8 ) 
      err = msg ( "E2error reading font data" ) ;
    
    if ( ! err && ( font->w / 256 > MAXFONTW || font->h > MAXFONTH ) ) {
      err = msg ( "E2font size (%dx%d) too large", font->w, font->h ) ;
    }

    if ( err ) {
      if ( f ) fclose ( f ) ;
      font->w = font->h = 0 ;
    } else {
      font->w /= 256 ;	 /* convert PBM width to character width */
      for ( i=0 ; i<256 ; i++ ) font->offset[i] = i*font->w ;
      msg ("Iread %dx%d font %s (%d bytes)", font->w, font->h, fname, n ) ;
    }
  }    

  if ( err || ! fname ) {	           /* use built-in font */

    font->w = STDFONTW ;
    font->h = STDFONTH ;

    for ( i=j=0 ; j<STDFONTBUF ; i++ )	   /* expand bit map */
      if ( stdfont [ i ] == 0 )
	for ( n = stdfont [ ++i ] ; n > 0 ; n-- ) 
	  font->buf [ j++ ] = 0 ;
      else
	font->buf [ j++ ] = stdfont [ i ] ;

    if ( i != 1980 ) err = msg ( "E2can't happen(readfont)" ) ;

    for ( i=0 ; i<256 ; i++ ) font->offset[i] = i*font->w ;
  }

  return err ;
}


/* Append nb bits from in[from] to bit-mapped scan line buffer
   where `from' is a bit (not byte) index.  Bits in bytes are
   ordered from MS to LS bit. Initialize before each scan line by
   calling with nb=0 and in pointing to output buffer.  Flush
   after each scan line by calling with nb=0 and in=NULL. */

#define putbits( c , b ) { x = ( x << (b) ) | (c) ; shift += (b) ; \
          if ( shift >= 0 ) { *out++ = x >> shift ; shift -= 8 ; } }

void copybits ( u_char *in , int from , short nb )
{
  u_char *f ;
  short bits ;
  static u_char *out ;
  static short x, shift ;
  static unsigned char right [ 9 ] = { 
    0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff } ;
  
  if ( ! nb ) {				/* reset for new scan line */
    if ( in ) out = in ;		   
    else putbits ( 0, -shift ) ;	/* or flush bit buffer */
    x = 0 ;
    shift = -8 ;
  } else {
    f = in + ( from >> 3 ) ;
    bits = 8 - ( from & 7 ) ;

    if ( nb >= bits ) {
      putbits ( *f++ & right [ bits ] , bits ) ;
      nb -= bits ;
    } else {
      putbits ( ( *f >> ( bits - nb ) ) & right [ bits ] , nb ) ;
      nb = 0 ;
    } 

    while ( nb >= 8 ) { putbits ( *f++ , 8 ) ; nb -= 8 ; }

    if ( nb > 0 ) putbits ( *f >> ( 8 - nb ) , nb );
  }
}   


/* Generate scan line 'line' of string 'txt' using font `font' and store
   the runs in 'runs'.  Sets 'pels' to line width if not null.  Returns
   number of runs coded. */

int texttorun( u_char *txt, pbmfont *font, short line, short *runs, int *pels )
{
  u_char *in, out [ MAXLINELEN * MAXFONTW / 8 + 1 ] ;
  int i, nc = 0, cw, nr ;

  cw = font->w ;
  if ( line >= font->h ) line = font->h - 1 ;
  in = font->buf + 256/8 * cw * line ;

  copybits ( out, 0, 0 ) ;
  for ( i=0 ; txt[i] && i < MAXLINELEN ; i++ ) {
    copybits ( in, font->offset [ txt[i] ], cw ) ;
    nc++ ;
    while ( ( txt[i] == HT ) && ( nc & 7 ) ) { /* tab */
      copybits ( in, font->offset [ ' ' ], cw ) ;
      nc++ ;
    }
  }
  copybits ( 0, 0, 0 ) ;

  if ( pels ) *pels = cw * nc ;

  nr = bittorun ( out, ( nc*cw + 7 )/8, runs ) ;

  return nr ;
}

		/* Image File Input Functions */

/* Read run lengths for one scan line from T.4-coded file f into buffer
   runs using decoder state d.  If pointer pels is not null it is used to
   save pixel count.  Returns number of runs stored, EOF on RTC, or -2 on
   EOF or other error. */

int readruns ( DECODER *d, FILE *f, short *runs, int *pels )
{
  int err=0, c=EOF, n ;
  register int x ;
  dtab *tab, *t ;
  short shift ;
  short *p, *maxp, *q, len=0, npad=0 ;

  maxp = ( p = runs ) + MAXRUNS ;

  x = d->x ; shift = d->shift ; tab = d->tab ; /* restore decoder state */

  do {
    do {
      while ( shift < 0 ) { 
	if ( ( c = getc ( f ) ) == EOF )  {
	  x = ( x << 15 ) | 1 ; shift += 15 ;  /* EOL pad at EOF */
	  npad++ ;
	} else {
	  x = ( x <<  8 ) | c ; shift +=  8 ; 
	}
      }
      t = tab + ( ( x >> shift ) & 0x1ff ) ;
      tab = t->next ;
      shift -= t->bits ;
    } while ( ! t->code ) ;
    if ( p < maxp ) *p++ = t->code ;
  } while ( t->code != -1 ) ;

  d->x = x ; d->shift = shift ; d->tab = tab ; /* save state */

  if ( npad > 1 ) msg ("W EOF before RTC" ) ;

  if ( p >= maxp ) msg ( "Wrun length buffer overflow" ) ;

  /* combine make-up and terminating codes and remove +1 offset
     in run lengths */

  n = p - runs - 1 ;
  for ( p = q = runs ; n-- > 0 ; )
    if ( *p > 64 && n-- > 0 ) {
      len += *q++ = p[0] + p[1] - 2 ;
      p+=2 ;
    } else {
      len += *q++ = *p++ - 1 ;
    }
  n = q - runs ;
  
  /* check for RTC and errors */

  if ( len )
    d->eolcnt = 0 ;
  else
    if ( ++(d->eolcnt) >= RTCEOL ) err = EOF ;

  if ( c == EOF ) 
    if ( ferror ( f ) ) 
      err = -msg ("ES2error reading fax file:") ;
    else 
      err = -2 ;

  if ( pels ) *pels = len ;
  
  return err ? err : n ;
}


/* Read a scan line from the current page of IFILE f.  Stores number of
   runs in runs and line width in pels if not null.  Pages ends at
   EOF. Text pages also end if a complete text line would not fit or if
   formfeed is the first character of a line.  PBM pages also end when all
   lines in the bitmap have been read. Fax pages also end at RTC. Returns
   number of runs stored or EOF at end of page. */

int readline ( IFILE *f, short *runs, int *pels )
{
  int nr = 0 ;
  u_char bits [ MAXBITS ] ;

  if ( f->lines )
    switch ( f->format ) {
    case TEXT :
      if ( f->txtlines <= 0 ) {		 /* need another text line */
	if ( fgets ( f->text, MAXLINELEN, f->f ) ) {
	  f->txtlines = f->font->h ;
	  if ( strchr ( f->text, FF ) ) {
	    f->lines = 0 ;	/* no more lines in this page */
	    nr = EOF ;		/* don't return any */
	  } 
	} else {
	  nr = EOF ;
	}
      }
      if ( nr != EOF ) {
	nr = texttorun( (u_char*) f->text, f->font, f->font->h - f->txtlines, 
		       runs, pels ) ;
	f->txtlines-- ;
      } 
      break ;

    case  PBM:
      if ( fread ( bits, 1, f->Bw, f->f ) != f->Bw ) {
	nr = EOF ;
      } else {
	nr = bittorun ( bits, f->Bw, runs ) ;
	if ( pels ) *pels = f->Bw * 8 ;
      }
      break ;

    case FAX:
      nr = readruns ( &(f->d), f->f, runs, pels ) ;
      break ;
    }
  else
    nr = EOF ;

  if ( nr >= 0 && f->lines > 0 ) f->lines-- ;

  return nr ;
}


/* Skip to start of next/same (if dp=0) page image, possibly within same
   file.  For text files, continues reading from the current file unless at
   EOF.  Other formats skip to next file in list of file names.  Uses stdin
   if file name is '-'.  Checks for correct file headers (if any) and
   initializes relevant parts of the IFILE.  Returns f if OK, NULL on
   error.  */


IFILE *nextipage ( IFILE *f, int dp )
{
  int err = 0, newfile=1, stdinput=0 ;
  short runs [ MAXRUNS ] ;
  char *fn = "next page" ;
  static pbmfont defaultfont ;		/* default font used for text files */

  /* seek to start of next/same page within current file (if any) */

  if ( f->f )
    switch ( f->format ) {
    case TEXT:
      if ( dp )
	while ( readline ( f, runs, 0 ) >= 0 ) ;
      else
	if ( fseek ( f->f, f->start, SEEK_SET ) ) 
	  err = msg ("ES2can't re-read page:" ) ;
      if ( ! feof ( f->f ) && ! ferror ( f->f ) ) 
	newfile = 0 ;
      break ;

    case  PBM:
      break ;

    case FAX:
      break ;
    }

  if ( newfile && ! err ) {			 /* open same/next file */

    if ( dp && f->fname && *f->fname ) 
      f->fname++ ;

    if ( ! f->fname || ! *f->fname ) 
      err = 1 ;
    else {
      fn = *f->fname ;
      if ( ! strcmp ( fn, "-" ) ) {
	stdinput = 1 ;
	if ( f->f != stdin ) fclose ( f->f ) ;
	rewind ( f->f = stdin ) ;
      } else {
	if ( f->f ) fclose ( f->f ) ;
	f->f = fopen ( fn, ( f->format==TEXT ) ? "r" : "rb" ) ;
      }
      if ( ! f->f )
	err = msg ( "ES2can't open %s:", fn ) ;
    }

  }

  /* read file/subfile header and set relevant IFILE struct variables */

  if ( ! err ) {
    switch ( f->format ) {			 /* initialize f  */

    case TEXT:
      if ( ! f->font ) readfont ( 0, f->font = &defaultfont ) ; 
      if ( ! f->pglines ) f->pglines = DEFPGLINES ;
      f->start = ftell ( f->f ) ;
      f->lines = ( ( f->pglines * f->font->h ) / f->font->h ) * f->font->h ;
      break ;

    case  PBM:					 /* get width & height */
      if ( openpbm ( f->f, &f->w, &f->h ) ) 
	err = 2 ;
      else
	if ( f->w % 8 ) 
	  err = msg ( "E2 PBM width must be multiple of 8" ) ;
	else {
	  f->Bw = f->w / 8 ;
	  f->lines = f->h ;
	}
      break ;

    case FAX:
      newDECODER ( &f->d ) ;
      if ( readruns ( &f->d, f->f, runs, 0 ) < 0 )	 /* skip first EOL */
	msg ( "Wno EOL: probably not fax file" ) ;
      f->lines = -1 ;
      break ;
    }

    msg ("I+opened %s", stdinput ? "standard input" : fn ) ;
    switch ( f->format ) {
    case  FAX: msg ( "I  as fax file" ) ; break ;
    case  PBM: msg ( "I  as %dx%d PBM file", f->w, f->h ) ; break ;
    case TEXT: msg ( "I  as %d-line/page text file with %dx%d font",
		    f->pglines, f->font->w, f->font->h ) ; break ;
    } 
  }

  if ( err ) f->lines = 0 ;

  return err ? 0 : f ;
}


/* Initialize an IFILE. */

void newIFILE ( IFILE *f, int format, char **fname )
{
  f->f = 0 ;
  f->format = format ;
  f->fname = fname ;
  f->lines = 0 ;
  f->txtlines = 0 ;
  f->font = 0 ;
  f->pglines = 0 ;
}


		/* Image File Output Functions */

/* Strings and function to write a bit map in HP-PCL format. The only
   compression is removal of trailing zeroes.  Margins and resolution are
   set before first write.  */

char *PCLBEGIN =
	 "\033E"		/* Printer reset. */
	 "\033&l0E"		/* top  margin = 0 */
	 "\033&a0L"		/* left margin = 0 */
	 "\033*t%dR"		/* Set raster graphics resolution */
	 "\033*r1A" ;		/* Start raster graphics, rel. adressing */

char *PCLEND = 
	 "\033*rB"		/* end raster graphics */
	 "\033E" ;		/* Printer reset. */

void pclwrite ( OFILE *f, unsigned char *buf, int n )
{
  while ( n > 0 && buf [ n-1 ] == 0 ) n-- ; 
  fprintf( f->f, "\033*b%dW", n ) ;
  fwrite ( buf, n, 1, f->f ) ;
}


/* Write a bit map as (raw) Portable Gray Map (PGM) format after
   decimating by a factor of 4.  Sums bits in each 4x4-pel square
   to compute sample value.  This function reduces each dimension
   of a bit map by 4 (it writes n*8/4 pixels per scan line and
   one scan line for every 4 in).  The 17 possible sample values
   are spread linearly over the range 0-255. */

void pgmwrite ( OFILE *f, u_char *buf, int n )
{
  static u_char gval [ MAXBITS * 8 / 4 ] ;
  static int init=0, lines=0 ;
  static u_char hbits [ 256 ] , lbits [ 256 ] ;
  static nybblecnt [ 16 ] = { 0,1,1,2, 1,2,2,3, 1,2,2,3, 2,3,3,4 } ;
  static u_char corr [ 17 ] = { 255, 239, 223, 207, 191, 175, 159, 143, 127, 
				111,  95,  79,  63,  47,  31,  15,   0 } ;
  int m ;
  u_char *p, *q ; 

  if ( ! init ) {	    /*  build table of bit counts in each nybble */
    short i ;
    for ( i=0 ; i<256 ; i++ ) {
	hbits [ i ] = nybblecnt [ i >> 4 & 0x0f ] ;
	lbits [ i ] = nybblecnt [ i & 0x0f ] ;
      }
    init = 1 ;
  }

  for ( m=n, p=gval, q=buf ; m-- > 0 ; q++ ) {
    *p++ += hbits [ *q ] ;
    *p++ += lbits [ *q ] ;
  }
  
  if ( ( lines++ & 0x03 ) == 0x03 ) {
    for ( p=gval, m=2*n ; m-- > 0 ; p++ ) *p = corr [ *p ] ;
    fwrite ( gval,  1, 2*n, f->f ) ;
    memset ( gval,  0, 2*n ) ;
  }
}


/* Postscript image data is differentially coded vertically and
   run-length coded horizontally.  A leading byte (n) defines the type
   of coding for subsequent data:

   0        repeat previous line
   1-127    n data bytes follow
   128-254  copy n-127 bytes from previous line
   255 n    repeat the next character 'n' times 

   The overhead for coding a copy is 2 bytes (copy count, data count),
   so copies > 2 bytes should be so coded.  The overhead for coding a
   run is 4 bytes (255, count, byte, data count), so runs > 4 bytes
   should be so coded.  Copies decode/execute faster and code more
   compactly so are preferred over runs.

*/

const char PSBEGIN [] =		/* start of file */
  "%%!PS-Adobe-2.0 EPSF-2.0 \n"
  "%%%%Creator: efax (Copyright 1995 Ed Casas) \n"
  "%%%%Title: efix output\n"
  "%%%%Pages: (atend) \n"
  "%%%%BoundingBox: 0 0 %d %d \n"
  "%%%%BeginComments \n"
  "%%%%EndComments \n"
  "/val 1 string def \n"
  "/buf %d string def   \n"
  "/getval { \n"
  "  currentfile val readhexstring pop 0 get \n"
  "} bind def \n"
  "/readbuf { \n"
  "  0  %% => index \n"
  "  { \n"
  "    dup buf length ge { exit } if \n"
  "    getval   %% => index run_length \n"
  "    dup 127 le { \n"
  "      dup 0 eq { \n"
  "        pop buf length \n"
  "      } { \n"
  "        currentfile buf 3 index 3 index getinterval readhexstring pop pop\n"
  "      } ifelse \n"
  "    } { \n"
  "      dup 255 eq { \n"
  "        pop getval getval %% => index run_length value \n"
  "        2 index 1 3 index 2 index add 1 sub  %% => ... start 1 end \n"
  "          { buf exch 2 index put } for \n"
  "        pop \n"
  "      } { \n"
  "        127 sub \n"
  "      } ifelse \n"
  "    } ifelse \n"
  "    add %% => index \n"
  "  } loop \n"
  "  pop \n"
  "  buf \n"
  "} bind def \n"
  "%%%%EndProlog \n" ;

const char PSPAGE [] =		/* start of page */
  "%%%%Page: %d %d \n"
  "gsave \n"
  "%f %f translate \n"
  "%f %f scale \n"
  "%d %d %d [ %d %d %d %d %d %d ] { readbuf } image \n" ;

const char PSPAGEEND [] =	/* end of page */
  "\n"
  "grestore \n"
  "showpage \n" ;

const char PSEND [] =		/* end of file */
  "%%Trailer \n"
  "%%%%Pages: %d \n" ;


void psinit ( OFILE *f, int newfile, int page, int w, int h, int n )
{
  float ptw, pth ;
  ptw = w/f->xres * 72.0 ;		   /* convert to points */
  pth = h/f->yres * 72.0 ;
  if ( newfile )
    fprintf ( f->f, PSBEGIN, 
	    (int) ptw, (int) pth,		 /* Bounding Box */
	    n ) ;				 /* buffer string length */
  fprintf ( f->f, PSPAGE, 
	  page, page,				 /* page number */
	  0.0, 0.0,				 /* shift */
	  ptw, pth,				 /* scaling */
	  w, h, 1,				 /* image size */
	  w, 0, 0, -h, 0, h ) ;			 /* CTM */
  f->firstpsline = newfile ;
  f->lastpageno = page ;
}



char nhexout = 0 , hexchars [ 16 ] = "0123456789abcdef" ;

#define hexputc( f, c ) ( \
        putc ( hexchars [ (c) >>   4 ] , f ), \
        putc ( hexchars [ (c) & 0x0f ] , f ), \
        ( ( ( nhexout++ & 31 ) == 31 ) ? putc ( '\n' , f ) : 0 ) )

void hexputs ( FILE *f, u_char *p , int n )
{
  u_char c ;
  if ( n > 0 ) {
    hexputc ( f, n ) ;
    while ( n-- ) { c = *p++ ^ 0xff ; hexputc ( f, c ) ; }
  }
}

/* Encode into postscript.  If not a repeated line, test (using
   index j) from current position (i) for possible encodings as:
   copy of > 2 bytes, runs of > 4 or data >=127.  Otherwise the
   byte is skipped. Uncoded bytes are output from the last
   uncoded byte (l) before output of runs/copies.  */

void pswrite ( OFILE *f, unsigned char *buf, int n )
{
  int i, j, l ;
  static unsigned char last [ MAXBITS ] ;
  
  l=i=0 ;

  for ( j=0 ; j<n && buf[j]==last[j] && ! f->firstpsline ; j++ ) ;
  if ( j == n ) {		/* repeat line */
    hexputc ( f->f, 0 ) ;
    l=i=n ;
  }

  while ( i<n ) {

    for ( j=i ; j<n && buf[j]==last[j] && j-i<127 && ! f->firstpsline ; j++ ) ;
    if ( j-i > 2 ) {		/* skip */
      hexputs ( f->f, buf+l , i-l ) ;
      hexputc ( f->f, j-i + 127 ) ; 
      l=i=j ;
    } else {
      for ( j=i ; j<n && buf[j]==buf[i] && j-i<255 ; j++ ) ;
      if ( j-i > 4 ) {		/* run */
	hexputs ( f->f, buf+l , i-l ) ;
	hexputc ( f->f, 255 ) ; 
	hexputc ( f->f, j-i ) ; 
	hexputc ( f->f, buf[i] ^ 0xff ) ;
	l=i=j ;
      } else {
	if ( i-l >= 127 ) {	/* maximum data length */
	  hexputs ( f->f, buf+l, i-l ) ;
	  l=i ;
	} else {		/* data */
	  i++ ;
	}
      }
    }

  }
  hexputs ( f->f, buf+l, i-l ) ;

  if ( n >= 0 ) 
    memcpy ( last, buf, n ) ;

  f->firstpsline = 0 ;
}


/* Convert array 'runs' of 'nr' run lengths into a bit map 'buf'. Returns
   the number of bytes filled. */

int runtobit ( short *runs, int nr, u_char *buf )
{
  static u_char zerofill [ 9 ] = { 
    0xff,  0xfe, 0xfc, 0xf8, 0xf0,  0xe0, 0xc0, 0x80, 0x00 } ;
  static u_char onefill [ 9 ] = { 
    0x00,  0x01, 0x03, 0x07, 0x0f,  0x1f, 0x3f, 0x7f, 0xff } ; 

  u_char col=0, *buf0 = buf ;
  register short len, b=8, bytes ;
  
  while ( nr-- > 0 ) {
    len = *runs++ ;
    if ( col ) *buf |= onefill  [ b ] ;		 /* right bits of cur. byte */
    else       *buf &= zerofill [ b ] ;
    if ( b > len ) {				 /* done fill */
      b -= len ;
    } else {					 /* continue to next byte */
      len -= b ; 
      buf++ ; 
      b = 8 ;
      if ( ( bytes = len>>3 ) > 0 ) {		 /* fill >1 byte */
	memset ( buf, col, bytes ) ;
	len -= bytes*8; 
	buf += bytes ;
      } 
      *buf = col ;				 /* flood the rest */
      b -= len ;
    }
    col ^= 0xff ;
  }

  return buf - buf0 + ( b < 8 ) ;
}


/* Begin/end output pages.  If not starting first page (0), terminate
   previous page.  If output filename pattern is defined, [re-]opens that
   file.  If not terminating last page (page==-1), writes file header.
   Returns 0 or 2 on errors. */

int nextopage ( OFILE *f, int page )
{
  int err = 0 ;
  int i ;
  u_char *p, codes [ ( RTCEOL * EOLBITS ) / 8 + 3 ] ;
  
  if ( page )				    /* terminate previous page */
    switch ( f->format ) {
    case PBM:
      break ;
    case PGM:
      break ;
    case FAX:
      for ( p = codes, i=0 ; i<RTCEOL ; i++ ) 
	p = putcode ( &f->e, EOLCODE, EOLBITS, p ) ;
      fwrite ( codes, 1, putcode ( &f->e, 0, 0, p ) - codes, f->f ) ;
      break ;
    case PCL:
      fprintf ( f->f, PCLEND ) ;
      break ;
    case PS:
      fprintf ( f->f, PSPAGEEND ) ;
      if ( f->fname || page<0 ) fprintf ( f->f, PSEND, f->lastpageno ) ;
      break ;
    }

  if ( ferror ( f->f ) ) err = msg ("ES2output error:" ) ;

  if ( ! err && f->fname && page >= 0 ) {	    /* start new file */
    sprintf ( f->cfname, f->fname, page+1, page+1, page+1 ) ;
    if ( ! freopen ( f->cfname, ( f->format == PS ) ? "w" : "wb", f->f )  ) 
      err = msg ("ES2unable to open output file\n%s:", f->cfname ) ;
  }    

  if ( ! err && page >= 0 ) {		    /* start new page */
    switch ( f->format ) {
    case  PBM:
      fprintf ( f->f, "P4 %d %d\n", f->w , f->h ) ;
      break ;
    case  PGM:
      fprintf ( f->f, "P5 %d %d %d\n", f->w/4, f->h/4, 255 ) ;
      break ;
    case FAX:
      p = putcode ( &f->e, EOLCODE, EOLBITS, codes ) ;
      fwrite ( codes, 1, p - codes, f->f ) ;
      break ;
    case PCL:
      fprintf ( f->f, PCLBEGIN, (int) f->xres ) ;
      break ;
    case PS:
      psinit ( f, ( f->fname || page==0 ), page+1, f->w, f->h, f->w/8 ) ;
      break ;
    }
  }
  
  if ( ! err && ferror ( f->f ) ) err = msg ("ES2output error:" ) ;

  return err ;
}


/* Output scan line of nr runs no times to output file f. */

void writeline ( OFILE *f, short *runs, int nr, int no )
{
  int nb = 0 ;
  u_char *p, buf [ MAXCODES ] ;

  /* if line to be output, convert to right format */

  if ( no > 0 )
    switch ( f->format ) {
    case PBM:
    case PGM:
    case PCL:
    case PS:
      nb = runtobit ( runs, nr, buf ) ;
      break ;
    case FAX:
      break ;
    }
  
  /* output `no' times. */
    
  while ( no-- > 0 )
    switch ( f->format ) {
    case PBM:
      fwrite ( buf, 1, nb, f->f ) ;
      break ;
    case PGM:
      pgmwrite ( f, buf, nb ) ;
      break ;
    case FAX:
      p = runtocode ( &f->e, runs, nr, buf ) ;
      p = putcode ( &f->e, EOLCODE, EOLBITS, p ) ;
      fwrite ( buf, 1, p - buf, f->f ) ;
      break ;
    case PCL:
      pclwrite ( f, buf, nb ) ;
      break ;
    case PS:
      pswrite ( f, buf, nb ) ;
      break ;
    }
}


/* Initialize new output file. If fname is NULL, stdout will be used for
   all images. */

void newOFILE ( OFILE *f, int format, char *fname, 
	       float xres, float yres, int w, int h )
{
  f->f = stdout ;
  f->format = format ;
  f->fname = fname ;
  f->xres = xres ;
  f->yres = yres ;
  f->w = w ;
  f->h = h ;
  newENCODER ( &(f->e) ) ;
}

		   /* T.4 Encoding/Decoding */

/* Table-lookup decoder for variable-bit-length codewords.  The table index
   is the N most recently undecoded bits with the first (oldest) undecoded
   bit as the MS bit.  If the N bits uniquely identify a codeword then the
   indexed 'code' member identifies the code, otherwise it is zero.  The
   'bits' member gives the number of bits to be considered decoded (to be
   removed from the bit stream) and the 'next' element is a pointer to the
   table to use for decoding the next part of the bit sequence.

   For T.4 decoding the longest T.4 codeword is 13 bits. The implementation
   below uses two tables of 512 elements (N=9 bits) for each colour.
   Codewords longer than 9 bits require a second lookup.  Since all
   codewords longer than than 9 bits have a 4-bit zero prefix it is
   possible to use only one secondary 9-bit lookup table by dropping only
   the first 4 bits after the first lookup. The code indentifier is the run
   length + 1. A separate table is used for decoding the variable-length
   FILL patterns.

   For undefined codewords, one bit is skipped and decoding continues at
   the white code table. */

/* the lookup tables for each colour and the fill lookup table */

dtab tw1 [ 512 ], tw2 [ 512 ], tb1 [ 512 ], tb2 [ 512 ], fill [ 512 ] ;
char tinit=0 ;

/* Add code cword shifted left by shift to decoding table tab. */

void addcode ( dtab *tab, int cword, int shift, 
	      short code, short bits, dtab *next )
{
  int i, n = 1 << shift ;
  
  for ( i = cword << shift ; n-- > 0 ; i++ ) {
    tab[i].code = code ;
    tab[i].bits = bits ;
    tab[i].next = next ;
  }
}

/* Initialize the decoding table for one colour using the codes in the T.4
   table p0.  t1 and t2 are the two decoding tables and ot is the first
   table of the other colour. */

void init1dtab ( t4tab *p0, dtab *t1, dtab *t2, dtab *ot )
{
  t4tab *p ;
  for ( p = p0 ; p->code ; p++ ) 
    if ( p->bits <= 9 ) {
      addcode ( t1, p->code, 9 - p->bits, p->rlen + 1, p->bits,   
	       ( p - p0 ) > 63 ? t1 : ot ) ;
    } else {
      addcode ( t1, p->code >> ( p->bits - 9 ), 0, 0, 4, t2 ) ;
      addcode ( t2, p->code, 13 - p->bits, p->rlen + 1, p->bits - 4, 
	       ( p - p0 ) > 63 ? t1 : ot ) ;
    }
}


/* Initialize a T.4 decoder.  First sets all  */

void newDECODER ( DECODER *d )
{
  int i ;

  if ( ! tinit ) {

    /* undefined codes */

    addcode ( tw1,  0, 9, 0, 1, tw1 ) ;
    addcode ( tw2,  0, 9, 0, 1, tw1 ) ;
    addcode ( tb1,  0, 9, 0, 1, tw1 ) ;
    addcode ( tb2,  0, 9, 0, 1, tw1 ) ;
    addcode ( fill, 0, 9, 0, 1, tw1 ) ;
  
    /* fill and EOL */

    addcode ( tw1, 0, 0, 0, 4, tw2 ) ;
    addcode ( tw2, 0, 2, 0, 7, fill ) ;
    addcode ( tb1, 0, 0, 0, 4, tb2 ) ;
    addcode ( tb2, 0, 2, 0, 7, fill ) ;

    addcode ( fill, 0, 0, 0, 9, fill ) ;
    for ( i=0 ; i<=8 ; i++ )
      addcode ( fill, 1, i, -1, 9-i, tw1 ) ;


    /* white and black runs */
    
    init1dtab ( wtab, tw1, tw2, tb1 ) ;
    init1dtab ( btab, tb1, tb2, tw1 ) ;

    tinit=1 ;
  }

  /* initialize decoder to starting state */

  d->x = 0 ;
  d->shift = -9 ;
  d->tab = tw1 ;
  d->eolcnt = 0 ;
}

      /* T.4 coding table and default font for efax/efix */

/* T.4 1-D run-length coding tables. codes must be in run length
   order for runtocode(). */

t4tab wtab [ ( 64 + 27 + 13 ) + 1 ] = {	/* runs of white */

/*  Terminating White Codes */

{53,8,0},    {7,6,1},     {7,4,2},     {8,4,3},     {11,4,4},    {12,4,5},
{14,4,6},    {15,4,7},    {19,5,8},    {20,5,9},    {7,5,10},    {8,5,11},
{8,6,12},    {3,6,13},    {52,6,14},   {53,6,15},   {42,6,16},   {43,6,17},
{39,7,18},   {12,7,19},   {8,7,20},    {23,7,21},   {3,7,22},    {4,7,23},
{40,7,24},   {43,7,25},   {19,7,26},   {36,7,27},   {24,7,28},   {2,8,29},
{3,8,30},    {26,8,31},   {27,8,32},   {18,8,33},   {19,8,34},   {20,8,35},
{21,8,36},   {22,8,37},   {23,8,38},   {40,8,39},   {41,8,40},   {42,8,41},
{43,8,42},   {44,8,43},   {45,8,44},   {4,8,45},    {5,8,46},    {10,8,47},
{11,8,48},   {82,8,49},   {83,8,50},   {84,8,51},   {85,8,52},   {36,8,53},
{37,8,54},   {88,8,55},   {89,8,56},   {90,8,57},   {91,8,58},   {74,8,59},
{75,8,60},   {50,8,61},   {51,8,62},   {52,8,63},   

/*  Make Up White Codes */

{27,5,64},   {18,5,128},  {23,6,192},  {55,7,256},  {54,8,320},  {55,8,384},
{100,8,448}, {101,8,512}, {104,8,576}, {103,8,640}, {204,9,704}, {205,9,768},
{210,9,832}, {211,9,896}, {212,9,960}, {213,9,1024},{214,9,1088},{215,9,1152},
{216,9,1216},{217,9,1280},{218,9,1344},{219,9,1408},{152,9,1472},{153,9,1536},
{154,9,1600},{24,6,1664}, {155,9,1728},

/*  Extended Make Up Codes (Black and White) */

{8,11,1792}, {12,11,1856},{13,11,1920},{18,12,1984},{19,12,2048},{20,12,2112},
{21,12,2176},{22,12,2240},{23,12,2304},{28,12,2368},{29,12,2432},{30,12,2496},
{31,12,2560},

{0,0,0}  } ;

t4tab btab [ ( 64 + 27 + 13 ) + 1 ] = {	/* runs of black */

/*  Terminating Black Codes */

{55,10,0},   {2,3,1},     {3,2,2},     {2,2,3},     {3,3,4},     {3,4,5},
{2,4,6},     {3,5,7},     {5,6,8},     {4,6,9},     {4,7,10},    {5,7,11},
{7,7,12},    {4,8,13},    {7,8,14},    {24,9,15},   {23,10,16},  {24,10,17},
{8,10,18},   {103,11,19}, {104,11,20}, {108,11,21}, {55,11,22},  {40,11,23},
{23,11,24},  {24,11,25},  {202,12,26}, {203,12,27}, {204,12,28}, {205,12,29},
{104,12,30}, {105,12,31}, {106,12,32}, {107,12,33}, {210,12,34}, {211,12,35},
{212,12,36}, {213,12,37}, {214,12,38}, {215,12,39}, {108,12,40}, {109,12,41},
{218,12,42}, {219,12,43}, {84,12,44},  {85,12,45},  {86,12,46},  {87,12,47},
{100,12,48}, {101,12,49}, {82,12,50},  {83,12,51},  {36,12,52},  {55,12,53},
{56,12,54},  {39,12,55},  {40,12,56},  {88,12,57},  {89,12,58},  {43,12,59},
{44,12,60},  {90,12,61},  {102,12,62}, {103,12,63}, 

/*  Make Up Black Codes */

{15,10,64},  {200,12,128},{201,12,192},{91,12,256}, {51,12,320}, {52,12,384},
{53,12,448}, {108,13,512},{109,13,576},{74,13,640}, {75,13,704}, {76,13,768},
{77,13,832}, {114,13,896},{115,13,960},{116,13,1024},{117,13,1088},
{118,13,1152},
{119,13,1216},{82,13,1280},{83,13,1344},{84,13,1408},{85,13,1472},{90,13,1536},
{91,13,1600},{100,13,1664},{101,13,1728},

/*  Extended Make Up Codes (Black and White) */

{8,11,1792}, {12,11,1856},{13,11,1920},{18,12,1984},{19,12,2048},{20,12,2112},
{21,12,2176},{22,12,2240},{23,12,2304},{28,12,2368},{29,12,2432},{30,12,2496},
{31,12,2560},

{0,0,0}  } ;


/* The built-in 8x16 font.  Runs of zeroes are coded as 0
   followed by the repetition count. */

u_char stdfont [ 1980 ] = {
0,255,0,255,0,194,8,4,12,10,18,0,3,16,4,8,20,8,4,8,20,0,1,10,8,4,
4,10,18,0,2,16,4,8,20,4,0,68,20,0,1,8,0,2,12,6,48,0,5,2,0,43,14,32,
56,0,2,12,0,1,32,0,1,2,0,1,14,0,1,32,8,4,32,56,0,14,6,8,48,0,40,8,
0,1,18,0,6,30,0,4,4,0,11,4,8,18,20,18,12,0,2,8,8,20,20,4,8,20,20,
0,1,20,4,8,10,20,18,0,2,8,8,20,20,8,0,1,24,8,4,8,10,20,12,0,2,8,4,
8,20,16,8,8,20,54,10,8,4,8,10,20,0,2,16,4,8,20,4,0,1,20,0,33,12,20,
18,28,48,12,12,8,8,8,0,4,2,28,8,28,28,4,62,28,62,28,28,0,5,60,28,
12,60,14,56,62,30,14,34,62,62,33,16,33,34,12,60,12,60,30,127,34,33,
65,34,34,62,8,32,8,8,0,1,24,0,1,32,0,1,2,0,1,16,0,1,32,8,4,32,8,0,
7,16,0,6,8,8,8,0,36,4,14,0,1,34,8,12,18,28,24,0,3,28,0,1,24,0,1,28,
28,8,0,1,30,0,2,8,28,0,1,100,100,98,0,6,18,31,14,0,8,56,0,7,13,0,
5,32,36,4,8,20,20,20,18,0,2,4,8,20,20,8,16,20,20,8,20,4,8,20,20,20,
0,2,8,8,20,20,8,32,20,0,33,12,20,18,42,73,18,24,8,8,42,8,0,3,4,34,
24,34,34,12,32,34,2,34,34,0,2,2,0,1,16,2,34,12,34,18,36,32,16,18,
34,8,8,34,16,51,50,18,34,18,34,32,8,34,33,73,34,34,2,8,16,8,8,0,1,
24,0,1,32,0,1,2,0,1,16,0,1,32,0,2,32,8,0,7,16,0,6,8,8,8,0,36,15,16,
65,34,8,18,0,1,34,4,0,3,34,0,1,36,8,2,2,0,2,58,0,2,56,34,0,1,36,36,
18,0,1,12,12,12,12,12,12,24,18,62,62,62,62,62,62,62,62,36,34,12,12,
12,12,12,0,1,18,34,34,34,34,34,32,36,0,5,12,0,10,52,0,6,8,0,6,32,
0,34,12,0,1,63,40,74,18,0,1,16,4,20,8,0,3,4,34,40,2,2,20,32,32,2,
34,34,24,24,4,0,1,8,2,78,18,34,32,34,32,16,32,34,8,8,36,16,51,50,
33,34,33,34,32,8,34,33,73,20,34,4,8,16,8,20,0,2,28,44,14,30,28,62,
30,44,56,60,34,8,82,44,28,44,30,22,30,62,34,34,65,34,34,62,8,8,8,
0,35,12,20,16,62,34,8,16,0,1,77,4,0,3,93,0,1,24,8,2,12,0,1,34,58,
0,2,8,34,0,1,40,40,100,4,12,12,12,12,12,12,40,32,32,32,32,32,8,8,
8,8,34,50,18,18,18,18,18,34,35,34,34,34,34,34,60,40,28,28,28,28,28,
28,54,14,28,28,28,28,56,56,56,56,2,44,28,28,28,28,28,8,29,34,34,34,
34,34,44,34,0,33,12,0,1,18,24,52,12,0,1,16,4,42,8,0,3,8,34,8,2,2,
36,60,32,4,34,34,24,24,8,127,4,2,82,18,34,32,34,32,16,32,34,8,8,40,
16,45,42,33,34,33,34,48,8,34,33,73,20,20,4,8,8,8,20,0,2,34,50,16,
34,34,16,34,50,8,4,36,8,109,50,34,50,34,24,32,16,34,34,73,34,34,2,
4,8,16,57,0,34,12,36,16,34,20,0,1,40,0,1,81,28,18,127,0,1,89,0,2,
127,12,2,0,1,34,58,28,0,1,8,34,36,40,40,24,4,18,18,18,18,18,18,40,
32,32,32,32,32,8,8,8,8,34,50,33,33,33,33,33,20,37,34,34,34,34,20,
34,40,34,34,34,34,34,34,9,16,34,34,34,34,8,8,8,8,30,50,34,34,34,34,
34,0,1,34,34,34,34,34,34,50,34,0,33,12,0,1,18,12,8,25,0,1,16,4,8,
127,0,1,127,0,1,8,34,8,4,12,68,2,60,8,28,30,0,2,16,0,1,2,28,82,18,
60,32,34,60,30,32,62,8,8,56,16,45,42,33,34,33,60,28,8,34,18,85,8,
20,8,8,8,8,34,0,2,2,34,32,34,34,16,34,34,8,4,40,8,73,34,34,34,34,
16,32,16,34,34,73,20,34,4,24,8,12,78,0,35,36,60,34,62,0,1,36,0,1,
81,36,36,1,28,85,0,2,8,16,2,0,1,34,26,28,0,1,8,34,18,18,22,106,0,
1,18,18,18,18,18,18,47,32,60,60,60,60,8,8,8,8,122,42,33,33,33,33,
33,8,45,34,34,34,34,20,34,36,2,2,2,2,2,2,9,32,34,34,34,34,8,8,8,8,
34,34,34,34,34,34,34,127,38,34,34,34,34,34,34,34,0,33,8,0,1,63,10,
22,37,0,1,16,4,0,1,8,0,3,8,34,8,8,2,126,2,34,8,34,2,0,2,8,127,4,16,
86,63,34,32,34,32,16,34,34,8,8,36,16,45,38,33,60,33,36,6,8,34,18,
54,20,8,16,8,8,8,34,0,2,30,34,32,34,62,16,34,34,8,4,56,8,73,34,34,
34,34,16,28,16,34,20,85,8,20,8,4,8,16,0,35,8,36,16,34,8,8,18,0,1,
81,26,72,1,0,1,34,0,2,8,30,28,0,1,34,10,28,0,1,8,28,9,22,17,22,4,
63,63,63,63,63,63,120,32,32,32,32,32,8,8,8,8,34,42,33,33,33,33,33,
20,41,34,34,34,34,8,34,34,30,30,30,30,30,30,63,32,62,62,62,62,8,8,
8,8,34,34,34,34,34,34,34,0,1,42,34,34,34,34,20,34,20,0,35,18,10,41,
34,0,1,16,4,0,1,8,0,3,16,34,8,16,2,4,2,34,16,34,2,0,2,4,0,1,8,0,1,
73,33,34,32,34,32,16,34,34,8,8,34,16,33,38,33,32,33,34,2,8,34,18,
34,20,8,16,8,4,8,0,3,34,34,32,34,32,16,38,34,8,4,36,8,73,34,34,34,
34,16,2,16,34,20,34,20,20,16,8,8,8,0,35,12,20,16,62,62,8,10,0,1,77,
0,1,36,1,0,1,28,0,6,34,10,0,4,18,42,34,42,28,33,33,33,33,33,33,72,
32,32,32,32,32,8,8,8,8,34,38,33,33,33,33,33,34,49,34,34,34,34,8,60,
34,34,34,34,34,34,34,72,32,32,32,32,32,8,8,8,8,34,34,34,34,34,34,
34,8,50,34,34,34,34,20,34,20,0,33,12,0,1,18,42,73,34,0,1,8,8,0,1,
8,12,0,1,24,16,34,8,32,34,4,34,34,16,34,34,24,24,2,0,1,16,16,32,33,
34,16,36,32,16,18,34,8,8,33,16,33,34,18,32,18,34,2,8,34,12,34,34,
8,32,8,4,8,0,3,34,34,16,38,34,16,26,34,8,4,34,8,73,34,34,34,38,16,
2,16,38,8,34,34,8,32,8,8,8,0,35,12,15,16,65,8,8,4,0,1,34,0,1,18,0,
5,127,0,3,54,10,0,4,36,79,68,79,32,33,33,33,33,33,33,72,16,32,32,
32,32,8,8,8,8,36,38,18,18,18,18,18,0,1,18,34,34,34,34,8,32,34,34,
34,34,34,34,34,72,16,34,34,34,34,8,8,8,8,34,34,34,34,34,34,34,8,34,
38,38,38,38,8,34,8,0,33,12,0,1,18,28,6,29,0,1,8,8,0,2,12,0,1,24,32,
28,8,62,28,4,28,28,16,28,28,24,24,0,3,16,28,33,60,14,56,62,16,14,
34,62,112,33,30,33,34,12,32,12,34,60,8,28,12,34,34,8,62,8,2,8,0,3,
29,60,14,26,28,16,2,34,8,4,33,8,73,34,28,60,26,16,60,14,26,8,34,34,
8,62,8,8,8,0,35,12,4,62,0,1,8,8,36,0,1,28,0,11,42,10,0,5,66,71,66,
32,33,33,33,33,33,33,79,14,62,62,62,62,62,62,62,62,56,34,12,12,12,
12,12,0,1,44,28,28,28,28,8,32,36,29,29,29,29,29,29,55,14,28,28,28,
28,8,8,8,8,28,34,28,28,28,28,28,0,1,92,26,26,26,26,8,60,8,0,36,8,
0,3,6,48,0,2,24,0,2,32,0,11,48,0,21,6,0,9,14,2,56,0,1,127,0,7,2,0,
2,4,0,5,32,2,0,7,16,0,1,6,8,48,0,35,12,0,4,8,24,0,13,32,10,0,1,4,
0,6,32,0,7,4,0,31,4,0,21,16,32,16,0,81,3,0,21,28,0,2,56,0,5,32,2,
0,7,48,0,39,12,0,19,32,0,2,24,0,6,30,0,7,24,0,31,24,0,21,48,32,48,
0,255,0,1
} ;
