/***********************************************************************/
/* COMM5.C - Commands T-Z                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1997 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * PO Box 203                    Phone:                    +617 3802 0800
 * Bellara                       http://www.gu.edu.au/gext/the/markh.html
 * QLD 4507                      **** Maintainer PDCurses & REXX/SQL ****
 * Australia                     ************* Author of THE ************
 */

/*
$Id: comm5.c 2.1 1995/06/24 16:28:59 MH Rel MH $
*/

#include <the.h>
#include <proto.h>

/*#define DEBUG 1*/
/*man-start*********************************************************************
COMMAND
     tabpre - switch between FILEAREA and PREFIX area

SYNTAX
     tabpre

DESCRIPTION
     The TABPRE command switches the focus of the editor from the
     <filearea> to the <prefix area> and vice versa, depending
     on which window is currently active.

     This command can only be used by assigning it to a function key.

     This command will be removed in a future version.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent of <SOS LEFTEDGE> and <SOS PREFIX>

SEE ALSO
     <SOS LEFTEDGE>, <SOS PREFIX>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Tabpre(CHARTYPE *params)
#else
short Tabpre(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Tabpre");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp((DEFCHAR *)params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the cursor is in the command line or there is no prefix on, exit.*/
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND
 ||  !CURRENT_VIEW->prefix)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 if (CURRENT_VIEW->current_window == WINDOW_FILEAREA)
    rc = Sos_prefix((CHARTYPE *)"");
 else
    rc = Sos_leftedge((CHARTYPE *)"");
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     text - simulate keyboard entry of characters

SYNTAX
     TEXT text

DESCRIPTION
     The TEXT command simulates the entry of 'text' from the
     keyboard. This command is actually called when you enter text
     from the keyboard.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.
            Does not allow trailing spaces in text.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Text(CHARTYPE *params)
#else
short Text(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern bool INSERTMODEx;
 extern bool prefix_changed;
 extern CHARTYPE *pre_rec;
 extern unsigned short pre_rec_len;
 extern bool readonly;
 extern VIEW_DETAILS *vd_mark;
 extern bool in_macro;
 extern bool clear_command;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 CHARTYPE real_key=0;
 chtype chtype_key=0;
 LENGTHTYPE x=0;
 unsigned short y=0;
 short len_params=0;
 short rc=RC_OK;
 LENGTHTYPE offl=0,offr=0;
#if defined(USE_EXTCURSES)
 ATTR attr=0;
#else
 chtype attr=0;
#endif
 bool need_to_build_screen=FALSE;
 bool save_in_macro=in_macro;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Text");
#endif
/*---------------------------------------------------------------------*/
/* If running in read-only mode, do not allow any text to be entered   */
/* in the main window.                                                 */
/*---------------------------------------------------------------------*/
 if (readonly && CURRENT_VIEW->current_window == WINDOW_FILEAREA)
   {
    display_error(56,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* If HEX mode is on, convert the hex string...                        */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->hex)
   {
    if ((len_params = convert_hex_strings(params)) == (-1))
      {
       display_error(32,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   len_params = strlen((DEFCHAR *)params);
 for (i=0;i<len_params;i++)
   {
    real_key = case_translate((CHARTYPE)*(params+i));
    chtype_key = (chtype)(real_key & A_CHARTEXT);
    getyx(CURRENT_WINDOW,y,x);
#if defined(USE_EXTCURSES)
    attr = CURRENT_WINDOW->_a[y][x];
    wattrset(CURRENT_WINDOW,attr);
    attr = 0;
#else
    attr = winch(CURRENT_WINDOW) & A_ATTRIBUTES;
#endif
    switch(CURRENT_VIEW->current_window)
      {
       case WINDOW_FILEAREA:
            if (CURRENT_SCREEN.sl[y].line_type != LINE_LINE)
               break;
            if ((LENGTHTYPE)(x+CURRENT_VIEW->verify_start) > (LENGTHTYPE)(CURRENT_VIEW->verify_end))
               break;
            if (INSERTMODEx)
              {
               rec = meminschr(rec,real_key,CURRENT_VIEW->verify_col-1+x,
                               max_line_length,rec_len);
               put_char(CURRENT_WINDOW,chtype_key|attr,INSCHAR);
              }
            else
              {
               rec[CURRENT_VIEW->verify_col-1+x] = real_key;
               if (x == CURRENT_SCREEN.cols[WINDOW_FILEAREA]-1)
                  put_char(CURRENT_WINDOW,chtype_key|attr,INSCHAR);
               else
                  put_char(CURRENT_WINDOW,chtype_key|attr,ADDCHAR);
              }
            rc = memrevne(rec,' ',max_line_length);
            if (rc == (-1))
               rec_len = 0;
            else
               rec_len = rc+1;
            /* check for the cursor moving past the right   */
            /* margin when WORDWRAP is ON. If true, then    */
            /* don't execute the cursor_right() function, as */
            /* this could cause a window scroll.            */
            if (CURRENT_VIEW->wordwrap
            &&  rec_len > CURRENT_VIEW->margin_right)
               execute_wrap_word(x+CURRENT_VIEW->verify_col);
            else    
               {
               /* this is done here so that the show_page() in */
               /* cursor_right() is executed AFTER we get the   */
               /* new length of rec_len.                       */
#if defined(USE_EXTCURSES)
               if (x == CURRENT_SCREEN.cols[WINDOW_FILEAREA]-1)
                 {
                  wmove(CURRENT_WINDOW,y,x);
     /*           wrefresh(CURRENT_WINDOW); */
                  cursor_right(TRUE,FALSE);
                 }
#else
               if (INSERTMODEx
               || x == CURRENT_SCREEN.cols[WINDOW_FILEAREA]-1)
                  cursor_right(TRUE,FALSE);
#endif
               }
/*---------------------------------------------------------------------*/
/* If HEXSHOW is on and we are on the current line, build screen...    */
/*---------------------------------------------------------------------*/
            if (CURRENT_VIEW->hexshow_on
            &&  CURRENT_VIEW->focus_line == CURRENT_VIEW->current_line)
               need_to_build_screen = TRUE;
            break;
       case WINDOW_COMMAND:
            if (INSERTMODEx)
              {
               cmd_rec = (CHARTYPE *)meminschr((CHARTYPE *)cmd_rec,
                               real_key,x,
                               COLS,cmd_rec_len);
               put_char(CURRENT_WINDOW,chtype_key,INSCHAR);
#if !defined(USE_EXTCURSES)
               cursor_right(TRUE,FALSE);
#endif
#ifndef OLD_CMD
               cmd_rec_len++;
#endif
              }
            else
              {
               cmd_rec[x] = real_key;
               put_char(CURRENT_WINDOW,chtype_key,ADDCHAR);
#ifndef OLD_CMD
               cmd_rec_len = max(x+1,cmd_rec_len);
#endif
              }
#ifdef OLD_CMD
            rc = memrevne(cmd_rec,' ',COLS);
            if (rc == (-1))
               cmd_rec_len = 0;
            else
               cmd_rec_len = rc+1;
#endif            
            /*
             * Ensure that the command line is not cleared if text is
             * entered here
             */
            clear_command = FALSE;
            break;
       case WINDOW_PREFIX:
            prefix_changed = TRUE;
            if (pre_rec_len == 0)
              {
               x = 0;
               wmove(CURRENT_WINDOW,y,x);
               my_wclrtoeol(CURRENT_WINDOW);
               wrefresh(CURRENT_WINDOW);
              }
            if (INSERTMODEx)
              {
               if (pre_rec_len == (CURRENT_VIEW->prefix_width-CURRENT_VIEW->prefix_gap))
                  break;
               pre_rec = (CHARTYPE *)meminschr((CHARTYPE *)pre_rec,
                               real_key,x,CURRENT_VIEW->prefix_width-CURRENT_VIEW->prefix_gap,pre_rec_len);
               put_char(CURRENT_WINDOW,chtype_key,INSCHAR);
              }
            else
              {
               pre_rec[x] = real_key;
               put_char(CURRENT_WINDOW,chtype_key,ADDCHAR);
              }
            wmove(CURRENT_WINDOW,y,min(x+1,CURRENT_VIEW->prefix_width-CURRENT_VIEW->prefix_gap-1));
            rc = memrevne(pre_rec,' ',CURRENT_VIEW->prefix_width);
            if (rc == (-1))
               pre_rec_len = 0;
            else
               pre_rec_len = rc+1;
            break;
      }
   }
 if (in_macro)
    need_to_build_screen = TRUE;
/*---------------------------------------------------------------------*/
/* Set in_macro to FALSE to allow for the refreshing of the screen.    */
/*---------------------------------------------------------------------*/
 in_macro = FALSE;
/*---------------------------------------------------------------------*/
/* If text is being inserted on a line which is in the marked block,   */
/* build and redisplay the window.                                     */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW == MARK_VIEW
 &&  CURRENT_VIEW->current_window == WINDOW_FILEAREA
 &&  INSERTMODEx
 &&  CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line
 &&  CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
     need_to_build_screen = TRUE;
 if (need_to_build_screen)
   {
    build_screen(current_screen);
    display_screen(current_screen);
   }
/*---------------------------------------------------------------------*/
/* Set in_macro back to its original value...                          */
/*---------------------------------------------------------------------*/
 in_macro = save_in_macro;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     the - edit another file or switch to next file

SYNTAX
     THE [filename]

DESCRIPTION
     The THE command allows the user to edit another 'file'. The new file
     is placed in the file <ring>. The previous file being edited remains
     in memory and can be returned to by issuing a THE command without
     any parameters. Several files can be edited at once, and all files
     are arranged in a ring, with subsequent THE commands moving through
     the ring, one file at a time.

COMPATIBILITY
     XEDIT: Does not provide options switches.
     KEDIT: Does not provide options switches.

SEE ALSO
     <XEDIT>, <EDIT>

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     top - move to the top of the file

SYNTAX
     TOP

DESCRIPTION
     The TOP command moves to the very start of the current file.
     The <Top-of-File line> is set to the <current line>.

     TOP is equivalent to <BACKWARD> *.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     <BACKWARD>, <BOTTOM>

STATUS
     Complete
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Top(CHARTYPE *params)
#else
short Top(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_TOF_EOF_REACHED;
 unsigned short x=0,y=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Top");
#endif
/*---------------------------------------------------------------------*/
/* No arguments are allowed; error if any are present.                 */
/*---------------------------------------------------------------------*/
 if (strcmp((DEFCHAR *)params,"") != 0)
   {
    display_error(1,(CHARTYPE *)params,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->current_line = 0L;
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL,TRUE);
 build_screen(current_screen); 
 if (!line_in_view(current_screen,CURRENT_VIEW->focus_line))
    CURRENT_VIEW->focus_line = 0L;
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 if (curses_started)
   {
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       getyx(CURRENT_PREV_WINDOW,y,x);
    else
       getyx(CURRENT_WINDOW,y,x);
    display_screen(current_screen);
    y = get_row_for_focus_line(current_screen,CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_row);
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
       wmove(CURRENT_PREV_WINDOW,y,x);
    else
       wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     up - move backward in the file a number of lines

SYNTAX
     Up [relative target]

DESCRIPTION
     The UP command moves the <current line> backwards the number of
     lines specified by the <relative target>. This <relative target> can
     only be a positive integer or the character "*". 

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     1

SEE ALSO
     <NEXT>, <DOWN>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Up(CHARTYPE *params)
#else
short Up(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 LINETYPE num_lines=0L,true_line=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Up");
#endif
 params = MyStrip(params,STRIP_BOTH,' ');
 if (strcmp("",(DEFCHAR *)params) == 0)
    params = (CHARTYPE *)"1";
 true_line = get_true_line(TRUE);
 if (strcmp("*",(DEFCHAR *)params) == 0)
    num_lines = true_line + 1L;
 else
   {
    if (!valid_integer(params))
      {
       display_error(4,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    num_lines = atol((DEFCHAR *)params);
    if (num_lines < 0L)
      {
       display_error(5,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 rc = advance_current_or_focus_line(-num_lines);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     uppercase - change lowercase characters to uppercase

SYNTAX
     UPPercase [target]

DESCRIPTION
     The UPPERCASE command changes all lowercase characters in all
     lines up to the <'target'> line to uppercase. All other characters
     remain untouched.

COMPATIBILITY
     XEDIT: Equivalent of UPPERCAS
     KEDIT: Compatible.

SEE ALSO
     <LOWERCASE>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Uppercase(CHARTYPE *params)
#else
short Uppercase(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Uppercase");
#endif
 rc = execute_change_case(params,CASE_UPPER);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     xedit - edit another file or switch to next file

SYNTAX
     Xedit [file]

DESCRIPTION
     The XEDIT command allows the user to edit another 'file'. The new file
     is placed in the file <ring>. The previous file being edited remains
     in memory and can be returned to by issuing an XEDIT command without
     any parameters. Several files can be edited at once, and all files
     are arranged in a ring, with subsequent XEDIT commands moving through
     the ring, one file at a time.

COMPATIBILITY
     XEDIT: Does not provide options switches.
     KEDIT: Does not provide options switches.

SEE ALSO
     <EDIT>, <THE>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Xedit(CHARTYPE *params)
#else
short Xedit(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern bool REPROFILEx;
 extern CHARTYPE number_of_files;
 extern CHARTYPE *prf_arg;
 extern CHARTYPE *local_prf;
 extern short file_start;
 extern bool curses_started;
 extern bool execute_profile;
 extern bool in_profile;
 extern bool in_reprofile;
 extern int profile_file_executions;
 extern CHARTYPE display_screens;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK,y=0,x=0;
 VIEW_DETAILS *save_current_view=NULL;
 VIEW_DETAILS *previous_current_view=NULL;
 CHARTYPE save_prefix=0;
 short save_gap=0;
 ROWTYPE save_cmd_line=0;
 bool save_id_line=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Xedit");
#endif
/*---------------------------------------------------------------------*/
/* With no arguments, edit the next file in the ring...                */
/*---------------------------------------------------------------------*/
 if (strcmp((DEFCHAR *)params,"") == 0)
   {
    rc = advance_view(NULL,DIRECTION_FORWARD);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* With "-" as argument, edit the previous file in the ring...         */
/*---------------------------------------------------------------------*/
 if (strcmp((DEFCHAR *)params,"-") == 0)
   {
    rc = advance_view(NULL,DIRECTION_BACKWARD);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If there are still file(s) in the ring, clear the command line and  */
/* save any changes to the focus line.                                 */
/*---------------------------------------------------------------------*/
 if (number_of_files > 0)
   {
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL,TRUE);
    memset(cmd_rec,' ',max_line_length);
    cmd_rec_len = 0;
   }
 previous_current_view = CURRENT_VIEW;
/*---------------------------------------------------------------------*/
/* Save the position of the cursor for the current view before getting */
/* the contents of the new file...                                     */
/*---------------------------------------------------------------------*/
 if (curses_started
 &&  number_of_files > 0)
   {
    if (CURRENT_WINDOW_COMMAND != NULL)
      {
       wmove(CURRENT_WINDOW_COMMAND,0,0);
       my_wclrtoeol(CURRENT_WINDOW_COMMAND);
      }
    getyx(CURRENT_WINDOW_FILEAREA,CURRENT_VIEW->y[WINDOW_FILEAREA],CURRENT_VIEW->x[WINDOW_FILEAREA]);
    if (CURRENT_WINDOW_PREFIX != NULL)
       getyx(CURRENT_WINDOW_PREFIX,CURRENT_VIEW->y[WINDOW_PREFIX],CURRENT_VIEW->x[WINDOW_PREFIX]);
   }
 if (number_of_files > 0)
   {
    save_prefix=CURRENT_VIEW->prefix;
    save_gap=CURRENT_VIEW->prefix_gap;
    save_cmd_line=CURRENT_VIEW->cmd_line;
    save_id_line=CURRENT_VIEW->id_line;
   }
/*---------------------------------------------------------------------*/
/* Read the contents of the new file into memory...                    */
/*---------------------------------------------------------------------*/
 if ((rc = get_file(strrmdup(strtrans(params,OSLASH,ISLASH),ISLASH))) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If more than one screen is displayed, sort out which view is to be  */
/* displayed...                                                        */
/*---------------------------------------------------------------------*/
 if (display_screens > 1)
   {
    save_current_view = CURRENT_VIEW;
    CURRENT_SCREEN.screen_view = CURRENT_VIEW = previous_current_view;
    advance_view(save_current_view,DIRECTION_FORWARD);
   }
 else
   {
    if (number_of_files > 0)
      {
/*---------------------------------------------------------------------*/
/* If the position of the prefix or command line for the new view is   */
/* different from the previous view, rebuild the windows...            */
/*---------------------------------------------------------------------*/
       if ((save_prefix&PREFIX_LOCATION_MASK) != (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK)
       ||  save_gap != CURRENT_VIEW->prefix_gap
       ||  save_cmd_line != CURRENT_VIEW->cmd_line
       ||  save_id_line != CURRENT_VIEW->id_line)
         {
          set_screen_defaults();
          if (curses_started)
            {
             if (set_up_windows(current_screen) != RC_OK)
               {
#ifdef TRACE
                trace_return();
#endif
                return(RC_OK);
               }
            }
         }
      }
/*---------------------------------------------------------------------*/
/* Re-calculate CURLINE for the new view in case the CURLINE is no     */
/* longer in the display area.                                         */
/*---------------------------------------------------------------------*/
    prepare_view(current_screen);
 }
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 build_screen(current_screen);
/*---------------------------------------------------------------------*/
/* Position the cursor in the main window depending on the type of file*/
/*---------------------------------------------------------------------*/
 if (curses_started)
   {
    if (CURRENT_VIEW->in_ring)
      {
       wmove(CURRENT_WINDOW_FILEAREA,CURRENT_VIEW->y[WINDOW_FILEAREA],CURRENT_VIEW->x[WINDOW_FILEAREA]);
       if (CURRENT_WINDOW_PREFIX != NULL)
          wmove(CURRENT_WINDOW_PREFIX,CURRENT_VIEW->y[WINDOW_PREFIX],CURRENT_VIEW->x[WINDOW_PREFIX]);
       getyx(CURRENT_WINDOW,y,x);
       wmove(CURRENT_WINDOW,y,x);
      }
    else
      {
       if (CURRENT_FILE->pseudo_file == PSEUDO_DIR)
          wmove(CURRENT_WINDOW_FILEAREA,CURRENT_VIEW->current_row,file_start-1);
       else
          wmove(CURRENT_WINDOW_FILEAREA,CURRENT_VIEW->current_row,0);
      }
   }
/*---------------------------------------------------------------------*/
/* Execute any profile file...                                         */
/*---------------------------------------------------------------------*/
 if (REPROFILEx || in_profile)
   {
    profile_file_executions++;
    in_reprofile = TRUE;
    if (execute_profile)
      {
       if (local_prf != (CHARTYPE *)NULL)
          rc = get_profile(local_prf,prf_arg);
      }
    in_reprofile = FALSE;
   }
/*---------------------------------------------------------------------*/
/* If the result of processing the profile file results in no files    */
/* in the ring, we need to get out NOW.                                */
/*---------------------------------------------------------------------*/
 if (number_of_files == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/* pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);*/
 build_screen(current_screen);
/*---------------------------------------------------------------------*/
/* If curses hasn't started, don't try to use curses functions...      */
/*---------------------------------------------------------------------*/
 if (curses_started)
   {
    display_screen(current_screen);
    if (CURRENT_WINDOW_COMMAND != NULL)
       wmove(CURRENT_WINDOW_COMMAND,0,0);
    if (CURRENT_WINDOW_PREFIX != NULL)
       touchwin(CURRENT_WINDOW_PREFIX);
    if (CURRENT_WINDOW_GAP != NULL)
       touchwin(CURRENT_WINDOW_GAP);
    if (CURRENT_WINDOW_COMMAND != NULL)
       touchwin(CURRENT_WINDOW_COMMAND);
    if (CURRENT_WINDOW_IDLINE != NULL)
       touchwin(CURRENT_WINDOW_IDLINE);
    touchwin(CURRENT_WINDOW_FILEAREA);
    show_statarea();
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     ? - retrieve - return the next/prior command on the command line

SYNTAX
     ?[+|?...]

DESCRIPTION
     The ? command returns the next or prior command from the command
     line ring and displays it on the command line.

     With the ['+'] argument, the next command in the command ring is
     retrieved.

     With no arguments, the previous command entered on the command
     line is retrieved.

     With multiple, concatenated ?s as argument, the pervious command
     entered on the command line is retrieved corresponding to the
     number of ?s entered.

     For Example:
     The command; ????? will retrieve the fifth last command entered.

COMPATIBILITY
     XEDIT: Compatible. Adds extra support for multiple ?s.
     KEDIT: See below..
     This command is bound to the up and down arrows when on the
     command line depending on the setting of <SET CMDARROWS>.

SEE ALSO
     <SET CMDARROWS>

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Retrieve(CHARTYPE *params)
#else
short Retrieve(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 CHARTYPE *current_command=NULL;
 CHARTYPE *save_params=NULL;
 int param_len=0;
 short direction=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Retrieve");
#endif
/*
 *---------------------------------------------------------------------
 * No parameters, get the last command...
 *---------------------------------------------------------------------
 */
 if (strcmp((DEFCHAR *)params,"") == 0)
    current_command = get_next_command(DIRECTION_FORWARD,1);
 else
   {
    /*
     *---------------------------------------------------------------------
     * Get a copy of the parameters, because we want to manipulate them,
     * and also retain the orignal for error reporting.
     *---------------------------------------------------------------------
     */
    if ((save_params = (CHARTYPE *)my_strdup(params)) == NULL)
      {
       display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_OUT_OF_MEMORY);
      }
    /*
     *---------------------------------------------------------------------
     * Strip all spaces from the parameters.  We want to be able to specify
     * ?  ? ? - as a valid set of arguments, equivalent to ???-
     *---------------------------------------------------------------------
     */
    save_params = MyStrip(save_params,STRIP_ALL,' ');
    param_len = strlen((DEFCHAR *)save_params);
    if (*(save_params+(param_len-1)) == (CHARTYPE)'+')
      {
       *(save_params+(param_len-1)) = '\0';
       direction = DIRECTION_BACKWARD;
      }
    else
      {
       if (*(save_params+(param_len-1)) == (CHARTYPE)'-')
         {
          *(save_params+(param_len-1)) = '\0';
          direction = DIRECTION_FORWARD;
         }
      }
    if (strzne(save_params,(CHARTYPE)'?') != (-1))
      {
       display_error(1,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    current_command = get_next_command(direction,strlen((DEFCHAR *)save_params)+1);
   }
 if (save_params)
    (*the_free)(save_params);
 wmove(CURRENT_WINDOW_COMMAND,0,0);
 my_wclrtoeol(CURRENT_WINDOW_COMMAND);
 if (current_command != (CHARTYPE *)NULL)
    Cmsg(current_command);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     = - re-execute the last command issued on the command line

SYNTAX
     =

DESCRIPTION
     The = command retrieves the most recently issued command from
     the <command line> and re-executes it.

COMPATIBILITY
     XEDIT: Does not support optional [subcommand] option.
     KEDIT: Does not support optional [command] option.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef HAVE_PROTO
short Reexecute(CHARTYPE *params)
#else
short Reexecute(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE last_command_for_reexecute[MAX_COMMAND_LENGTH];
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm5.c:   Reexecute");
#endif
 if (strcmp((DEFCHAR *)params,""))
    {
     display_error(1,params,FALSE);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Retrieve the last command and execute it.                           */
/*---------------------------------------------------------------------*/
 rc = command_line(last_command_for_reexecute,COMMAND_ONLY_FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     ! - execute an operating system command

SYNTAX
     ! [command]

DESCRIPTION
     The ! command executes the supplied operating system 'command'
     or runs an interactive shell if no 'command' is supplied.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent to DOS command.

SEE ALSO
     <DOS>, <OS>

STATUS
     Complete.
**man-end**********************************************************************/

/*man-start*********************************************************************
COMMAND
     & - execute and re-display command

SYNTAX
     &[command]

DESCRIPTION
     The & command executes the supplied 'command' in the normal
     way, but when the command completes, instead of clearing
     the THE command line, the command, and the & are 
     re-displayed.  This makes it easy to repeat the same
     command, or make changes to it.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
