/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1998 */
/* See the file NOTICE for conditions of use and distribution. */


/* General functions concerned with directing, and generic director options. */


#include "exim.h"



/* Generic options for directors, all of which live inside director_instance
data blocks and hence have the opt_public flag set. */

optionlist optionlist_directors[] = {
  { "*expand_transport",  opt_stringptr|opt_public|opt_hidden,
                 (void *)offsetof(director_instance, expand_transport) },
  { "condition",          opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, condition) },
  { "domains",            opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, domains) },
  { "driver",             opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, driver_name) },
  { "errors_to",          opt_stringptr|opt_public,
                 (void *)(offsetof(director_instance, errors_to)) },
  { "except_domains",     opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, except_domains) },
  { "except_local_parts", opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, except_local_parts) },
  { "except_senders",     opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, except_senders) },
  { "expn",               opt_bool|opt_public,
                 (void *)offsetof(director_instance, expn) },
  { "fail_verify",        opt_bool_verify|opt_hidden|opt_public,
                 (void *)offsetof(director_instance, fail_verify_sender) },
  { "fail_verify_recipient", opt_bool|opt_public,
                 (void *)offsetof(director_instance, fail_verify_recipient) },
  { "fail_verify_sender", opt_bool|opt_public,
                 (void *)offsetof(director_instance, fail_verify_sender) },
  { "headers_add",        opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, extra_headers) },
  { "headers_remove",     opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, remove_headers) },
  { "local_parts",        opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, local_parts) },
  { "more",               opt_bool|opt_public,
                 (void *)offsetof(director_instance, more) },
  { "new_director",       opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, new_director) },
  { "prefix",             opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, prefix) },
  { "prefix_optional",    opt_bool|opt_public,
                 (void *)offsetof(director_instance, prefix_optional) },
  { "require_files",      opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, require_files) },
  { "senders",            opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, senders) },
  { "suffix",             opt_stringptr|opt_public,
                 (void *)offsetof(director_instance, suffix) },
  { "suffix_optional",    opt_bool|opt_public,
                 (void *)offsetof(director_instance, suffix_optional) },
  { "transport",          opt_transportptr|opt_public,
                 (void *)offsetof(director_instance, transport) },
  { "unseen",             opt_bool|opt_public,
                 (void *)offsetof(director_instance, unseen) },
  { "verify",             opt_bool_verify | opt_hidden|opt_public,
                 (void *)offsetof(director_instance, verify_sender) },
  { "verify_only",        opt_bool|opt_public,
                 (void *)offsetof(director_instance, verify_only) },
  { "verify_recipient",   opt_bool|opt_public,
                 (void *)offsetof(director_instance, verify_recipient) },
  { "verify_sender",      opt_bool|opt_public,
                 (void *)offsetof(director_instance, verify_sender) }
};

int optionlist_directors_size =
  sizeof(optionlist_directors)/sizeof(optionlist);




/*************************************************
*    Get errors address for a director/router    *
*************************************************/

/* This function is called by both directors and routers to sort out the errors
address for a particular address. If there is a setting in the director (or
router) block, then expand and verify it, and if it works, use it. Otherwise
use any setting that is in the address itself. This might be NULL, meaning
unset (the message's sender is then used). Verification isn't done when the
original address is just being verified, as otherwise there might be directing
loops if someone sets up a silly configuration.

Arguments:
  addr         the input address
  dblock       the director or router instance (the value used comes from
                 the part of the block that they have in common)
  type         "director" or "router"
  verify       TRUE if verifying
  errors_to    point the errors address here

Returns:       OK if no problem
               DEFER if verifying the address caused a deferment
               ERROR big disaster (e.g. expansion failure)
*/

int
direct_get_errors_address(address_item *addr, director_instance *dblock,
  char *type, BOOL verify, char **errors_to)
{
char *s;

*errors_to = addr->errors_address;
if (dblock->errors_to == NULL) return OK;

s = expand_string(dblock->errors_to);
if (s == NULL)
  {
  addr->message = string_sprintf("%s %s failed to expand \"%s\": %s",
    dblock->name, type, dblock->errors_to, expand_string_message);
  if (search_find_defer) return DEFER;
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }

/* While verifying, set the sender address to null, because that's what
it will be when sending an error message, and there are now configuration
options that control the running of directors and routers by checking
the sender address. When testing an address, there may not be a sender
address. */

if (verify) *errors_to = s; else
  {
  char *snew;
  int save1 = 0;
  if (sender_address != NULL)
    {
    save1 = sender_address[0];
    sender_address[0] = 0;
    }
  if (verify_address(s, NULL, NULL, &snew, vopt_is_recipient | vopt_local)
    == OK) *errors_to = snew;
  if (sender_address != NULL) sender_address[0] = save1;
  }

return OK;
}




/*************************************************
*  Get additional headers for a director/router  *
*************************************************/

/* This function is called by both directors and routers to sort out the
additional headers and header remove list for a particular address.

Arguments:
  addr           the input address
  dblock         the director or router instance (the value used comes from
                   the part of the block that they have in common)
  type           "director" or "router"
  extra_headers  points to where to hang the header chain
  remove_headers points to where to hang the remove list

Returns:         OK if no problem
                 DEFER if expanding a string caused a deferment
                 ERROR big disaster (e.g. expansion failure)
*/

int
direct_get_munge_headers(address_item *addr, director_instance *dblock,
  char *type, header_line **extra_headers, char **remove_headers)
{
if (dblock->extra_headers == NULL)
  *extra_headers = addr->extra_headers;
else
  {
  int slen;
  BOOL addnl = FALSE;
  header_line *h;
  char *s = expand_string(dblock->extra_headers);
  if (s == NULL)
    {
    addr->message = string_sprintf("%s %s failed to expand \"%s\": %s",
      dblock->name, type, dblock->extra_headers, expand_string_message);
    if (search_find_defer) return DEFER;
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }
  slen = (int)strlen(s);
  if (slen == 0 || s[slen-1] != '\n')
    {
    addnl = TRUE;
    slen++;
    }
  h = store_malloc(sizeof(header_line) + slen);
  strcpy(h->text, s);
  if (addnl) strcpy(h->text + slen - 1, "\n");
  h->next = addr->extra_headers;
  h->type = htype_other;
  h->slen = slen;
  *extra_headers = h;
  store_free(s);
  }

if (dblock->remove_headers == NULL)
  *remove_headers = addr->remove_headers;
else
  {
  int slen;
  char *s = expand_string(dblock->remove_headers);
  if (s == NULL)
    {
    addr->message = string_sprintf("%s %s failed to expand \"%s\": %s",
      dblock->name, type, dblock->remove_headers, expand_string_message);
    if (search_find_defer) return DEFER;
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }

  if (addr->remove_headers == NULL)
    *remove_headers = s;
  else
    {
    *remove_headers = string_sprintf("%s:%s", addr->remove_headers, s);
    store_free(s);
    }
  }

return OK;
}




/*************************************************
*  Check files, sender, condition for a director *
*************************************************/

/* This is a utility function for use by directors, and called from within
them. It carries out file existence, sender checking, and general condition
tests, in that order, because the latter may be using the files mentioned in
the former. There are two failure yields, partly for historical reasons. FAIL
causes the directing not to continue of the director has no_more set;
FAIL_CONDITION does not. Retaining FAIL is for backwards compatibility, but it
may be that changing them all to FAIL_CONDITION one day might be reasonable.

Argument:
  dblock    points to the director instance
  addr      address item (for defer message)

Returns:    OK               no conditions failed
            DEFER            some condition couldn't be checked
            FAIL             a condition failed; no_more means no more
            FAIL_CONDITION   a condition failed; ignore no_more
*/

int
direct_check_fsc(director_instance *dblock, address_item *addr)
{
int rc;

switch (match_exists(dblock->require_files))
  {
  case FAIL:
  DEBUG(9) debug_printf("%s director skipped: file existence failure\n",
    dblock->name);
  return FAIL;

  case DEFER:
  addr->message = string_sprintf("file existence defer in %s director: %s",
    dblock->name, strerror(errno));
  return DEFER;
  }

if ((dblock->except_senders != NULL &&
     (rc = match_sender(dblock->except_senders, &(dblock->re_except_senders)))
       == OK) ||
    (dblock->senders != NULL &&
     (rc = match_sender(dblock->senders, &(dblock->re_senders))) != OK))
  {
  if (rc == DEFER)
    {
    addr->message = string_sprintf("sender check defer in %s director",
      dblock->name);
    return DEFER;
    }
  DEBUG(9) debug_printf("%s director: sender match failure\n",
    dblock->name);
  return FAIL;
  }

if (dblock->condition != NULL &&
    !expand_check_condition(dblock->condition, dblock->name, "director"))
  {
  DEBUG(9) debug_printf("%s director skipped: condition failure\n",
    dblock->name);
  return FAIL_CONDITION;
  }

return OK;
}




/*************************************************
*           Find a local user                    *
*************************************************/

/* Try several times (if configured) to find a local user, in case delays in
NIS or NFS whatever cause an incorrect refusal. It's a pity that getpwnam()
doesn't have some kind of indication as to why it has failed. If the string
given consists entirely of digits, and the third argument is not NULL, assume
the string is the numerical value of the uid. Otherwise it is looked up using
getpwnam(). The uid is passed back via return_uid, if not NULL, and the
pointer to a passwd structure, if found, is passed back via pw, if not NULL.

Because this may be called several times in succession for the same user for
different directors, cache the result of the previous getpwnam call so that it
can be re-used. Note that we can't just copy the structure, as the store it
points to can get trashed.

Arguments:
  s           the login name or textual form of the numerical uid of the user
  pw          if not NULL, return the result of getpwnam here, or set NULL
                if no call to getpwnam is made (s numeric, return_uid != NULL)
  return_uid  if not NULL, return the uid via this address

Returns:      TRUE if s is numerical or was looked up successfully

*/

static struct passwd pwcopy;
static struct passwd *lastpw = NULL;
static char lastname[48] = { 0 };
static char lastdir[128];

BOOL
direct_finduser(char *s, struct passwd **pw, uid_t *return_uid)
{
if (strcmp(lastname, s) != 0)
  {
  int i = 0;

  if (return_uid != NULL && (isdigit(*s) || *s == '-') &&
       s[strspn(s+1, "0123456789")+1] == 0)
    {
    *return_uid = (uid_t)atoi(s);
    if (pw != NULL) *pw = NULL;
    return TRUE;
    }

  strncpy(lastname, s, sizeof(lastname));
  lastname[sizeof(lastname)-1] = 0;          /* just in case */

  for (;;)
    {
    if ((lastpw = getpwnam(s)) != NULL) break;
    if (++i > finduser_retries) break;
    sleep(1);
    }

  if (lastpw != NULL)
    {
    pwcopy.pw_uid = lastpw->pw_uid;
    pwcopy.pw_gid = lastpw->pw_gid;
    strncpy(lastdir, lastpw->pw_dir, sizeof(lastdir));
    lastdir[sizeof(lastdir)-1] = 0;          /* just in case */
    pwcopy.pw_name = lastname;
    pwcopy.pw_dir = lastdir;
    lastpw = &pwcopy;
    }
  }

if (lastpw == NULL) return FALSE;

if (return_uid != NULL) *return_uid = lastpw->pw_uid;
if (pw != NULL) *pw = lastpw;

return TRUE;
}




/*************************************************
*           Find a local group                   *
*************************************************/

/* Try several times (if configured) to find a local group, in case delays in
NIS or NFS whatever cause an incorrect refusal. It's a pity that getgrnam()
doesn't have some kind of indication as to why it has failed.

Arguments:
  s           the group namd or textual form of the numerical gid
  return_gid  return the gid via this address

Returns:      TRUE if the group was found; FALSE otherwise

*/

BOOL
direct_findgroup(char *s, gid_t *return_gid)
{
int i = 0;
struct group *gr;

if ((isdigit(*s) || *s == '-') && s[strspn(s+1, "0123456789")+1] == 0)
  {
  *return_gid = (gid_t)atoi(s);
  return TRUE;
  }

for (;;)
  {
  if ((gr = getgrnam(s)) != NULL)
    {
    *return_gid = gr->gr_gid;
    return TRUE;
    }
  if (++i > finduser_retries) break;
  sleep(1);
  }

return FALSE;
}




/*************************************************
*          Find user by expanding string         *
*************************************************/

/* Expands a string, and then looks up the result in the passwd file.

Arguments:
  string       the string to be expanded, yielding a login name or a numerical
                 uid value (to be passed to direct_finduser())
  driver_name  caller name for panic error message (only)
  driver_type  caller type for panic error message (only)
  pw           return passwd entry via this pointer
  uid          return uid via this pointer
  errmsg       where to point a message on failure

Returns:       TRUE if user found, FALSE otherwise
*/

BOOL
direct_find_expanded_user(char *string, char *driver_name,
  char *driver_type, struct passwd **pw, uid_t *uid, char **errmsg)
{
BOOL yield = TRUE;
char *user = expand_string(string);

if (user == NULL)
  {
  *errmsg = string_sprintf("Failed to expand user string \"%s\" from the "
    "%s %s: %s", string, driver_name, driver_type, expand_string_message);
  log_write(0, LOG_MAIN|LOG_PANIC, "%s", *errmsg);
  return FALSE;
  }

if (!direct_finduser(user, pw, uid))
  {
  *errmsg = string_sprintf("Failed to find user \"%s\" from expanded string "
    "\"%s\" from the %s %s", user, string, driver_name, driver_type);
  log_write(0, LOG_MAIN|LOG_PANIC, "%s", *errmsg);
  yield = FALSE;
  }

store_free(user);
return yield;
}



/*************************************************
*          Find group by expanding string        *
*************************************************/

/* Expands a string and then looks up the result in the group file.

Arguments:
  string       the string to be expanded, yielding a group name or a numerical
                 gid value (to be passed to direct_findgroup())
  driver_name  caller name for panic error message (only)
  driver_type  caller type for panic error message (only)
  gid          return gid via this pointer
  errmsg       return error message via this pointer

Returns:       TRUE if found group, FALSE otherwise
*/

BOOL
direct_find_expanded_group(char *string, char *driver_name, char *driver_type,
  gid_t *gid, char **errmsg)
{
BOOL yield = TRUE;
char *group = expand_string(string);

if (group == NULL)
  {
  *errmsg = string_sprintf("Failed to expand group string \"%s\" from the "
    "%s %s: %s", string, driver_name, driver_type, expand_string_message);
  log_write(0, LOG_MAIN|LOG_PANIC, "%s", *errmsg);
  return FALSE;
  }

if (!direct_findgroup(group, gid))
  {
  *errmsg = string_sprintf("Failed to find group \"%s\" from expanded string "
    "\"%s\" from the %s %s", group, string, driver_name, driver_type);
  log_write(0, LOG_MAIN|LOG_PANIC, "%s", *errmsg);
  yield = FALSE;
  }

store_free(group);
return yield;
}



/*************************************************
*             Initialize director list           *
*************************************************/

/* Read the directors configuration file, and set up a chain of director
instances according to its contents. Each director has generic options and may
also have its own private options. This function is only ever called when
directors == NULL. We use generic code in readconf to do the work. */

void
direct_init(void)
{
director_instance *d;
readconf_driver_init("director",
  (driver_instance **)(&directors),     /* chain anchor */
  (driver_info *)directors_available,   /* available drivers */
  sizeof(director_info),                /* size of info blocks */
  &director_defaults,                   /* default values for generic options */
  sizeof(director_instance),            /* size of instance block */
  optionlist_directors,                 /* generic options */
  optionlist_directors_size);

/* A director may not have more=false and unseen=true. Also, check that
any settings of new_director are valid, and set up the actual pointers. */

for (d = directors; d != NULL; d = d->next)
  {
  if (d->unseen && !d->more )
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG,
      "the combination of \"unseen\" and \"no_more\" is not permitted on "
      "a director,\n  but was set for the %s director", d->name);

  if (d->new_director != NULL)
    {
    director_instance *dd;
    for (dd = directors; dd != NULL; dd = dd->next)
      {
      if (strcmp(d->new_director, dd->name) == 0)
        {
        d->new = dd;
        break;
        }
      }
    if (dd == NULL)
      log_write(0, LOG_PANIC_DIE|LOG_CONFIG,
        "new_director \"%s\" not found for \"%s\" director", d->new_director,
          d->name);
    }
  }
}




/*************************************************
*           Tidy up after directing              *
*************************************************/

/* Directors are entitled to keep hold of certain resources in their instance
blocks so as to save setting them up each time. An example is the open file for
the aliasfile director. Such directors must provide a tidyup entry point which
is called when all directing is finished via this function. */

void
direct_tidyup(void)
{
director_instance *d;
for (d = directors; d != NULL; d = d->next)
  if (d->info->tidyup != NULL) (d->info->tidyup)(d);
}





/*************************************************
*                 Direct one address             *
*************************************************/

/* This function is passed in one address item, for processing by the
directors. It has been determined that the address is for one of the local
domains. The action can be one of:

  . adding the address to the chain on addr_local, for local delivery, and
    returning OK;
  . adding the address to the chain on addr_remote, for remote delivery, and
    returning OK;
  . returning FAIL;
  . returning DEFER;
  . adding one or more new addresses, with this one as parent, to the chain
    on addr_new, for reprocessing ab initio, and returning OK, or
    OK_CONTINUE.

The verify flag is set if this is being called for verification rather than
delivery. If the director doesn't have its "verify" flag set, it is skipped.
Otherwise, the flag is passed to the director in case it needs to know.

If the director has the "more" flag set false (i.e. "no_more" has been
specified) then, if it fails to match a name, no further directors are tried.
This includes the case when the director would have been called, but for the
verify flag's being FALSE.

If the new_director field of the address is set, then directing must start
at the director it points to. This has been checked to exist at start-up
time.

The generic options "domain", "prefix", and "suffix" are handled at this top
level.

Arguments
  addr         the address to direct
  addr_local   the chain onto which to hang the address for local delivery
  addr_remote  the chain onto which to hang the address for remote delivery
  addr_new     the chain onto which to hang newly-generated addresses
  addr_succeed the chain onto which to hang addresses that we are finished
                 with (e.g. they've been expanded into something else)
  verify       v_none if delivering rather than verifying,
               v_sender if verifying a sender address
               v_recipient if verifying a recipient address
               v_expn if processing an EXPN address

Returns:       OK, OK_CONTINUE, FAIL, or DEFER, as described above
*/


int
direct_address(address_item *addr,
  address_item **addr_local,
  address_item **addr_remote,
  address_item **addr_new,
  address_item **addr_succeed,
  int verify)
{
director_instance *d;
address_item *parent;
BOOL more = TRUE;

DEBUG(9) debug_printf(">>>>>>>>>>>>>>>>>>>>>>>>\ndirecting %s\n", addr->orig);

/* Loop through all director instances, possibly starting at a specific
point. */

for (d = (addr->start_director == NULL)? directors : addr->start_director;
     more && d != NULL;
     d = d->next)
  {
  int loopcount = 0;
  int yield, check_yield;
  int suffixchar = 0;
  int suffixend = -1;
  char *oldlocal_part = NULL;
  BOOL loop_detected = FALSE;

  /* Check the conditions that are common to both routers and directors. The
  relevant items in the instance block are carefully put in the same position,
  which is why we can coerce the director instance pointer to a router instance
  pointer here. */

  if (route_skip_driver((router_instance *)d, addr->local_part, addr->domain,
    verify, &more, "director")) continue;

  /* Skip this director if processing EXPN and it doesn't have expn set */

  if (verify == v_expn && !d->expn)
    {
    DEBUG(9) debug_printf("%s director skipped: no_expn set\n", d->name);
    continue;
    }

  /* Handle any configured prefix by replacing the local_part address,
  saving it for restoration if the director fails. Skip the director
  if the prefix doesn't match, unless the prefix is optional. */

  if (d->prefix != NULL)
    {
    BOOL matched = FALSE;
    char *prefix;
    char *listptr = d->prefix;
    char prebuf[64];

    /* Loop for possible list of prefixes. If any one begins with '*',
    special wildcard checking is done. */

    for (prefix = string_nextinlist(&listptr, ':', prebuf, sizeof(prebuf));
         prefix != NULL;
         prefix = string_nextinlist(&listptr, ':', prebuf, sizeof(prebuf)))
      {
      int plen = (int)strlen(prefix);
      if (prefix[0] == '*')
        {
        char *p;
        prefix++;
        for (p = addr->local_part + (int)strlen(addr->local_part) - (--plen);
             p >= addr->local_part; p--)
          {
          if (strncmpic(prefix, p, plen) == 0)
            {
            matched = TRUE;
            plen += p - addr->local_part;
            break;
            }
          }
        }
      else matched = (strncmpic(prefix, addr->local_part, plen) == 0);

      if (matched)
        {
        oldlocal_part = addr->local_part;
        addr->local_part += plen;
        addr->prefix = string_copyn(oldlocal_part, plen);
        DEBUG(9) debug_printf("stripped prefix %s\n", addr->prefix);
        break;
        }
      }

    if (!matched && !d->prefix_optional)
      {
      DEBUG(9) debug_printf("%s director skipped: prefix mismatch\n",
        d->name);
      continue;
      }
    }

  /* Handle any configured suffix likewise, but for this we have to
  fudge the end of the address; save the character so it can be put
  back if the director fails. */

  if (d->suffix != NULL)
    {
    BOOL matched = FALSE;
    char *suffix;
    char *listptr = d->suffix;
    char sufbuf[64];

    for (suffix = string_nextinlist(&listptr, ':', sufbuf, sizeof(sufbuf));
         suffix != NULL;
         suffix = string_nextinlist(&listptr, ':', sufbuf, sizeof(sufbuf)))
      {
      int slen = (int)strlen(suffix);
      int alen = (int)strlen(addr->local_part);

      if (suffix[slen-1] == '*')
        {
        char *p;
        char *pend = addr->local_part + alen - (--slen) + 1;
        for (p = addr->local_part; p < pend; p++)
          {
          if (strncmpic(suffix, p, slen) == 0)
            {
            matched = TRUE;
            slen = alen - (p - addr->local_part);
            break;
            }
          }
        }

      else matched = alen > slen &&
        strncmpic(suffix, addr->local_part + alen - slen, slen) == 0;

      if (matched)
        {
        suffixend = alen - slen;
        suffixchar = addr->local_part[suffixend];
        addr->suffix = string_copy(addr->local_part+suffixend);
        addr->local_part[suffixend] = 0;
        DEBUG(9) debug_printf("stripped suffix %s\n", addr->suffix);
        break;
        }
      }

    /* If failed to match the suffix, restore any change the prefix
    handler made. */

    if (!matched && !d->suffix_optional)
      {
      DEBUG(9) debug_printf("%s director skipped: suffix mismatch\n",
        d->name);
      if (oldlocal_part != NULL)
        {
        addr->local_part = oldlocal_part;
        addr->prefix = NULL;
        }
      continue;
      }
    }


  /* Loop protection: If this address has a parent with the same address that
  was directed by this director, we skip this director. This prevents a variety
  of possibly looping states, and saves us having to do anything special for
  the forwardfile director. Just in case someone does put it into a loop
  (possible with a smartuser director just adding to an address, for example),
  put a long stop counter on the number of parents. */

  for (parent = addr->parent; parent != NULL; parent = parent->parent)
    {
    if (parent->director == d &&
        strcmpic(parent->local_part, addr->local_part) == 0 &&
        strcmpic(parent->domain, addr->domain) == 0
        &&
        ((parent->prefix == NULL && addr->prefix == NULL) ||
         (parent->prefix != NULL && addr->prefix != NULL &&
          strcmpic(parent->prefix, addr->prefix) == 0))
        &&
        ((parent->suffix == NULL && addr->suffix == NULL) ||
         (parent->suffix != NULL && addr->suffix != NULL &&
          strcmpic(parent->suffix, addr->suffix) == 0)))
      {
      DEBUG(9) debug_printf("%s director skipped: previously directed "
        "%s%s%s@%s\n",
        d->name,
        (addr->prefix == NULL)? "" : addr->prefix,
        addr->local_part,
        (addr->suffix == NULL)? "" : addr->suffix,
        addr->domain);
      loop_detected = TRUE;
      break;
      }

    if (loopcount++ > 100)
      log_write(0, LOG_PANIC_DIE, "directing loop for %s", addr->orig);
    }

  if (loop_detected)
    {
    yield = FAIL;            /* Behave as if director failed */
    goto DIRECTOR_FAILED;
    }

  /* Prefix and suffix (if any) have been stripped; ready to call the
  director, first setting up values that may be used in string expansions.
  Reset them afterwards. Also, we must arrange for any dynamic transport to get
  set up. The code in route_set_transport() handles this for both routers and
  directors, using items in the data blocks that are at the same offsets. */

  deliver_set_expansions(addr);

  if (route_set_transport((router_instance *)d, "director", &(addr->message)))
    {
    DEBUG(9) debug_printf("calling %s director\n", d->name);
    yield = (d->info->code)(d, addr, addr_local, addr_remote, addr_new,
      addr_succeed, verify != v_none && verify != v_expn);
    }
  else
    {
    DEBUG(9) debug_printf("%s", addr->message);
    yield = DEFER;
    }

  deliver_set_expansions(NULL);

  /* If yield = FAIL_CONDITION it means that a condition that has to be tested
  from within the director (because of setting up variables like $home or
  whatever) has failed, and the condition is one that should logically be
  tested before the "more" flag. We therefore force "more" back to be TRUE, and
  convert the yield to FAIL. */

  if (yield == FAIL_CONDITION) { more = TRUE; yield = FAIL; }

  /* If yield = OK_CONTINUE the director has generated new address(es) that
  should always be themselves processed when verifying. Without this, secondary
  generated addresses are not necessarily processed (if an alias works, you
  often don't want to know if its children work.) Save a bit of time in
  avoiding duplicated tests below. */

  check_yield = (yield == OK_CONTINUE)? OK : yield;

  /* Success or deferral means we are finished with this address, as does an
  internal or configuration failure. If we succeed on a director that has
  "fail_verify" set, convert the result into a fail. If we succeed on a
  director that has "unseen" set on it, we must make a copy of the address to
  hand on to the subsequent directors. Actually, we can't pass it on directly;
  we have to put it on the new queue, but since it has the same address as
  the current one, it will be passed by all previous directors and also the
  one that has just succeeded, by the loop-avoidance code. */

  if (check_yield == OK || check_yield == DEFER || check_yield == ERROR)
    {
    addr->director = d;

    if (check_yield == OK)
      {
      if ((verify == v_sender && d->fail_verify_sender) ||
          (verify == v_recipient && d->fail_verify_recipient))
        {
        yield = check_yield = FAIL;
        addr->message = string_sprintf("%s director forced verify failure",
          d->name);
        }
      else if (d->unseen)
        {
        address_item *new = deliver_make_addr(string_copy(addr->orig));
        new->parent = addr;
        new->ignore_error |= addr->ignore_error;
        new->errors_address = addr->errors_address;
        addr->child_count++;
        new->next = *addr_new;
        *addr_new = new;
        }
      }

    DEBUG(2)
      {
      if (check_yield == OK)
        {
        debug_printf("%s director succeeded for %s\n", d->name,
          addr->local_part);
        debug_printf("  transport: %s\n", (addr->transport == NULL)?
          "<none>" : addr->transport->name);
        if (d->unseen)
          debug_printf("\"unseen\" set: replicated %s\n", addr->orig);
        }
      else if (check_yield == DEFER)
        {
        debug_printf("%s director deferred %s\n", d->name, addr->local_part);
        debug_printf("  message: %s\n", (addr->message == NULL)?
          "<none>" : addr->message);
        }
      else
        {
        debug_printf("%s director: error for %s\n", d->name, addr->local_part);
        debug_printf("  message: %s\n", (addr->message == NULL)?
          "<none>" : addr->message);
        }
      }

    return yield;
    }

  DIRECTOR_FAILED:

  /* Restore prefix & suffix for the next director. */

  if (suffixend >= 0) addr->local_part[suffixend] = suffixchar;
  if (oldlocal_part != NULL) addr->local_part = oldlocal_part;
  addr->prefix = addr->suffix = NULL;

  /* Break the loop if a director forced a failure. */

  if (yield == FORCEFAIL) break;
  }

/* Ensure store for domain and local part data is freed */

if (domain_data != NULL)
  {
  store_free(domain_data);
  domain_data = NULL;
  }

if (local_part_data != NULL)
  {
  store_free(local_part_data);
  local_part_data = NULL;
  }

/* No directors accepted this address; fail it. */

if (addr->message == NULL)
  addr->message = string_sprintf("unknown local-part \"%s\" in domain \"%s\"",
    addr->local_part, addr->domain);
return FAIL;
}

/* End of direct.c */
