# file nnet/multinom.q copyright (C) 1994-8 W. N. Venables and B. D. Ripley
#
multinom <- function(formula, data=sys.parent(), weights, subset, na.action,
	    contrasts=NULL, Hess=F, summ=0, censored=F, ...)
{
  class.ind <- function(cl)
  {
    n <- length(cl)
    x <- matrix(0, n, length(levels(cl)))
    x[(1:n) + n * (as.vector(codes(cl)) - 1)] <- 1
    dimnames(x) <- list(names(cl), levels(cl))
    x
  }
  summ2 <- function(X, Y)
  {
    X <- as.matrix(X)
    Y <- as.matrix(Y)
    n <- dim(X)[1]
    p <- dim(X)[2]
    q <- dim(Y)[2]
    Z <- t(cbind(X, Y))
    storage.mode(Z) <- "double"
    z <- .C("VR_summ2",
	    as.integer(n),
	    as.integer(p),
	    as.integer(q),
	    Z = Z,
	    na = integer(1))
    Za <- t(z$Z[, 1:z$na, drop = F])
    list(X = Za[, 1:p, drop = F], Y = Za[, p + 1:q])
  }

  call <- match.call()
  m <- match.call(expand = F)
  m$summ <- m$Hess <- m$contrasts <- m$censored <- m$... <- NULL
  m[[1]] <- as.name("model.frame")
  m <- eval(m, sys.parent())
  Terms <- attr(m, "terms")
  X <- model.matrix(Terms, m, contrasts)
  Xr <- qr(X)$rank
  Y <- model.extract(m, response)
  if(!is.matrix(Y)) Y <- as.factor(Y)
  w <- model.extract(m, weights)
  if(length(w) == 0) 
    if(is.matrix(Y)) w <- rep(1, dim(Y)[1]) 
      else w <- rep(1, length(Y))
  lev <- levels(Y)
  if(is.factor(Y)) {
    counts <- table(Y)
    if(any(counts == 0)) {
      warning(paste("group(s)", paste(lev[counts == 0], collapse=" "), 
		    "are empty"))
      Y <- factor(Y, levels=lev[counts > 0])
      lev <- lev[counts > 0]
    }
    if(length(lev) == 2) Y <- as.vector(codes(Y)) - 1
    else Y <- class.ind(Y)
  }
  if(summ==1) {
    Z <- cbind(X,Y)
    assign("z1", cumprod(apply(Z, 2, max)+1), frame=1)
    Z1 <- apply(Z, 1, function(x) sum(z1*x))
    oZ <- sort.list(Z1)
    Z2 <- !duplicated(Z1[oZ])
    oX <- ((1:length(Z1))[oZ])[Z2]
    X <- X[oX,]
    if(is.matrix(Y)) Y <- Y[oX,] else Y <- Y[oX]
    w <- diff(c(0,cumsum(w))[c(Z2,T)])
    print(dim(X))
  }
  if(summ==2) {
    Z <- summ2(cbind(X, Y), w)
    X <- Z$X[, 1:ncol(X)]
    Y <- Z$X[, ncol(X) + 1:ncol(Y), drop = F]
    w <- Z$Y
    print(dim(X))
  }
  if(summ==3) {
    Z <- summ2(X, Y*w)
    X <- Z$X
    Y <- Z$Y[,1:ncol(Y), drop = F]
    w <- rep(1, nrow(X))
    print(dim(X))
  }
  offset <- model.extract(m, offset)
  r <- ncol(X)
  if(is.matrix(Y)) {
    p <- ncol(Y)
    sY <- Y %*% rep(1, p)
    if(any(sY==0)) stop("some case has no observations")
    if(!censored) {
      Y <- Y / matrix(sY, nrow(Y), p)
      w <- w*sY
    }
    if(length(offset) > 1) {
      if(ncol(offset) !=  p) stop("ncol(offset) is wrong")
      mask <- c(rep(0, r+1+p), rep(c(0, rep(1, r), rep(0, p)), p-1) )
      X <- cbind(X, offset)
      Wts <- as.vector(rbind(matrix(0, r+1, p), diag(p)))
      fit <- nnet.default(X, Y, w, Wts=Wts, mask=mask, size=0, skip=T, softmax=T, censored=censored, rang=0, ...)
    } else {
      mask <- c(rep(0, r+1), rep(c(0, rep(1, r)), p-1) )
      fit <- nnet.default(X, Y, w, mask=mask, size=0, skip=T, softmax=T, censored=censored, rang=0, ...)
    }
  } else {
    if((p <- length(lev)) == 2) {
      if(length(offset) == 1) {
	mask <- c(0, rep(1, r))
	fit <- nnet.default(X, Y, w, mask=mask, size=0, skip=T, entropy=T, rang=0, ...)
      } else {
	mask <- c(0, rep(1, r), 0)
	Wts <- c(rep(0, r+1), 1)
	X <- cbind(X, offset)
	fit <- nnet.default(X, Y, w, Wts=Wts, mask=mask, size=0, skip=T, entropy=T, rang=0, ...)
      }
    } else {
      mask <- c(rep(0, ncol(X)+1), rep(1, (p-1)*(ncol(X)+1)))
      fit <- nnet.default(X, Y, w, mask=mask, size=0, skip=T, softmax=T, 
			  rang=0, ...)
    }
  }
  fit$formula <- as.vector(attr(Terms, "formula"))
  fit$terms <- Terms
  fit$call <- call
  fit$weights <- w
  fit$lev <- lev
  fit$deviance <- 2 * fit$value
  fit$rank <- Xr
  edf <- ifelse(length(lev) == 2, 1, length(lev)-1)*Xr
  if(is.matrix(Y)) {
    edf <- (ncol(Y)-1)*Xr
    if(length(dn <- dimnames(Y)[[2]]) > 0) fit$lab <- dn
      else fit$lab <- 1:ncol(Y)
  }
  fit$coefnames <- dimnames(X)[[2]]
  fit$vcoefnames <- fit$coefnames[1:r] # remove offset cols
  fit$edf <- edf
  fit$AIC <- 2 * (fit$value + edf)
  attr(fit, "class") <- c("multinom", "nnet")
  if(Hess) {
    mask <- as.logical(mask)
    fit$Hessian <- nnet.Hess(fit, X, Y, w)[mask, mask]
    cf <- fit$vcoefnames
    if(length(fit$lev) != 2) {
     if(length(fit$lev)) bf <- fit$lev else bf <- fit$lab
     cf <- t(outer(bf[-1], cf, function(x,y)paste(x,y,sep=":")))
    }
    dimnames(fit$Hessian) <- list(cf, cf)
  }
  fit
}

predict.multinom <- function(object, newdata, type=c("class","probs"))
{
  if(!inherits(object, "multinom")) stop("Not a multinom fit")
  type <- match.arg(type)
  if(missing(newdata)) 
    Y <- object$fitted.values
  else {
    form <- delete.response(terms(object$form))
    X <- model.matrix(form, newdata)
    Y <- predict.nnet(object, X)
  }
  switch(type, class={
    if(length(object$lev) > 2)
      Y <- factor(max.col(Y), levels=seq(along=object$lev), labels=object$lev)
    if(length(object$lev) == 2)
      Y <- factor(1 + (Y > 0.5), levels=1:2, labels=object$lev)
    if(length(object$lev) == 0)
      Y <- factor(max.col(Y), levels=seq(along=object$lab), labels=object$lab)
  },)
  drop(Y)
}

print.multinom <- function(x, ...)
{        
  if(!is.null(cl <- x$call)) {
    cat("Call:\n")
    dput(cl)
  }
  cat("\nCoefficients:\n")
  print(coef(x), ...)
  cat("\nResidual Deviance:", format(x$deviance), "\n")
  cat("AIC:", format(x$AIC), "\n")
  invisible(x)
}

coef.multinom <- function(object, ...)
{
  r <- length(object$vcoefnames)
  if(length(object$lev) == 2) {
    coef <- object$wts[1+(1:r)]
    names(coef) <- object$vcoefnames
  } else {
    coef <- matrix(object$wts, nrow = object$n[3], byrow=T)[, 1+(1:r), drop=F]
    if(length(object$lev)) dimnames(coef) <- list(object$lev, object$vcoefnames)
    if(length(object$lab)) dimnames(coef) <- list(object$lab, object$vcoefnames)
    coef <- coef[-1, , drop=F]
  }
  coef
}

drop1.multinom <- function(object, scope, sorted=T, trace=F)
{
  if(!inherits(object, "multinom")) stop("Not a multinom fit")
  if(missing(scope)) scope <- drop.scope(object)
    else {
      if(!is.character(scope))
	scope <- attr(terms(update.formula(object, scope)), "term.labels")
      if(!all(match(scope, attr(object$terms, "term.labels"), F)))
	stop("scope is not a subset of term labels")
    }
  n <- length(scope)
  ans <- matrix(nrow=n+1, ncol=2)
  dimnames(ans) <- list(c("<none>",paste("-",scope,sep="")), 
			c("df", "AIC"))
  ans[1,] <- c(object$edf, object$AIC)
  i <- 2
  for(tt in scope) {
    cat("trying -", tt,"\n")
    nobject <- update(object, paste("~ . -", tt), trace=trace)
    if(nobject$edf == object$edf) nobject$AIC <- NA
    ans[i,] <- c(nobject$edf, nobject$AIC)
    i <- i+1
  }
  if(sorted) ans <- ans[sort.list(ans[,2]),]
  ans
}

add1.multinom <- function(object, scope, sorted=T, trace=F)
{
  if(!inherits(object, "multinom")) stop("Not a multinom fit")
  if(!is.character(scope))
    scope <- add.scope(object, update.formula(object, scope, 
					   evaluate = F))
  if(!length(scope))
    stop("no terms in scope for adding to object")	
  n <- length(scope)
  ans <- matrix(nrow=n+1, ncol=2)
  dimnames(ans) <- list(c("<none>",paste("+",scope,sep="")), 
			c("df", "AIC"))
  ans[1,] <- c(object$edf, object$AIC)
  i <- 2
  for(tt in scope) {
    cat("trying +", tt,"\n")
    nobject <- update(object, paste("~ . +", tt), trace=trace)
    if(nobject$edf == object$edf) nobject$AIC <- NA
    ans[i,] <- c(nobject$edf, nobject$AIC)
    i <- i+1
  }
  if(sorted) ans <- ans[sort.list(ans[,2]),]
  ans
}
extractAIC.multinom <- function(fit, ...) c(fit$edf, c(fit$AIC))

vcov.multinom <- function(object)
{
  if(is.null(object$Hessian)) {
    cat("\nRe-fitting to get Hessian\n\n")
    object <- update(object, Hess=T, trace=F)
  }
  solve(object$Hessian)
}

summary.multinom <- function(object, correlation=T, digits=NULL, ...)
{        
  if(!is.null(cl <- object$call)) {
    cat("Call:\n")
    dput(cl)
  }
  if(is.null(digits)) digits <- options()$digits
  vc <- vcov(object)
  r <- length(object$vcoefnames)
  cat("\nCoefficients:\n")
  if(length(object$lev) == 2) {
    coef <- cbind(object$wts[1+(1:r)], sqrt(diag(vc)))
    dimnames(coef) <- list(object$vcoefnames, c("Values", "Std. Error"))
    print(coef, digits=digits)
  } else {
    coef <- matrix(object$wts, nrow = object$n[3], byrow=T)[, 1+(1:r), drop=F]
    if(length(object$lev)) dimnames(coef) <- list(object$lev, object$vcoefnames)
    if(length(object$lab)) dimnames(coef) <- list(object$lab, object$vcoefnames)
    coef <- coef[-1, , drop=F]
    print(coef, digits=digits)
    cat("\nStd. Errors:\n")
    coef <- matrix(sqrt(diag(vc)), nrow = object$n[3]-1, byrow=T)
    if(length(object$lev)) dimnames(coef) <- list(object$lev[-1], object$vcoefnames)
    if(length(object$lab)) dimnames(coef) <- list(object$lab[-1], object$vcoefnames)
    print(coef, digits=digits)    
  }
  cat("\nResidual Deviance:", format(object$deviance), "\n")
  cat("AIC:", format(object$AIC), "\n")
  if(correlation) {
    stddev<- sqrt(diag(vc))
    correl <- vc / outer(stddev, stddev)
    p <- dim(correl)[2]
    if(p > 1) {
      cat("\nCorrelation of Coefficients:\n")
      ll <- lower.tri(correl)
      correl[ll] <- format(round(correl[ll], digits))
      correl[!ll] <- ""
      print(correl[-1,  - p, drop = F], quote = F, ...)
    }
  }
  invisible()
}
