/*
   Siag, Scheme In A Grid
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * fileio_siag.c
 * 960209 New file format
 * 970206 Added string support
 * 970215 Broken out in a separate module
 * 970223 Added support for different numeric styles
 * 970630 Support for different interpreters
 * 970701 Removed explicit string format (now same as expression)
 * 970719 New tag m for embedding
 */

/*

   Brief description of the file format:

   #comment                     	A comment; ignored
   .sw width                    	Standard column width
   .sh height                   	Standard row height
   .sf format                   	Standard format code
   .cw col width                	Column width in pixels
   .rh row height               	Column height in pixels
   .fmtN				Format N
   .pN key				Property: length N bytes,
						value on following lines
   row col format       #               Empty cell
   row col format       "label          Cell containing a label
   row col format       =expression     Cell containing a SIOD expression
   row col format	==expression	Cell containing a C expression
   row col format	+int,expr	Expression with named interpreter
   row col format	$expression	String expression
   row col format	mObject		Embedded object

   Leading or trailing white space is not allowed
   Empty lines are ignored
   All widths and heights are in pixels
   Format codes are in decimal

 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../common/cmalloc.h"
#include "../common/fonts.h"
#include "../common/common.h"
#include "calc.h"
#include "types.h"

extern char *embed_load(char *);

extern char *getenv();

char *siag_format;

static int col_last_changed(buffer *b, int row, int sf)
{
	int i;

	if (row > b->alloc_lines) return 0;
	for (i = b->alloc_cols[row]; i > 1; i--)
		if (ret_type(b, row, i) != EMPTY ||
			ret_format(b, row, i) != sf)
			break;
	return i;
}

static int save(char *fn, buffer * buf)
/* Returns: 0 if successful, otherwise 1 */
{
	int i, j, lastcell, lr;
	FILE *fp;
	int sw = buf->sw, sh = buf->sh, sf = SIZE_10 | HELVETICA;
	int intp, fmt;
	char *taddr;
	property_list *p;

	if ((fp = fopen(fn, "w")) == (FILE *) 0)
		return 1;

	/* start by saving standard values */
	fprintf(fp, "# Creator: %s\n", VERSION);
	fprintf(fp, ".sw %d\n", sw);
	fprintf(fp, ".sh %d\n", sh);
	fprintf(fp, ".sf %d\n", sf);
	for (i = FMT_INTEGER; i <= FMT_USER5; i += FMT_INVISIBLE)
		fprintf(fp, ".fmt%d %s\n", i, fmt_get(NULL, i));

	lr = line_last_used(buf);
	fprintf(fp, "# %s\n# %d lines\n#\n", fn, lr);

	/* properties */
	for (p = buf->p_list; p; p = p->next) {
		fprintf(fp, ".p%d %s\n", strlen(p->value), p->key);
		fwrite(p->value, strlen(p->value), 1, fp);
		fprintf(fp, "\n");	/* terminate last line */
	}

	for (i = 1; i <= buf->used_lines; i++) {
		int height = cell_height(buf, i);
		if (height != sh)
			fprintf(fp, ".rh %d %d\n", i, height);
	}
	for (i = 1; i <= buf->used_cols; i++) {
		int width = cell_width(buf, i);
		if (width != sw)
			fprintf(fp, ".cw %d %d\n", i, width);
	}
	for (i = 1; i <= lr; i++) {
		fprintf(fp, "# Line %d\n", i);
		lastcell = col_last_changed(buf, i, sf);
		for (j = 1; j <= lastcell; j++) {
			switch (ret_type(buf, i, j)) {
			case EMPTY:
				if (ret_format(buf, i, j) != sf)
					fprintf(fp, "%d %d %d\t#\n", i, j, ret_format(buf, i, j));
				break;
			case LABEL:
				fprintf(fp, "%d %d %d\t\"%s\n", i, j,
					ret_format(buf, i, j),
					ret_text(buf, i, j));
				break;
			case EMBED:
				fprintf(fp, "%d %d %d\tm%s\n", i, j,
					ret_format(buf, i, j),
					ret_text(buf, i, j));
				break;
		/* ERRORs are stored as EXPRESSION */
			default:	/* EXPRESSION */
	/* Special case SIOD and C for backward compatibility. */
				intp = ret_interpreter(buf, i, j);
				taddr = ret_text(buf, i, j);
				if (intp == siod_interpreter &&
						taddr[0] == '=') {
					intp = C_interpreter;
					taddr++;
				}
				fmt = ret_format(buf, i, j);
				if (intp == siod_interpreter) {
					fprintf(fp, "%d %d %d\t=%s\n",
						i, j, fmt, taddr);
				} else if (intp == C_interpreter) {
					fprintf(fp, "%d %d %d\t==%s\n",
						i, j, fmt, taddr);
				} else {
					fprintf(fp, "%d %d %d\t+%s,%s\n",
						i, j, fmt,
						interpreter2name(intp), taddr);
				}
			}
		}
	}
	fprintf(fp, "# End of file %s\n", fn);
	fclose(fp);
	return 0;
}				/* save */

static int load(char *fn, buffer * buf)
/* Returns: 0 if successful, otherwise 1 */
{
	int i, j, intp;
	char *texti;
	FILE *fp;
	char instring[256], *p, *coords, *contents, *comma;
	int sw = 80, sh = 20, sf = SIZE_10 | HELVETICA, f;
	cval value;
	value.number = 0;

TRACEME((f,"load(%s)",fn))

	if ((fp = fopen(fn, "r")) == NULL)
		return 1;

	for (i = line_last_used(buf); i > 0; i--) {
		for (j = col_last_changed(buf, i, sf); j > 0; j--) {
			ins_data(buf, siod_interpreter, NULL,
				value, EMPTY, i, j);
			ins_format(buf, i, j, sf);
		}
	}

	while (fgets(instring, 250, fp) != NULL) {

TRACEME((f,"loaded '%s'",instring))

		/* strip off \r\n */
/*
   if ((p = strpbrk(instring, "\r\n")) != NULL) *p = '\0';
 */

		switch (instring[0]) {
		case '\0':
		case '#':
			break;
		case '.':
			if (!strncmp(instring, ".sw", 3)) {
				sscanf(instring, ".sw %d", &sw);
				buf->sw = sw;
			} else if (!strncmp(instring, ".sh", 3)) {
				sscanf(instring, ".sh %d", &sh);
				buf->sh = sh;
			} else if (!strncmp(instring, ".sf", 3)) {
				sscanf(instring, ".sf %d", &sf);
			} else if (!strncmp(instring, ".cw", 3)) {
				int w;
				sscanf(instring, ".cw %d %d", &i, &w);
				set_width(buf, i, w);
			} else if (!strncmp(instring, ".rh", 3)) {
				int h;
				sscanf(instring, ".rh %d %d", &i, &h);
				set_height(buf, i, h);
			} else if (!strncmp(instring, ".fmt", 4)) {
				int h;
				char b[256];
				sscanf(instring, ".fmt%d %[^\n]", &h, b);
				fmt_put(b, h);
			} else if (!strncmp(instring, ".p", 2)) {
				int n;
				char key[256], *value;
				sscanf(instring, ".p%d %s", &n, key);
				value = cmalloc(n+1);
				fread(value, n, 1, fp);
				value[n] = '\0';
				put_property(buf, key, value);
				cfree(value);
			}
			break;
		default:	/* cell definition */
			if ((coords = strtok(instring, "\t")) == NULL)
				break;
			if ((contents = strtok((char *) 0, "\t\n\r")) == NULL)
				break;
			if ((p = strtok(coords, " ")) == NULL)
				break;
			i = atoi(p);
			if ((p = strtok(NULL, " ")) == NULL)
				break;
			j = atoi(p);
			if ((p = strtok(NULL, " ")) == NULL)
				break;
			f = atoi(p);
			switch (contents[0]) {
			case '$':	/* string expression */
				if ((texti = (contents+1)) == 0)
					return 1;
				ins_data(buf, siod_interpreter, texti,
					value, STRING, i, j);
				ins_format(buf, i, j, f);
				break;
			case '=':	/* old style expression */
	/* Special case SIOD and C for backward compatibility. */
				if (contents[1] == '=') {	/* C */
					intp = C_interpreter;
					texti = (contents+2);
				} else {	/* SIOD */
					intp = siod_interpreter;
					texti = (contents+1);
				}
				if (texti == 0) return 1;
				ins_data(buf, intp, texti,
					value, EXPRESSION, i, j);
				ins_format(buf, i, j, f);
				break;
			case '+':	/* new style expression */
				if ((comma = strchr(contents, ',')) == NULL)
					return 1;
				*comma = '\0';
				intp = name2interpreter(contents+1);
				if (intp < 0) return 1;
				if (intp == siod_interpreter &&
						comma[1] == '=') {
			/* C expression masquerading as SIOD */
					intp = C_interpreter;
					texti = (comma+2);
				} else {
					texti = (comma+1);
				}
				if (texti == 0) return 1;
				ins_data(buf, intp,
					texti, value, EXPRESSION, i, j);
				ins_format(buf, i, j, f);
				break;
			case '\"':	/* label */
				if ((texti = (contents + 1)) == 0)
					return 1;
				ins_data(buf, siod_interpreter, texti,
					value, LABEL, i, j);
				ins_format(buf, i, j, f);
				break;
			case 'm':	/* embedded */
				if ((texti = (contents + 1)) == 0)
					return 1;
				embed_load(contents+1);
				ins_data(buf, siod_interpreter, texti,
					value, EMBED, i, j);
				ins_format(buf, i, j, f);
				break;
			default:	/* anything else means empty */
				ins_format(buf, i, j, f);
				break;
			}
			break;
		}
	}

	fclose(fp);

TRACEME((f,"successfully loaded"))

	return 0;
} /* load */

/* conservative file format guessing:
   1. extension .siag
   2. starts with the string "# Creator: Siag"
*/
static int myformat(char *fn)
{
	char *ext;
	FILE *fp;
	char b[10];

	ext = rindex(fn, '.');
	if (!ext) return 0;	/* no extension */
	if (cstrcasecmp(ext, ".siag"))
		return 0;	/* wrong extension */
	if ((fp = fopen(fn, "r")) == NULL)
		return 0;	/* can't open */
	if (fgets(b, 10, fp) && strncmp(b, "# Creator: Siag", 15)) {
		fclose(fp);
		return 1;
	}
	fclose(fp);
	return 0;
}

void fileio_siag_init()
{
	register_format(load, save, myformat, siag_format = "Siag (*.siag)");
}

