
/***

log.c - the iptraf logging facility
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***/

#include <curses.h>
#include <panel.h>
#include <stdio.h>
#include <time.h>
#include <netinet/in.h>
#include "if_ether.h"
#include "dirs.h"
#include "error.h"
#include "ifstats.h"
#include "serv.h"
#include "hostmon.h"

void openlogerr()
{
    int resp;
    errbox("Unable to open log file", ANYKEY_MSG, &resp);
}

FILE *opentlog()
{
    FILE *fd;

    fd = fopen(LOGFILE, "a");

    if (fd == NULL)
	openlogerr();

    return fd;
}

void genatime(time_t now, char *atime)
{
    strcpy(atime, ctime(&now));
    atime[strlen(atime) - 1] = '\0';
}

void writelog(int logging, FILE * fd, char *msg)
{
    char atime[26];

    if (logging) {
	genatime(time((time_t *) NULL), atime);
	fprintf(fd, "%s: %s\n", atime, msg);
    }
}

void writegstatlog(struct iftab *table, unsigned long nsecs, FILE * fd)
{
    struct iflist *ptmp = table->head;
    char atime[26];

    genatime(time((time_t *) NULL), atime);
    fprintf(fd, "\n*** General interface statistics log generated %s\n\n", atime);

    while (ptmp != NULL) {

	fprintf(fd, "%s: %lu total, %lu IP, %lu non-IP, %lu IP checksum errors",
		ptmp->ifname, ptmp->total, ptmp->iptotal, ptmp->noniptotal, ptmp->badtotal);

	if (nsecs > 5) {
	    fprintf(fd, ", average activity %.2f Kbits/s", (float) (ptmp->br * 8 / 1000) / (float) nsecs);
	    fprintf(fd, ", peak activity %.2f Kbits/s", ptmp->peakrate);
	    fprintf(fd, ", last 5-second activity %.2f kbits/s", ptmp->rate);
	}
	fprintf(fd, "\n");

	ptmp = ptmp->next_entry;
    }

    fprintf(fd, "\n%lu seconds running time\n", nsecs);
}

void writedstatlog(char *ifname, float activity, float pps,
		   float peakactivity, float peakpps,
		   struct iftotals *ts, unsigned long nsecs,
		   FILE * fd)
{
    char atime[26];

    genatime(time((time_t *) NULL), atime);

    fprintf(fd, "\n*** Detailed statistics for interface %s, generated %s\n\n",
	    ifname, atime);

    fprintf(fd, "Total: %lu packets, %lu bytes\n", ts->total, ts->bytestotal);
    fprintf(fd, "IP: %lu packets, %lu bytes\n", ts->iptotal, ts->ipbtotal);
    fprintf(fd, "TCP: %lu packets, %lu bytes\n", ts->tcptotal, ts->tcpbtotal);
    fprintf(fd, "UDP: %lu packets, %lu bytes\n", ts->udptotal, ts->udpbtotal);
    fprintf(fd, "ICMP: %lu packets, %lu bytes\n", ts->icmptotal, ts->icmpbtotal);
    fprintf(fd, "Other IP: %lu packets, %lu bytes\n", ts->othtotal, ts->othbtotal);
    fprintf(fd, "Non-IP: %lu packets, %lu bytes\n\n", ts->noniptotal, ts->nonipbtotal);

    fprintf(fd, "Packet Size Counts\n");
    fprintf(fd, "    1 to 100 bytes: %lu\n", ts->max100);
    fprintf(fd, "    101 to 200 bytes: %lu\n", ts->max200);
    fprintf(fd, "    201 to 300 bytes: %lu\n", ts->max300);
    fprintf(fd, "    301 to 400 bytes: %lu\n", ts->max400);
    fprintf(fd, "    401 to 500 bytes: %lu\n", ts->max500);
    fprintf(fd, "    501 to 600 bytes: %lu\n", ts->max600);
    fprintf(fd, "    601 to 700 bytes: %lu\n", ts->max700);
    fprintf(fd, "    701 to 800 bytes: %lu\n", ts->max800);
    fprintf(fd, "    801 to 900 bytes: %lu\n", ts->max900);
    fprintf(fd, "    901 to 1000 bytes: %lu\n", ts->max1000);
    fprintf(fd, "    1001 to 1100 bytes: %lu\n", ts->max1100);
    fprintf(fd, "    1101 to 1200 bytes: %lu\n", ts->max1200);
    fprintf(fd, "    1201 to 1300 bytes: %lu\n", ts->max1300);
    fprintf(fd, "    1301 to 1400 bytes: %lu\n", ts->max1400);
    fprintf(fd, "    1401 to 1500 bytes: %lu\n", ts->max1500);
    fprintf(fd, "    Larger than 1500 bytes: %lu\n\n", ts->g1500);

    if (nsecs > 5) {
	fprintf(fd, "Average activity: %.2f kbits/s, %.2f packets/s\n",
		((float) (ts->bytestotal * 8 / 1000) / (float) nsecs),
		((float) (ts->total) / (float) nsecs));

	fprintf(fd, "Peak activity: %.2f kbits/s, %.2f packets/s\n",
		peakactivity, peakpps);

	fprintf(fd, "Last 5-second interface activity: %.2f kbits/s, %.2f packets/s\n",
		activity, pps);
    }
    fprintf(fd, "IP checksum errors: %lu\n\n", ts->badtotal);
    fprintf(fd, "Running time: %lu seconds\n", nsecs);
}

void writeutslog(struct portlistent *list, unsigned long nsecs, FILE * fd)
{
    char atime[26];
    struct portlistent *ptmp = list;

    genatime(time((time_t *) NULL), atime);

    fprintf(fd, "\n*** TCP/UDP traffic log, generated %s\n\n", atime);

    while (ptmp != NULL) {
	if (ptmp->protocol == IPPROTO_TCP)
	    fprintf(fd, "TCP/%u: ", ptmp->port);
	else
	    fprintf(fd, "UDP/%u: ", ptmp->port);

	fprintf(fd, "%lu packets, %lu bytes total; ", ptmp->count, ptmp->bcount);
	fprintf(fd, "%lu packets, %lu bytes incoming; ", ptmp->icount, ptmp->ibcount);
	fprintf(fd, "%lu packets, %lu bytes outgoing\n\n", ptmp->ocount, ptmp->obcount);

	ptmp = ptmp->next_entry;
    }

    fprintf(fd, "\nRunning time: %lu seconds\n", nsecs);
}

void writeethlog(struct ethtabent *list, unsigned long nsecs, FILE * fd)
{
    char atime[26];
    struct ethtabent *ptmp = list;

    genatime(time((time_t *) NULL), atime);

    fprintf(fd, "\n*** Ethernet traffic log, generated %s\n\n", atime);

    while (ptmp != NULL) {
	if (ptmp->type == 0)
	    fprintf(fd, "\nHW Address: %s:\n", ptmp->un.desc.ascaddr);
	else {
	    fprintf(fd, "\tIncoming total %lu packets, %lu bytes; %lu IP packets\n",
		    ptmp->un.figs.inpcount, ptmp->un.figs.inbcount, ptmp->un.figs.inippcount);
	    fprintf(fd, "\tOutgoing total %lu packets, %lu bytes; %lu IP packets\n",
		    ptmp->un.figs.outpcount, ptmp->un.figs.outbcount, ptmp->un.figs.outippcount);

	    fprintf(fd, "\tAverage rates: %.2f kbits/s incoming, %.2f kbits/s outgoing\n",
	     (float) (ptmp->un.figs.inbcount * 8 / 1000) / (float) nsecs,
	    (float) (ptmp->un.figs.outbcount * 8 / 1000) / (float) nsecs);

	    if (nsecs > 5)
		fprintf(fd, "\tLast 5-second rates: %.2f kbits/s incoming, %.2f kbits/s outgoing\n",
			ptmp->un.figs.inrate, ptmp->un.figs.outrate);
	}

	ptmp = ptmp->next_entry;
    }

    fprintf(fd, "\nRunning time: %lu seconds\n", nsecs);
}
