/*
 * xpm.c - XPM stuff.
 */


#include <stdio.h>
#include <stdlib.h>

#include <string.h>
#include <ctype.h>
#include <math.h>

#include "arena.h"
#ifdef HAVE_LIBXPM
#  include XPM_HEADER_FILE_FOR_INCLUSION_IN_ARENA
#endif

#include "defs.h"
#include "types.h"
#include "bridge.h"
#ifndef HAVE_LIBXPM
#  include "colour.h"
#endif
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
#  include "debug.h"
# else
#  include "error.h"
#endif
#include "dither.h"
#include "image.h"
#include "main.h"
#include "status.h"
#include "util.h"


#ifndef HAVE_LIBXPM
/* Return pointer to the beginning of the last word in the string.
 * WARNING! The routine does NOT bother stripping trailing spaces!
 */
char* LastWord_in_the_String(char* theString)
{
 char *p;
 char* x = NULL;


 for (p = theString; p; )
   {
    while (*p == ' ') p++;
    if (*p == '\0') break;

    x = p;

    p =  ARENA_Index(p, ' ');
   }

 return x;
}
#endif


/*
 * Process loaded XPM. Allocate colours.
 */
ImageXyoke* ProcessXPM(char** xpm, unsigned int depth)
{
 ImageXyoke* theImageXcopule = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "ProcessXPM";
#endif


 if (xpm == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The XPM is NULL.\n");
#endif
    return NULL;
   }


#ifdef HAVE_LIBXPM
 if ((theImageXcopule = NewImageXyoke()))
   {
    XpmCreateImageFromData(display, xpm,
			   &(theImageXcopule->image), &(theImageXcopule->mask),
			   NULL);
    return theImageXcopule;
   }
  else
   {
    Warn(_("Failed to allocate space for X pair"));
    return NULL;
   }
# else
 {
  Bool IsTransparent = False;
  int c, i, j, ncolours, map[256];
  unsigned int width, height;
  unsigned short cr, cg, cb;
  ImageData* theImageData = NULL;
  char *name, *s;
  char** xpm_p = NULL;
  Colour *colours, colour;
  XColor xcolour;


  xpm_p = xpm;

  s = *(xpm_p++);
  sscanf(s, "%d %d %d", &width, &height, &ncolours);
  /* Actually 4 integers should be read: width, height, colours, chars/pixel.
   * I.e. it should be: sscanf(s, "%d %d %d %d", &w, &h, &ncolours, &cpp);
   */

  if (width && height && depth && ncolours)
    {
#ifdef ARENA_DEBUG
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam,
			" processing %dx%d XPM image.\n", width, height);
#endif
    }
   else
    {
#ifdef ARENA_DEBUG
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam, " ERROR! Got zero size for the XPM image.\n");
#endif
     return NULL;
    }

  switch (depth)
    {
     case 1:
     case 2:
     case 4:
     case 8:
     case 16:
     case 24:
       break;

     default:
#ifdef ARENA_DEBUG
       Arena_TracePrint(Iam, " Display depth %d unsupported.\n", depth);
# else
       Arena_PrintError(_("XPM for display depth %d unsupported\n"), depth);
#endif
       return NULL;
       break;
    }
  /* End ``switch (depth)'' */


  colours = (Colour*)Arena_MAlloc(ncolours * sizeof(Colour), False);

  for (i = 255; i--;) map[i] = -1;

  for (i = 0; i < ncolours; i++)
    {
     if ((s = *(xpm_p++)))
       {
	c = *s;
	name = LastWord_in_the_String(s);
       }
      else
       {
#ifdef ARENA_DEBUG
	if (IMAGE_TRACE)
	  Arena_TracePrint(Iam, " ERROR! Premature XPM buffer end.\n");
#endif
	Free(colours);
	return NULL;
       }

     if (name)
       {
	if (XParseColor(display, colourmap, name, &xcolour))
	  {
	   map[c] = i;

	   /* apply Gamma correction to map voltages to brightness values */
	   VoltageVector2BrightnessVector(xcolour.red,
					  xcolour.green,
					  xcolour.blue,
					  depth,
					  &(colours[i].red),
					  &(colours[i].green),
					  &(colours[i].blue),
					  &(colours[i].grey));
	  }
         else
          {
           map[c] = -1; /* The sign of transparent pixel */
           if (!IsTransparent) IsTransparent = True;
          }
       }
      else
       {
#ifdef ARENA_DEBUG
	if (IMAGE_TRACE)
	  Arena_TracePrint(Iam, " Failed to get a colour %d in XPM.\n", i);
#endif
       }
    }

  {
   unsigned int m;
   unsigned char theColourL[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
   ImageAssembly* theImageAssembly = NewImageAssembly(imaging, width, height,
						      depth,
						      (IsTransparent ? 1 : 0));

   if (theImageAssembly)
     {
      for (i = 0; i < height; i++)
	{
	 s = *(xpm_p++);
	 if (s == NULL) break;

	 for (j = 0; j < width; j++)
	   {
	    c = map[(int)(*s++)];
	    m = (c == -1 ? 0 : 1);

	    if (IsTransparent && !m)
	      {
	      /* Transparent */
	       cr = (windowColour >> 16) & 0xFF;
	       cg = (windowColour >>  8) & 0xFF;
	       cb = (windowColour      ) & 0xFF;
	      }
	     else
	      {
	       colour = colours[c];
	       cr = colour.red;
	       cg = colour.green;
	       cb = colour.blue;
	      }

	    if (ArenaDither(cr, cg, cb, j, i, depth, imaging, theColourL))
	      {
	       if (!NextPixel_to_ImageAssembly(theImageAssembly,
					       theColourL, m))
		 {
#ifdef ARENA_DEBUG
		  if (IMAGE_TRACE)
		    Arena_TracePrint(Iam, " ERROR! Failed to put pixel.\n");
#endif
		  FreeImageData(FinishImageAssembly(theImageAssembly));
		  Free(colours);
		  return NULL;
		 }
	      }
	     else
	      {
#ifdef ARENA_DEBUG
	       if (IMAGE_TRACE)
		 Arena_TracePrint(Iam, " ERROR! Failed to dither pixel.\n");
#endif
	       FreeImageData(FinishImageAssembly(theImageAssembly));
	       Free(colours);
	       return NULL;
	      }
	   }
	}

      theImageData = FinishImageAssembly(theImageAssembly);
     }
    else
     {
#ifdef ARENA_DEBUG
      if (IMAGE_TRACE)
	Arena_TracePrint(Iam, " ERROR! Failed to create image assembly.\n");
#endif
      Free(colours);
      return NULL;
     }
  }

  Free(colours);

  theImageXcopule = processImage_data2image(theImageData,
					    width, height, depth);

  theImageData->image = theImageData->mask = NULL;
  FreeImageData(theImageData);
 }
#endif /* ifndef HAVE_LIBXPM */

 return theImageXcopule;
}


/*
 * Load data from an XPM file and
 *   either call libXpm  XpmCreateImageFromBuffer()
 *       or use our own code
 * to process it.
 */
ImageXyoke* LoadXPMimage(Block* theBlock, unsigned int depth)
{
 ImageXyoke* theImageXcopule = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "LoadXPMimage";
#endif


 if (theBlock == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The block structure is NULL.\n");
#endif
    return NULL;
   }

 if (theBlock->buffer == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The block buffer is NULL.\n");
#endif
    return NULL;
   }


#ifdef HAVE_LIBXPM
 {
  if ((theImageXcopule = NewImageXyoke()))
    {
     char* p = theBlock->buffer + theBlock->next;

     XpmCreateImageFromBuffer(display, p,
			      &(theImageXcopule->image),
			      &(theImageXcopule->mask),
			      NULL);
     return theImageXcopule;
    }
   else
    {
#  ifdef ARENA_DEBUG
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam, " Failed to alloc image X structure.\n");
#  endif
     return NULL;
    }
 }
# else
 {
  char** xpm_buffer = NULL;
  char*  xpm_b = NULL;


  if ((xpm_b = (char*)Arena_CAlloc(theBlock->size - theBlock->next,
				   sizeof(char),
				   False)))
    {
     int theStrLen;
     int image_data_length = 0, image_data_strings = 0;
     char* s;
     char* p;
     char* pEnd;
     char* xpm_p = xpm_b;


     for (p = theBlock->buffer + theBlock->next,
	  pEnd = theBlock->buffer + theBlock->size;
	  pEnd - p >= 0;
	 )
       {
       /* Jump to the next quote character and extract quoted string */
        s =  ARENA_Index(++p, '"');
        if (s == NULL) break;

        p =  ARENA_Index(++s, '"');
        if (p == NULL) break;

        theStrLen = p - s;
        strncpy(xpm_p, s, theStrLen);
        xpm_p += theStrLen;
        *(xpm_p++) = '\0';

        image_data_strings++;
        image_data_length += (theStrLen + 1);

        /* Next line */
        p =  ARENA_Index(p, '\n');
        if (p == NULL) break;
       }

#  ifdef ARENA_DEBUG
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam,
		        " Read %d lines from the XPM, total %d bytes.\n",
		        image_data_strings, image_data_length);
#  endif

     if (image_data_strings)
       {
        int i;

        /* Now create strings array (char*[]) */

        if ((xpm_buffer = (char**)Arena_CAlloc(image_data_strings + 1,
					       sizeof(char*),
					       False)))
	  {
	   for (xpm_p = xpm_b, i = 0; i < image_data_strings; i++ )
	     {
	      xpm_buffer[i] = xpm_p;
	      xpm_p += Arena_StrLen(xpm_p) + 1;
	     }

	   xpm_buffer[image_data_strings] = NULL;

	   theImageXcopule = ProcessXPM(xpm_buffer, depth);

	   Free(xpm_b);
	   Free(xpm_buffer);

	   return theImageXcopule;
	  }
         else
	  {
#  ifdef ARENA_DEBUG
	   if (IMAGE_TRACE)
	     Arena_TracePrint(Iam, " Failed to alloc pointers for XPM.\n");
#  endif
	   Free(xpm_b);
	   return NULL;
	  }
       }
      else
       {
#  ifdef ARENA_DEBUG
        if (IMAGE_TRACE)
	  Arena_TracePrint(Iam, " No image in XPM.\n");
#  endif
        Free(xpm_b);
        return NULL;
       }
    }
   else
    {
#  ifdef ARENA_DEBUG
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam, " Failed to alloc buffer for XPM.\n");
#  endif
     return NULL;
    }
 }
#endif
}
