/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 */

#ifndef ICEWM_H
#define ICEWM_H

#include "config.h"

#define ICON_SMALL 16   // small: 16x16
#define ICON_LARGE 32   // large: 32x32

#define SCROLLBAR_SIZE 16

#define XPM_PATH LIBPATH
#define ICON_PATH LIBPATH "/icons"
#define ETC_PATH ETCPATH

#ifdef __EMX__ 
#define PATHSEP ';'
#define SLASH '\\'
#define ISSLASH(c) ((c) == '/' || (c) == '\\')
#else
#define PATHSEP ':'
#define SLASH '/'
#define ISSLASH(c) ((c) == '/')
#endif

extern "C" {
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <limits.h>
#include <unistd.h>
#include <stdarg.h>
#include <signal.h>
#include <ctype.h>
#include <fcntl.h>

#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include <X11/xpm.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#ifndef NO_MWM_HINTS
#include "MwmUtil.h"
#endif
#ifdef __EMX__
char* __XOS2RedirRoot(const char*);
#define REDIR_ROOT(path) __XOS2RedirRoot(path)
#else
#define REDIR_ROOT(path) (path)
#endif
}

#ifndef O_TEXT
#define O_TEXT 0
#endif
#ifndef O_BINARY
#define O_BINARY 0
#endif

#ifdef NEED_BOOL
enum bool_t { false = 0, true = 1};
typedef int bool;
#endif

#include "default.h"

#ifdef CONFIG_GUIEVENTS
#include "guievent.h"
#endif

#ifndef NO_CONFIGURE_MENUS
extern char *menuFile;
extern char *programFile;
#endif
#ifndef NO_WINDOW_OPTIONS
extern char *winOptFile;
#endif

extern char **progNames;
extern char **progCmds;
extern char ***progArgs;
extern int progCount;

#define MAXWORKSPACES 32 /* don't change */
#define ALLWORKSPACES 0xFFFFFFFF

extern int workspaceCount;
extern char *workspaceNames[MAXWORKSPACES];

#define DO_BORDER(x) 0 // (x)      // 0 looks much better
#define KEY_MODMASK(x) ((x) & (ControlMask | ShiftMask | Mod1Mask))
#define BUTTON_MODMASK(x) ((x) & (ControlMask | ShiftMask | Mod1Mask | Button1Mask | Button2Mask | Button3Mask))

#define ISMASK(w,e,n) (((w) & ~(n)) == (e))
#define HASMASK(w,e,n) ((((w) & ~(n)) & (e)) == (e))

#define ISLOWER(c) ((c) >= 'a' && (c) <= 'z')
#define TOUPPER(c) (ISLOWER(c) ? (c) - 'a' + 'A' : (c))

void die(int exitcode, const char *msg, ...);

class YApplication;
class YTimer;
class YWindow;
class YToolTip;
class YFocusWindow;
class YPopupWindow;
class YMenu;
class YButton;
class YRootWindow;
class YFrameWindow;
class YFrameControl;
class YFrameTitle;
class YFrameButton;
class YFrameClient;
class YListBox;
class YScrollBar;
class TaskBar;
class TaskBarApp;
class DesktopIcon;
class YListItem;

class TimerListener;

#ifdef DEBUG
extern int debug;
extern int debug_z;

void msg(const char *msg, ...);
void logEvent(XEvent xev);
void dumpZorder(const char *oper, YFrameWindow *w, YFrameWindow *a = 0);

#define DBG if (debug)
#define MSG(x) DBG msg x
#else

#define DBG if (0)
#define MSG(x)
#endif

extern YApplication *app;
extern TaskBar *taskBar;

enum WMCommand {
    cmdNone = 0,
    cmdSubmenu,
    cmdSeparator,
    cmdRestore,
    cmdMove,
    cmdSize,
    cmdMinimize, cmdHide,
    cmdMaximize, cmdMaximizeVert, cmdMaximizeHorz,
    cmdRollup,
    cmdLower,    cmdRaise,
    cmdClose,    cmdKill,
    cmdSysMenu,
    cmdWindowList,
    cmdActivate,
    cmdActivateWindow,
    cmdRefresh,
    cmdCloseSession,
    cmdExit,
    cmdExec,
    cmdRun,
    cmdRestart,
    cmdOccupyAll,
    cmdOccupyAllOrCurrent,
    cmdOccupyWorkspace,
    cmdOccupyOnlyWorkspace,
    cmdMoveToWorkspace,
    cmdActivateWorkspace,
    cmdTermProcess,
    cmdKillProcess,

    cmdCascade,
    cmdTileVertical,
    cmdTileHorizontal,
    cmdShow
};

enum PhaseType {
    phaseStartup,
    phaseShutdown,
    phaseRunning
};

struct WindowOption {
    char *name;
    char *icon;
    unsigned long functions, function_mask;
    unsigned long decors, decor_mask;
    unsigned long options, option_mask;
    int workspace;
};

typedef int FrameState;

class YColor {
public:
    YColor(unsigned short red, unsigned short green, unsigned short blue);
    YColor(const char *clr);

    void alloc();
    unsigned long pixel() { return color.pixel; }

    YColor *darker();
    YColor *brighter();
private:
    XColor color;
    YColor *fDarker;
    YColor *fBrighter;
};

class YFont {
public:
    YFont(const char *name);
    ~YFont();

    void alloc();
    XFontStruct *getFontStruct() { return font; }

    int height() { return font->max_bounds.ascent + font->max_bounds.descent; }
    int descent() { return font->max_bounds.descent; }
    int ascent() { return font->max_bounds.ascent; }
private:
    XFontStruct *font;
};

class YPixmap {
public:
    YPixmap(char *fileName);
    ~YPixmap();

    Pixmap pixmap() { return fPixmap; }
    unsigned int width() { return fWidth; }
    unsigned int height() { return fHeight; }
private:
    Pixmap fPixmap;
    unsigned int fWidth, fHeight;
};

class YMaskPixmap {
public:
    YMaskPixmap(char *fileName);
    ~YMaskPixmap();

    Pixmap pixmap() { return fPixmap; }
    Pixmap mask() { return fMask; }
    unsigned int width() { return fWidth; }
    unsigned int height() { return fHeight; }
private:
    Pixmap fPixmap;
    Pixmap fMask;
    unsigned int fWidth, fHeight;
};

class YIcon {
public:
    YIcon(char *fileName);
    ~YIcon();

    YMaskPixmap *large();
    YMaskPixmap *small();

    char *iconName() { return fPath; }
    YIcon *next() { return fNext; }
private:
    char *fPath;
    YMaskPixmap *fSmall;
    YMaskPixmap *fLarge;
    YIcon *fNext;
    int loadedS;
    int loadedL;

    int findIcon(char **fullPath, int size);
    YMaskPixmap *loadIcon(int size);
};

extern YIcon *defaultAppIcon;

class CommandListener {
public:
    virtual void handleCommand(WMCommand command, void *context, unsigned int modifiers) = 0;
    virtual void setPopupActive(YPopupWindow *popup) = 0;
};

class TimerListener {
public:
    virtual bool handleTimer(YTimer *timer) = 0;
};

class Graphics {
public:
    Graphics(YWindow *window);
    Graphics(YPixmap *pixmap);
    virtual ~Graphics();

    void copyArea(int x, int y, int width, int height, int dx, int dy) {
        XCopyArea(display, drawable, drawable, gc,
                  x, y, width, height, dx, dy);
    }
    void copyPixmap(YPixmap *pixmap, int x, int y, int width, int height, int dx, int dy) {
        XCopyArea(display, pixmap->pixmap(), drawable, gc,
                  x, y, width, height, dx, dy);
    }
    void drawPoint(int x, int y) {
        XDrawPoint(display, drawable, gc, x, y);
    }
    void drawLine(int x1, int y1, int x2, int y2) {
        XDrawLine(display, drawable, gc, x1, y1, x2, y2);
    }
    void drawRect(int x, int y, int width, int height) {
        XDrawRectangle(display, drawable, gc, x, y, width, height);
    }
    void drawChars(char *data, int offset, int len, int x, int y) {
        XDrawString(display, drawable, gc, x, y, data + offset, len);
    }
    void drawCharUnderline(int x, int y, char *str, int charPos) {
        int left = XTextWidth(font->getFontStruct(), str, charPos);
        int right = XTextWidth(font->getFontStruct(), str, charPos + 1) - 1;

        drawLine(x + left, y + 2, x + right, y + 2);
    }

    void drawPixmap(YPixmap *pix, int x, int y) {
        XCopyArea(display, pix->pixmap(), drawable, gc,
                  0, 0, pix->width(), pix->height(), x, y);
    }
    void drawClippedPixmap(Pixmap pix, Pixmap clip,
                           int x, int y, int w, int h, int toX, int toY);
    void drawMaskPixmap(YMaskPixmap *maskpix, int x, int y) {
        drawClippedPixmap(maskpix->pixmap(),
                          maskpix->mask(),
                          0, 0, maskpix->width(), maskpix->height(), x, y);
    }
    void fillRect(int x, int y, int width, int height) {
        XFillRectangle(display, drawable, gc,
                       x, y, width, height);
    }
    void fillPolygon(XPoint *points, int n, int shape, int mode) {
        XFillPolygon(display, drawable, gc, points, n, shape, mode);
    }
    void setColor(YColor *aColor) {
        color = aColor;
        XSetForeground(display, gc, color->pixel());
    }
    void setFont(YFont *aFont) {
        font = aFont;
        XSetFont(display, gc, font->getFontStruct()->fid);
    }

    void draw3DRect(int x, int y, int w, int h, bool raised);
    void drawBorderW(int x, int y, int w, int h, bool raised);
    void drawCenteredPixmap(int x, int y, int w, int h, YPixmap *pixmap);
    void drawOutline(int l, int t, int r, int b, int iw, int ih);
    void repHorz(YPixmap *pixmap, int x, int y, int w);
    void repVert(YPixmap *pixmap, int x, int y, int h);

    YColor *getColor() { return color; }
    YFont *getFont() { return font; }
    GC handle()  { return gc; }
private:
    Display *display;
    Drawable drawable;
    GC gc;

    YColor *color;
    YFont *font;
};

class YWindow {
public:
    YWindow(YWindow *aParent, Window win = 0);
    virtual ~YWindow();

    void setStyle(int aStyle);

    void show();
    void hide();
    virtual void raise();
    virtual void lower();

    void repaint();
    void repaintFocus();

    void reparent(YWindow *parent, int x, int y);

    void setFocus();

    void setGeometry(int x, int y, unsigned int width, unsigned int height);
    void setSize(unsigned int width, unsigned int height);
    void setPosition(int x, int y);
    virtual void configure(int x, int y, unsigned int width, unsigned int height);

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void paintFocus(Graphics &g, int x, int y, unsigned int w, unsigned int h);

    virtual void handleEvent(const XEvent &event);

    virtual void handleExpose(const XExposeEvent &expose);
    virtual void handleGraphicsExpose(const XGraphicsExposeEvent &graphicsExpose);
    virtual void handleConfigure(const XConfigureEvent &configure);
    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);
    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual void handleProperty(const XPropertyEvent &property);
    virtual void handleColormap(const XColormapEvent &colormap);
    virtual void handleFocus(const XFocusChangeEvent &focus);
    virtual void handleClientMessage(const XClientMessageEvent &message);
#if 0
    virtual void handleVisibility(const XVisibilityEvent &visibility);
    virtual void handleCreateWindow(const XCreateWindowEvent &createWindow);
#endif
    virtual void handleMap(const XMapEvent &map);
    virtual void handleUnmap(const XUnmapEvent &unmap);
    virtual void handleDestroyWindow(const XDestroyWindowEvent &destroyWindow);
    virtual void handleConfigureRequest(const XConfigureRequestEvent &configureRequest);
    virtual void handleMapRequest(const XMapRequestEvent &mapRequest);
#ifdef SHAPE
    virtual void handleShapeNotify(const XShapeEvent &shape);
#endif

    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleEndDrag(const XButtonEvent &down, const XButtonEvent &up);

    virtual void handleClose();
    
    void setPointer(Cursor pointer);
    void setGrabPointer(Cursor pointer);
    void grabKey(int key, unsigned int modifiers);

    void captureEvents();
    void releaseEvents();

    Window handle();
    YWindow *parent() { return fParentWindow; }

    Graphics &getGraphics();

    int x() { return fX; }
    int y() { return fY; }
    unsigned int width() { return fWidth; }
    unsigned int height() { return fHeight; }

    int visible() { return (flags & wfVisible) ? 1 : 0; }
    int created() { return (flags & wfCreated) ? 1 : 0; }
    int adopted() { return (flags & wfAdopted) ? 1 : 0; }
    int destroyed() { return (flags & wfDestroyed) ? 1 : 0; }
    int unmapped() { return (flags & wfUnmapped) ? 1 : 0; }

    virtual void donePopup(YPopupWindow * /*command*/);

    typedef enum {
        wsOverrideRedirect = 1 << 0,
        wsSaveUnder        = 1 << 1
    } WindowStyle;

    virtual bool isFocusTraversable();
    bool isFocused();
    bool isEnabled() { return fEnabled; }
    void setEnabled(bool enable);
    void nextFocus();
    void prevFocus();
    void requestFocus();
    void setFocus(YWindow *window);
    virtual void gotFocus();
    virtual void lostFocus();

    bool isToplevel() { return fToplevel; }
    void setToplevel(bool toplevel) { fToplevel = toplevel; }

    void installAccelerator(unsigned int key, unsigned int mod, YWindow *win);
    void removeAccelerator(unsigned int key, unsigned int mod, YWindow *win);

    void setToolTip(const char *tip);

    void mapToGlobal(int &x, int &y);
    void mapToWindow(int &x, int &y);

private:
    typedef enum {
        wfVisible   = 1 << 0,
        wfCreated   = 1 << 1,
        wfAdopted   = 1 << 2,
        wfDestroyed = 1 << 3,
        wfUnmapped  = 1 << 4
    } WindowFlags;

    void create();
    void destroy();

    void insertWindow();
    void removeWindow();
    
    YWindow *fParentWindow;
    YWindow *fNextWindow;
    YWindow *fPrevWindow;
    YWindow *fFirstWindow;
    YWindow *fLastWindow;
    YWindow *fFocusedWindow;

    Window fHandle;
    unsigned long flags;
    unsigned long fStyle;
    int fX, fY;
    unsigned int fWidth, fHeight;
    Cursor fPointer;
    int unmapCount;
    Graphics *fGraphics;

    bool fEnabled;
    bool fToplevel;

    typedef struct YAccelerator {
        unsigned int key;
        unsigned int mod;
        YWindow *win;
        YAccelerator *next;
    } YAccelerator;

    YAccelerator *accel;
    YToolTip *fToolTip;

    static XButtonEvent fClickEvent;
    static YWindow *fClickWindow;
    static Time fClickTime;
    static int fClickCount;
    static int fClickDrag;
    static unsigned int fClickButton;
    static unsigned int fClickButtonDown;
    static YTimer *fToolTipTimer;
};

class YToolTip: public YWindow, public TimerListener {
public:
    YToolTip(YWindow *aParent, Window win = 0);
    virtual ~YToolTip();
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    void setText(const char *tip);
    virtual bool handleTimer(YTimer *t);
    void locate(YWindow *w, const XCrossingEvent &crossing);

private:
    void display();

    char *fText;
};

class MoveSizeStatus: public YWindow {
public:
    MoveSizeStatus(YWindow *aParent, Window win = 0);
    virtual ~MoveSizeStatus();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    void setStatus(YFrameWindow *frame, int x, int y, int width, int height);
    void setStatus(YFrameWindow *frame);
    void begin(YFrameWindow *frame);
    void end() { hide(); }
private:
    int fX, fY, fW, fH;
};

class YPopupWindow: public YWindow {
public:
    YPopupWindow(YWindow *aParent);
    virtual ~YPopupWindow();

    virtual void sizePopup();
    
    void popup(YWindow *forWindow, CommandListener *commandWindow,
               int x, int y,
               int x_delta, int y_delta,
               unsigned int flags);
    void popup(YWindow *forWindow, CommandListener *commandWindow,
               unsigned int flags);
    void popdown();

    virtual void updatePopup();
    void finishPopup(WMCommand command, void *context, unsigned int modifiers);
    void cancelPopup();

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);

    virtual void activatePopup();
    virtual void deactivatePopup();

    unsigned int popupFlags() { return fFlags; }

    CommandListener *commandListener() { return fCommandListener; }
    YPopupWindow *prevPopup() { return fPrevPopup; }
    void setPrevPopup(YPopupWindow *prevPopup) { fPrevPopup = prevPopup; }

    enum {
        pfButtonDown        = 1 << 0,
        pfCanFlipVertical   = 1 << 1,
        pfCanFlipHorizontal = 1 << 2,
        pfFlipVertical      = 1 << 3,
        pfFlipHorizontal    = 1 << 4,
        pfNoPointerChange   = 1 << 5,
        pfPopupMenu         = 1 << 6
    } PopupFlags;

private:
    unsigned int fFlags;
    YWindow *fForWindow;
    CommandListener *fCommandListener;
    YPopupWindow *fPrevPopup;
};

class YMenu: public YPopupWindow {
public:
    class YMenuItem {
    public:
        YMenuItem(const char *name, int hotCharPos, const char *param, WMCommand command, YMenu *submenu, void *context = 0);
        YMenuItem() { fName = 0; fHotCharPos = -1; fParam = 0; fCommand = cmdSeparator; fContext = 0; fEnabled = 0; fSubmenu = 0; }
        ~YMenuItem();
        char *name() { return fName; }
        char *param() { return fParam; }
        WMCommand command() { return fCommand; }
        void *context() { return fContext; }
        YMenu *submenu() { return fSubmenu; }
        int hotChar() { return (fName && fHotCharPos >= 0) ? fName[fHotCharPos] : -1; }
        int hotCharPos() { return fHotCharPos; }

        YMaskPixmap *getPixmap() { return fPixmap; }
        void setPixmap(YMaskPixmap *pixmap);
        void setChecked(bool c);
        int isChecked() { return fChecked; }
        int isEnabled() { return fEnabled; }
        void setEnabled(bool e) { fEnabled = e; }
    private:
        char *fName;
        char *fParam;
        WMCommand fCommand;
        void *fContext;
        int fHotCharPos;
        YMenu *fSubmenu;

        YMaskPixmap *fPixmap;
        bool fChecked;
        bool fEnabled;
    };

    YMenu(YWindow *parent);
    virtual ~YMenu();

    virtual void sizePopup();
    virtual void activatePopup();
    virtual void deactivatePopup();
    virtual void donePopup(YPopupWindow *popup);
    
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);
    
    YMenuItem *add(YMenuItem *item);
    YMenuItem *addItem(const char *name, int hotCharPos, const char *param, WMCommand command, void *context = 0);
    YMenuItem *addSubmenu(const char *name, int hotCharPos, YMenu *submenu);
    YMenuItem *addSeparator();
    void removeAll();
    YMenuItem *findItem(WMCommand command);
    YMenuItem *findSubmenu(YMenu *sub);

    void enableCommand(WMCommand cmd); // cmdNone == All
    void disableCommand(WMCommand cmd); // cmdNone == All

    int itemCount() { return fItemCount; }
    YMenuItem *item(int n) { return fItems[n]; }

private:
    int fItemCount;
    YMenuItem **fItems;
    int selectedItem;
    int paintedItem;
    int paramPos;
    YPopupWindow *fPopup;
    YPopupWindow *fPopupActive;

    void paintItem(Graphics &g, int i, int &y, int paint);
    void paintItems();
    int findItemPos(int item, int &x, int &y);
    int findItem(int x, int y);
    int findActiveItem(int cur, int direction);
    void focusItem(int item, int submenu, int byMouse);
    int activateItem(int no, int byMouse, unsigned int modifiers);
    int onCascadeButton(int selectedItem, int x, int y);
};

class YWindowListMenu: public YMenu {
public:
    YWindowListMenu(YWindow *parent);
    virtual void updatePopup();
};

class SwitchWindow: public YPopupWindow {
public:
    SwitchWindow(YWindow *parent);
    virtual ~SwitchWindow();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    void begin(int next);

    virtual void activatePopup();
    virtual void deactivatePopup();
    
    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);

    void destroyedFrame(YFrameWindow *frame);
private:
    YFrameWindow *fActiveWindow;
};

class YRootWindow: public YWindow {
public:
    YRootWindow(YWindow *parent, Window win = 0);
    virtual ~YRootWindow();

    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual bool handleKey(const XKeyEvent &key);

    virtual void handleConfigureRequest(const XConfigureRequestEvent &configureRequest);
    virtual void handleMapRequest(const XMapRequestEvent &mapRequest);
    virtual void handleMap(const XMapEvent &map);
    virtual void handleUnmap(const XUnmapEvent &unmap);
    virtual void handleDestroyWindow(const XDestroyWindowEvent &destroyWindow);
    virtual void handleClientMessage(const XClientMessageEvent &message);
};

class YFrameControl: public YWindow {
public:
    YFrameControl(YWindow *parent, YFrameWindow *frame, Window win = 0): YWindow((YWindow *)parent, win) { fFrame = frame; }
    virtual ~YFrameControl() { }

    YFrameWindow *frame() { return fFrame; }
    virtual void setFrame(YFrameWindow *newFrame);
private:
    YFrameWindow *fFrame;
};

class YFrameClient: public YFrameControl {
public:
    YFrameClient(YWindow *parent, YFrameWindow *frame, Window win = 0);
    virtual ~YFrameClient();

    void configureClient(const XConfigureRequestEvent &configureRequest);

    virtual void handleProperty(const XPropertyEvent &property);
    virtual void handleColormap(const XColormapEvent &colormap);
    virtual void handleUnmap(const XUnmapEvent &unmap);
    virtual void handleMap(const XMapEvent &map);
    virtual void handleDestroyWindow(const XDestroyWindowEvent &destroyWindow);
#ifdef SHAPE
    virtual void handleShapeNotify(const XShapeEvent &shape);
#endif

    Window clientWindow() { return fClient; }

    unsigned int border() { return fBorder; }
    void setBorder(unsigned int border) { fBorder = border; }
    void setFrame(YFrameWindow *newFrame);

    enum {
        wpDeleteWindow = 1 << 0,
        wpTakeFocus    = 1 << 1
    } WindowProtocols;

    unsigned long protocols() { return fProtocols; }
    void getProtocols();

    void getTransient();
    Window ownerWindow() { return fTransientFor; }

    unsigned char *windowTitle() { return fWindowTitle; }
    unsigned char *iconTitle() { return fIconTitle; }

    void setWindowTitle(unsigned char *aWindowTitle);
    void setIconTitle(unsigned char *aIconTitle);

    void sendMessage(Atom msg, Time timeStamp = CurrentTime);

    XSizeHints *sizeHints() { return fSizeHints; }
    void getSizeHints();

    XClassHint *classHint() { return fClassHint; }
    void getClassHint();

    enum {
        csKeepX = 1,
        csKeepY = 2,
        csRound = 4
    };
    
    void constrainSize(int &w, int &h, int flags = 0);
    void gravityOffsets (int &xp, int &yp);

    XWMHints *hints() { return fHints; }

    Colormap colormap() { return fColormap; }
    void setColormap(Colormap cmap);

    void setWMState(FrameState state);
    FrameState getWMState();

#ifdef SHAPE
    int shaped() { return fShaped; }
    void queryShape();
#endif

#ifndef NO_MWM_HINTS
    MwmHints *mwmHints() { return fMwmHints; }
    void getMwmHints();
    CARD32 mwmFunctions();
    CARD32 mwmDecors();
#endif
    
    void setClientWorkspaces();
    void getClientWorkspaces();
    int allWorkspaces() { return fAllWorkspaces; }
    void setAllWorkspaces(int allWorkspaces);
    void setWorkspaces(unsigned long workspace);
    unsigned long workspaces() { return fWorkspaces; }

private:
    int fProtocols;
    int haveButtonGrab;
    Window fClient;
    unsigned int fBorder;
    XSizeHints *fSizeHints;
    XClassHint *fClassHint;
    XWMHints *fHints;
    Colormap fColormap;
    int fShaped;

    unsigned char *fWindowTitle;
    unsigned char *fIconTitle;

    MwmHints *fMwmHints;

    unsigned long fWorkspaces;
    int fAllWorkspaces;

    Window fTransientFor;
};

class YClientContainer: public YFrameControl {
public:
    YClientContainer(YWindow *parent, YFrameWindow *frame);
    //: YFrameControl(parent, frame) { isActive = 0; movingWindow = 0; }
    virtual ~YClientContainer();

    virtual void handleButton(const XButtonEvent &button);
    virtual void handleConfigureRequest(const XConfigureRequestEvent &configureRequest);
    virtual void handleMapRequest(const XMapRequestEvent &mapRequest);
    virtual void handleCrossing(const XCrossingEvent &crossing);

    void grabButtons();
    void releaseButtons();
    int haveGrab() { return haveButtonGrab; }
private:
    int haveButtonGrab;
};

class YFrameTitleBar: public YFrameControl {
public:
    YFrameTitleBar(YWindow *parent, YFrameWindow *frame): YFrameControl(parent, frame) { isActive = 0; movingWindow = 0; }
    virtual ~YFrameTitleBar() { }

    void activate();
    void deactivate();
    
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);

    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleEndDrag(const XButtonEvent &down, const XButtonEvent &up);

    int active() { return isActive; }

private:
    int buttonDownX, buttonDownY;
    int movingWindow;
    int isActive;
};

class YFrameButton: public YFrameControl {
public:
    YFrameButton(YWindow *parent, YFrameWindow *frame, WMCommand command, WMCommand command2 = cmdNone);
    virtual ~YFrameButton();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    void activate();
    void deactivate();

    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleCrossing(const XCrossingEvent &crossing);

    void setCommand(WMCommand command, WMCommand command2 = cmdNone);

    virtual void donePopup(YPopupWindow *popup);
    void popupMenu();
    void popdownMenu();
        
private:
    WMCommand fCommand;
    WMCommand fCommand2;

    int isDown;
    int isHilited;
    int isActive;
    int wasPopupActive;
    YPopupWindow *fPopup;
};

class YFocusWindow: public YWindow {
public:
    YFocusWindow(YWindow *parent, Window win = 0): YWindow(parent, win) { }
    virtual ~YFocusWindow() { }
};

class YFrameWindow: public YFocusWindow, public CommandListener {
public:
    YFrameWindow(YWindow *parent, YFrameClient *client);
    virtual ~YFrameWindow();

    void manage(YFrameClient *client);
    void unmanage();
    void sendConfigure();

    void createPointerWindows();
    void grabKeys();

    void activate();
    void deactivate();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleEndDrag(const XButtonEvent &down, const XButtonEvent &up);
    virtual void handleMotion(const XMotionEvent &motion);
    virtual void handleCrossing(const XCrossingEvent &crossing);

    virtual void handleCommand(WMCommand command, void *context, unsigned int modifiers);
    
    void wmRestore();
    void wmMinimize();
    void wmMaximize();
    void wmMaximizeVert();
    void wmMaximizeHorz();
    void wmRollup();
    void wmHide();
    void wmShow();
    void wmLower();
    void doLower();
    void wmRaise();
    void doRaise();
    void wmClose();
    void wmKill();
    void wmNextWindow();
    void wmPrevWindow();
    void wmLastWindow();
    void wmMove();
    void wmSize();
    void wmOccupyAll();
    void wmOccupyAllOrCurrent();
    void wmOccupyWorkspace(int workspace);
    void wmOccupyOnlyWorkspace(int workspace);
    void wmMoveToWorkspace(int workspace);

    void DoMaximize(int what);

    void loseFocus();
    void setFocus();
    bool focused() { return fFocused; }
    void focusOnMap();

    YFrameClient *client() { return fClient; }
    YFrameTitleBar *titlebar() { return fTitleBar; }
    YClientContainer *container() { return fClientContainer; }

    void startMoveSize(int doMove, int byMouse,
                       int sideX, int sideY,
                       int mouseXroot, int mouseYroot);
    void endMoveSize();
    void moveWindow(int newX, int newY);
    void manualPlace();

    void drawOutline(int x, int y, int w, int h);
    int handleMoveKeys(const XKeyEvent &xev, int &newX, int &newY);
    int handleResizeKeys(const XKeyEvent &key,
                         int &newX, int &newY, int &newWidth, int &newHeight,
                         int incX, int incY);
    void handleMoveMouse(const XMotionEvent &motion, int &newX, int &newY);
    void handleResizeMouse(const XMotionEvent &motion,
                           int &newX, int &newY, int &newWidth, int &newHeight);

    void outlineMove();
    void outlineResize();
    
    void constrainPositionByModifier(int &x, int &y, const XMotionEvent &motion);
    void constrainMouseToWorkspace(int &x, int &y);

    void getDefaultOptions();

    bool canSize();
    bool canMove();
    bool canClose();
    bool canMaximize();
    bool canMinimize();
    bool canRollup();
    bool canHide();
    bool canLower();
    bool canRaise();

    void insertFrame();
    void removeFrame();
    void setAbove(YFrameWindow *aboveFrame); // 0 = at the bottom
    void setBelow(YFrameWindow *belowFrame); // 0 = at the top
    YFrameWindow *next() { return fNextFrame; }
    YFrameWindow *prev() { return fPrevFrame; }
    void setNext(YFrameWindow *next) { fNextFrame = next; }
    void setPrev(YFrameWindow *prev) { fPrevFrame = prev; }

    typedef enum {
        fwfVisible    = 1 << 0, // visible windows only
        fwfCycle      = 1 << 1, // cycle when bottom(top) reached
        fwfBackward   = 1 << 2, // go up in zorder (default=down)
        fwfNext       = 1 << 3, // start from next window
        fwfFocusable  = 1 << 4, // only focusable windows
        fwfWorkspace  = 1 << 5, // current workspace only
        fwfSame       = 1 << 6, // return same if no match and same matches
        fwfLayers     = 1 << 7  // find windows in other layers
    } FindWindowFlags;

    YFrameWindow *findWindow(int flag);
    int maximized() { return (style & fsMaximized) ? 1 : 0; }
    int minimized() { return (style & fsMinimized) ? 1 : 0; }
    int rolledup() { return (style & fsRolledup) ? 1 : 0; }
    int hidden() { return (style & fsHidden) ? 1 : 0; }

    YFrameButton *menuButton() { return fMenuButton; }
    YFrameButton *closeButton() { return fCloseButton; }
    YFrameButton *minimizeButton() { return fMinimizeButton; }
    YFrameButton *maximizeButton() { return fMaximizeButton; }
    void updateMenu();

    virtual void raise();
    virtual void lower();

    void setPopupActive(YPopupWindow *popup);
    void popupSystemMenu();

    virtual void configure(int x, int y, unsigned int width, unsigned int height);
    
    void configureClient(const XConfigureRequestEvent &configureRequest);

    FrameState frameState() { return fState; }
    void setFrameState(FrameState state);
    void changeState(FrameState state);

#ifdef SHAPE
    void setShape();
#endif

    void updateTitle();
    void updateIconTitle();

    enum {
        ffMove          = (1 << 0),
        ffResize        = (1 << 1),
        ffClose         = (1 << 2),
        ffMinimize      = (1 << 3),
        ffMaximize      = (1 << 4),
        ffHide          = (1 << 5),
        ffRollup        = (1 << 6)
    } YFrameFunctions;

    enum {
        fdTitleBar      = (1 << 0),
        fdSysMenu       = (1 << 1),
        fdBorder        = (1 << 2),
        fdResize        = (1 << 3),
        fdClose         = (1 << 4),
        fdMinimize      = (1 << 5),
        fdMaximize      = (1 << 6),
        fdHide          = (1 << 7)
    } YFrameDecors;

    enum YFrameOptions {
        foOnTop         = (1 << 0),
        foAllWorkspaces = (1 << 1),
        foIgnoreTaskBar = (1 << 2),
        foIgnoreWinList = (1 << 3),
        foFullKeys      = (1 << 4)
    };
        
    
    unsigned long frameFunctions() { return fFrameFunctions; }
    unsigned long frameDecors() { return fFrameDecors; }
    unsigned long frameOptions() { return fFrameOptions; }
    void getFrameHints();
#ifndef NO_WINDOW_OPTIONS
    void getWindowOptions(WindowOption &opt);
#endif

    YMenu *windowMenu();

    YIcon *clientIcon() { return fFrameIcon ? fFrameIcon : defaultAppIcon; }
    void getClientIcon();

    unsigned int borderX() {
        return (frameDecors() & fdResize) ? wsBorderX : wsDlgBorderX;
    }
    unsigned int borderY() {
        return (frameDecors() & fdResize) ? wsBorderX : wsDlgBorderX;
    }
    unsigned int titleY() {
        return (frameDecors() & fdTitleBar) ? wsTitleBar : 0;
    }
    
    void layoutTitleBar();
    void layoutButtons();
    void layoutResizeIndicators();
    void layoutClient();

    void workspaceShow();
    void workspaceHide();

    void addToMenu(YMenu *menu);

    int visibleOn(int workspace) {
        return (client()->allWorkspaces() ||
                (client()->workspaces() & (1 << workspace))) ? 1 : 0;
    }
    int visibleNow();

    int layer() {
        return fWindowLayer;
    }
    YFrameWindow *nextLayer();
    YFrameWindow *prevLayer();
    YListItem *winListItem() { return fWinListItem; }
    void setWinListItem(YListItem *i) { fWinListItem = i; }

    bool isMinimized() { return (style & fsMinimized) ? true : false; }

    void showIcon();
    void hideIcon();

    void addAsTransient();
    void removeAsTransient();
    void addTransients();
    void removeTransients();

    void setTransient(YFrameWindow *transient) { fTransient = transient; }
    void setNextTransient(YFrameWindow *nextTransient) { fNextTransient = nextTransient; }
    void setOwner(YFrameWindow *owner) { fOwner = owner; }
    YFrameWindow *transient() { return fTransient; }
    YFrameWindow *nextTransient() { return fNextTransient; }
    YFrameWindow *owner() { return fOwner; }

    bool isModal();
    bool hasModal();
    bool isFocusable();

private:
    typedef enum {
        fsMinimized       = 1 << 0,
        fsMaximized       = 1 << 1,
        fsRolledup        = 1 << 2,
        fsHidden          = 1 << 3,
        fsWorkspaceHidden = 1 << 4
    } FrameStateFlags;

    bool fFocused;
    unsigned long fFrameFunctions;
    unsigned long fFrameDecors;
    unsigned long fFrameOptions;

    int normalX, normalY;
    unsigned int normalWidth, normalHeight;

    YFrameClient *fClient;
    YClientContainer *fClientContainer;
    YFrameTitleBar *fTitleBar;
    YFrameButton *fCloseButton;
    YFrameButton *fMenuButton;
    YFrameButton *fMaximizeButton;
    YFrameButton *fMinimizeButton;

    YPopupWindow *fPopupActive;

    unsigned long style;
    FrameState fState;
    FrameState fSavedFrameState;

    int buttonDownX, buttonDownY;
    int grabX, grabY;
    int movingWindow, sizingWindow;
    int sizeByMouse;
    int origX, origY, origW, origH;

    YFrameWindow *fNextFrame; // window below this one
    YFrameWindow *fPrevFrame; // window above this one

    Window topSide, leftSide, rightSide, bottomSide;
    Window topLeftCorner, topRightCorner, bottomLeftCorner, bottomRightCorner;
    int indicatorsVisible;

    TaskBarApp *fTaskBarApp;
    DesktopIcon *fDesktopIcon;
    YListItem *fWinListItem;
    YIcon *fFrameIcon;
    int fWindowLayer;

    YFrameWindow *fOwner;
    YFrameWindow *fTransient;
    YFrameWindow *fNextTransient;
};

class YButtonListener {
public:
    virtual void ButtonClick(YButton *button) = 0;
};

class YButton: public YWindow {
public:
    YButton(YWindow *parent, WMCommand command, void *context);
    YButton(YWindow *parent, YPopupWindow *popup);
    virtual ~YButton();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void paintFocus(Graphics &g, int x, int y, unsigned int w, unsigned int h);
    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleCrossing(const XCrossingEvent &crossing);

    void setCommand(WMCommand command, void *context);
    void setPopup(YPopupWindow *popup);
    void setPixmap(YPixmap *pixmap);
    void setText(const char *str, int hot = -1);
    void setPressed(int pressed);

    virtual bool isFocusTraversable();

    virtual void donePopup(YPopupWindow *popup);

    void setListener(YButtonListener *listener) { fListener = listener; }

    void popupMenu();

private:
    WMCommand fCommand;
    YPopupWindow *fPopup;
    YPixmap *fPixmap;
    char *fText;
    void *fContext;
    int fPressed;
    int fHotCharPos;
    int hotKey;

    int selected;
    int isActive;
    int wasPopupActive;

    YButtonListener *fListener;

    void popup();
    void popdown();
};

class DesktopIcon: public YFocusWindow {
public:
    DesktopIcon(YFrameWindow *frame, YWindow *aParent, Window win = 0);
    virtual ~DesktopIcon();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);

    YFrameWindow *frame() { return fFrame; }
private:
    YFrameWindow *fFrame;
    int selected;
};

class TaskBarApp: public YWindow {
public:
    TaskBarApp(YFrameWindow *frame, YWindow *aParent, Window win = 0);
    virtual ~TaskBarApp();

    virtual bool isFocusTraversable();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleCrossing(const XCrossingEvent &crossing);

    YFrameWindow *frame() { return fFrame; }

    void setShown(int show);
    int getShown() { return fShown; }
    
    TaskBarApp *getNext() { return fNext; }
    TaskBarApp *getPrev() { return fPrev; }
    void setNext(TaskBarApp *next) { fNext = next; }
    void setPrev(TaskBarApp *prev) { fPrev = prev; }
private:
    YFrameWindow *fFrame;
    TaskBarApp *fPrev, *fNext;
    int fShown;
    int selected;
};

class YClock: public YWindow, public TimerListener {
public:
    YClock(YWindow *aParent, Window win = 0);
    virtual ~YClock();

    virtual void handleCrossing(const XCrossingEvent &crossing);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);

    virtual bool handleTimer(YTimer *t);

    YPixmap *getPixmap(char ch);
    int calcWidth(char *s, int count);
private:
    int selected;
    YTimer *clockTimer;
};

class TaskBar: public YWindow {
public:
    TaskBar(YWindow *aParent, Window win = 0);
    virtual ~TaskBar();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);

    void insert(TaskBarApp *tapp);
    void remove(TaskBarApp *tapp);

    TaskBarApp *addApp(YFrameWindow *frame);
    void removeApp(YFrameWindow *frame);

    void relayout();

    YClock *clock() { return fClock; }
    YButton *workspaceButton(int n) { return fWorkspaceButton[n]; }

    void popupStartMenu();
    void popupWindowListMenu();
    TaskBarApp *getFirst() { return fFirst; }
    TaskBarApp *getLast() { return fLast; }
private:
    TaskBarApp *fFirst, *fLast;
    int fCount;
    YClock *fClock;
    YButton *fWinList;
    YButton *fApplications;
    YButton **fWorkspaceButton;
    int leftX, rightX;

    friend class WindowList;
    friend class YListBox;
};

class YScrollListener {
public:
    virtual void scroll(YScrollBar *scroll, int delta) = 0;
    virtual void move(YScrollBar *scroll, int pos) = 0;
};

class YListItem {
public:
    YListItem(YFrameWindow *frame);
    virtual ~YListItem();
    
    YListItem *getNext();
    YListItem *getPrev();
    void setNext(YListItem *next);
    void setPrev(YListItem *prev);
    
    YFrameWindow *frame();
    int getSelected();
    void setSelected(int aSelected);
private:
    YFrameWindow *fFrame;
    YListItem *fPrevItem, *fNextItem;
    int fSelected;
};

class YListBox: public YWindow, public YScrollListener, public CommandListener {
public:
    YListBox(YScrollBar *vert, YWindow *aParent, Window win = 0);
    virtual ~YListBox();

    int addItem(YListItem *item);
    void removeItem(YListItem *item);
    YListItem *findItem(int mouseY, int &no);
    int itemCount();
    YListItem *item(int no);

    virtual bool handleKey(const XKeyEvent &key);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);
    //virtual void handleBeginDrag(const XButtonEvent &down, const XMotionEvent &motion);
    virtual void handleDrag(const XButtonEvent &down, const XMotionEvent &motion);
    //virtual void handleEndDrag(const XButtonEvent &down, const XButtonEvent &up);
    virtual void handleMotion(const XMotionEvent &motion);

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void scroll(YScrollBar *sb, int delta);
    virtual void move(YScrollBar *sb, int pos);

    void selectItem(YListItem *i);
    int getLineHeight();
    void ensureVisibility(int item);
    void focusVisible();

    virtual void handleCommand(WMCommand command, void *context, unsigned int modifiers);
    virtual void setPopupActive(YPopupWindow *popup);

    virtual bool isFocusTraversable();

    YListItem *getFirst() { return fFirst; }
    YListItem *getLast() { return fLast; }

private:
    YScrollBar *fVerticalScroll;
    int fOffsetY;
    YListItem *fFirst, *fLast;
    int fFocusedItem;
    int fSelectStart, fSelectEnd;
    int fSelecting;

    void setSelection();
    void updateSelection(int apply);
};

class YScrollBar: public YWindow {
public:
    enum Orientation {
        Vertical, Horizontal
    };

    YScrollBar(YWindow *aParent);
    YScrollBar(Orientation anOrientation, YWindow *aParent);
    YScrollBar(Orientation anOrientation,
               int aValue, int aVisibleAmount, int aMin, int aMax,
               YWindow *aParent);
    virtual ~YScrollBar();

    Orientation getOrientation() { return fOrientation; }
    int getMaximum() { return fMaximum; }
    int getMinimum() { return fMinimum; }
    int getVisibleAmount() { return fVisibleAmount; }
    int getUnitIncrement() { return fUnitIncrement; }
    int getBlockIncrement() { return fBlockIncrement; }
    int getValue() { return fValue; }

    void setOrientation(Orientation anOrientation);
    void setMaximum(int aMaximum);
    void setMinimum(int aMinimum);
    void setVisibleAmount(int aVisibleAmount);
    void setUnitIncrement(int anUnitIncrement);
    void setBlockIncrement(int aBlockIncrement);
    void setValue(int aValue);
    void setValues(int aValue, int aVisibleAmount, int aMin, int aMax);
    
private:
    Orientation fOrientation;
    int fMaximum;
    int fMinimum;
    int fValue;
    int fVisibleAmount;
    int fUnitIncrement;
    int fBlockIncrement;
public:
    void scroll(int delta);
    void move(int pos);

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void handleButton(const XButtonEvent &button);
    virtual void handleMotion(const XMotionEvent &motion);
    void setListener(YScrollListener *notify) { fListener = notify; }
private:
    enum {
        goUp, goDown, goPageUp, goPageDown, goPosition, goNone
    } fScrollTo;

    int fGrabDelta;
    YScrollListener *fListener;
};

class WindowList: public YFrameClient {
public:
    WindowList(YWindow *aParent, Window win = 0);
    virtual ~WindowList();

    void handleFocus(const XFocusChangeEvent &focus);
    virtual void handleClose();

    virtual void configure(int x, int y, unsigned int width, unsigned int height);
    void relayout();

    YListItem *addWindowListApp(YFrameWindow *frame);
    void removeWindowListApp(YListItem *item);

    void showFocused();

    YListBox *getList() { return list; }

private:
    YListBox *list;
    YScrollBar *scrollVert;
};

class MailBoxStatus: public YWindow {
public:
    enum MailBoxState {
        mbxNoMail,
        mbxHasMail,
        mbxHasUnreadMail,
        mbxHasNewMail
    };
    
    MailBoxStatus(const char *mailBox, const char *mailCommand, YWindow *aParent);
    virtual ~MailBoxStatus();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual void handleClick(const XButtonEvent &down, const XButtonEvent &up, int count);

    void checkMail();
    void newMailArrived();
    
private:
    const char *fMailBox;
    const char *fMailCommand;
    time_t fLastCheckTime;
    unsigned long fLastSize;
    MailBoxState fState;
};

class CtrlAltDelete: public YWindow, public YButtonListener {
public:
    CtrlAltDelete(YWindow *parent);
    virtual ~CtrlAltDelete();

    virtual void paint(Graphics &g, int x, int y, unsigned int width, unsigned int height);
    virtual bool handleKey(const XKeyEvent &key);
    virtual void ButtonClick(YButton *button);

    void activate();
    void deactivate();
private:
    YButton *lockButton;
    YButton *logoutButton;
    YButton *cancelButton;
};

class YTimer {
public:
    YTimer();
    ~YTimer();

    void setListener(TimerListener *listener) { fListener = listener; }
    TimerListener *getListener() { return fListener; }
    
    void setInterval(long ms) { fInterval = ms; }
    long getInterval() { return fInterval; }

    void startTimer();
    void stopTimer();
    bool isRunning() { return fRunning; }

private:
    TimerListener *fListener;
    long fInterval;
    bool fRunning;
    YTimer *fPrev;
    YTimer *fNext;

    struct timeval timeout;

    friend class YApplication;
};

class YApplication {
public:
    YApplication(const char *displayName = 0);
    virtual ~YApplication();

    int mainLoop();
    void exitLoop(int exitCode);
    void exit(int exitCode);
    
    void wmCloseSession();

    YRootWindow *root() { return fRoot; }
    Display *display() { return fDisplay; }

    void saveEventTime(XEvent &xev);
    Time getEventTime() { return lastEventTime; }

    void manageClients();
    void unmanageClients();

    YFrameClient *manageWindow(Window win);
    YFrameWindow *mapWindow(Window win, int newClient = 1);
    YFrameClient *unmapWindow(Window win);
    void unmanageWindow(Window win);
    void destroyedWindow(Window win);
    YFrameClient *manageClient(Window win);
    YFrameClient *findClient(Window win);
    
    void unmanageFrame(YFrameWindow *frame);

    YFrameWindow *getFrame(Window win);
    YFrameWindow *createFrame(YFrameClient *client);

    YFrameClient *getClient(Window win);
    YFrameClient *createClient(Window win);

    void setFocus(YFrameWindow *window);
    void loseFocus(YFrameWindow *window);
    YFrameWindow *focus() { return fFocus; }
    void activate(YFrameWindow *frame);

    void installColormap(Colormap cmap);
    void setColormapWindow(YFrameWindow *frame);
    YFrameWindow *colormapWindow() { return fColormapWindow; }

    int minX();
    int minY();
    int maxX();
    int maxY();
    int maxWidth() { return maxX() - minX(); }
    int maxHeight() { return maxY() - minY(); }

    void placeWindow(YFrameWindow *frame, int x, int y, int newClient = 1);

    YFrameWindow *top(int layer) { return fTop[layer]; }
    void setTop(int layer, YFrameWindow *top);
    YFrameWindow *bottom(int layer) { return fBottom[layer]; }
    void setBottom(int layer, YFrameWindow *bottom) { fBottom[layer] = bottom; }

    YFrameWindow *topLayer() { return fTop[1] ? fTop[1] : fTop[0]; }
    YFrameWindow *bottomLayer() { return fBottom[0] ? fBottom[0] : fBottom[1]; }

    void restackWindows(YFrameWindow *win, int oper = 0);
    void focusTopWindow();
    void relocateWindows(int dx, int dy);

    YMenu *createWindowMenu(YMenu *menu, int workspace);
    int windowCount(int workspace);
    void popupWindowListMenu(int x, int y);
    virtual void handleCommand(WMCommand command, void *context, unsigned int modifiers);

    YPopupWindow *popup() { return fPopup; }
    void popup(YWindow *forWindow, YPopupWindow *popup);
    void popdown(YPopupWindow *popdown);

    int activeWorkspace() { return fActiveWorkspace; }
    void activateWorkspace(int workspace);
    int workspaceCount() { return ::workspaceCount; }
    char *workspaceName(int workspace) { return ::workspaceNames[workspace]; }

    int grabEvents(YWindow *win, Cursor ptr, unsigned int eventMask, int grabMouse = 1, int grabKeyboard = 1, int grabTree = 0);
    int releaseEvents();
    void handleGrabEvent(YWindow *win, XEvent &xev);

    void captureGrabEvents(YWindow *win);
    void releaseGrabEvents(YWindow *win);

    void dispatchEvent(YWindow *win, XEvent &e);

#ifdef CONFIG_GUIEVENTS
    void signalGuiEvent(GUIEvent ge);
#endif

    void popupStartMenu();
    void popupWindowListMenu();
    
private:
    Display *fDisplay;
    YRootWindow *fRoot;
    YFrameWindow *fFocus;
    Time lastEventTime;
    YPopupWindow *fPopup;

    YFrameWindow *fTop[2];
    YFrameWindow *fBottom[2];

    YFrameWindow *fColormapWindow;

    int fGrabTree;
    YWindow *fXGrabWindow;
    int fGrabMouse;
    YWindow *fGrabWindow;

    int fActiveWorkspace;

    YTimer *fFirstTimer, *fLastTimer;

    int fLoopLevel;
    int fExitLoop;
    int fExitCode;
    int fExitApp;

    friend class YTimer;
    
    void registerTimer(YTimer *t);
    void unregisterTimer(YTimer *t);
    void getTimeout(struct timeval *timeout);
    void handleTimeouts();
};

extern Colormap defaultColormap;

extern GC blackGC;
extern GC whiteGC;
extern GC outlineGC;
extern GC clipPixmapGC;

extern YColor *black;
extern YColor *white;

extern YColor *buttonBg;
extern YColor *buttonFg;

extern YColor *scrollBarBg;
extern YColor *scrollBarArrow;
extern YColor *scrollBarSlider;

extern YColor *listBoxBg;
extern YColor *listBoxFg;
extern YColor *listBoxSelBg;
extern YColor *listBoxSelFg;

extern YColor *menuBg;
extern YColor *menuItemFg;
extern YColor *activeMenuItemBg;
extern YColor *activeMenuItemFg;
extern YColor *disabledMenuItemFg;

extern YColor *activeBorderBg;
extern YColor *inactiveBorderBg;

extern YColor *activeTitleBarBg;
extern YColor *activeTitleBarFg;

extern YColor *inactiveTitleBarBg;
extern YColor *inactiveTitleBarFg;

extern YColor *taskBarBg;

extern YColor *normalTaskBarAppFg;
extern YColor *normalTaskBarAppBg;

extern YColor *activeTaskBarAppFg;
extern YColor *activeTaskBarAppBg;

extern YColor *minimizedTaskBarAppFg;
extern YColor *minimizedTaskBarAppBg;

extern YColor *statusFg;
extern YColor *statusBg;

extern YColor *switchFg;
extern YColor *switchBg;

extern YColor *toolTipBg;
extern YColor *toolTipFg;

extern YColor *clockBg;
extern YColor *clockFg;

#ifdef CONFIG_LOOK_PIXMAP
extern YPixmap *frameTL[2][2];
extern YPixmap *frameT[2][2];
extern YPixmap *frameTR[2][2];
extern YPixmap *frameL[2][2];
extern YPixmap *frameR[2][2];
extern YPixmap *frameBL[2][2];
extern YPixmap *frameB[2][2];
extern YPixmap *frameBR[2][2];

extern YPixmap *titleL[2];
extern YPixmap *titleT[2];
extern YPixmap *titleM[2];
extern YPixmap *titleB[2];
extern YPixmap *titleR[2];

extern YPixmap *menuButton[2];
#endif

extern YPixmap *closePixmap[2];
extern YPixmap *minimizePixmap[2];
extern YPixmap *maximizePixmap[2];
extern YPixmap *restorePixmap[2];

extern YPixmap *startPixmap;
extern YPixmap *windowsPixmap;
extern YPixmap *mailPixmap;
extern YPixmap *unreadMailPixmap;
extern YPixmap *newMailPixmap;

extern YPixmap *PixNum[10];
extern YPixmap *PixSpace;
extern YPixmap *PixColon;
extern YPixmap *PixSlash;
extern YPixmap *PixA;
extern YPixmap *PixP;
extern YPixmap *PixM;
extern YPixmap *PixDot;

/* small pixmaps (16x16) */
extern YPixmap pixWinRestore;
extern YPixmap pixWinMove;
extern YPixmap pixWinSize;
extern YPixmap pixWinMinimize;
extern YPixmap pixWinMaximize;
extern YPixmap pixWinRaise;
extern YPixmap pixWinLower;
extern YPixmap pixWinRollup;
extern YPixmap pixWinHide;
extern YPixmap pixWinMoveToWorkspace;
extern YPixmap pixWinOccupy;
extern YPixmap pixWinClose;
extern YPixmap pixWinList;

extern YFont *titleFont;
extern YFont *menuFont;
extern YFont *statusFont;
extern YFont *switchFont;
extern YFont *normalTaskBarFont;
extern YFont *activeTaskBarFont;
extern YFont *windowListFont;
extern YFont *toolTipFont;
extern YFont *clockFont;

extern Cursor leftPointer;
extern Cursor rightPointer;

extern Cursor movePointer;

extern Cursor sizeRightPointer;
extern Cursor sizeTopRightPointer;
extern Cursor sizeTopPointer;
extern Cursor sizeTopLeftPointer;
extern Cursor sizeLeftPointer;
extern Cursor sizeBottomLeftPointer;
extern Cursor sizeBottomPointer;
extern Cursor sizeBottomRightPointer;

#ifdef SHAPE
extern int shapesSupported;
extern int shapeEventBase, shapeErrorBase;
#endif

extern XContext windowContext;
extern XContext frameContext;
extern XContext clientContext;

extern YMenu *windowMenu;
extern YMenu *occupyMenu;
extern YMenu *moveMenu;
extern YMenu *windowListMenu;
extern YMenu *windowListPopup;
extern YMenu *rootMenu;
extern MoveSizeStatus *statusMoveSize;
extern SwitchWindow *switchWindow;
extern MailBoxStatus *mailBoxStatus;
extern WindowList *windowList;
extern CtrlAltDelete *ctrlAltDelete;

extern Atom _XA_WM_PROTOCOLS;
extern Atom _XA_WM_TAKE_FOCUS;
extern Atom _XA_WM_DELETE_WINDOW;
extern Atom _XA_WM_STATE;
extern Atom _XA_WM_CHANGE_STATE;
extern Atom _XATOM_MWM_HINTS;
extern Atom _XA_WM_COLORMAP_WINDOWS;
extern Atom _XA_WM_WORKSPACE;
extern Atom _XA_WM_WORKSPACE_NAMES;

extern const char *clrBlack;
extern const char *clrWhite;

extern int configurationLoaded;
extern PhaseType phase;

void getOptions(int &argc, char **argv);
void loadConfiguration(const char *fileName);
void loadMenus(const char *fileName, bool programs = false);
#ifndef NO_WINDOW_OPTIONS
void loadWinOptions(const char *optFile);
#endif
int addProgram(const char *name, const char *command, char **args = 0);
void addWorkspace(const char *name);
int findPath(const char *path, int mode, const char *name, char **fullname);
char *findProgFile(const char *name);

YIcon *getIcon(char *name);

void runProgram(const char *str, char **args);
void runCommand(const char *prog);

WindowOption *getWindowOption(char *name, int create);

#endif
