/****************************************************************************
 * This module is all original code 
 * by Rob Nation 
 * Copyright 1993, Robert Nation
 *     You may use this code for any purpose, as long as the original
 *     copyright remains in the source code and all documentation
 ****************************************************************************/

/****************************************************************************
 *
 * Assorted odds and ends
 *
 **************************************************************************/


#include <config.h>

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <stdarg.h>

#include "scwm.h"
#include <X11/Xatom.h>
#include "menus.h"
#include "misc.h"
#include "screen.h"
#include "window.h"
#include "events.h"

ScwmWindow *FocusOnNextTimeStamp = NULL;

char NoName[] = "Untitled";	/* name if no name in XA_WM_NAME */
char NoClass[] = "NoClass";	/* Class if no res_class in class hints */
char NoResource[] = "NoResource";	/* Class if no res_name in class hints */

/**************************************************************************
 * 
 * Releases dynamically allocated space used to store window/icon names
 *
 **************************************************************************/
void 
free_window_names(ScwmWindow * tmp, Bool nukename, Bool nukeicon)
{
  if (!tmp)
    return;

  if (nukename && nukeicon) {
    if (tmp->name == tmp->icon_name) {
      if (tmp->name != NoName && tmp->name != NULL)
	XFree(tmp->name);
      tmp->name = NULL;
      tmp->icon_name = NULL;
    } else {
      if (tmp->name != NoName && tmp->name != NULL)
	XFree(tmp->name);
      tmp->name = NULL;
      if (tmp->icon_name != NoName && tmp->icon_name != NULL)
	XFree(tmp->icon_name);
      tmp->icon_name = NULL;
    }
  } else if (nukename) {
    if (tmp->name != tmp->icon_name
	&& tmp->name != NoName
	&& tmp->name != NULL)
      XFree(tmp->name);
    tmp->name = NULL;
  } else {			/* if (nukeicon) */
    if (tmp->icon_name != tmp->name
	&& tmp->icon_name != NoName
	&& tmp->icon_name != NULL)
      XFree(tmp->icon_name);
    tmp->icon_name = NULL;
  }

  return;
}

/***************************************************************************
 *
 * Handles destruction of a window 
 *
 ****************************************************************************/
void 
Destroy(ScwmWindow * sw)
{
  int i;
  extern ScwmWindow *ButtonWindow;
  extern ScwmWindow *colormap_win;
  extern Bool PPosOverride;

  /*
   * Warning, this is also called by HandleUnmapNotify; if it ever needs to
   * look at the event, HandleUnmapNotify will have to mash the UnmapNotify
   * into a DestroyNotify.
   */
  if (!sw)
    return;

  XUnmapWindow(dpy, sw->frame);

  if (!PPosOverride)
    XSync(dpy, 0);

  if (sw == Scr.Hilite)
    Scr.Hilite = NULL;

  Broadcast(M_DESTROY_WINDOW, 3, sw->w, sw->frame,
	    (unsigned long) sw, 0, 0, 0, 0);

  if (Scr.PreviousFocus == sw)
    Scr.PreviousFocus = NULL;

  if (ButtonWindow == sw)
    ButtonWindow = NULL;

  if ((sw == Scr.Focus) && (sw->flags & ClickToFocus)) {
    if (sw->next) {
      HandleHardFocus(sw->next);
    } else
      SetFocus(Scr.NoFocusWin, NULL, 1);
  } else if (Scr.Focus == sw)
    SetFocus(Scr.NoFocusWin, NULL, 1);

  if (sw == FocusOnNextTimeStamp)
    FocusOnNextTimeStamp = NULL;

  if (sw == Scr.Ungrabbed)
    Scr.Ungrabbed = NULL;

  if (sw == Scr.pushed_window)
    Scr.pushed_window = NULL;

  if (sw == colormap_win)
    colormap_win = NULL;

  XDestroyWindow(dpy, sw->frame);
  XDeleteContext(dpy, sw->frame, ScwmContext);

  XDestroyWindow(dpy, sw->Parent);

  XDeleteContext(dpy, sw->Parent, ScwmContext);

  XDeleteContext(dpy, sw->w, ScwmContext);

  if ((sw->icon_w) && (sw->flags & PIXMAP_OURS) &&
    sw->picIcon)
    XFreePixmap(dpy, sw->picIcon->picture);

  if (sw->icon_w) {
    XDestroyWindow(dpy, sw->icon_w);
    XDeleteContext(dpy, sw->icon_w, ScwmContext);
  }
  if ((sw->flags & ICON_OURS) && (sw->icon_pixmap_w != None))
    XDestroyWindow(dpy, sw->icon_pixmap_w);
  if (sw->icon_pixmap_w != None)
    XDeleteContext(dpy, sw->icon_pixmap_w, ScwmContext);

  if (sw->flags & TITLE) {
    XDeleteContext(dpy, sw->title_w, ScwmContext);
    for (i = 0; i < Scr.nr_left_buttons; i++)
      XDeleteContext(dpy, sw->left_w[i], ScwmContext);
    for (i = 0; i < Scr.nr_right_buttons; i++)
      if (sw->right_w[i] != None)
	XDeleteContext(dpy, sw->right_w[i], ScwmContext);
  }
  if (sw->flags & BORDER) {
    for (i = 0; i < 4; i++)
      XDeleteContext(dpy, sw->sides[i], ScwmContext);
    for (i = 0; i < 4; i++)
      XDeleteContext(dpy, sw->corners[i], ScwmContext);
  }
  sw->prev->next = sw->next;
  if (sw->next != NULL)
    sw->next->prev = sw->prev;
  free_window_names(sw, True, True);
  if (sw->wmhints)
    XFree((char *) sw->wmhints);
  /* removing NoClass change for now... */
  if (sw->class.res_name && sw->class.res_name != NoResource)
    XFree((char *) sw->class.res_name);
  if (sw->class.res_class && sw->class.res_class != NoClass)
    XFree((char *) sw->class.res_class);
  if (sw->mwm_hints)
    XFree((char *) sw->mwm_hints);

  if (sw->cmap_windows != (Window *) NULL)
    XFree((void *) sw->cmap_windows);

  /* XSCM */
  invalidate_window(sw->schwin);

  free((char *) sw);

  if (!PPosOverride)
    XSync(dpy, 0);
  return;
}



/**************************************************************************
 *
 * Removes expose events for a specific window from the queue 
 *
 *************************************************************************/
int 
flush_expose(Window w)
{
  XEvent dummy;
  int i = 0;

  while (XCheckTypedWindowEvent(dpy, w, Expose, &dummy))
    i++;
  return i;
}

/* CoerceEnterNotifyOnCurrentWindow()
 * Pretends to get a HandleEnterNotify on the
 * window that the pointer currently is in so that
 * the focus gets set correctly from the beginning
 * Note that this presently only works if the current
 * window is not click_to_focus;  I think that
 * that behaviour is correct and desirable. --11/08/97 gjb */
void
CoerceEnterNotifyOnCurrentWindow()
{
  extern ScwmWindow *swCurrent; /* from events.c */
  Window child, root;
  int root_x, root_y;
  int win_x, win_y;
  Bool f = XQueryPointer(dpy, Scr.Root, &root,
			 &child, &root_x, &root_y, &win_x, &win_y, &JunkMask);
  if (f && child != None) {
    Event.xany.window = child;
    swCurrent = SwFromWindow(dpy,child);
    HandleEnterNotify();
    swCurrent = None;
  }
}


/***********************************************************************
 *
 *  Procedure:
 *	RestoreWithdrawnLocation
 * 
 *  Puts windows back where they were before Scwm took over 
 *
 ************************************************************************/
void 
RestoreWithdrawnLocation(ScwmWindow * tmp, Bool restart)
{
  int a, b, w2, h2;
  unsigned int bw, mask;
  XWindowChanges xwc;

  if (!tmp)
    return;

  if (XGetGeometry(dpy, tmp->w, &JunkRoot, &xwc.x, &xwc.y,
		   &JunkWidth, &JunkHeight, &bw, &JunkDepth)) {
    XTranslateCoordinates(dpy, tmp->frame, Scr.Root, xwc.x, xwc.y,
			  &a, &b, &JunkChild);
    xwc.x = a + tmp->xdiff;
    xwc.y = b + tmp->ydiff;
    xwc.border_width = tmp->old_bw;
    mask = (CWX | CWY | CWBorderWidth);

    /* We can not assume that the window is currently on the screen.
     * Although this is normally the case, it is not always true.  The
     * most common example is when the user does something in an
     * application which will, after some amount of computational delay,
     * cause the window to be unmapped, but then switches screens before
     * this happens.  The XTranslateCoordinates call above will set the
     * window coordinates to either be larger than the screen, or negative.
     * This will result in the window being placed in odd, or even
     * unviewable locations when the window is remapped.  The followin code
     * forces the "relative" location to be within the bounds of the display.
     *
     * gpw -- 11/11/93
     *
     * Unfortunately, this does horrendous things during re-starts, 
     * hence the "if(restart)" clause (RN) 
     *
     * Also, fixed so that it only does this stuff if a window is more than
     * half off the screen. (RN)
     */

    if (!restart) {
      /* Don't mess with it if its partially on the screen now */
      if ((tmp->frame_x < 0) || (tmp->frame_y < 0) ||
	  (tmp->frame_x >= Scr.MyDisplayWidth) ||
	  (tmp->frame_y >= Scr.MyDisplayHeight)) {
	w2 = (tmp->frame_width >> 1);
	h2 = (tmp->frame_height >> 1);
	if ((xwc.x < -w2) || (xwc.x > (Scr.MyDisplayWidth - w2))) {
	  xwc.x = xwc.x % Scr.MyDisplayWidth;
	  if (xwc.x < -w2)
	    xwc.x += Scr.MyDisplayWidth;
	}
	if ((xwc.y < -h2) || (xwc.y > (Scr.MyDisplayHeight - h2))) {
	  xwc.y = xwc.y % Scr.MyDisplayHeight;
	  if (xwc.y < -h2)
	    xwc.y += Scr.MyDisplayHeight;
	}
      }
    }
    XReparentWindow(dpy, tmp->w, Scr.Root, xwc.x, xwc.y);

    if ((tmp->flags & ICONIFIED) && (!(tmp->flags & SUPPRESSICON))) {
      if (tmp->icon_w)
	XUnmapWindow(dpy, tmp->icon_w);
      if (tmp->icon_pixmap_w)
	XUnmapWindow(dpy, tmp->icon_pixmap_w);
    }
    XConfigureWindow(dpy, tmp->w, mask, &xwc);
    if (!restart)
      XSync(dpy, 0);
  }
}


/****************************************************************************
 *
 * Records the time of the last processed event. Used in XSetInputFocus
 *
 ****************************************************************************/
Time lastTimestamp = CurrentTime;	/* until Xlib does this for us */

Bool 
StashEventTime(XEvent * ev)
{
  Time NewTimestamp = CurrentTime;

  switch (ev->type) {
  case KeyPress:
  case KeyRelease:
    NewTimestamp = ev->xkey.time;
    break;
  case ButtonPress:
  case ButtonRelease:
    NewTimestamp = ev->xbutton.time;
    break;
  case MotionNotify:
    NewTimestamp = ev->xmotion.time;
    break;
  case EnterNotify:
  case LeaveNotify:
    NewTimestamp = ev->xcrossing.time;
    break;
  case PropertyNotify:
    NewTimestamp = ev->xproperty.time;
    break;
  case SelectionClear:
    NewTimestamp = ev->xselectionclear.time;
    break;
  case SelectionRequest:
    NewTimestamp = ev->xselectionrequest.time;
    break;
  case SelectionNotify:
    NewTimestamp = ev->xselection.time;
    break;
  default:
    return False;
  }
  /* Only update is the new timestamp is later than the old one, or
   * if the new one is from a time at least 30 seconds earlier than the
   * old one (in which case the system clock may have changed) */
  if ((NewTimestamp > lastTimestamp) || ((lastTimestamp - NewTimestamp) > 30000))
    lastTimestamp = NewTimestamp;
  if (FocusOnNextTimeStamp) {
    SetFocus(FocusOnNextTimeStamp->w, FocusOnNextTimeStamp, 1);
    FocusOnNextTimeStamp = NULL;
  }
  return True;
}

/*****************************************************************************
 *
 * Grab the pointer and keyboard
 *
 ****************************************************************************/
Bool 
GrabEm(int cursor)
{
  int i = 0, val = 0;
  unsigned int mask;

  XSync(dpy, 0);
  /* move the keyboard focus prior to grabbing the pointer to
   * eliminate the enterNotify and exitNotify events that go
   * to the windows */
  if (Scr.PreviousFocus == NULL)
    Scr.PreviousFocus = Scr.Focus;
  SetFocus(Scr.NoFocusWin, NULL, 0);
  mask = ButtonPressMask | ButtonReleaseMask | ButtonMotionMask | PointerMotionMask
    | EnterWindowMask | LeaveWindowMask;
  while ((i < 1000) && (val = XGrabPointer(dpy, Scr.Root, True, mask,
				     GrabModeAsync, GrabModeAsync, Scr.Root,
				    Scr.ScwmCursors[cursor], CurrentTime) !=
			GrabSuccess)) {
    i++;
    /* If you go too fast, other windows may not get a change to release
     * any grab that they have. */
    sleep_ms(1);
  }

  /* If we fall out of the loop without grabbing the pointer, its
     time to give up */
  XSync(dpy, 0);
  if (val != GrabSuccess) {
    return False;
  }
  return True;
}


/*****************************************************************************
 *
 * UnGrab the pointer and keyboard
 *
 ****************************************************************************/
void 
UngrabEm()
{
  Window w;

  XSync(dpy, 0);
  XUngrabPointer(dpy, CurrentTime);

  if (Scr.PreviousFocus != NULL) {
    w = Scr.PreviousFocus->w;

    /* if the window still exists, focus on it */
    if (w) {
      SetFocus(w, Scr.PreviousFocus, 0);
    }
    Scr.PreviousFocus = NULL;
  }
  XSync(dpy, 0);
}



/****************************************************************************
 *
 * Keeps the "StaysOnTop" windows on the top of the pile.
 * This is achieved by clearing a flag for OnTop windows here, and waiting
 * for a visibility notify on the windows. Exeption: OnTop windows which are
 * obscured by other OnTop windows, which need to be raised here.
 *
 ****************************************************************************/
void 
KeepOnTop()
{
  ScwmWindow *t;

  /* flag that on-top windows should be re-raised */
  for (t = Scr.ScwmRoot.next; t != NULL; t = t->next) {
    if ((t->flags & ONTOP) && !(t->flags & VISIBLE)) {
      RaiseWindow(t);
      t->flags &= ~RAISED;
    } else
      t->flags |= RAISED;
  }
}


/**************************************************************************
 * 
 * Unmaps a window on transition to a new desktop
 *
 *************************************************************************/
void 
UnmapIt(ScwmWindow * t)
{
  XWindowAttributes winattrs;
  unsigned long eventMask;

  /*
   * Prevent the receipt of an UnmapNotify, since that would
   * cause a transition to the Withdrawn state.
   */
  XGetWindowAttributes(dpy, t->w, &winattrs);
  eventMask = winattrs.your_event_mask;
  XSelectInput(dpy, t->w, eventMask & ~StructureNotifyMask);
  if (t->flags & ICONIFIED) {
    if (t->icon_pixmap_w != None)
      XUnmapWindow(dpy, t->icon_pixmap_w);
    if (t->icon_w != None)
      XUnmapWindow(dpy, t->icon_w);
  } else if (t->flags & (MAPPED | MAP_PENDING)) {
    XUnmapWindow(dpy, t->frame);
  }
  XSelectInput(dpy, t->w, eventMask);
}

/**************************************************************************
 * 
 * Maps a window on transition to a new desktop
 *
 *************************************************************************/
void 
MapIt(ScwmWindow * t)
{
  if (t->flags & ICONIFIED) {
    if (t->icon_pixmap_w != None)
      XMapWindow(dpy, t->icon_pixmap_w);
    if (t->icon_w != None)
      XMapWindow(dpy, t->icon_w);
  } else if (t->flags & MAPPED) {
    XMapWindow(dpy, t->frame);
    t->flags |= MAP_PENDING;
    XMapWindow(dpy, t->Parent);
  }
}




void 
RaiseWindow(ScwmWindow * t)
{
  ScwmWindow *t2;
  int count, i;
  Window *wins;

  /* raise the target, at least */
  count = 1;
  Broadcast(M_RAISE_WINDOW, 3, t->w, t->frame, (unsigned long) t, 0, 0, 0, 0);

  for (t2 = Scr.ScwmRoot.next; t2 != NULL; t2 = t2->next) {
    if (t2->flags & ONTOP)
      count++;
    if ((t2->flags & TRANSIENT) && (t2->transientfor == t->w) &&
	(t2 != t)) {
      count++;
      Broadcast(M_RAISE_WINDOW, 3, t2->w, t2->frame, (unsigned long) t2,
		0, 0, 0, 0);
      if ((t2->flags & ICONIFIED) && (!(t2->flags & SUPPRESSICON))) {
	count += 2;
      }
    }
  }
  if ((t->flags & ICONIFIED) && (!(t->flags & SUPPRESSICON))) {
    count += 2;
  }
  wins = (Window *) safemalloc(count * sizeof(Window));

  i = 0;

  /* ONTOP windows on top */
  for (t2 = Scr.ScwmRoot.next; t2 != NULL; t2 = t2->next) {
    if (t2->flags & ONTOP) {
      Broadcast(M_RAISE_WINDOW, 3, t2->w, t2->frame, (unsigned long) t2,
		0, 0, 0, 0);
      wins[i++] = t2->frame;
    }
  }

  /* now raise transients */
#ifndef DONT_RAISE_TRANSIENTS
  for (t2 = Scr.ScwmRoot.next; t2 != NULL; t2 = t2->next) {
    if ((t2->flags & TRANSIENT) &&
	(t2->transientfor == t->w) &&
	(t2 != t) &&
	(!(t2->flags & ONTOP))) {
      wins[i++] = t2->frame;
      if ((t2->flags & ICONIFIED) && (!(t2->flags & SUPPRESSICON))) {
	if (!(t2->flags & NOICON_TITLE))
	  wins[i++] = t2->icon_w;
	if (!(t2->icon_pixmap_w))
	  wins[i++] = t2->icon_pixmap_w;
      }
    }
  }
#endif
  if ((t->flags & ICONIFIED) && (!(t->flags & SUPPRESSICON))) {
    if (!(t->flags & NOICON_TITLE))
      wins[i++] = t->icon_w;
    if (t->icon_pixmap_w)
      wins[i++] = t->icon_pixmap_w;
  }
  if (!(t->flags & ONTOP))
    wins[i++] = t->frame;
  if (!(t->flags & ONTOP))
    Scr.LastWindowRaised = t;

  if (i > 0)
    XRaiseWindow(dpy, wins[0]);

  XRestackWindows(dpy, wins, i);
  free(wins);
  raisePanFrames();
}


void 
LowerWindow(ScwmWindow * t)
{
  XLowerWindow(dpy, t->frame);

  Broadcast(M_LOWER_WINDOW, 3, t->w, t->frame, (unsigned long) t, 0, 0, 0, 0);

  if ((t->flags & ICONIFIED) && (!(t->flags & SUPPRESSICON))) {
    XLowerWindow(dpy, t->icon_w);
    XLowerWindow(dpy, t->icon_pixmap_w);
  }
  Scr.LastWindowRaised = (ScwmWindow *) 0;
}


void 
HandleHardFocus(ScwmWindow * t)
{
  int x, y;

  FocusOnNextTimeStamp = t;
  Scr.Focus = NULL;
  /* Do something to guarantee a new time stamp! */
  XQueryPointer(dpy, Scr.Root, &JunkRoot, &JunkChild,
		&JunkX, &JunkY, &x, &y, &JunkMask);
  GrabEm(WAIT);
  XWarpPointer(dpy, Scr.Root, Scr.Root, 0, 0, Scr.MyDisplayWidth,
	       Scr.MyDisplayHeight,
	       x + 2, y + 2);
  XSync(dpy, 0);
  XWarpPointer(dpy, Scr.Root, Scr.Root, 0, 0, Scr.MyDisplayWidth,
	       Scr.MyDisplayHeight,
	       x, y);
  UngrabEm();
}


/*
   ** Scwm_msg: used to send output from Scwm to files and or stderr/stdout
   **
   ** type -> DBG == Debug, ERR == Error, INFO == Information, WARN == Warning
   ** id -> name of function, or other identifier
 */
void 
scwm_msg(scwm_msg_levels type, char *id, char *msg,...)
{
  char *typestr;
  va_list args;

  switch (type) {
  case DBG:
    typestr = "<<DEBUG>>";
    break;
  case ERR:
    typestr = "<<ERROR>>";
    break;
  case WARN:
    typestr = "<<WARNING>>";
    break;
  case INFO:
  default:
    typestr = "";
    break;
  }

  va_start(args, msg);

  fprintf(stderr, "[Scwm][%s]: %s ", id, typestr);
  vfprintf(stderr, msg, args);
  fprintf(stderr, "\n");

  if (type == ERR) {
    char tmp[1024];		/* I hate to use a fixed length but this will do for now */

    sprintf(tmp, "[Scwm][%s]: %s ", id, typestr);
    vsprintf(tmp + strlen(tmp), msg, args);
    tmp[strlen(tmp) + 1] = '\0';
    tmp[strlen(tmp)] = '\n';
    BroadcastName(M_ERROR, 0, 0, 0, tmp);
  }
  va_end(args);
}				/* Scwm_msg */

/* Local Variables: */
/* tab-width: 8 */
/* c-basic-offset: 2 */
/* End: */
