#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <varargs.h>
#include "stdstuff.h"
#include "error.h"

extern int sys_nerr;
extern char *sys_errlist[];

extern char *ProgramName;
extern void (*FatalErrorHook) ();

static char *UnknownError = "(unknown error)";

static char **Local_errlist = NULL;
static int Local_nerr = 0;
static BOOLEAN ErrorInitialized = FALSE;

void
ErrorInit (nerr, errlist)
     int nerr;
     char **errlist;
{
  Local_errlist = errlist;
  assert (Local_errlist != NULL);

  Local_nerr = sys_nerr + nerr;

  ErrorInitialized = TRUE;
}

char *
ErrorMessage (ErrorCode)
     int ErrorCode;
{
  char *Msg;

  if (ErrorCode < 0)
    Msg = UnknownError;
  else if (ErrorCode < sys_nerr)
    Msg = sys_errlist [ErrorCode];
  else if (ErrorCode <= Local_nerr)
    {
      if (ErrorInitialized == TRUE)
	Msg = Local_errlist [ErrorCode - sys_nerr];
      else
	Msg = UnknownError;
    }
  else
    Msg = UnknownError;
    
  return Msg;
}

/*
  Error (File, Line, ErrorNumber, format_string, ...)
     char *File;
     unsigned long Line;
     int ErrorNumber;
     char *format_string;
*/
void
Error (va_alist)
     va_dcl
{
  char *File;
  unsigned long Line;
  int ErrorNumber;
  char *printfFormat;
  va_list va_marker;

  va_start (va_marker);

  File = va_arg (va_marker, char *);
  Line = va_arg (va_marker, unsigned long);
  ErrorNumber = va_arg (va_marker, int);
  printfFormat = va_arg (va_marker, char *);

  fprintf (stderr, "\n");
  if (ProgramName != NULL)
    fprintf (stderr, "%s: ", ProgramName);

  if (printfFormat != NULL)
    vfprintf (stderr, printfFormat, va_marker);

  if (ErrorNumber != 0)
    fprintf (stderr, ": %s", ErrorMessage (ErrorNumber));

  if (File != NULL)
    fprintf (stderr, ": file `%s' line %ld", File, Line);

  fprintf (stderr, "\n");

  va_end (va_marker);
}

/*
  FatalError (File, Line, ErrorNumber, format_string, ...)
     char *File;
     unsigned long Line;
     int ErrorNumber;
     char *format_string;
*/
void
FatalError (va_alist)
     va_dcl
{
  char *File;
  unsigned long Line;
  int ErrorNumber;
  char *printfFormat;
  va_list va_marker;

  if (FatalErrorHook != NULL)
    (*FatalErrorHook) ();

  va_start (va_marker);

  File = va_arg (va_marker, char *);
  Line = va_arg (va_marker, unsigned long);
  ErrorNumber = va_arg (va_marker, int);
  printfFormat = va_arg (va_marker, char *);

  fprintf (stderr, "\n");
  if (ProgramName != NULL)
    fprintf (stderr, "%s: ", ProgramName);

  if (printfFormat != NULL)
    vfprintf (stderr, printfFormat, va_marker);

  if (ErrorNumber != 0)
    fprintf (stderr, ": %s", ErrorMessage (ErrorNumber));

  if (File != NULL)
    fprintf (stderr, ": file `%s' line %ld", File, Line);

  fprintf (stderr, ".  Stop.\n");

  va_end (va_marker);

  exit (EXIT_FAILURE);
}


/* Set Emacs C-mode style.
   Local Variables:
   c-style: GNU
   End:
 */
