package biss.awt;

import biss.ObserverSocket;
import biss.VectorLib;
import java.awt.Container;
import java.awt.Rectangle;
import java.util.Observable;
import java.util.Vector;

/**
 * editable List (replaces java.awt.List). Items can be
 * directly modified appended or inserted by "popup" TextFields. Items
 * can also be removed or shifted. Uses the following hotkeys:
 *   ctrl-e : edit
 *   ctrl-i : insert
 *   ctrl-a : append
 *   ctrl-r : remove
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class EditList
  extends List
{
/**
 * triggered by end-edit event (Enter key from the TextField)
 * just in case of at least one registered observer.
 * This observer is responsible to update the Data list
 * by calling the replaceElementWith( selection, "selText") method.
 * Default replacement text is the TextField contents.
 * parameters: current selection, TextField contents
 */
	public ObserverSocket OsEndEdit = new ObserverSocket( this);
/**
 * triggered by modify request ( Ctrl-e or via pop-up menu)
 * just in case of at least one registered observer.
 * This observer is responsible to set the modified entry's
 * text by calling the modifyEntry( "myText") method.
 * Default modification text is the selections string representation.
 * parameter: current selection
 */
	public ObserverSocket OsModify = new ObserverSocket( this);
/**
 * triggered by insert request ( Ctrl-i or via pop-up menu)
 * just in case of at least one registered observer.
 * This observer is responsible to set the new entry's
 * template text by calling the insertEntry( "myTeyt") method.
 * Default template text is an empty String ( "").
 * parameter: current selection
 */
	public ObserverSocket OsInsert = new ObserverSocket( this);
/**
 * triggered by remove request ( Ctrl-r, Del, or via pop-up menu)
 * just in case of at least one registered observer.
 * This observer is responsible for removing the entry
 * by calling the removeEntry( selection) method.
 * Default action is remove the current selection from the data list.
 * parameter: current selection
 */
	public ObserverSocket OsRemove = new ObserverSocket( this);
/**
 * triggered by add request ( Ctrl-a or via pop-up menu)
 * just in case of at least one registered observer.
 * This observer is responsible to set the new entry's
 * template text by calling the addEntry( "myTeyt") method.
 * Default template text is an empty String ( "").
 * parameter: current selection
 */
	public ObserverSocket OsAdd = new ObserverSocket( this);

public EditList () {
	this( 0);
}

public EditList ( int type) {
	super( type);
	setEditMenu();
	OsCommand.addObserver( this);
}

void addEntry () {
	if ( ! OsAdd.isEmpty() )
		OsAdd.notifyObservers();
	else
		addEntry( "");
}

public void addEntry ( Object ao) {
	addElement( ao);
	setSelection( ao, false);
	modifyEntry();
}

/**
 *
 */
boolean insertEntry () {
	if ( Cv.Selection == null)
		return false;

	if ( ! OsInsert.isEmpty() )
		OsInsert.notifyObservers( Cv.Selection);
	else
		insertEntry( "");

	return true;
}

public boolean insertEntry ( Object io) {
	if ( Cv.Selection == null)
		return false;
		
	insertElementBefore( io, Cv.Selection);
	setSelection( io, false);
	modifyEntry();
	
	return true;
}

boolean modifyEntry () {
	if ( Cv.Selection == null)
		return false;
		
	if ( ! OsModify.isEmpty() )
		OsModify.notifyObservers( Cv.Selection);
	else
		modifyEntry( Cv.Selection.toString() );

	return true;
}

public void modifyEntry ( String defStr) {
	new Prompter( "item text", defStr, "edit", this, Cv.getTopWindow());
}

boolean removeEntry () {

	if ( Cv.Selection == null)
		return false;

	if ( !OsRemove.isEmpty() )
		OsRemove.notifyObservers( Cv.Selection);
	else
		removeEntry( Cv.Selection);
		
	return true;
}

public void removeEntry (Object ro) {

	if ( ro == null)
		return;

	int rIdx = Cv.Lines.indexOf( ro);
	if ( rIdx > -1) {
		removeElement( ro);
		if ( rIdx > Cv.Lines.size()-1 )
			rIdx--;
		setSelection( rIdx, false);
	}

}

void setEditMenu () {
	Menu m = new Menu();
	m.addItem( "~edit");
	m.addSeparator();
	m.addItem( "~insert");
	m.addItem( "~append");
	m.addItem( "~remove");
	m.addSeparator();
	m.addItem( "~up");
	m.addItem( "~down");
	setMenu( m);

	addHotKey( Key.Delete, HotKey.Lone, "remove");
}

void setEditResult ( String er) {

	if ( ! OsEndEdit.isEmpty() )
		OsEndEdit.notifyObservers( Cv.Selection, er );
	else if ( Cv.Selection instanceof String)
		replaceElementWith( Cv.Selection, er );

	redrawObject( Cv.Selection);
}

public void update( Observable obs, Object arg){
	if ( obs == OsCommand) {
		if ( arg.equals("edit"))					modifyEntry();
		else if ( arg.equals( "insert"))	insertEntry();
		else if ( arg.equals( "up"))			shiftUp();
		else if ( arg.equals( "down"))	  shiftDown();
		else if ( arg.equals( "append"))	addEntry();
		else if ( arg.equals( "remove"))	removeEntry();
	}
	else if ( obs instanceof Prompter) {
		if ( "edit".equals( arg))
			setEditResult( ((Prompter)obs).getContents() );
	}
	else
		super.update( obs, arg);
}
}
