#include "cthugha.h"
#include "display.h"
#include "options.h"
#include "interface.h"
#include "disp-sys.h"
#include "action.h"

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include "initial_palettes.c"

int change_palette_imm(int);
feature palettes = {
    "palette",
    NULL,
    0,
    CHANGE_RANDOM,
    f_palette
};
char palette_first[256] = "";			/* Start with this palette */

int display_internal_pal = 1;			/* use internal pal. */
int display_external_pal = 1;			/* use external pal. */

unsigned int bitmap_colors[256];		/* "compiled" palette */

feature_entry* read_palette(FILE * file, char * name, char * dir);
char * palette_path[] = {
    "./",
    "./map/",
    INST_LIB "/map/",
    ""	
};

int colormapped=1;				/* 0 .. True/Direct color
						   1 .. All 256 color cells
						   2 .. Only some cells */

palette rgb_pal;				/* palette for RGB and 256 colors */


/* when using only some colorcells (e.g. when drawing on the root window)
   update to the palette are done immediately without smoothing.
   */

int load_palettes() {
    int i,j,k,l;
	
    /* Builtin palettes */
    if ( display_internal_pal ) {
	feature_entry new_pal;
	palette * pal;

	printfv(2,"  preparing internal palettes...\n");

	new_pal.desc[0] = '\0';
	new_pal.use = 1;
	    
	for(i=0; i < sizeof(initial_palettes)/sizeof(palette); i++) {
	    new_pal.data = initial_palettes[i];
	    sprintf( new_pal.name, "Internal_%d", i);

	    add_feature(&palettes, &new_pal); 
	}

	/* create one general palette */
	new_pal.data = cth_memory(NULL, sizeof(palette), 
		"Can not allocate memory for internal palette.\n");
	pal = (palette*)new_pal.data;
	for(i=0; i < 64; i++) {
	    (*pal)[i][0]	= i << 2;
	    (*pal)[i][1]	= 0;
	    (*pal)[i][2]	= 0;

	    (*pal)[i+64][0] 	= 0;
	    (*pal)[i+64][1]	= i << 2;
	    (*pal)[i+64][2] 	= 0;

	    (*pal)[i+128][0] 	= 0;
	    (*pal)[i+128][1]	= 0;
	    (*pal)[i+128][2] 	= i << 2;

	    (*pal)[i+192][0] 	= i << 2;
	    (*pal)[i+192][1]	= i << 2;
	    (*pal)[i+192][2] 	= i << 2;
	}
	sprintf(new_pal.name, "Internal_%d", palettes.nr_entries);

	add_feature(&palettes, &new_pal);
    }

    /* create the RGB pal for 256 colors */
    for(l=0, i=0; i < 8; i++)
	for(j=0; j < 8; j++)
	    for(k=0; k < 4; k++, l++) {
		rgb_pal[l][0] = i << 5;
		rgb_pal[l][1] = j << 5;
		rgb_pal[l][2] = k << 6;
	    }
    
    /* read palettes from file */
    if ( display_external_pal) {
	printfv(2,"  loading external palettes...\n");
	load(palette_path, "/map/", ".map", read_palette, &palettes);
	printfv(2,"  number of loaded palettes: %d\n", palettes.nr_entries);
    }

    
    /* brighten up palettes, that are a bit dark */
    for(i=0; i < palettes.nr_entries; i++) {
	palette * pal = (palette*)palettes.entries[i].data;
	double P = 0;

	for(l=0; l < 256; l++) {
	    P = max(P, (*pal)[l][0] + (*pal)[l][1] + (*pal)[l][2]);
	}
	if( (P > 0) && (P < 3.0*255.0) ) {
	    printfv(10, "Brightening palette %d (%s). Faktor: %3.0f\n", 
		    i, palettes.entries[i].name, P);

	    for(l=0; l < 256; l++) {
		(*pal)[l][0] = min((int) ( (double)(*pal)[l][0] * 3.0*255.0 / P), 255);
		(*pal)[l][1] = min((int) ( (double)(*pal)[l][1] * 3.0*255.0 / P), 255);
		(*pal)[l][2] = min((int) ( (double)(*pal)[l][2] * 3.0*255.0 / P), 255);
	    }
	}
    }

    return 0;
}

/*
 * store the palette with 8 bytes per color
 */
feature_entry* read_palette(FILE * file, char * name, char * dir) {
    char dummy[256];
    int i, j, r,g,b;
    static feature_entry new_pal;
    palette * pal;

    new_pal.data = cth_memory(NULL, sizeof(palette), 
		     "Can not allocate memory for external palette\n");
    pal = (palette*)new_pal.data;

    for(i=0; i < 256; i++) { 
	if ( fscanf( file, "%d %d %d", &r, &g, &b) < 3) {
	    printfv(3,"\n    Can't read at line: %d (%s)", i,name);
	    if ( i == 0) {				/* nothing read */
		printfv(3," ... skipping file");
		free(new_pal.data);
		return NULL;
	    }
	    printfv(3," ... filling with black");
	    for(; i < 256; i++)				/* fill with black */
		for(j=0;j<3;j++)
		    (*pal)[i][j] = 0;
	    break;
	}
	fgets( dummy, 255, file);
	(*pal)[i][0] = r;
	(*pal)[i][1] = g;
	(*pal)[i][2] = b;
    }

    strncpy(new_pal.name, name, MAX_NAME_LEN);
    new_pal.desc[0] = '\0';
    new_pal.use = 1;

    return &new_pal;
}

/*
 * Set the first palette
 *   - allocate palette
 */
int init_palettes() {
    if( init_palettes_sys() )
	return 1;

    if(flame_rgb)
	set_rgb_palette();

    update_palette(); 
    return 0;
}

int exit_palettes() {
    exit_palettes_sys();
    return 0;
}

/*
 * Functions to change/selct the palette
 *
 * changes (smooth_palette) by "Stanislav V. Voronyi" <stas@use.kharkov.ua>
 */
palette cur_palette;
static palette new_palette;
static int pal_changed = 0;
static int pal_text = 0;	/* a "text-palette" is on screen */


void cth_setpalette(palette pal, int immed) {
    static int cur_empty = 1;

    memcpy( new_palette, pal, sizeof(palette) );	/* save new palette */

    if ( display_text_on_screen) {			/* text-mode */
	int i,j;

	/* create a dimmed palette */

	if(colormapped ) {		/* use only colors 0 - 127, save rest for text */
	    for(i=0; i < 128; i++)
		for(j=0; j < 3; j++)
		    new_palette[i][j]  = (new_palette[i*2][j] + new_palette[i*2+1][j]) >> 2;

	    /* text colors */
	    for(i=0; i < display_text_colors; i++)
		for(j=0; j < 3; j++) {
		    cur_palette[255-i][j] = new_palette[255-i][j] = display_text_color[i][j];
		}
	} else {			/* here we can use all colors */
	    for(i=0; i < 256; i++)
		for(j=0; j < 3; j++)
		    new_palette[i][j]  >>= 1;
	}

	/* use only some of the colors, when text is on screen */
	if( draw_mode == DM_direct) {
	    draw_mode = DM_tmp_mapped;

	    for(i=0; i < 256; i++)
		bitmap_colors[i] = i / 2;
	}
	    
	pal_text = 1;
    } else {
	pal_text = 0;
	if(draw_mode == DM_tmp_mapped)		/* switch back to direct draw mode */
	    draw_mode = DM_direct;
    }

    pal_changed = 1;				/* palette entries changed */

    if( cur_empty || immed || (colormapped == 2) ) {
 	memcpy( cur_palette, new_palette, sizeof(palette) );
	cur_empty=0;

	smooth_setpalette();			/* really set the palette */
    }
}

void set_rgb_palette() {
    palette tmp;

    memcpy(tmp, cur_palette, sizeof(palette));
    memcpy(cur_palette, rgb_pal, sizeof(palette));
    set_palette_sys();
    memcpy(cur_palette, tmp, sizeof(palette));

}

/*
 * brings the current palette nearer to the desired palette (new_palette)
 * smoothing one palette to the next needs at most 1 second.
 */
void smooth_setpalette(void) {
    int i,j;
    int max_chg;

    if (display_text_on_screen && !pal_text) {	/* text on screen, but no text pal yet */
	update_palette();
	return;
    }

    if (!display_text_on_screen && pal_text) {	/* text no longer on screen, but still text pal */
	update_palette();
	return;
    }

    if (! pal_changed )				/* no update needed */
	return;

    max_chg = (fps > 0) ? max(255 / fps, 1) : 4;

    /* count if anything changed */
    pal_changed = 256*3;
    for(i=0; i < 256; i++)
	for(j=0; j < 3; j++) {
	    int d = new_palette[i][j] - cur_palette[i][j];
	    d = min( max(d, -max_chg), max_chg);	/* d between -max_chg and +max_chg */
	    if( d != 0)
		cur_palette[i][j] += d;
	    else
		pal_changed --;
	}

    /* really set the palette now, only if not running in RGB mode */
    if(! flame_rgb)
	set_palette_sys();
}

/*
 * set a new palette
 */
int update_palette() {
    change(f_palette, palettes.current, 0);
    return 0;
}












