/*
 * Copyright (c) 1997 Massachusetts Institute of Technology
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to use, copy, modify, and distribute the Software without
 * restriction, provided the Software, including any modified copies made
 * under this license, is not distributed for a fee, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY BE LIABLE
 * FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Massachusetts
 * Institute of Technology shall not be used in advertising or otherwise
 * to promote the sale, use or other dealings in this Software without
 * prior written authorization from the Massachusetts Institute of
 * Technology.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "rfftw.h"

#define NUM_ITER 2500000L

#define N_TESTS 17

void initialize_fft_data(FFTW_REAL * arr, long n);

void main(void)
{
    int n[N_TESTS] =
    {
	2,
	4,
	8,
	16,
	32,
	64,
	128,
	256,
	512,
	1024,
	2048,
	4096,
	8192,
	16384,
	32768,
	65536,
	131072
    };
    int i, test, iter, max_iter;
    fftw_time start_t, end_t, init_t;
    FFTW_COMPLEX *cin, *out;
    FFTW_REAL *rin;
    int max_size = 0;
    fftw_plan cplan;
    rfftw_plan rplan;

    for (i = 0; i < N_TESTS; ++i)
	if (n[i] > max_size)
	    max_size = n[i];

    cin = fftw_malloc(max_size * sizeof(FFTW_COMPLEX));
    out = fftw_malloc(max_size * sizeof(FFTW_COMPLEX));
    rin = (FFTW_REAL *) cin;

    if (!cin || !out) {
	printf("Not enough memory!  At least %d bytes needed.\n",
	       max_size * sizeof(FFTW_COMPLEX) * 2);
	exit(1);
    }
    printf("%10s%20s%20s%20s\n", "N", "FFTW", "FFTW / 2", "RFFTW");

    for (test = 0; test < N_TESTS; ++test) {
	printf("%10d", n[test]);
	fflush(stdout);

	cplan = fftw_create_plan(n[test], FFTW_FORWARD, FFTW_MEASURE);
	rplan = rfftw_create_plan(n[test], FFTW_FORWARD, FFTW_MEASURE, REAL_TO_COMPLEX);

	max_iter = NUM_ITER / (n[test] * log(2.0 * n[test]));

	if (max_iter < 1) {
	    printf("Error!  max_iter < 1!!!\n");
	    exit(1);
	}
	initialize_fft_data(rin, n[test]);
	start_t = fftw_get_time();
	for (iter = 0; iter < max_iter; ++iter)
	    initialize_fft_data(rin, n[test]);
	end_t = fftw_get_time();
	init_t = fftw_time_diff(end_t,start_t);

	/* Time FFTW: */

	initialize_fft_data(rin, n[test]);
	fftw(cplan, 1, cin, 1, 0, out, 1, 0);
	start_t = fftw_get_time();
	for (iter = 0; iter < max_iter; ++iter) {
	    initialize_fft_data(rin, n[test]);
	    fftw(cplan, 1, cin, 1, 0, out, 1, 0);
	}
	end_t = fftw_get_time();
	printf("%20g", fftw_time_to_sec(fftw_time_diff(fftw_time_diff(end_t,start_t),init_t)));
	printf("%20g", fftw_time_to_sec(fftw_time_diff(fftw_time_diff(end_t,start_t),init_t)) * 0.5);
	fflush(stdout);

	/* Time RFFTW: */

	initialize_fft_data(rin, n[test] >> 1);
	start_t = fftw_get_time();
	for (iter = 0; iter < max_iter; ++iter)
	    initialize_fft_data(rin, n[test] >> 1);
	end_t = fftw_get_time();
	init_t = fftw_time_diff(end_t,start_t);

	initialize_fft_data(rin, n[test] >> 1);
	rfftw(rplan, 1, cin, 1, 0, out, 1, 0);
	start_t = fftw_get_time();
	for (iter = 0; iter < max_iter; ++iter) {
	    initialize_fft_data(rin, n[test] >> 1);
	    rfftw(rplan, 1, cin, 1, 0, out, 1, 0);
	}
	end_t = fftw_get_time();
	printf("%20g", fftw_time_to_sec(fftw_time_diff(fftw_time_diff(end_t,start_t),init_t)));
	fflush(stdout);

	/* Done. */

	printf("\n");

	fftw_destroy_plan(cplan);
	rfftw_destroy_plan(rplan);
    }

    fftw_free(cin);
    fftw_free(out);
}

void initialize_fft_data(FFTW_REAL * arr, long n)
{
    long i;

    for (i = 0; i < 2 * n; i += 2) {	/* initialize to some arbitrary values: */
	arr[i] = 0.56923456;
	arr[i + 1] = 0.23858572;
    }
}
