//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  The 2600 System class.

  @author  Bradford W. Mott
  @version $Id: System.cxx,v 1.2 1997/05/17 19:00:08 bwmott Exp $
*/

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "BasTerm.hxx"
#include "Error.hxx"
#include "Joystick.hxx"
#include "M6507Hi.hxx"
#include "M6507Lo.hxx"
#include "Paddle.hxx"
#include "Random.hxx"
#include "System.hxx"

#if defined (WIN32)
#include <mmsystem.h>
#endif

//============================================================================
// Constructor
//============================================================================
System::System(BasicTerminal& terminal, const Properties& props)
    : myProperties(&props),
      myTerminal(terminal),
      myConsole(*this),
      myNumberOfFrames(0)
{
  Random random;

  // Setup the memory map
  myMap = new Map[8192];

  myUnmappedDummyData = (uWord)random;
  myUnmappedDummyDataOffset = 0;
 
  for(int i = 0; i < 8192; ++i)
  {
    myMap[i].peekBase = (uByte*)&myUnmappedDummyData;
    myMap[i].peekOffset = &myUnmappedDummyDataOffset;
    myMap[i].pokeBase = (uByte*)&myUnmappedDummyData;
    myMap[i].device = 0;
  }

  myCartridge = Cartridge::create(*this, *myProperties);

  // At this point I don't have any controllers
  myController[0] = myController[1] = 0;

  // Configure controllers by setting my properties
  properties(props);

  // Figure out the minimum frame rate
  switch(myMinimumFrameRate = properties().integer("Display.FrameRate"))
  {
    case 60:
      myFrameRateMultiple = 1; break;
    case 30:
      myFrameRateMultiple = 2; break;
    case 20:
      myFrameRateMultiple = 3; break;
    case 15:
      myFrameRateMultiple = 4; break;
    case 10:
      myFrameRateMultiple = 6; break;
    default:
      myFrameRateMultiple = 1; break;
  }

  // Create each of my sub-systems
  if(strcmp(properties().get("M6507.Compatibility"), "High") == 0)
  {
    myM6507 = new M6507High(*this);
  }
  else
  {
    myM6507 = new M6507Low(*this);
  }

  myM6532 = new M6532(*this);
  myTIA = new TIA(*this);

  // Now that all of the parts are mapped into the system they need resetting
  myCartridge->reset();
  myM6532->reset();
  myM6507->reset();
  myTIA->reset();

  throttleReset(); 
}

//============================================================================
// Destructor
//============================================================================
System::~System()
{
  // Free each of my sub-systems
  delete myCartridge;
  delete myM6532;
  delete myM6507;
  delete myTIA;

  // Free the controllers
  delete myController[0];
  delete myController[1];

  // Free my memory map
  delete myMap;
}

//============================================================================
// Reset myself and all my components
//============================================================================
void System::reset()
{
  myNumberOfFrames = 0;

  myM6532->reset();
  myM6507->reset();
  myTIA->reset();
}

//============================================================================
// Change my properties and configure controllers
//============================================================================
void System::properties(const Properties& props)
{
  // Change my property pointer
  myProperties = &props;

  // Delete old controllers
  delete myController[0];
  delete myController[1];

  // Create the controllers
  const char* joystick0 = properties().find("Controller.Joystick0");
  const char* joystick1 = properties().find("Controller.Joystick1");
  const char* paddle0 = properties().find("Controller.Paddle0");
  const char* paddle1 = properties().find("Controller.Paddle1");
  const char* paddle2 = properties().find("Controller.Paddle2");
  const char* paddle3 = properties().find("Controller.Paddle3");

  // Create the left controller object
  if(strcmp(joystick0, "None") != 0)
  {
    Joystick::PhysicalPort port = Joystick::Zero;

    if(strcmp(joystick0, "0") == 0)
      port = Joystick::Zero;
    else if(strcmp(joystick0, "1") == 0)
      port = Joystick::One;

    myController[0] = new Joystick(*this, port);
  }
  else if((strcmp(paddle0, "None") != 0) || (strcmp(paddle1, "None") != 0))
  {
    Paddle::PhysicalPort paddle0Port = Paddle::None;
    Paddle::PhysicalPort paddle1Port = Paddle::None;

    if(strcmp(paddle0,"None") == 0)
      paddle0Port = Paddle::None;
    else if(strcmp(paddle0, "0") == 0)
      paddle0Port = Paddle::Zero;
    else if(strcmp(paddle0, "1") == 0)
      paddle0Port = Paddle::One;
    else if(strcmp(paddle0, "2") == 0)
      paddle0Port = Paddle::Two;
    else if(strcmp(paddle0, "3") == 0)
      paddle0Port = Paddle::Three;

    if(strcmp(paddle1,"None") == 0)
      paddle1Port = Paddle::None;
    else if(strcmp(paddle1, "0") == 0)
      paddle1Port = Paddle::Zero;
    else if(strcmp(paddle1, "1") == 0)
      paddle1Port = Paddle::One;
    else if(strcmp(paddle1, "2") == 0)
      paddle1Port = Paddle::Two;
    else if(strcmp(paddle1, "3") == 0)
      paddle1Port = Paddle::Three;

    myController[0] = new Paddle(*this, paddle0Port, paddle1Port);
  }
  else
  { 
    // No real device active so just attach the standard controller
    myController[0] = new Controller(*this);
  }

  // Create the right controller object
  if(strcmp(joystick1, "None") != 0)
  {
    Joystick::PhysicalPort port = Joystick::Zero;

    if(strcmp(joystick1, "0") == 0)
      port = Joystick::Zero;
    else if(strcmp(joystick1, "1") == 0)
      port = Joystick::One;

    myController[1] = new Joystick(*this, port);
  }
  else if((strcmp(paddle2, "None") != 0) || (strcmp(paddle3, "None") != 0))
  {
    Paddle::PhysicalPort paddle2Port = Paddle::None;
    Paddle::PhysicalPort paddle3Port = Paddle::None;

    if(strcmp(paddle2,"None") == 0)
      paddle2Port = Paddle::None;
    else if(strcmp(paddle2, "0") == 0)
      paddle2Port = Paddle::Zero;
    else if(strcmp(paddle2, "1") == 0)
      paddle2Port = Paddle::One;
    else if(strcmp(paddle2, "2") == 0)
      paddle2Port = Paddle::Two;
    else if(strcmp(paddle2, "3") == 0)
      paddle2Port = Paddle::Three;

    if(strcmp(paddle3,"None") == 0)
      paddle3Port = Paddle::None;
    else if(strcmp(paddle3, "0") == 0)
      paddle3Port = Paddle::Zero;
    else if(strcmp(paddle3, "1") == 0)
      paddle3Port = Paddle::One;
    else if(strcmp(paddle3, "2") == 0)
      paddle3Port = Paddle::Two;
    else if(strcmp(paddle3, "3") == 0)
      paddle3Port = Paddle::Three;

    myController[1] = new Paddle(*this, paddle2Port, paddle3Port);
  }
  else
  { 
    // No real device active so just attach the standard controller
    myController[1] = new Controller(*this);
  }
}
 
//============================================================================
// This routine should be called on a regular basis to update the system
//============================================================================
void System::update()
{
  // Update my components
  myConsole.update();
  myController[0]->update();
  myController[1]->update();

  if((myNumberOfFrames % myFrameRateMultiple) == 0)
  {
    tia().frame(true);
    terminal().update(tia());
    ++myNumberOfFrames;
  }
  else
  {
    tia().frame(false);
    ++myNumberOfFrames;
  }
}

//============================================================================
// Throttle the frame rate so games don't run to fast
//============================================================================
void System::throttle()
{
#ifdef THROTTLE
  if(myM6507->cycles() >= myCycleLimit)
  {
#if defined (WIN32)
    while((double)timeGetTime() < myClockLimit)
#else
    while((double)clock() < myClockLimit)
#endif
    {
      if(myTerminal.eventState(BasicTerminal::Quit)) 
        break;
    }

    myCycleLimit += myCyclesPerThrottle;
    myClockLimit += myClocksPerThrottle;
  }
#endif
}

//============================================================================
// Reset the throttle variables based on current system state
//============================================================================
void System::throttleReset()
{
#ifdef THROTTLE
  // Some systems don't define CLOCKS_PER_SEC so we'll assume microseconds
  #ifdef CLOCKS_PER_SEC
    double clocks = CLOCKS_PER_SEC;
  #else
    double clocks = 1000000.0;
  #endif

  myClocksPerThrottle = clocks / (double)myMinimumFrameRate;
  myCyclesPerThrottle = 1193333.0 / (double)myMinimumFrameRate;
  myCycleLimit = (double)m6507().cycles();

  #if defined(WIN32)
    myClockLimit = (double)clock();
  #else
    myClockLimit = (double)clock();
  #endif
#endif
}

//============================================================================
// Answer a reference to the controller
//============================================================================
Controller& System::controller(uByte port) const
{
  if(port == 0)
    return *myController[0];
  else
    return *myController[1];
}

//============================================================================
// Map all peeks at address to the given device using base and offset
//============================================================================
void System::mapPeek(uWord address, Device& device, uByte* base, uWord* offset)
{
  address = address & 0x1fff;

  // An address cannot be mapped to two different devices
  assert((myMap[address].device == 0) || (myMap[address].device == &device));

  myMap[address].device = &device;
  myMap[address].peekBase = base;
  myMap[address].peekOffset = offset;
}

//============================================================================
// Map all peeks at address to the given device
//============================================================================
void System::mapPeek(uWord address, Device& device)
{
  address = address & 0x1fff;

  myMap[address].device = &device;
  myMap[address].peekBase = (uByte*)&myUnmappedDummyData;
  myMap[address].peekOffset = &myUnmappedDummyDataOffset;
}

//============================================================================
// Map all pokes at address to the given device using base and offset
//============================================================================
void System::mapPoke(uWord address, Device& device, uByte* base)
{
  address = address & 0x1fff;

  myMap[address].device = &device;
  myMap[address].pokeBase = base;
}

//============================================================================
// Map all pokes at address to the given device
//============================================================================
void System::mapPoke(uWord address, Device& device)
{
  address = address & 0x1fff;

  myMap[address].device = &device;
  myMap[address].pokeBase = (uByte*)&myUnmappedDummyData;
}

//============================================================================
// Remove any mappings associated with the given device
//============================================================================
void System::unmap(Device& device)
{
  for(int i = 0; i < 8192; ++i)
  {
    if(myMap[i].device == &device)
    {
      myMap[i].device = 0;
      myMap[i].peekBase = (uByte*)&myUnmappedDummyData;
      myMap[i].peekOffset = &myUnmappedDummyDataOffset;
      myMap[i].pokeBase = (uByte*)&myUnmappedDummyData;
    }
  }
}

