 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Fri May  9 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Track maps provide a mapping from symbolic track names to the        */
 /* actual position of the information within a track frame.             */
 /*                                                                      */
 /* Channel name maps map textual names for track channels to symbolic   */
 /* names, they are just a special case of named enums.                  */
 /*                                                                      */
 /************************************************************************/

#ifndef __EST_TRACKMAP_H__
#define __EST_TRACKMAP_H__

#include <limits.h>
#include "EST_TNamedEnum.h"
#include "EST_ChannelType.h"

typedef EST_TNamedEnum<EST_ChannelType> EST_ChannelNameMap;

typedef EST_TValuedEnumDefinition<EST_ChannelType, const char *, NO_INFO> 
	EST_ChannelNameTable[];

// arrays of structs used to define a track map at compile time.

typedef struct EST_ChannelMappingElement {
  EST_ChannelType type;
  unsigned short channel;
} EST_ChannelMapping[];

#define NO_SUCH_CHANNEL (-1)
#define NOT_REFCOUNTED (INT_MAX) 
#define TURN_ON_REFCOUNTING (1)

class EST_TrackMap {
private:

  int p_refcount;
  short p_map[num_channel_types];

protected:
  EST_TrackMap(int dummy_arg);	// protected creation function
				// turns on refounting

  void clear(void);
  void init(void);
  void inc_refcount(void) {if (p_refcount!=NOT_REFCOUNTED) p_refcount++;};
  void dec_refcount(void) {if (p_refcount!=NOT_REFCOUNTED) p_refcount--;};
  void make_refcounted(void) { if (p_refcount==NOT_REFCOUNTED) p_refcount = 0; };
  int is_unreferenced(void) const {return p_refcount == 0;};
  int is_refcounted(void) const {return p_refcount!= NOT_REFCOUNTED;};

public:
  EST_TrackMap(void);
  EST_TrackMap(EST_ChannelMapping map);

  void set(EST_ChannelType type, short pos) { p_map[(int)type] = pos; };
  
  short get(EST_ChannelType type) const { return p_map[(int)type]; };
  short operator() (EST_ChannelType type) const { return p_map[(int)type]; };

  short last_channel(void) const;

  EST_ChannelType channel_type(unsigned short channel) const;

  friend class EST_Track;
};

extern EST_ChannelNameMap EST_default_channel_names;
extern EST_ChannelNameMap esps_channel_names;

#endif
