/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  February 1996                            */
/*-----------------------------------------------------------------------*/
/*                    Event RFC labelling                                */
/*                                                                       */
/*=======================================================================*/

/* Please note: This software isn't very good - its the result of
trying to make ancient hacky code robust. Hence there are lots of wee
hacky things here and there. It should do though, and rather than tidy
this up, I think the best thing would be to re-implement the whole
thing using for example a viterbi decoder to esnure optimal global
matching.*/

#include "EST_String.h"
#include "EST.h"
#include "EST_Event.h"
#include "EST_rfc.h"
#include "math.h"
#include "float.h"

static void add_silences(EST_Track &fz, EST_Stream &ev);
static int add_end_silence(EST_Track &fz, EST_Stream &ev, float min_duration);

static int match_rf_point(EST_Track &fz, int b_start, int b_stop, int e_start, int
		    e_stop, int &mi, int &mj);

static EST_Stream_Item make_int_item(EST_String name, float end, float start_pos,
			  float start_f0, float peak_pos, float peak_f0);

/*
static void add_int_item(EST_Stream &ev, EST_Stream_Item *cur, float end, float
		 start_f0, float peak_pos, float peak_f0, EST_String name);
*/

static int rf_match(EST_Track &fz, EventSI &ev, float range);
static int zero_cross(EST_Track &fz);

static int comp_extract(EST_Track &fz, EST_Track &part, float &start, float
			&end, float min);

static void conn_ends(EST_Track &fz, EST_Stream &ev, float min_length);
static void adjust_overlaps(EST_Track &fz, EST_Stream &ev, float min_length);
static void minium_duration(EST_Stream &ev, float min_dur);

static void fill_f0_values(EST_Track &fz, EST_Stream &ev);

// find event portions of fz in contour, cut out, and send one by one
// to individual labeller.

// This routine takes an Fz contour and a list of potential events,
// and peforms RFC matching on them. It returns a list of events with RFC
// parameters marked.
// 
// The algorithm works as follows:
// 
// make a list of events, with start and stop times.
// 
// for every event
// {
//    find start and stop times.
//    call comp_extract() to get best section of contour that 
//         falls between these times. If no suitable contour is found the
//         event is deleted and not labelled.
//    call rf_match to determine the optimal start and end times for
//           that section
// }
//
// Now add connections between non-overlapping events. Overlapping events
// get readjusted to make them simply adjacent


void default_rfc_params(EST_Option &op)
{
    op.override_fval("rfc_start_range", 0.1);
    op.override_fval("rfc_stop_range", 0.1);
    op.override_fval("rfc_search_range", 0.3);
    op.override_fval("rfc_min_connection", 0.01);
    op.override_fval("rfc_min_event", 0.03);
}

EST_Stream rfc_label(EST_Track &fz, EST_Stream &event_list, EST_Option &op)
{
    EST_Stream_Item *e, tmp;
    float start, end, min_dur;
    EST_Track part;
    EventSI ev;
    EST_Stream rfc_ev;

    fz.change_type(0.0, "MANY");

    // set up values in label list pointing to events
    for (e = event_list.head(); e != 0; e = next(e))
    {
	if (!pos_event(*e))
	    continue;
	tmp = make_int_item(e->name(),e->end(),e->start(),
			    0.0, 0.0, 0.0);
	rfc_ev.append(tmp); 
    }

    // fill empty values in event labels using matching algorithms
    for (e = rfc_ev.head(); e != 0; e = next(e))
	if (pos_event(*e))
	{
	    start = RFCS(*e)->start_pos() - op.fval("rfc_start_range");
	    end = RFCS(*e)->end_pos() + op.fval("rfc_stop_range");
	    if (comp_extract(fz, part, start, end, op.fval("rfc_min_event")))
		rf_match(part, *(RFCS(*e)), op.fval("rfc_search_range"));
	    else
		e = rfc_ev.remove(e);
	}

    rfc_ev.set_stream_name("rfc_event");
    cout << "hello1\n";
    print_rfc_events(rfc_ev);

    // hack to deal with cases where no events exist 
    if (rfc_ev.head() == 0)
    {
	cout << "Empty\n";
	tmp = make_int_item("sil",  fz.t(0), fz.t(fz.num_frames() - 1), 
			    0.0, 0.0, 0.0);
	rfc_ev.append(tmp);
    }
    else
    {	
	// make sure events don't overlap
	adjust_overlaps(fz, rfc_ev, op.fval("rfc_min_connection"));
	// fill empty values in non-event labels
	conn_ends(fz, rfc_ev, op.fval("rfc_min_connection"));

	add_silences(fz, rfc_ev);
	fill_f0_values(fz, rfc_ev);
    }

    min_dur = 0.005;
    minium_duration(rfc_ev, min_dur);    
    add_end_silence(fz, rfc_ev, min_dur);
    validate_rfc_stream(rfc_ev);
    
    // fill fields in labels with RFC info
    write_rfc_fields(rfc_ev, "rfc2");

    cout << "ref print\n";
    print_rfc_events(rfc_ev);

    return rfc_ev;
}

// Create a section of fz contour, bounded by times "start" and "end".
// The created contour is defined to be the largest single continuous
// section of contour bounded by the two times. If no fz contour exits within
// the limits an error is returned.

static int comp_extract(EST_Track &fz, EST_Track &part, float &start, float
			&end, float min_length)
{
    int i;
    int continuous = 1;
    cout.precision(6);
   
    if (start > end)
    {
	cerr << "Illegal start and end times\n";
	return 0;
    }
    
    part = extract(fz, start, end);
    part.rm_trailing_breaks();
    
    if ((part.end() - part.start()) < min_length)
    {
	cout << "Contour too small for analysis\n";
	return 0;
    }
    
    for (i = 0; i < part.num_frames(); ++i)
	if (part.track_break(i))
	    continuous = 0;
    
    // if no breaks are found in this section return. 
    if (continuous)
	return 1;
    
    // Otherwise, find the longest single section
    cout << "This contour has a break in it at point\n";
    
    int longest, s_c, s_l;
    longest = s_c = s_l = 0;
    
    for (i = 0; i < part.num_frames(); ++i)
	if (part.track_break(i))
	{
	    if ((i - s_c) > longest)
	    {
		longest = i - s_c - 1;
		s_l = s_c;
	    }
	    // skip to next real values
	    for (;(i < part.num_frames()) && (part.track_break(i)); ++i)
		s_c = i;
	}

    if ((i - s_c) > longest)
    {
	longest = i - s_c - 1;
	s_l = s_c;
    } 
    
    //    cout << "Lonest fragment is " << longest << " starting at " << s_l <<endl;
    //    cout << "Times: " << part.t(s_l) << " : " <<part.t(s_l + longest) << endl;
    part = extract(part, part.t(s_l), part.t(s_l + longest));
    part.rm_trailing_breaks();
    start = part.t(0);
    end = part.t(part.num_frames()-1);
    //    cout << part;

    return 1;
}

static int add_end_silence(EST_Track &fz, EST_Stream &ev, float min_duration)
{
    if (sil_event(*(ev.tail())))
	return 0;

    cout << "Adding end silence\n";
    EST_Stream_Item tmp;
    float f;

    f = fz.prev_non_break(fz.num_frames() - 1);

    tmp = make_int_item("sil", ev.tail()->end() + min_duration,
			ev.tail()->end(), f, 0.0, 0.0);

    ev.append(tmp); 
    return 1;
}

static int zero_cross(EST_Track &fz)
{
    for (int i = 0; i < fz.num_frames() - 1; ++i)
	if ((fz.a(i) >= 0.0) && (fz.a(i + 1) < 0.0))
	    return i;
    
    return -1;
}

// 1. There should be a more sophisticated decision about whether there
// should be a risefall analysis, and if so, where the peak (zero cross)
// region should be.
// 2. There should be minimum endforced distances for rises and falls.

static int rf_match(EST_Track &fz, EventSI &ev, float range)
{ 
    int n;
    EST_Track diff;
    int start, stop;
    int b_start, b_stop, e_start, e_stop, region;
    
    if (fz.num_frames() <= 0)
    {
	ev.set_start_f0(0.0);
	ev.set_start_pos(0.0);
	ev.set_peak_f0(0.0);
	ev.set_peak_pos(0.0);
    }
    
    diff = differentiate(fz);
    diff.set_contour_type(EST_ContourType::make("pish"));

    n = zero_cross(diff);

    if (n >= 0)		// rise + fall combination
    {
	cout << "zero crossing at point " << n << " time " << fz.t(n) << endl;
	b_start = 0;
	stop = n;
	// find rise part
	region = (int)(range * float(stop - b_start));
	// ensure region is bigger than 0
	region = region > 0 ? region : 1;
	
	b_stop = b_start + region;
	e_start = stop - region;
	e_stop = stop + region;
	// ensure regions are separate
	e_start = (e_start < b_stop)? b_stop : e_start;
	
	printf("rise: b_start  %d, b_stop %d, end %d, end stop%d\n", b_start,
	       b_stop, e_start, e_stop);
	match_rf_point(fz, b_start, b_stop, e_start, e_stop, start, stop);
	cout << "Rise is at start: " << start << " Stop = " << stop << endl;
	
	ev.set_start_f0(fz.a(start));
	ev.set_start_pos(fz.t(start));
	
	// find fall part. The start of the search is FIXED by the position
	// of where the rise stopped
	
	b_start = n;
	b_stop = n + 1;
	e_stop = fz.num_frames() - 1;
	region = (int)(range * float(e_stop - b_start));
	region = region > 0 ? region : 1;
	e_start = e_stop - region;
	
	printf("fall: b_start  %d, b_stop %d, end %d, end stop%d\n", b_start,
	       b_stop, e_start, e_stop);
	
	match_rf_point(fz, b_start, b_stop, e_start, e_stop, start, stop);
	cout << "Fall is at start: " << start << " Stop = " << stop << endl;
	cout << "region: " << region << endl;
	cout << "stop could be " << e_stop << " value " << fz.t(e_stop) << endl;
	cout << "start could be " << e_start << " value " << fz.t(e_start) << endl;
	ev.set_peak_f0(fz.a(start));
	ev.set_peak_pos(fz.t(start));
	
	ev.set_end_f0(fz.a(stop));
	ev.set_end_pos(fz.t(stop));
	
	ev.type = "RISEFALL";
	cout << "labelled event: " << ev << endl;
    }
    else			// separate rise or fall
    {
	b_start = 0;
	e_stop = fz.num_frames() - 1;
	
	region = (int)(range * float(e_stop - b_start));
	region = region > 0 ? region : 1;
	
	b_stop = b_start + region;
	e_start = e_stop - region;
	
	printf("b_start  %d, b_stop %d, end start %d, end stop%d\n", b_start,
	       b_stop, e_start, e_stop);

	match_rf_point(fz, b_start, b_stop, e_start, e_stop, start, stop);

	ev.set_start_f0(fz.a(start));
	ev.set_start_pos(fz.t(start));
	ev.set_peak_f0(0.0);
	ev.set_peak_pos(0.0);
	
	ev.set_end_f0(fz.a(stop));
	ev.set_end_pos(fz.t(stop));
	
	cout  << "start " << fz.t(start) << " end " << fz.t(stop) << endl;
	
	if (fz.a(fz.index(fz.start())) < fz.a(fz.index(fz.end())))
	    ev.type = "RISE";
	else
	    ev.type = "FALL";

	cout << "labelled event: " << ev << endl;
    }    
    
    return 0;
}

static void minium_duration(EST_Stream &ev, float min_dur)
{
    EST_Stream_Item *e;
//    cout << "duration check\n";
    for (e = ev.head(); e != 0; e = next(e))
    {
//	cout << *e << endl;
//	cout << e->dur() << endl;
	if (e->dur() < min_dur)
	{
//	    cout << " delete\n";
	    e = ev.remove(e);
	}
//	cout << endl;
    }
}

static void wedge(EST_Stream &ev, EST_Stream_Item *n, EST_Track &fz, int i, int j)
{
    EST_Stream_Item before, in, *p;
    float min_length = 0.015;
    float start, end;
    
    start = i > 0 ? fz.t(i - 1) : 0.0;
    end = fz.t(j);
    
    in = make_int_item("sil", end, start, 0.0, 0.0, 0.0);
    RFCS(in)->type = "SIL";
    
    before = make_int_item(n->name(), start, 0.0, 0.0, 0.0, 0.0);
    
    ev.insert_before(n, before);
    p = prev(n);
//    cout << "BEFORE inserted " << *p << " dur " << p->dur() << endl;
//    cout << "\nInserting between " << *p << " and " << *n << endl;
//    cout << "\nInserting between " << *RFCS(*p) << " and " << *RFCS(*n) << endl;
    
    if (p->dur() < min_length)
	ev.remove(p);
    
    ev.insert_before(n, in);
    cout << "ev dur " << *n << ": " << n->dur()<< endl;
    if (n->dur() < min_length)
	ev.remove(n);
}

static void add_silences(EST_Track &fz, EST_Stream &ev)
{
    int i, j;
    EST_Stream_Item tmp, *e;
    
    //    cout << endl << "Adding silence\n";
    
    for (i = 0; i < fz.num_frames(); ++i)
	if (fz.track_break(i))
	{
	    for (j = i; j < fz.num_frames(); ++j)
		if (fz.val(j))
		    break;
	    if (j == fz.num_frames()) // off end of array
		break;
	    //	    cout << "silence between " <<i << " and " << j << endl;
	    //	    cout << "  " << fz.t(i) << " and " << fz.t(j) << endl;
	    
	    for (e = ev.head(); e != 0; e = next(e))
		if (e->end() >= fz.t(j))
		    break;
	    wedge(ev, e, fz, i, j);
	    //	    for (e = ev.head(); e != 0; e = next(e))
	    //		cout << *e << " : " << *RFCS(*e) << endl;
	    
	    i = j;
	}
}

static void fill_f0_values(EST_Track &fz, EST_Stream &ev)
{
    EST_Stream_Item *e;
    float start_a;
    int pos;
    float prev = 0.0;
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	if (e->name() == "sil")
	{
	    pos =  fz.index(prev);
	    pos = fz.prev_non_break(pos);
	    start_a = pos > 0 ? fz.a(pos) : 0.0;
	}
	else if (e->name() == "c")
	{
	    pos =  fz.index(prev);;
	    pos = fz.next_non_break(pos);
	    start_a = fz.a(pos);
	}
	else 
	    start_a = fz.a(prev);
	
	RFCS(*e)->set_start_f0(start_a);
	RFCS(*e)->set_start_pos(prev);
	//	cout << "setting start to be " << start_a << " at pos " << pos << endl;
	//	cout << *e << " " << *RFCS(*e) << endl;
	
	if (RFCS(*e)->type == "RISEFALL")
	{
	    start_a = fz.a(RFCS(*e)->peak_pos());
	    RFCS(*e)->set_peak_f0(start_a);
	}
	prev = e->end();
    }
    //    cout << "in filling values\n";
    //    for (e = ev.head(); e != 0; e = next(e)) 
    //	cout << *e << " : " <<	*RFCS(*e) << endl;
    
    
}

static void adjust_overlaps(EST_Track &fz, EST_Stream &ev, float min_length)
{
    EST_Stream_Item *e, *n, tmp;
    float pos=0.0;
    (void)min_length;
    (void)fz;
    
    for (e = ev.head(); next(e) != 0; e = next(e))
    {
	n = next(e);
	if (RFCS(*e)->end_pos() > RFCS(*n)->start_pos())
	{
	    cout << "Overlapping events " << *e <<":" << *n << endl;
	    cout << "Overlapping events " << *RFCS(*e)<<":"<<*RFCS(*n) 
		<< endl;
	    // case a: genunine overlap
	    if (RFCS(*n)->end_pos() > RFCS(*e)->end_pos())
	    {
		pos = RFCS(*n)->start_pos() + (RFCS(*e)->end_pos() -
					       RFCS(*n)->start_pos()) / 2.0;
	    }
	    // case b: second element is enclosed by first
	    else if (RFCS(*n)->end_pos() <= RFCS(*e)->end_pos())
		pos = RFCS(*n)->start_pos();
	    
	    // case c: second element is before first
	    else if ((RFCS(*n)->end_pos() < RFCS(*e)->end_pos()) &&
		     (RFCS(*n)->start_pos() < RFCS(*e)->start_pos()))
		pos = RFCS(*e)->start_pos() + (RFCS(*n)->end_pos() -
					       RFCS(*e)->start_pos()) / 2.0;
	    else
		cout << "No overlap conditions met\n";
	    //	    cout << "pos =" << pos << endl;
	    RFCS(*e)->set_end_pos(pos);
	    RFCS(*n)->set_start_pos(pos);
	    
	}
    }
    
    // The overlap adjustments may cause the peak position to lie outside
    // the start and end points. This checks for this and makes an
    // arbitrary adjustment
    for (e = ev.head(); next(e) != 0; e = next(e))
	if ((RFCS(*e)->type == "RISEFALL") && (RFCS(*e)->peak_pos() <
					       RFCS(*e)->start_pos()))
	{
	    RFCS(*e)->set_peak_pos(RFCS(*e)->start_pos() + 
				   (RFCS(*e)->end_pos() - RFCS(*e)->start_pos()) / 2.0);
	}
}

static void conn_ends(EST_Track &fz, EST_Stream &ev, float min_length)
{
    EST_Stream_Item *e, *n, tmp;
    float t, f0;
    const float ARB_DISTANCE = 0.1;

    cout << "ZZZ\n";
    
    for (e = ev.head(); next(e) != 0; e = next(e))
    {
	n = next(e);
	//	cout << "here c\n";
		cout << "analysis of " <<  *RFCS(*e) << endl;
		cout << "end " << RFCS(*e)->end_pos() << " start " <<
		    RFCS(*n)->start_pos() << endl;
	
	e->set_end(RFCS(*e)->end_pos());
	if ((RFCS(*n)->start_pos() - RFCS(*e)->end_pos()) > min_length)
	{
//	    cout << "making connection\n";
	    //	    add_int_item(ev, n, RFCS(*n)->start_pos(), RFCS(*e)->end_f0(), 
	    //			 0.0, 0.0, "c");
	    
	    tmp = make_int_item("c", RFCS(*n)->start_pos(),
				RFCS(*e)->start_pos(),
				RFCS(*e)->end_f0(), 0.0, 0.0);
	    ev.insert_before(n, tmp);
//	    cout << "\njust made " << tmp << "\n";;
	    e = next(e);		// advance after new connection
	} 
	//	cout << "\njust made " << *e << "\n";;
    }
    
    e->set_end(RFCS(*e)->end_pos()); // last one

    t = RFCS(*(ev.head()))->start_pos(); // store time of start of first event
    // insert silence at beginning if contour doesn't start at near time 0
    if (fz.start() > fz.shift())
    {
	tmp = make_int_item("sil", fz.start(), 0.0, 0.0, 0.0, 0.0);
	//	ev.prepend(tmp);
    }
    
    // add connection between silence and first event
    
    tmp = make_int_item("c", t, 0.0, fz.a(fz.start()), 0.0, 0.0);
    ev.insert_before(ev.head(), tmp);
    
    if ((ev.tail()->end() + min_length)< fz.end())
    {
	f0 = fz.a(ev.tail()->end());
	// add connection after last event.
	tmp = make_int_item("c", fz.end(), 0.0, f0, 0.0, 0.0);
	ev.insert_after(ev.tail(), tmp);
    }
    
    // add silence, an arbitrary distance after end - what a hack!
    tmp = make_int_item("sil", fz.end() + ARB_DISTANCE, 
			0.0, fz.a(fz.end()), 0.0, 0.0);
    ev.insert_after(ev.tail(), tmp);
}

/*static void add_int_item(EST_Stream &ev, EST_Stream_Item *cur, float end, float
  start_f0, float peak_pos, float peak_f0, EST_String name)
  
  {
  EST_Stream_Item tmp;
  EventSI *pev;
  
  pev = new EventSI;
  pev->init();
  
  tmp.set_end(end);
  tmp.set_name(name);
  
  pev->set_end_pos(end);
  pev->set_start_f0(start_f0);
  
  if ((name == "a") || (name == "b"))
  {
  pev->set_peak_pos(peak_pos);
  pev->set_peak_f0(peak_f0);
  }
  
  tmp.set_val(pev);
  ev.insert_before(cur, tmp);
  }
  */
static EST_Stream_Item make_int_item(EST_String name, float end, float start_pos,
				 float start_f0, float peak_pos, float peak_f0)

{
    EST_Stream_Item tmp;
    EventSI *pev;
    
    pev = new EventSI;
    pev->init();
    
    tmp.set_end(end);
    tmp.set_name(name);
    
    pev->set_start_pos(start_pos);
    pev->set_start_f0(start_f0);
    pev->set_end_pos(end);
    
    if ((name != "sil") && (name != "c"))
    {
	pev->set_peak_pos(peak_pos);
	pev->set_peak_f0(peak_f0);
	tmp.set_feature("pos", 1);
    }

    tmp.set_contents(pev, gc_eventsi);
    return tmp;
}

// Return indexs in fz to best fitting region of monomial curve to
// fz contour. The search is bounded by the b/e_start an b/e_stop
// values. The contour fz, should have no breaks in it.

static int match_rf_point(EST_Track &fz, int b_start, int b_stop, int e_start, int
			  e_stop, int &mi, int &mj)
{
    int i, j, k;
    float s_pos, e_pos, s_freq, e_freq, t;
    float amp, duration, dist, ndist;
    float min_dist = FLT_MAX;
    int length;
    EST_Track new_fz(fz.num_frames(), 1);
    float f_shift;
    
    mi = mj = 0;		// set values to zero for saftey
    
    if ((b_start >= b_stop) || (b_start < 0))
    {
	cerr << "Illegal beginning search region in match_rf_point:" <<
	    b_start << "-" << b_stop << endl;
	return -1;
    }
    if ((e_start >= e_stop) || (e_stop > fz.num_frames()))
    {
	cerr << "Illegal ending search region in match_rf_point:" <<
	    e_start << "-" << e_stop << endl;
	return -1;
    }
    
    f_shift = fz.shift();
    duration = 0.0;
    
    for (i = b_start; i < b_stop; ++i)
	for (j = e_start; j < e_stop; ++j)
	{
	    s_pos = fz.t(i);
	    s_freq = fz.a(i);
	    
	    e_pos = fz.t(j);
	    e_freq = fz.a(j);
	    
	    duration = e_pos - s_pos;
	    amp = e_freq - s_freq;
	    length = j - i;
	    
	    for (k = 0; k < length + 1; ++k)
	    {
		t = ((float) k) * f_shift;
		new_fz.a(k) = (amp * fncurve(duration, t, 2.0)) 
		    + s_freq;
	    }
	    
	    dist = distance(fz, i, new_fz, length);
	    ndist = dist / (duration * 100.0);
	    ndist *= weight(duration);
	    
	    if (ndist < min_dist)
	    {
		min_dist = ndist;
		mi = i;
		mj = j;
	    }
	}
    return 0;
}

