/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Paul Taylor                               */
/*                   Date   :  May 1995                                  */
/*-----------------------------------------------------------------------*/
/*                EST_Utterance class source file                        */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include "EST_Token.h"
#include "EST_Utterance.h"
#include "EST_Stream.h"
#include "stream_io.h"

EST_Regex RXaddr("a[0-9]+");

EST_Utterance::EST_Utterance()
{
    refcount = 0;
    s.init();
}

void EST_Utterance::init()
{
    s.init();
}

int EST_Utterance::create_stream(const EST_String &inname)
{
    EST_TBI *p;

    for (p = s.head(); p; p = next(p))
	if (s(p).stream_name() == inname)
	{
	    cerr << "Stream " << inname << " already exists!\n";
	    return -1;
	}
    
    EST_Stream a;
    a.init(inname);

    s.append(a);

    return 0;
}

void EST_Utterance::remove_stream(const EST_String &inname)
{
    EST_TBI *p;

    for (p = s.head(); p; p = next(p))
	if (s(p).stream_name() == inname)
	  {
	    s(p).clear();
	    s.remove(p);
	    return;
	  }
    cerr << "Couldn't find stream " << inname << " for removal\n";
}

void EST_Utterance::clear()
{
    s.clear();
}

EST_Stream &EST_Utterance::stream(const EST_String &inname)
{
    EST_TBI *p;

    for (p = s.head(); p; p = next(p))
	if (s(p).stream_name() == inname)
	    return s(p);

    cerr << "Tried to access undefined stream " << inname << endl;
    return  s(s.head()); // dummy error
}

int EST_Utterance::stream_present(const EST_String &inname)
{
    EST_TBI *p;

    for (p = s.head(); p; p = next(p))
	if (s(p).stream_name() == inname)
	    return 1;

    return 0;
}

void EST_Utterance::remove_item(const EST_String &relname, int a)
{
    EST_TBI *sp, *r, *nr;
    EST_Stream_Item *ptr;
    EST_String sn;

    // remove item from utterance and traverse all relations lists to
    // delete mentions of it

    for (sp = s.head(); sp; sp = next(sp))
	for (ptr = s(sp).head(); ptr; ptr = next(ptr))
	{
	    sn = s(sp).stream_name();
	    for (r = rlink(sn, ptr->addr(), relname).head(); r; r=nr )
	    {
		nr = next(r);
		if (rlink(sn, ptr->addr(), relname).item(r)==a)
		    rlink(sn, ptr->addr(), relname).remove(r);
	    }
		
	}
    stream(relname).remove(a);
}


EST_Stream_Item *EST_Utterance::item(const EST_String &inname, int a)
{
    return stream(inname).item(a);
}

EST_Stream_Item &EST_Utterance::ritem(const EST_String &inname, int a)
{
    return *stream(inname).item(a);
}

EST_Relation &EST_Utterance::rlink(const EST_String &ns, int a, const EST_String &r)
{   // return "r" relations for stream_item "a" in stream "s" 
    return ritem(ns, a).rlink(r);
}

static EST_read_status parse_field_relations(const EST_String &s, 
					    EST_Stream_Item &item)
{
    //  Parse the relations 
    EST_String rel_stream;
    EST_TokenStream ts;
    ts.open_string(s);

    if (ts.eof()) return format_ok;
    rel_stream = ts.get().string();
    while (!ts.eof() && (ts.peek() != ""))
    {
	if (!ts.peek().string().matches(RXint))
	{
	    cerr << "Label file error: invalid relation address for " <<
		rel_stream << " " << ts.peek().string() 
		    << " on " << item.stream_name() << " line labelled " 
			<< item.end() << endl;
	    return misc_read_error;
	}
	else
	    item.make_link(rel_stream, atoi(ts.get().string()));
    }
    return format_ok;
}
	
static EST_read_status parse_field_addr_features(const EST_String &s, 
						 EST_Stream_Item &item)
{
    // Parse address name and features 
    EST_TokenStream ts;
    EST_String feature, val;
    ts.open_string(s);
    ts.set_PunctuationSymbols("");
    ts.set_PrePunctuationSymbols("");
    ts.set_quotes('"','\\');

    if (!ts.peek().string().matches(RXaddr))
    {
	cerr << "Label file error: address does not start with `a' "
	    << " on " << item.stream_name() << " line labelled " 
		<< item.end() << endl;
	return misc_read_error;
    }
    item.set_addr(atoi(ts.get().string().after("a")));

    // The rest are treated as feature name and value
    while ((!ts.eof()) && (ts.peek() != ""))
    {
	feature = ts.get().string();
	val = ts.get().string();
	item.set_feature(feature,val);
    }
    return format_ok;
}

static EST_read_status parse_field_labels(EST_Stream_Item &s)
{
    // Parse the info in the EST_StrList fields into address name,
    // features and relations
    EST_StrList &flist = s.fields;

    if (flist.length() > 0)
    {
	if (parse_field_addr_features(flist.ritem(0),s) == misc_read_error)
	    return misc_read_error;
	for (EST_TBI *p=next(flist.head()); p != 0; p=next(p))
	    if (parse_field_relations(flist.ritem(p),s) == misc_read_error)
		return misc_read_error;
    }
    return format_ok;
}    

EST_read_status EST_Utterance::load(const EST_String &filename)
{   // A rewrite using TokenStreams and no restrictions on sizes
    EST_Stream_Item item;
    EST_String stream_name, full_addr, addr, name;
    EST_TokenStream ts;
    EST_Token t;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "can't open utterance input file " << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");
    ts.set_quotes('"','\\');

    // Skip over header
    for(; !ts.eof(); ts.get())
	if ((ts.peek().col() == 0) && (ts.peek() == "#"))
	    break;
    
    if (ts.eof())
    {
	cerr << "couldn't find header in label file " << filename << endl;
	return wrong_format;
    }
    
    init();
    
    while(!ts.eof())
    {
	if (ts.peek() == "#")
	{			// new stream
	    ts.get();			// skip hash
	    stream_name = ts.get().string();
	    create_stream(stream_name);
	    while (!ts.eoln()) 
		ts.get(); // ignore rest of line 
	}
	else
	{
	    // reads all labels up to eof or next stream and 
	    // appends them to stream.
	    if (read_esps_portion(ts, stream(stream_name)) != format_ok)
		return misc_read_error;
	    for (EST_Stream_Item *st=stream(stream_name).head(); 
		 st != 0; st=next(st))
	    {
		// ; separated chunks should now be in the EST_StrList fields
		// parse them as address, features and relations
		if (parse_field_labels(*st) == misc_read_error)
		    return misc_read_error;
	    }
	}
    }
    ts.close();
    return format_ok;
}

EST_write_status EST_Utterance::save(const EST_String &filename, int num)
{
    EST_Stream_Item *ptr;
    EST_TBI *sptr, *iptr, *fptr;
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (outf == 0)  {
	cerr << "Can't open utterance output file " << filename << endl;
	return misc_write_error;
    }
    
    *outf << "separator ;\n";
    *outf << "nfields 1\n";
    outf->precision(4);
    
    for (sptr = s.head(); sptr; sptr = next(sptr))
    {
	*outf << "# " << s(sptr).stream_name() << endl;
	for (ptr = s(sptr).head(); ptr != 0; ptr = next(ptr))
	{
	    *outf << "\t" << ptr->end() << " 26 \t";
	    *outf << quote_string(ptr->name()) ;
	    // address
	    if (num > 0)
		*outf << "; a" << ptr->addr();
	    // features
	    for (fptr=ptr->features.list.head(); fptr != 0; fptr=next(fptr))
		*outf << " " << ptr->features.list(fptr).k <<
		    " " << quote_string(ptr->features.list(fptr).v.string());
	    if (num > 1)
	    {
		*outf << "; ";
		for (iptr = ptr->links.head(); iptr != 0; iptr = next(iptr))
		{
		    *outf << ptr->links(iptr);
		    *outf << (ptr->links(iptr).length() ? "; " : "");
		}
	    }
	    *outf << endl;
	}
    }
    
    if (outf != &cout)
	delete outf;

    return write_ok;
}

ostream& operator << (ostream &st, const EST_Utterance &u)
{
    EST_TBI *p;
    
    for (p = u.s.head(); p; p = next(p))
    {
	st << "# " << u.s(p).stream_name() << endl;
	st << u.s(p);
    }
    return st;
}

